/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Image
{
    /// <summary>
    /// Swaps array of bytes.
    /// </summary>
    static class BitSwapper
    {
		/// <summary>
		/// Swaps 4-byte words.
		/// </summary>
		/// <param name="buf">byte array</param>
		/// <param name="startindex">starting index</param>
		/// <param name="length">number of bytes to swap</param>
		public static byte[] swapWordBytes(byte[] buf, int startindex, int length)
		{
			byte b;
			//swap inside words
			if (startindex + length > buf.Length) throw new TPCInvalidArgumentsException("Swap region [" + startindex + ".." + (startindex + length - 1) + "] goes out of buffer bounds [0.." + (buf.Length - 1) + "]");
			for (int i = startindex; i < startindex + length; i += 4)
			{
				b = buf[i];
				buf[i] = buf[i + 3];
				buf[i + 3] = b;
				b = buf[i + 1];
				buf[i + 1] = buf[i + 2];
				buf[i + 2] = b;
			}
			return buf;
		}

		/// <summary>
		/// Swaps bytes and 4-byte words.
		/// </summary>
		/// <param name="buf">byte array</param>
		/// <param name="startindex">starting index</param>
		/// <param name="length">number of bytes to swap</param>
		public static byte[] swapWordAndArrayBytes(byte[] buf, int startindex, int length)
		{
			byte b;
			if (startindex + length > buf.Length) throw new TPCInvalidArgumentsException("Swap region [" + startindex + ".." + (startindex + length - 1) + "] goes out of buffer bounds [0.." + (buf.Length - 1) + "]");
			//swap bytes in array
			buf = swapArrayBytes(buf, startindex, length);
			//swap inside words
			for (int i = startindex; i < startindex + length; i += 4)
			{
				b = buf[i];
				buf[i] = buf[i + 3];
				buf[i + 3] = b;
				b = buf[i + 1];
				buf[i + 1] = buf[i + 2];
				buf[i + 2] = b;
			}
			return buf;
		}

		/// <summary>
		/// Swaps bytes of array
		/// </summary>
		/// <param name="buf">byte array</param>
		/// <param name="startindex">first byte to swap</param>
		/// <param name="length">number of bytes to swap</param>
		/// <returns>swapped byte array</returns>
		public static byte[] swapArrayBytes(byte[] buf, int startindex, int length)
		{
			byte b;
			if (startindex + length > buf.Length) throw new TPCInvalidArgumentsException("Swap region [" + startindex + ".." + (startindex + length - 1) + "] goes out of buffer bounds [0.." + (buf.Length - 1) + "]");
			for (int i = 1; i < length; i += 2)
			{
				b = buf[startindex + i];
				buf[startindex + i] = buf[startindex + i - 1];
				buf[startindex + i - 1] = b;
			}
			return buf;
		}

		/// <summary>
		/// Read 32-bit VAX floats
		/// </summary>
		/// <param name="buf">array of 32-bit float tacs</param>
		/// <param name="isvax">true if VAX format</param>
		/// <param name="islittle">true if little endian</param>
		/// <returns>converted byte array</returns>
		public static byte[] VAX32toFloat(byte[] buf, bool isvax, bool islittle)
		{
			int ul;
			for (int i = 0; i < buf.Length; i += 4)
			{
				ul = BitConverter.ToInt32(buf, i);
				if (ul == 0) continue;
				if (isvax)
				{
					/* if input is in VAX format */
					/* Swap words on i386 and bytes on SUN */
					if (islittle)
						swapWordBytes(buf, i, 4);
					else
						swapArrayBytes(buf, i, 4);
					ul -= (2 << 23); /* subtract 2 from exp */
				}
				else
				{
					/* input is in i386 format */
					if (!islittle) swapWordAndArrayBytes(buf, i, 4); /* Switch words and bytes on SUN */
				}
			}
			return buf;
		}

		/// <summary>
		/// Swabs bytes according to tacs type if necessary.
		/// </summary>
		/// <param name="buf">byte array</param>
		/// <param name="datatype">tacs type</param>
		public  static void swapBytes(byte[] buf, ImageFile.DataType datatype)
		{
			switch (datatype)
			{
				case ImageFile.DataType.BIT16_S:
				case ImageFile.DataType.VAXI16:
					// byte conversion necessary on big endian platform
					if (BitConverter.IsLittleEndian)
					{
						swapArrayBytes(buf, 0, buf.Length);
					}
					break;
				case ImageFile.DataType.VAXI32:
					// Swap both words and bytes on SUN 
					if (BitConverter.IsLittleEndian) swapWordAndArrayBytes(buf, 0, buf.Length);
					break;
				case ImageFile.DataType.VAXFL32: //ecat7rInt
					buf = VAX32toFloat(buf, true, BitConverter.IsLittleEndian);
					break;
				case ImageFile.DataType.FLT32:
				// IEEE float ; byte conversion necessary on little endian platforms
				case ImageFile.DataType.SUNI4:
					// SUN int ; byte conversion necessary on little endian platforms
					if (BitConverter.IsLittleEndian) swapWordBytes(buf, 0, buf.Length);
					break;
				case ImageFile.DataType.SUNI2:
					// SUN short ; byte conversion necessary on little endian platforms
					if (BitConverter.IsLittleEndian) swapArrayBytes(buf, 0, buf.Length);
					break;
				default:
					break;
			}
		}
    }
}