/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Class Analyze75Header
	/// </summary>
	public class Analyze75Header : AnalyzeHeader
	{
		#region Analyze 7.5 header field description

		private static readonly string[] analyze75Names =
        {
			// Header key substructure
			"sizeof_hdr", "data_type", "db_name", "extents", "session_error", "regular", "hkey_un0",
			// Image dimension substructure; scl_slope and scl_intercept are nonstandard, original specification has "funused1" and "funused2" instead
			"dim", "unused", "datatype", "bitpix", "dim_un0", "pixdim", "vox_offset", "scl_slope", "scl_inter", "funused3",
            "cal_max", "cal_min", "compressed", "verified", "glmax", "glmin",
			// Data history substruct
			"descrip", "aux_file", "orient", "originator", "generated", "scannum", "patient_id", "exp_date", "exp_time", "hist_un0",
			"views", "vols_added", "start_field", "field_skip", "omax", "omin", "smax", "smin"
		};

		private static readonly HeaderFieldType[] analyze75Types =
        {
			// Header key substructure
			HeaderFieldType.INT, HeaderFieldType.STRING, HeaderFieldType.STRING, HeaderFieldType.INT, HeaderFieldType.SHORT,
            HeaderFieldType.CHAR, HeaderFieldType.CHAR,
			// Image dimension substructure
			HeaderFieldType.SHORT, HeaderFieldType.SHORT, HeaderFieldType.SHORT, HeaderFieldType.SHORT, HeaderFieldType.SHORT,
            HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT,
            HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.INT,
            HeaderFieldType.INT,
			// Data history substruct
			HeaderFieldType.STRING, HeaderFieldType.STRING, HeaderFieldType.CHAR, HeaderFieldType.SHORT, HeaderFieldType.STRING,
			HeaderFieldType.STRING, HeaderFieldType.STRING, HeaderFieldType.STRING, HeaderFieldType.STRING, HeaderFieldType.CHAR,
			HeaderFieldType.INT, HeaderFieldType.INT, HeaderFieldType.INT, HeaderFieldType.INT, HeaderFieldType.INT,
			HeaderFieldType.INT, HeaderFieldType.INT, HeaderFieldType.INT
		};

		private static readonly int[] analyze75Lengths =
        {
			// Header key substructure
			1, 10, 18, 1, 1, 1, 1, 
			// Image dimension substructure
			8, 7, 1, 1, 1, 8, 1, 1, 1, 1,
            1, 1, 1, 1, 1, 1,
			// Data history substruct
			80, 24, 1, 5, 10, 10, 10, 10, 10, 3,
            1, 1, 1, 1, 1, 1, 1, 1
		};

		#endregion

		public static readonly Orientation DefaultOrientation = Orientation.Radiological;

		/// <summary>
		/// Default constructor
		/// </summary>
		public Analyze75Header()
			: base()
		{
			Orientation = DefaultOrientation;
		}

		/// <summary>
		/// Create an Analyze header from an existing image header
		/// </summary>
		/// <param name="ih">Image header</param>
		public Analyze75Header(ImageHeader ih)
			: base(ih)
		{
			if (ih is Analyze75Header)
				this.Originator = (ih as Analyze75Header).Originator;
		}


		/// <summary>
		/// Create a header from file
		/// </summary>
		/// <param name="filename">File to read</param>
		public Analyze75Header(string filename)
			: this()
		{
			Read(filename);

			// To force similar behaviour as the previous Analyze class, use float types instead of
			// complex, despite the header
			if ((AnalyzeFile.AnalyzeDataType)this[@"datatype"] == AnalyzeFile.AnalyzeDataType.COMPLEX)
			{
				if ((short)this[@"bitpix"] == 32) this.Datatype = ImageFile.DataType.FLT32;
				else this.Datatype = ImageFile.DataType.FLT64;
			}
		}

		/// <summary>
		/// Read header info from a file
		/// </summary>
		/// <param name="filename">File to read</param>
		/// <returns>True, if reading was succesfull</returns>
		public bool Read(string filename)
		{
			bool success = false;

			using (FileStream s = new FileStream(filename, FileMode.Open))
			{
				success = Read(s);
			}

			return success;
		}

		/// <summary>
		/// Scaling function intercept value. Not used to modify datavalues in Analyze 7.5, but saved into the header
		/// </summary>
		public override float Intercept
		{
			get
			{
				return 0.0f;
			}
			set
			{
				base.Intercept = value;
			}
		}

		/// <summary>
		/// Patient id
		/// </summary>
		public override string PatientID
		{
			get
			{
				return (String)(this[@"patient_id"]);
			}
			set
			{
				this[@"patient_id"] = value;
			}
		}

		/// <summary>
		/// Originator coördinates (x,y,z). Refer to the voxel nearest to the real space origo.
		/// </summary>
		public short[] Originator
		{
			get
			{
				IConvertible[] orig = (IConvertible[])this[@"originator"];
				short[] origin = Array.ConvertAll<IConvertible, short>(orig, Convert.ToInt16);
				return origin;
			}
			set
			{
				IConvertible[] orig = new IConvertible[5];

				int i;
				for (i = 0; i < value.Length && i < orig.Length; i++)
				{
					orig[i] = value[i];
				}

				for (; i < orig.Length; i++)
				{
					orig[i] = 0;
				}

				this[@"originator"] = orig;
			}
		}

		/// <summary>
		/// Description of the header bytes
		/// </summary>
		protected override AnalyzeHeader.HeaderDescription ByteDescription
		{
			get { return new HeaderDescription(analyze75Names, analyze75Types, analyze75Lengths); }
		}
	}
}
