/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Base class for all derivatives of Analyze file format
	/// </summary>
	public abstract class AnalyzeFile : ImageFile, IStreamReader
	{
		#region Private/protected

		// Delegates for IO operations
		private delegate float GetValue(BinaryReader br);
		private delegate void WriteValue(float f, BinaryWriter bw);

		/// <summary>
		/// 
		/// </summary>
		/// <param name="slope"></param>
		/// <param name="intercept"></param>
		/// <param name="dt"></param>
		/// <param name="min"></param>
		/// <param name="max"></param>
		protected virtual void GetScalingFactors(out float slope, out float intercept, DataType dt, float min, float max)
		{
			slope = 1.0f;
			intercept = 0.0f;

			if (dt != DataType.FLT32 && dt != DataType.FLT64)
			{
				double typeMax = MaxValue(dt);
				double typeMin = MinValue(dt);

				// Calculate slope and intercept values

				// Scale to fill the available datatype range
				slope = (float)((max - min) / (typeMax - typeMin));
				intercept = (float)((min - max * typeMin / typeMax) / (1.0 - typeMin / typeMax));
			}
		}

		/// <summary>
		/// Scale Nifti file data to real values
		/// </summary>
		/// <param name="x">Nifti file value</param>
		/// <param name="s">Scaling factor</param>
		/// <param name="i">Intercept</param>
		/// <returns>Real value</returns>
		protected static float Scale(float x, float s, float i)
		{
			return x * s + i; ;
		}

		/// <summary>
		/// Scale "real" data values to Nifti file values by using the slope and intercept
		/// defined in the header
		/// </summary>
		/// <param name="x">Nifti file value</param>
		/// <param name="s">Scaling factor != 0</param>
		/// <param name="i">Intercept</param>
		/// <returns>Nifti file value</returns>
		protected static float ScaleBack(float x, float s, float i)
		{
			return (x - i) / s;
		}

		/// <summary>
		/// Read an AnalyzeHeader.
		/// </summary>
		/// <returns></returns>
		protected abstract AnalyzeHeader ReadAnalyzeHeader();

		/// <summary>
		/// 
		/// </summary>
		/// <param name="datatype"></param>
		/// <returns></returns>
		private GetValue GetReader(DataType datatype)
		{
			GetValue getValue;

			switch (datatype)
			{
				case DataType.BIT8_S:
					getValue = delegate(BinaryReader br)
					{
						sbyte value = br.ReadSByte();
						return value;
					};
					break;
				case DataType.BIT8_U:
					getValue = delegate(BinaryReader br)
					{
						byte value = br.ReadByte();
						return value;
					};
					break;
				case DataType.BIT16_S:
					getValue = delegate(BinaryReader br)
					{
						Int16 value = br.ReadInt16();
						return value;
					};
					break;
				case DataType.BIT16_U:
					getValue = delegate(BinaryReader br)
					{
						UInt16 value = br.ReadUInt16();
						return value;
					};
					break;
				case DataType.BIT32_S:
					getValue = delegate(BinaryReader br)
					{
						Int32 value = br.ReadInt32();
						return value;
					};
					break;
				case DataType.FLT32:
					getValue = delegate(BinaryReader br)
					{
						return br.ReadSingle();
					};
					break;
				default:
					throw new TPCAnalyzeFileException("Unsupported datatype");
			}
			return getValue;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		private WriteValue GetWriter()
		{
			WriteValue WValue;
			DataType dt = imgHeader.Datatype;

			switch (dt)
			{
				case DataType.BIT8_U:
					WValue = delegate(float val, BinaryWriter bw)
					{
						Byte ui8 = Convert.ToByte(val);
						bw.Write(ui8);
					};
					break;
				case DataType.BIT8_S:
					WValue = delegate(float val, BinaryWriter bw)
					{
						SByte i8 = Convert.ToSByte(val);
						bw.Write(i8);
					};
					break;
				case DataType.BIT16_S:
					WValue = delegate(float val, BinaryWriter bw)
					{
						Int16 i16 = Convert.ToInt16(val);
						bw.Write(i16);
					};
					break;
				case DataType.BIT16_U:
					WValue = delegate(float val, BinaryWriter bw)
					{
						UInt16 ui16 = Convert.ToUInt16(val);
						bw.Write(ui16);
					};
					break;
				case DataType.BIT32_S:
					WValue = delegate(float val, BinaryWriter bw)
					{
						Int32 i32 = Convert.ToInt32(val);
						bw.Write(i32);
					};
					break;
				case DataType.BIT32_U:
					WValue = delegate(float val, BinaryWriter bw)
					{
						UInt32 ui32 = Convert.ToUInt32(val);
						bw.Write(ui32);
					};
					break;
				case DataType.BIT64_S:
					WValue = delegate(float val, BinaryWriter bw)
					{
						Int64 i64 = Convert.ToInt64(val);
						bw.Write(i64);
					};
					break;
				case DataType.BIT64_U:
					WValue = delegate(float val, BinaryWriter bw)
					{
						UInt64 ui64 = Convert.ToUInt64(val);
						bw.Write(ui64);
					};
					break;
				case DataType.FLT32:
					WValue = delegate(float val, BinaryWriter bw)
					{
						float f32 = Convert.ToSingle(val);
						bw.Write(f32);
					};
					break;
				case DataType.FLT64:
					WValue = delegate(float val, BinaryWriter bw)
					{
						double f64 = Convert.ToDouble(val);
						bw.Write(f64);
					};
					break;
				default:
					throw new TPCException("Unsupported datatype");
			}
			return WValue;
		}

		#endregion

		#region Public

		/// <summary>
		/// I/O processing event handler
		/// </summary>
		public static event IOProcessEventHandler IOProgress;

		/// <summary>
		/// Analyze datatype codes
		/// </summary>
		public enum AnalyzeDataType : short
		{
			// Analyze75 names

			/// <summary>
			/// No format specified (Analyze 7.5)
			/// </summary>
			NONE = 0,

			/// <summary>
			/// Unknown format (Analyze 7.5)
			/// </summary>
			UNKNOWN = 0,

			/// <summary>
			/// Single bit
			/// </summary>
			BINARY = 1,

			/// <summary>
			/// Unsigned char (Analyze 7.5)
			/// </summary>
			UNSIGNED_CHAR = 2,

			/// <summary>
			/// Signed short (Analyze 7.5)
			/// </summary>
			SIGNED_SHORT = 4,

			/// <summary>
			/// Signed int (Analyze 7.5)
			/// </summary>
			SIGNED_INT = 8,

			/// <summary>
			/// Single precision floating point (Analyze 7.5)
			/// </summary>
			FLOAT = 16,

			/// <summary>
			/// Complex number (32 bit) (Analyze 7.5)
			/// </summary>
			COMPLEX = 32,

			/// <summary>
			/// Double precision floating point (Analyze 7.5)
			/// </summary>
			DOUBLE = 64,

			/// <summary>
			/// RGB value (3 x 8 bit)  (Analyze 7.5)
			/// </summary>
			RGB = 128,

			/// <summary>
			/// All  (Analyze 7.5)
			/// </summary>
			ALL = 255,

			// Nifti names

			/// <summary>
			/// Unsigned 8-b integer
			/// </summary>
			UINT8 = 2,
			/// <summary>
			/// Signed 16-b integer
			/// </summary>
			INT16 = 4,
			/// <summary>
			/// Signed 32-b integer
			/// </summary>
			INT32 = 8,
			/// <summary>
			/// Single precision (32-b) float
			/// </summary>
			FLOAT32 = 16,
			/// <summary>
			/// 64-b complex = 2 x 32-b floats
			/// </summary>
			COMPLEX64 = 32,
			/// <summary>
			/// Double precision (64-b) float
			/// </summary>
			FLOAT64 = 64,
			/// <summary>
			/// 3 x 8-b bytes
			/// </summary>
			RGB24 = 128,
			/// <summary>
			/// Signed 8-b integer
			/// </summary>
			INT8 = 256,
			/// <summary>
			/// Unsigned 16-b integer
			/// </summary>
			UINT16 = 512,
			/// <summary>
			/// Unsigned 32-b integer
			/// </summary>
			UINT32 = 768,
			/// <summary>
			/// Signed 64-b integer
			/// </summary>
			INT64 = 1024,
			/// <summary>
			/// Unsigned 64-b integer
			/// </summary>
			UINT64 = 1280,
			/// <summary>
			/// 128-b float
			/// </summary>
			FLOAT128 = 1536,
			/// <summary>
			/// 128-b complex = 2 x 64-b floats
			/// </summary>
			COMPLEX128 = 1792,
			/// <summary>
			/// 256-b complex = 2 x 128-b floats
			/// </summary>
			COMPLEX256 = 2048,
			/// <summary>
			/// 4 x 8-b bytes
			/// </summary>
			RGBA32 = 2304
		}

		/// <summary>
		/// Analyze header file extension for images in two files
		/// </summary>
		public const string HEADEREXTENSION = @".hdr";

		/// <summary>
		/// Analyze data file extension for images in two files
		/// </summary>
		public const string DATAEXTENSION = @".img";

		/// <summary>
		/// Analyze file extension for files with frame time information
		/// </summary>
		public const string SIFEXTENSION = @".sif";

		/// <summary>
		/// Header file name
		/// </summary>
		public virtual string HeaderFilename
		{
			get
			{
				string dir = Path.GetDirectoryName(this.filename);
				string file = Path.GetFileNameWithoutExtension(this.filename);
				return (dir + Path.DirectorySeparatorChar + file + HEADEREXTENSION);
			}
		}

		/// <summary>
		/// Data file name
		/// </summary>
		public virtual string DataFilename
		{
			get
			{
				string dir = Path.GetDirectoryName(this.filename);
				string file = Path.GetFileNameWithoutExtension(this.filename);
				return (dir + Path.DirectorySeparatorChar + file + DATAEXTENSION);
			}
		}

		/// <summary>
		/// Frame time information file name
		/// </summary>
		public virtual string SifFileName
		{
			get
			{
				string dir = Path.GetDirectoryName(this.filename);
				string file = Path.GetFileNameWithoutExtension(this.filename);
				return (dir + Path.DirectorySeparatorChar + file + SIFEXTENSION);
			}
		}

		/// <summary>
		/// Resolve TPClib equivalent datatype from a Nifti datatype
		/// </summary>
		/// <param name="ndt">Nifti datatype</param>
		/// <returns>TPClib datatype</returns>
		public static DataType ResolveDataType(AnalyzeDataType ndt)
		{
			DataType dt;
			switch (ndt)
			{
				case AnalyzeDataType.BINARY:
					dt = DataType.BIT1;
					break;
				case AnalyzeDataType.INT8:
					dt = DataType.BIT8_S;
					break;
				case AnalyzeDataType.INT16:
					dt = DataType.BIT16_S;
					break;
				case AnalyzeDataType.INT32:
					dt = DataType.BIT32_S;
					break;
				case AnalyzeDataType.INT64:
					dt = DataType.BIT64_S;
					break;
				case AnalyzeDataType.UINT8:
					dt = DataType.BIT8_U;
					break;
				case AnalyzeDataType.UINT16:
					dt = DataType.BIT16_U;
					break;
				case AnalyzeDataType.UINT32:
					dt = DataType.BIT32_U;
					break;
				case AnalyzeDataType.UINT64:
					dt = DataType.BIT64_U;
					break;
				case AnalyzeDataType.FLOAT32:
					dt = DataType.FLT32;
					break;
				case AnalyzeDataType.FLOAT64:
					dt = DataType.FLT64;
					break;
				case AnalyzeDataType.RGB24:
					dt = DataType.COLRGB;
					break;
				default:
					throw new TPCException(@"Unsupported datatype.");
			}
			return dt;
		}

		/// <summary>
		/// Resolve equivalent Nifti datatype from a TPClib datatype
		/// </summary>
		/// <param name="dt">TPClib datatype</param>
		/// <returns>Nifti datatype</returns>
		public static AnalyzeDataType ResolveAnalyzeDataType(DataType dt)
		{
			AnalyzeDataType ndt;
			switch (dt)
			{
				case DataType.BIT1:
					ndt = AnalyzeDataType.BINARY;
					break;
				case DataType.BIT8_S:
					ndt = AnalyzeDataType.INT8;
					break;
				case DataType.BIT16_S:
					ndt = AnalyzeDataType.INT16;
					break;
				case DataType.BIT32_S:
					ndt = AnalyzeDataType.INT32;
					break;
				case DataType.BIT64_S:
					ndt = AnalyzeDataType.INT64;
					break;
				case DataType.BIT8_U:
					ndt = AnalyzeDataType.UINT8;
					break;
				case DataType.BIT16_U:
					ndt = AnalyzeDataType.UINT16;
					break;
				case DataType.BIT32_U:
					ndt = AnalyzeDataType.UINT32;
					break;
				case DataType.BIT64_U:
					ndt = AnalyzeDataType.UINT64;
					break;
				case DataType.FLT32:
					ndt = AnalyzeDataType.FLOAT32;
					break;
				case DataType.FLT64:
					ndt = AnalyzeDataType.FLOAT64;
					break;
				case DataType.COLRGB:
					ndt = AnalyzeDataType.RGB24;
					break;
				default:
					throw new TPCException(@"Unsupported datatype.");
			}
			return ndt;
		}

		/// <summary>
		/// Bit per pixel for a datatype
		/// </summary>
		/// <param name="dt">Datatype</param>
		/// <returns>Bits used per value in datatype dt</returns>
		public static int BitsPerPixel(DataType dt)
		{
			switch (dt)
			{
				case DataType.BIT1:
					return 1;
				case DataType.BIT8_S:
				case DataType.BIT8_U:
					return 8;
				case DataType.BIT16_U:
				case DataType.BIT16_S:
					return 16;
				case DataType.COLRGB:
					return 24;
				case DataType.BIT32_U:
				case DataType.BIT32_S:
				case DataType.FLT32:
					return 32;
				case DataType.BIT64_U:
				case DataType.BIT64_S:
				case DataType.FLT64:
					return 64;
				default:
					throw new TPCException("Unsupported datatype");
			}
		}

		/// <summary>
		/// Check the endianness of a datastream
		/// </summary>
		/// <param name="s">Stream to check</param>
		/// <returns>Endianness of the data stream</returns>
		public static Endianness CheckEndian(Stream s)
		{
			long start = s.Position;
			s.Seek(40, SeekOrigin.Current);
			byte[] check = new byte[2];
			s.Read(check, 0, check.Length);
			s.Position = start;
			int dimCheck = BitConverter.ToInt16(check, 0);

			Endianness endian = Endianness.LittleEndian;
			if (dimCheck < 0 || dimCheck > 7)
			{
				endian = Endianness.BigEndian;
			}

			return endian;
		}

		/// <summary>
		/// Get a stream to this image's data
		/// </summary>
		/// <param name="limits">Stream window limits</param>
		/// <returns>An image data stream</returns>
		public ImageStream GetStream(IntLimits limits, FileAccess a)
		{
			// Offset from file start is the header size for combined file, 0 for a separate data file.
			int headerOffset = this.DataFilename == this.HeaderFilename ? (this.imgHeader as AnalyzeHeader).HeaderSize : 0;
			ImageStream imgs = new ImageStream(this.DataFilename, this.imgHeader.Dim, limits, headerOffset, this.imgHeader.Datatype, a);
			imgs.SetScaling((this.imgHeader as AnalyzeHeader).Slope, (this.imgHeader as AnalyzeHeader).Intercept);
			imgs.StreamOrientation = imgHeader.Orientation;
			return imgs;
		}

		/// <summary>
		/// Get a stream to this image's data
		/// </summary>
		/// <returns>An image data stream</returns>
		public ImageStream GetStream(FileAccess a)
		{
			return GetStream(this.imgHeader.Dim, a);
		}
		
		/// <summary>
		/// Write image from a stream
		/// </summary>
		/// <param name="imgs">Imagestream</param>
		public void WriteFile(ImageStream imgs)
		{
			this.imgHeader.Dim = imgs.Dim;
			this.imgHeader.Datatype = imgs.DataType.Type;

			// Write header
			using (FileStream fs = new FileStream(this.HeaderFilename, FileMode.Create))
			{
				(this.imgHeader as AnalyzeHeader).Write(fs);
			}

			// Write the frame times to a .sif
			this.WriteSif();

			int imageSize = imgs.Dim.GetProduct();
			ImageStream analyzeStream = this.GetStream(FileAccess.Write);
			ImageStream.StreamCopy(imgs, analyzeStream, imageSize);
		}

		/// <summary>
		/// Write the frame time information to a .sif file.
		/// </summary>
		public void WriteSif()
		{
			// Write frame time information, if needed
			if (this.imgHeader.IsDynamic)
			{
				SifFile sf = new SifFile(this);
				sf.StartTimes = this.imgHeader.FrameStartTimes;
				sf.EndTimes = this.imgHeader.FrameEndTimes;
				sf.WriteFile();
			}
		}

		/// <summary>
		/// Read frame time information from a .sif file.
		/// </summary>
		private bool ReadSif(ImageHeader hdr)
		{
			if (File.Exists(this.SifFileName))
			{
				SifFile sf = new SifFile(this.SifFileName);
				sf.ReadFile();
				hdr.FrameStartTimes = sf.StartTimes;
				hdr.FrameEndTimes = sf.EndTimes;
				return true;
			}
			return false;
		}

		#endregion

		#region ImageFile interface

		/// <summary>
		/// Read the header of this file.
		/// </summary>
		/// <returns>File header</returns>
		public override ImageHeader ReadHeader()
		{
			AnalyzeHeader anh = ReadAnalyzeHeader();
			anh.IsDynamic = ReadSif(anh);
			return anh;
		}

		/// <summary>
		/// Read an Analyze file
		/// </summary>
		public override void ReadFile()
		{
			try
			{
				this.imgHeader = this.ReadHeader();
			}
			catch (Exception e)
			{
				throw new TPCAnalyzeFileException("Failed to read file " + this.HeaderFilename + " : " + e.Message);
			}

			AnalyzeHeader anHeader = this.imgHeader as AnalyzeHeader;

			// If slope is exactly zero or data is of the type RGB, slope and intercept are not used.
			if (anHeader.Slope == 0.0f || anHeader.Datatype == DataType.COLRGB)
			{
				anHeader.Slope = 1.0f;
				anHeader.Intercept = 0.0f;
			}

			// Create a new image matching the header data
			this.image = new Image(this.imgHeader);

			// Read image data
			string datafile = this.DataFilename;
			try
			{
				using (FileStream fs = new FileStream(datafile, FileMode.Open))
				{
					int headerOffset = 0;
					// If data is in the same file as the header, skip the header
					if (this.DataFilename == this.HeaderFilename) headerOffset = anHeader.HeaderSize;
					GetPixelData(ref this.image, 0, fs, anHeader.Dim, anHeader.Datatype, headerOffset, anHeader.Slope, anHeader.Intercept);
				}
			}
			catch (Exception e)
			{
				throw new TPCAnalyzeFileException("Failed to read file " + this.DataFilename + ". " + e.Message);
			}
		}

		/// <summary>
		/// Write image to this file. Modifies the header to match the image.
		/// </summary>
		/// <param name="img">Image to write</param>
		public override void WriteFile(ref Image img)
		{
			this.image = img;

			// Update header dimension information
			this.imgHeader.Dim = img.Dim;

			// Calculate scaling factors
			float scaleSlope, scaleIntercept;
			float[] minMax = img.GetMinMax();

			this.GetScalingFactors(out scaleSlope, out scaleIntercept, this.imgHeader.Datatype, minMax[0], minMax[1]);

			(this.imgHeader as AnalyzeHeader).Intercept = scaleIntercept;
			(this.imgHeader as AnalyzeHeader).Slope = scaleSlope;

			// Write header
			using (FileStream fs = new FileStream(this.HeaderFilename, FileMode.Create))
			{
				(this.imgHeader as AnalyzeHeader).Write(fs);
			}

			// If using single file, append. Otherwise, create new file
			FileMode fm = FileMode.Create;
			if (this.DataFilename == this.HeaderFilename) fm = FileMode.Append;

			// Write image data
			using (FileStream fs = new FileStream(this.DataFilename, fm))
			{
				WriteImage(img, fs);
			}

			// Write frame time information
			WriteSif();
		}

		/// <summary>
		/// Write image to a stream.
		/// </summary>
		/// <param name="img">Input Image</param>
		/// <param name="s">Output stream</param>
		/// <returns>True, if succesful</returns>
		public bool WriteImage(Image img, Stream s)
		{
			long bytesWritten = -s.Position;

			float scaleIntercept = (this.imgHeader as AnalyzeHeader).Intercept;
			float scaleSlope = (this.imgHeader as AnalyzeHeader).Slope;

			using (BinaryWriter bw = new BinaryWriter(s))
			{
				WriteValue WValue = GetWriter();

				int j = 0;
				if (scaleSlope == 0.0f)
				{
					float pixelValue = scaleIntercept;
					for (j = 0; j < image.DataLength; j++)
					{
						WValue(pixelValue, bw);
					}
				}
				else if (scaleIntercept == 0.0f)
				{
					for (j = 0; j < image.DataLength; j++)
					{
						WValue(image[j] / scaleSlope, bw);
					}
				}
				else
				{
					for (j = 0; j < image.DataLength; j++)
					{
						WValue((image[j] - scaleIntercept) / scaleSlope, bw);
					}
				}

				bytesWritten += bw.BaseStream.Position;
				return (bytesWritten == img.DataLength);
			}
		}

		/// <summary>
		/// Write image frame to a stream
		/// </summary>
		/// <param name="img">Image to read the frame from</param>
		/// <param name="frameIndex">Frame to write</param>
		/// <param name="s">Stream to write to</param>
		public void WriteFrame(ref Image img, int frameIndex, Stream s)
		{
			IntLimits limits = new IntLimits(this.imgHeader.Dim);
			limits.SetLimits(Limits.FRAMES, frameIndex, frameIndex);
			Image subImg = img.GetSubImage(limits);
			WriteImage(subImg, s);
		}

		/// <summary>
		/// Write a single frame from an image to this file
		/// </summary>
		/// <param name="img">Image to read from</param>
		/// <param name="frame_No">Frame to write</param>
		public override void WriteFrame(ref Image img, int frame_No)
		{
			if (frame_No < 1 || frame_No > this.imgHeader.Dim.Frames + 1)
				throw new TPCInvalidArgumentsException("Frame number " + frame_No + " out of bounds [1..no frames +1]");

			// Update the dimension information
			if (frame_No > this.imgHeader.Dim.Frames + 1)
			{
				IntLimits newlimits = new IntLimits(this.imgHeader.Dim);
				newlimits.SetLimit(Limits.FRAMES, Limits.Limit.HIGH, this.imgHeader.Dim.Frames + 1);
				this.imgHeader.Dim = newlimits;
			}

			string dataFile = this.DataFilename;
			int offset = (int)(imgHeader[@"vox_offset"]);
			using (FileStream stream = File.Open(dataFile, FileMode.OpenOrCreate))
			{
				// Skip the header
				stream.Position += offset;

				// Write the frame
				WriteFrame(ref img, frame_No, stream);
			}
		}

		/// <summary>
		/// Get a subimage of this image, discard data outside of limits
		/// </summary>
		/// <param name="region">Sub image limits</param>
		public override void ReadSubImage(IntLimits region)
		{
			imgHeader.Dim = region;
			image = image.GetSubImage(region);
		}

		/// <summary>
		/// Read image data from a stream and save it to an image
		/// </summary>
		/// <param name="data">Image</param>
		/// <param name="offset">Starting position for image writing</param>
		/// <param name="stream">Stream to read for image data</param>
		/// <param name="dim">Image dimensions</param>
		/// <param name="datatype">Image data datatype</param>
		/// <param name="position">Starting position for stream reading</param>
		/// <param name="scale">Scaling factor</param>
		/// <param name="intercept">Scaling intercept</param>
		public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale, float intercept = 0.0f)
		{
			//initialization for event sending
			IOProcessEventHandler pevent = null;

			// opens binary reader
			bool isBigendian = (this.imgHeader as AnalyzeHeader).bigEndian;
			Endianness endian = isBigendian ? Endianness.BigEndian : Endianness.LittleEndian;
			using (BinaryReader br = EndianReader.CreateReader(stream, endian))
			{
				// Skip to start position
				br.BaseStream.Position = position;

				int voxels_in_plane = imgHeader.DimX * imgHeader.DimY;

				GetValue getValue = GetReader(datatype);

				for (int i = offset; i < data.DataLength; i++)
				{
					// Read value and scale
					data[i] = Scale(getValue(br), scale, intercept);

					//send event after each plane
					if (i > 0 && i % voxels_in_plane == 0)
					{
						pevent = IOProgress;
						if (pevent != null)
							pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)image.DataLength), System.Reflection.MethodBase.GetCurrentMethod()));
					}
				}
			}
			pevent = IOProgress;
			if (pevent != null)
				pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
		}

		#endregion
	}
}
