/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using TPCDicom.DataStructure;
using TPCDicom.DataStructure.DataSet;
using TPCDicom.Registry;
using TPCDicom.Encoding;
using TPClib.Common;

namespace TPClib.Image
{
	/// <summary>
	/// DICOM file.
	/// </summary>
	public class DicomFile : ImageFile, IStreamableImage, IComparable<DicomFile>
	{
		#region Delegates

		private delegate float GetValue(int index);
		private delegate void ValueWriter(float f);

		#endregion

		#region Utility classes

		/// <summary>
		/// Class for writing of recursive sequence fields
		/// </summary>
		public class NestedDataSetWriteData
		{
			/// <summary>
			/// 
			/// </summary>
			protected List<Object> elements = new List<object>();

			/// <summary>
			/// 
			/// </summary>
			public void Clear()
			{
				elements.Clear();
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="obj"></param>
			public void Add(Object obj)
			{
				elements.Add(obj);

			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public List<Object>.Enumerator GetEnumerator()
			{
				return elements.GetEnumerator();
			}

			/// <summary>
			/// 
			/// </summary>
			public int Count
			{
				get { return elements.Count; }
			}
		}

		/// <summary>
		/// Class for writing of recursive sequences
		/// </summary>
		public class SequenceWriteData
		{
			/// <summary>
			/// 
			/// </summary>
			public Tag tag = new Tag(0, 0);

			/// <summary>
			/// 
			/// </summary>
			protected List<Object> elements = new List<object>();

			/// <summary>
			/// 
			/// </summary>
			/// <param name="group"></param>
			/// <param name="element"></param>
			public void SetTag(UInt16 group, UInt16 element)
			{
				tag = new Tag(group, element);
			}

			/// <summary>
			/// 
			/// </summary>
			public void Clear()
			{
				elements.Clear();
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="obj"></param>
			public void Add(Object obj)
			{
				elements.Add(obj);

			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public List<Object>.Enumerator GetEnumerator()
			{
				return elements.GetEnumerator();
			}

			/// <summary>
			/// 
			/// </summary>
			public int Count
			{
				get { return elements.Count; }
			}
		}

		#endregion

		#region Public fields

		/// <summary>
		/// Event that is sent when reading of file has progressed.
		/// </summary>
		public static event IOProcessEventHandler IOProgress;

		/// <summary>
		/// Dictionary containing all recognized DICOM tacs elements for reading and writing.
		/// </summary>
		public static readonly DataElementDictionary dataElementDictionary;

		/// <summary>
		/// Dictionary containing all recognized DICOM unique identifiers (Uids) for reading and writing.
		/// </summary>
		public static readonly UidDictionary uidDictionary;

		/// <summary>
		/// Header as a DicomHeader.
		/// </summary>
		public DicomHeader DicomHeader { get { return this.Header as DicomHeader; } }

		#endregion

		#region Constructors

		/// <summary>
		/// Static constructor that reads dictionary files for elements and uids, when this 
		/// class is used first time.
		/// </summary>
		static DicomFile()
		{
			DataElementDictionary dataDict = new DataElementDictionary();
			UidDictionary uidDict = new UidDictionary();
			try
			{
				dataDict.LoadFromString(TPClib.Properties.Resources.TPC_elems3, DictionaryFileFormat.PropertyFile);
				uidDict.LoadFromString(TPClib.Properties.Resources.TPC_uids2, DictionaryFileFormat.PropertyFile);
			}
			catch (Exception dictionaryException)
			{
				throw new TPCDicomFileError("Problems processing dictionaries:\n" + dictionaryException);
			}
			DataElementDictionary.Global = dataDict;
			UidDictionary.Global = uidDict;
			dataElementDictionary = DataElementDictionary.Global;
			uidDictionary = UidDictionary.Global;
		}

		/// <summary>
		/// Constructs DICOM file with filename.
		/// </summary>
		/// <param name="filename">full path to a single file</param>
		public DicomFile(string filename)
		{
			this.filename = filename;
			this.filetype = FileType.DICOM;
			imgHeader = new DicomHeader();
		}

		/// <summary>
		/// Constructs DICOM file.
		/// </summary>
		/// <param name="filename">basename of DICOM file</param>
		/// <param name="img">image tacs</param>
		/// <param name="hdr">image header</param>
		public DicomFile(string filename, Image img, ImageHeader hdr)
			: this(filename)
		{
			this.filetype = FileType.DICOM;
			this.image = img;
			this.Header = hdr;
			this.Header.Dim = new IntLimits(img.Dim);
		}

		/// <summary>
		/// Constructs an empty (all-zero) DICOM file with filename and dimensions.
		/// </summary>
		/// <param name="filename">filename of DICOM file</param>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		public DicomFile(string filename, int dimx, int dimy, int dimz) : this(filename, new int[] { dimx, dimy, dimz }) { }

		/// <summary>
		/// Constructs an empty (all-zero) DICOM file with filename and dimensions.
		/// </summary>
		/// <param name="filename">filename of DICOM file</param>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		/// <param name="dimt">t-dimension (time points)</param>
		public DicomFile(string filename, int dimx, int dimy, int dimz, int dimt) : this(filename, new int[] { dimx, dimy, dimz, dimt }) { }

		/// <summary>
		/// Constructs an empty (all-zero) DICOM file with filename and dimensions.
		/// </summary>
		/// <param name="filename">filename of DICOM file</param>
		/// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
		public DicomFile(string filename, int[] dim)
			: this(filename)
		{
			image = new Image(new IntLimits(dim));
			image.IsDynamic = (dim.Length > IntLimits.FRAMES);
			Header.IsDynamic = image.IsDynamic;
			Header.Dim = new IntLimits(dim);
			Header.Datatype = DataType.BIT16_S;
		}

		#endregion

		#region Static members

		/// <summary>
		/// Converts tacs object to byte array
		/// </summary>
		/// <param name="data">object that has DICOM tag tacs</param>
		/// <returns>DICOM tacs as byte array</returns>
		public static byte[] ToBytesArray(Object data)
		{

			if (data is Array)
			{
				Array a = ((Array)data);
				List<byte[]> values = new List<byte[]>();
				foreach (Object obj in a)
				{
					values.Add(ToBytesArray(obj));
				}
				if (values.Count == 0) return new byte[0];
				byte[] r = new byte[values.Count * values[0].Length];
				int r_i = 0;
				for (int i = 0; i < values.Count; i++)
				{
					for (int j = 0; j < values[i].Length; j++)
					{
						r[r_i++] = values[i][j];
					}
				}
				return r;
			}
			if (data is byte)
			{
				return new byte[1] { (byte)data };
			}
			if (data is char)
			{
				return new byte[1] { (byte)((char)data) };
			}
			else if (data is UInt32)
			{
				return BitConverter.GetBytes((UInt32)data);
			}
			else if (data is Int32)
			{
				return BitConverter.GetBytes((Int32)data);
			}
			else if (data is UInt16)
			{
				return BitConverter.GetBytes((UInt16)data);
			}
			else if (data is Int16)
			{
				return BitConverter.GetBytes((Int16)data);
			}
			else if (data is Int64)
			{
				return BitConverter.GetBytes((Int64)data);
			}
			else if (data is Single)
			{
				return BitConverter.GetBytes((Single)data);
			}
			else if (data is Double)
			{
				return BitConverter.GetBytes((Double)data);
			}
			else if (data is string)
			{
				return ASCIIEncoding.ASCII.GetBytes((string)data);
			}
			else if (data is Uid)
			{
				return ASCIIEncoding.ASCII.GetBytes(((Uid)data).ToString());
			}
			else if (data is TPCDicom.Encoding.Type.Age)
			{
				TPCDicom.Encoding.Type.Age age = (TPCDicom.Encoding.Type.Age)data;
				if (age.IsDays)
					return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "D");
				else if (age.IsMonths)
					return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "M");
				else if (age.IsWeeks)
					return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "W");
				else
					return ASCIIEncoding.ASCII.GetBytes(age.AgeValue.ToString("000") + "Y");
			}
			else if (data is System.DateTime)
			{
				return ASCIIEncoding.ASCII.GetBytes(((System.DateTime)data).Year.ToString("0000") +
					((System.DateTime)data).Month.ToString("00") +
					((System.DateTime)data).Day.ToString("00"));
			}
			else if (data is System.TimeSpan)
			{
				string str = ((System.TimeSpan)data).Hours.ToString("00") +
					((System.TimeSpan)data).Minutes.ToString("00") +
					((System.TimeSpan)data).Seconds.ToString("00");
				if (((System.TimeSpan)data).Milliseconds > 0)
					str += ((System.TimeSpan)data).Milliseconds.ToString("000000");
				return ASCIIEncoding.ASCII.GetBytes(str);
			}
			else if (data is TPCDicom.Encoding.Type.PersonName)
			{
				return ASCIIEncoding.ASCII.GetBytes(((TPCDicom.Encoding.Type.PersonName)data).ToString());
			}
			else if (data is Decimal)
			{
				return ASCIIEncoding.ASCII.GetBytes(data.ToString());
			}
			else if (data is TPCDicom.DataStructure.DataSet.Sequence)
			{
				//sequence delimeter has no tacs
				return new byte[0];
			}
			else
			{
				throw new TPCDicomFileException("Failed to read DICOM tag in file (" + data.GetType() + ") tacs");
			}
		}

		/// <summary>
		/// Resolves names of available DICOM files in path including subdirectories
		/// </summary>
		/// <param name="path">full path to the files</param>
		/// <returns>array of files</returns>
		public static FileInfo[] ResolveDicomFilenames(string path)
		{
			char[] chars;
			string pathroot = Path.GetDirectoryName(path);
			string filename = Path.GetFileName(path);
			chars = Path.GetInvalidFileNameChars();
			for (int i = 0; i < chars.Length; i++)
			{
				if (chars[i].CompareTo('*') == 0) continue;
				if (filename.Contains(chars[i].ToString())) throw new TPCDicomFileException("Invalid character '" + chars[i] + "' in filename");
			}
			chars = Path.GetInvalidPathChars();
			for (int i = 0; i < chars.Length; i++)
				if (pathroot.Contains(chars[i].ToString())) throw new TPCDicomFileException("Invalid character '" + chars[i] + "' in path");
			try
			{
				DirectoryInfo dir = new DirectoryInfo(pathroot);
				FileInfo[] files = dir.GetFiles(filename, SearchOption.TopDirectoryOnly);
				List<FileInfo> recognized_files = new List<FileInfo>();
				for (int i = 0; i < files.Length; i++)
				{
					try
					{
						if (DicomFile.ResolveFileFormat(files[i].FullName) == FileType.DICOM)
							recognized_files.Add(files[i]);
					}
					catch { }
				}
				return recognized_files.ToArray();
			}
			catch (ArgumentException e)
			{
				throw new TPCDicomFileException("Path contains invalid characters:" + e.Message);
			}
			catch (Exception e)
			{
				throw new TPCDicomFileException("While creating DirectoryInfo from [" + pathroot + "]:" + e.Message);
			}
		}

		/// <summary>
		/// Find all Dicom files that belong to the same series in a directory
		/// </summary>
		/// <param name="path">Directory to scan</param>
		/// <param name="combinedHeader">Header constructed from the found files</param>
		/// <returns>List of files in the directory, sorted in plane order</returns>
		public static List<DicomFile> ReadFilesFromPath(string path, out DicomHeader combinedHeader)
		{
			//all filenames found in path
			FileInfo[] filenames = ResolveDicomFilenames(path);
			//1st read DICOM holding header information
			DicomFile dcmfile_1st = null;
			//currently read DICOM holding header information
			DicomFile dcmfile_current;
			//DICOM files holding header information
			List<DicomFile> dcmfiles = new List<DicomFile>();
			//Event handler where progress events are sent to
			IOProcessEventHandler pevent = null;

			DicomHeader firstHeader;

			// Series UID
			string seriesUID = String.Empty;

			//gather recognized files
			for (int i = 0; i < filenames.Length; i++)
			{
				try
				{
					//read all header fields except the actual image tacs
					dcmfile_current = new DicomFile(filenames[i].FullName);
					dcmfile_current.ReadHeader();

					pevent = IOProgress;
					if (i % 5 == 0 && pevent != null)
						pevent.Invoke(dcmfile_current, new IOProgressEventArgs(100.0f * (float)i / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
					// Get the series id from the first file
					if (dcmfile_1st == null)
					{
						dcmfile_1st = dcmfile_current;
						seriesUID = dcmfile_1st.DicomHeader.SeriesInstanceUID;
					}
					else
					{
						//ensure consistency of tacs
						if (!dcmfile_1st.Header.Dim.Equals(dcmfile_current.Header.Dim))
							throw new TPCDicomFileException("Inconsistent dimensions in read DICOM files. Cannot combine into single file.");
					}
					// Only add those files that belong to the same series with the first file
					if (dcmfile_current.DicomHeader.SeriesInstanceUID == seriesUID) dcmfiles.Add(dcmfile_current);
				}
				catch (TPCUnrecognizedFileFormatException e)
				{
					// just skip the file 
					Console.WriteLine("Skipping file " + filenames[i].FullName + ":" + e.Message);
				}

				catch (TPCException e)
				{
					// print error and skip the file 
					Console.WriteLine("Cannot read DICOM file " + filenames[i].FullName + ":" + e.Message);
				}
			}

			dcmfiles.Sort();

			firstHeader = dcmfile_1st.DicomHeader;

			int frames = firstHeader.Frames;
			int planes = firstHeader.Planes;
			//fix number of planes if planes and frames does not multiply to number of found images
			if (frames * planes < dcmfiles.Count)
			{
				List<float> slice_locations = new List<float>();
				for (int i = 0; i < dcmfiles.Count; i++)
				{
					if (!slice_locations.Contains(dcmfiles[i].DicomHeader.SliceLocation))
					{
						slice_locations.Add(dcmfiles[i].DicomHeader.SliceLocation);
					}
				}
				if (slice_locations.Count > 1)
				{
					planes = slice_locations.Count;
				}
				else
				{
					if (frames > 0)
						planes = dcmfiles.Count / frames;
					else
						planes = dcmfiles.Count;
				}
				frames = dcmfiles.Count / planes;
			}

			combinedHeader = new DicomHeader(firstHeader);
			combinedHeader.Dim = new IntLimits(firstHeader.DimX, firstHeader.DimY, planes, frames);

			for (int i = 0; i < frames; i++)
			{
				combinedHeader.SetFrameDuration(i, dcmfiles[i * planes].DicomHeader.frame_duration);
				combinedHeader.SetFrameStart(i, dcmfiles[i * planes].DicomHeader.frame_start_time);
			}

			// Calculate the voxel thickness from the slice positions, if there are more than one slice
			if (dcmfiles.Count > 1)
			{
				int lastSlice = dcmfiles.Count - 1;
				combinedHeader.Siz.SizeZ = Math.Abs((dcmfiles[lastSlice].DicomHeader.SliceLocation - dcmfiles[0].DicomHeader.SliceLocation) / (planes - 1));
			}

			combinedHeader.imagepositions = new Point[dcmfiles.Count];
			for (int i = 0; i < dcmfiles.Count; i++)
				combinedHeader.imagepositions[i] = dcmfiles[i].DicomHeader.imagepositions[0];

			return dcmfiles;
		}

		/// <summary>
		/// Reads all DICOM files that match search criteria into a single file. 
		/// All dimensions higher than or equal to 4th are stacked together as frames. 
		/// Only the files that share the same seriesinstanceUID (with 1st encountered file) are read.
		/// </summary>
		/// <param name="file">full path to the files</param>
		/// <returns>array of DICOM files</returns>
		public static DicomFile ReadMultiple(string file)
		{
			IOProcessEventHandler pevent = null;

			DicomHeader fullHeader;
			IList<DicomFile> dcmfiles = ReadFilesFromPath(file, out fullHeader);

			// Create new Dicomfile for the complete image
			DicomFile dcmfile = new DicomFile(file, fullHeader.Dim.ToArray());
			dcmfile.Header = fullHeader;

			// Ensure that image and header data match
			dcmfile.image.IsDynamic = dcmfile.Header.IsDynamic;
			dcmfile.image.SetFrameDurations(fullHeader.FrameDurations);
			dcmfile.image.SetFrameStartTimes(fullHeader.FrameStartTimes);

			//read tacs from previously acquired positions
			int location = 0;
			int planelength = fullHeader.DimX * fullHeader.DimY;
			for (int i = 0; i < dcmfiles.Count; i++)
			{
				dcmfiles[i].GetPixelData(ref dcmfile.image,
										location,
										dcmfiles[i].filename,
										new IntLimits(fullHeader.DimX, fullHeader.DimY),
										dcmfile.Header.Datatype,
										dcmfiles[i].DicomHeader.stream_data_position,
										dcmfiles[i].DicomHeader.scalefactor,
										dcmfiles[i].DicomHeader.scaleintercept);
				location += planelength;
				pevent = IOProgress;
				if (pevent != null)
					pevent.Invoke(dcmfiles[i], new IOProgressEventArgs(100.0f * (float)i / dcmfiles.Count, System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
			}
			return dcmfile;
		}

		/// <summary>
		/// Reads all DICOM files from path.
		/// </summary>
		/// <param name="path">full path to the files</param>
		/// <returns>array of DICOM files</returns>
		public static DicomFile[] ReadDicomPath(string path)
		{
			FileInfo[] filenames = ResolveDicomFilenames(path);
			List<DicomFile> dicomlist = new List<DicomFile>();
			DicomFile[] r;
			DicomFile dfile = null;
			int current_instance_number = 0;
			IOProcessEventHandler pevent = null;
			//gather tacs into array and return it
			for (int i = 0; i < filenames.Length; i++)
			{
				try
				{
					dfile = new DicomFile(filenames[i].FullName);
					dfile.ReadSingle();
					//start new instance numbering if series is changed
					if (dfile.DicomHeader.instance_nr == 0 &&
						(dicomlist.Count == 0 || dicomlist[dicomlist.Count - 1].DicomHeader.series_nr != dfile.DicomHeader.series_nr))
						current_instance_number = 0;
					dfile.DicomHeader.instance_nr = current_instance_number++;
					dicomlist.Add(dfile);
					pevent = IOProgress;
					if (pevent != null)
						pevent.Invoke(dfile, new IOProgressEventArgs(100.0f * (float)i / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
				}
				catch (TPCUnrecognizedFileFormatException)
				{
					// just skip the file 
				}
				catch (TPCException e)
				{
					// print error and skip the file 
					Console.WriteLine("Cannot read file " + filenames[i].FullName + ":" + e.Message);
				}
			}
			r = dicomlist.ToArray();
			if (pevent != null)
				pevent.Invoke(dfile, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
			return r;
		}

		/// <summary>
		/// Resolves number of gates in dicom image. 
		/// </summary>
		/// <param name="files">planes of dicom file</param>
		/// <returns>number of gates in image (allways >= 1)</returns>
		/// <exception cref="TPCInvalidArgumentsException">if there is no input files</exception>
		public static int ResolveNoGates(DicomFile[] files)
		{
			int j = files[0].DicomHeader.slice_start;
			int k = 1;
			DataElement d;

			if (files.Length == 0) throw new TPCInvalidArgumentsException("No input files");

			//try to read value straight from DICOM field (0054,0101) 'NumberOfTimeSlices'
			try
			{
				d = files[0].ReadTAG(54, 71);
				k = (UInt16)(d.Value.ToArray()[0]);
				if (k > 0) return k;
			}
			catch (TPCDicomFileException)
			{
				//do nothing, failed to resolve gates
			}
			if (k > 1) return k;
			return 1;
		}

		/// <summary>
		/// Resolves number of frames in dicom image. 
		/// </summary>
		/// <param name="files">planes of dicom file</param>
		/// <returns>number of frames in image (allways >= 1)</returns>
		/// <exception cref="TPCInvalidArgumentsException">if there is no input files</exception>
		public static int ResolveNoFrames(DicomFile[] files)
		{
			int i = 0;
			int j = files[0].DicomHeader.slice_start;
			int k = 1;
			DataElement d;

			if (files.Length == 0) throw new TPCInvalidArgumentsException("No input files");

			//try to read value straight from DICOM field (0054,0101) 'NumberOfTimeSlices'
			try
			{
				d = files[0].ReadTAG(54, 101);
				k = (UInt16)(d.Value.ToArray()[0]);
				if (k > 0) return k;
			}
			catch (Exception)
			{
				//do nothing, determine value with slice_start fields
				//Console.WriteLine("Field (0054,0101) not found for frame number information:"+e.Message);
			}
			k = 0;
			for (; i < files.Length; i++)
			{
				// use frame start tacs 
				if (j != files[i].DicomHeader.slice_start)
				{
					j = files[i].DicomHeader.slice_start;
					k++;
				}
			}
			if (k > 1) return k;
			return 1;
		}

		/// <summary>
		/// Resolves number of planes in dicom image. 
		/// </summary>
		/// <param name="files">planes of dicom file</param>
		/// <returns>number of planes in image (allways >= 1)</returns>
		/// <exception cref="TPCInvalidArgumentsException">if there is no input files</exception>
		public static int ResolveNoPlanes(DicomFile[] files)
		{
			int j = files[0].DicomHeader.slice_start;
			int k = 1;
			DataElement d;

			if (files.Length == 0) throw new TPCInvalidArgumentsException("No input files");

			//try to read value straight from DICOM field (0054,0081) 'NumberOfSlices'
			try
			{
				d = files[0].ReadTAG(54, 81);
				k = (UInt16)(d.Value.ToArray()[0]);
				if (k > 0) return k;
			}
			catch (Exception)
			{
				//do nothing, determine using number of frames and number of images
			}
			k = files.Length / ResolveNoFrames(files);
			if (k > 1) return k;
			return 1;
		}

		/// <summary>
		/// Creates DICOM dataelement for writing
		/// </summary>
		/// <param name="element">element number</param>
		/// <param name="group">group number</param>
		/// <param name="data">tacs array</param>
		public static DataElement CreateDataElement(UInt16 group, UInt16 element, byte[] data)
		{
			Tag tag = new Tag(group, element);
			tag.TransferSyntax = TransferSyntax.Default;
			if (data.Length % 2 != 0)
			{
				byte[] even = new byte[data.Length + 1];
				data.CopyTo(even, 0);
				data = even;
			}
			if (dataElementDictionary[tag] == null)
				throw new TPCDicomFileError("Tag " + tag + " not found in dictionary, cannot write element.");
			Value value = new Value(dataElementDictionary[tag].VR, new ValueLength(null, data.Length));
			if (!BitConverter.IsLittleEndian)
				BitSwapper.swapArrayBytes(data, 0, data.Length);
			value.Add(data);
			return new DataElement(tag, value);
		}

		/// <summary>
		/// Writes DICOM tag element into stream
		/// </summary>
		/// <param name="stream">writable stream</param>
		/// <param name="element">element number</param>
		/// <param name="group">group number</param>
		/// <param name="data">tacs array</param>
		public static void WriteElement(Stream stream, UInt16 group, UInt16 element, byte[] data)
		{
			Tag tag = new Tag(group, element);
			tag.TransferSyntax = TransferSyntax.Default;
			if (data.Length % 2 != 0)
			{
				byte[] even = new byte[data.Length + 1];
				data.CopyTo(even, 0);
				data = even;
			}
			if (dataElementDictionary[tag] == null)
				throw new TPCDicomFileError("Tag " + tag + " not found in dictionary, cannot write element.");
			Value value = new Value(dataElementDictionary[tag].VR, new ValueLength(null, data.Length));
			if (!BitConverter.IsLittleEndian)
				BitSwapper.swapArrayBytes(data, 0, data.Length);
			value.Add(data);
			DataElement dataelement = new DataElement(tag, value);
			dataelement.SaveTo(stream);
		}

		/// <summary>
		/// Writes DICOM info sequence. Data element length is set to 
		/// 0xFFFFFFFF (undefined).
		/// </summary>
		/// <param name="stream">writable stream</param>
		/// <param name="group">group number</param>
		/// <param name="element">element number</param>
		/// <returns>position at tacs element length writing</returns>
		public static long WriteInfoSeq(Stream stream, UInt16 group, UInt16 element)
		{
			Tag tag = new Tag(group, element);
			tag.TransferSyntax = TransferSyntax.Default;
			tag.SaveTo(stream);
			BinaryWriter binarystream = new BinaryWriter(stream);
			binarystream.Write(ToBytesArray("SQ".ToCharArray()));
			binarystream.Write(new byte[2]);
			long pos = stream.Position;
			binarystream.Write(new byte[] { 0xFF, 0xFF, 0xFF, 0xFF });
			return pos;
		}

		/// <summary>
		/// Writes info sequene recursively. Writing mode is EXPLICIT, while still
		/// defining sequence length, and without ending tags (mixed representation understood by 
		/// Vinci, DCMTK).
		/// </summary>
		/// <param name="stream">output stream</param>
		/// <param name="dataset">dataset that is written</param>
		public static void WriteDataSetRecursive(Stream stream, NestedDataSetWriteData dataset)
		{
			long length_pos = WriteItemHeader(stream, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			long start_pos = stream.Position;
			foreach (Object d in dataset)
			{
				if (d is NestedDataSetWriteData)
				{
					WriteDataSetRecursive(stream, (NestedDataSetWriteData)d);
				}
				else if (d is SequenceWriteData)
				{
					WriteSequenceRecursive(stream, (SequenceWriteData)d);
				}
				else if (d is DataElement)
				{
					//write field
					((DataElement)d).SaveTo(stream);
				}
			}
			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes info sequene recursively. Writing mode is EXPLICIT, while still
		/// defining sequence length, and without ending tags (mixed representation understood by 
		/// Vinci, DCMTK).
		/// </summary>
		/// <param name="stream">output stream</param>
		/// <param name="sequence">sequence that is written</param>
		public static void WriteSequenceRecursive(Stream stream, SequenceWriteData sequence)
		{

			long length_pos = WriteInfoSeq(stream,
				UInt16.Parse(sequence.tag.Group, System.Globalization.NumberStyles.HexNumber),
				UInt16.Parse(sequence.tag.Element, System.Globalization.NumberStyles.HexNumber));
			long start_pos = stream.Position;
			foreach (Object d in sequence)
			{
				if (d is NestedDataSetWriteData)
				{
					WriteDataSetRecursive(stream, (NestedDataSetWriteData)d);
				}
				else if (d is SequenceWriteData)
				{
					WriteSequenceRecursive(stream, (SequenceWriteData)d);
				}
				else if (d is DataElement)
				{
					//write field
					((DataElement)d).SaveTo(stream);
				}
			}
			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes length of item to stream into defined positions. 
		/// The stream is set back original position after writing.
		/// Used for writing item length after items has been written.
		/// </summary>
		/// <param name="stream">stream</param>
		/// <param name="position"></param>
		/// <param name="length"></param>
		public static void WriteLengthToStream(Stream stream, long position, uint length)
		{
			long pos = stream.Position;
			BinaryWriter binarystream = new BinaryWriter(stream);
			stream.Seek(position, SeekOrigin.Begin);
			binarystream.Write(TransferSyntax.Default.CorrectByteOrdering(BitConverter.GetBytes(length)));
			binarystream.Flush();
			stream.Seek(pos, SeekOrigin.Begin);
		}

		/// <summary>
		/// Write item header tag into stream. This method is used when writing nested tags.
		/// </summary>
		/// <param name="stream">output stream</param>
		/// <param name="data">data that is written file, four empty bytes for explicit length</param>
		/// <returns>position at tacs element length writing</returns>
		public static long WriteItemHeader(Stream stream, byte[] data)
		{
			Tag tag = new Tag(0xFFFE, 0xE000);
			tag.TransferSyntax = TransferSyntax.Default;
			tag.SaveTo(stream);
			BinaryWriter binarystream = new BinaryWriter(stream);
			long pos = stream.Position;
			binarystream.Write(data);
			binarystream.Flush();
			return pos;
		}

		/// <summary>
		/// Stacks dicomfiles containing planes, frames and/or gates into one DICOM image. Planes are ordered.
		/// Filename is set to "stacked.dcm".
		/// </summary>
		/// <param name="dicomfiles">array of images</param>
		/// <returns>new DicomFile containing all files</returns>
		/// <exception cref="TPCInvalidArgumentsException">if tacs is inconsistent</exception>
		public static DicomFile Stack(DicomFile[] dicomfiles)
		{
			int g = 0;
			int f = 0;
			int p = 0;
			int file_i = 0;
			DicomFile r;
			int frames_total;
			IOProcessEventHandler pevent = null;

			if (!(dicomfiles is DicomFile[])) throw new TPCInvalidArgumentsException("No input array.");
			if (dicomfiles.Length == 0) throw new TPCInvalidArgumentsException("No files.");

			Array.Sort(dicomfiles);
			DicomFile[] gates = new DicomFile[DicomFile.ResolveNoGates(dicomfiles)];
			DicomFile[] frames = new DicomFile[DicomFile.ResolveNoFrames(dicomfiles)];
			DicomFile[] planes = new DicomFile[DicomFile.ResolveNoPlanes(dicomfiles)];

			frames_total = gates.Length * frames.Length;
			if (gates.Length * frames.Length * planes.Length != dicomfiles.Length)
				throw new TPCInvalidArgumentsException("Dimension information [gates=" + gates.Length + " frames=" + frames.Length + " planes=" + planes.Length + " is inconsistent with number of input files " + dicomfiles.Length + ".");
			for (g = 0; g < gates.Length; g++)
			{
				for (f = 0; f < frames.Length; f++)
				{
					for (p = 0; p < planes.Length; p++)
					{
						planes[p] = dicomfiles[file_i++];
					}
					frames[f] = DicomFile.StackDimension(planes, Limits.PLANES);
					pevent = IOProgress;
					if (pevent != null)
						pevent.Invoke(frames[f], new IOProgressEventArgs(100.0f * (float)((g + 1) * (f + 1)) / frames_total, System.Reflection.MethodBase.GetCurrentMethod()));
				}
				gates[g] = DicomFile.StackDimension(frames, Limits.FRAMES);
			}
			r = DicomFile.StackDimension(gates, Limits.GATES);

			//copy all general header tacs
			r.Header.Clear();
			System.ComponentModel.TypeConverter t = new System.ComponentModel.TypeConverter();
			for (int i = 0; i < dicomfiles[0].Header.Count; i++)
			{
				r.Header[dicomfiles[0].Header[i].Key] = dicomfiles[0].Header[i].Value;
			}

			pevent = IOProgress;
			if (pevent != null)
				pevent.Invoke(r, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
			return r;
		}

		/// <summary>
		/// Stacks dicomfiles into one DICOM image according to dimension. Images are sorted.
		/// Filename is set to "stacked.dcm". Result header information is copied from first 
		/// file.
		/// </summary>
		/// <see cref="DicomFile"/>
		/// <param name="dicomfiles">array of images (non-null)</param>
		/// <param name="dimension">dimension that is stacked (0-based)</param>
		/// <returns>new DicomFile object containing stacked tacs</returns>
		/// <exception cref="TPCInvalidArgumentsException">if tacs is inconsistent, unsupported stack dimension</exception>
		public static DicomFile StackDimension(DicomFile[] dicomfiles, int dimension)
		{
			float[] mm = new float[2];

			if (dicomfiles.Length < 1) throw new TPCInvalidArgumentsException("No DICOM files to stack.");
			if (dimension < IntLimits.PLANES || dimension > IntLimits.GATES)
				throw new TPCInvalidArgumentsException("Unsupported dimension for stacking:" + dimension);

			//ensure consistensy
			for (int i = 0; i < dicomfiles.Length; i++)
			{
				if (dicomfiles[i] == null) throw new TPCInvalidArgumentsException("null found at index " + i + ".");
				if (dicomfiles[i].Header == null) throw new TPCInvalidArgumentsException("no header at index " + i + ".");
				if (dicomfiles[i].Header.Siz != dicomfiles[0].Header.Siz) throw new TPCInvalidArgumentsException("Inconsistent gate voxel sizes.");
				if (dicomfiles[i].image == null) throw new TPCInvalidArgumentsException("no image at index " + i + ".");
				if (dicomfiles[i].image.Dim != dicomfiles[0].image.Dim) throw new TPCInvalidArgumentsException("Inconsistent dimensions.");
				if ((dicomfiles[i].image.IsDynamic) != (dicomfiles[0].image.IsDynamic))
					throw new TPCInvalidArgumentsException("Tried to stack dynamic and static images together.");
			}

			//sort dicomfiles
			Array.Sort(dicomfiles);

			DicomFile r = new DicomFile("stacked.dcm");
			r.Header = dicomfiles[0].Header;

			Voxel siz = r.Header.Siz;
			IntLimits dim = r.Header.Dim;

			switch (dimension)
			{
				case Limits.PLANES: r.DicomHeader.imagepositions[0].Z = dicomfiles[0].DicomHeader.SliceLocation; break;
				case Limits.FRAMES: r.DicomHeader.slice_start = dicomfiles[0].DicomHeader.slice_start; break;
				case Limits.GATES: r.DicomHeader.gate_number = dicomfiles[0].DicomHeader.gate_number; break;
			}

			//set result dimensions to comply with stacking dimension
			while (dim.Length <= dimension) dim.AddLimit(0, 1);
			dim.SetLimit(dimension, Limits.Limit.LOW, 0);
			dim.SetLimit(dimension, Limits.Limit.HIGH, dicomfiles.Length);

			r.image = new Image(dim);
			r.image.IsDynamic = r.Header.IsDynamic;
			r.Header.Dim = new IntLimits(dim);
			r.Header.Siz = new Voxel(siz);

			//gather image tacs
			if (dicomfiles.Length > 1)
			{
				for (int i = 0; i < dicomfiles.Length; i++)
				{
					dim = r.image.GetDimensionLimits(i, dimension);
					//upgrade set region to target dimensions
					while (dim.Length < r.image.Dim.Length) dim.AddLimit(0, 1);
					r.image.SetSubImage(dim, ref dicomfiles[i].image);
				}
				//take only first frame information even if files contain multiple frames, this may need update
				for (int i = 0; i < dicomfiles.Length; i++)
				{
					if (dimension == Limits.FRAMES)
					{
						r.image.SetFrameStartTime(i, dicomfiles[i].image.GetFrameStartTime(0));
						r.image.SetFrameDuration(i, dicomfiles[i].image.GetFrameDuration(0));
					}
					else
					{
						r.image.SetFrameStartTime(0, dicomfiles[i].image.GetFrameStartTime(0));
						r.image.SetFrameDuration(0, dicomfiles[i].image.GetFrameDuration(0));
					}
				}
			}
			else
			{
				r.image = dicomfiles[0].image;
			}
			r.Header.FrameStartTimes = r.image.GetFrameStartTimes();
			r.Header.FrameDurations = r.image.GetFrameDurations();
			//copy fileinfo
			return r;
		}

		/// <summary>
		/// Checks file format.
		/// </summary>
		/// <param name="filename">full path to file</param>
		/// <returns>true if file is a DICOM file</returns>
		/// <exception cref="TPCDicomFileException">if there was a read problem</exception>
		public static bool CheckFormat(string filename)
		{
			//try to resolve single DICOM file
			try
			{
				return (TPCDicom.File.DicomFile.IsDicomFile(filename));
			}
			catch (Exception)
			{
				//not a single DICOM file? try first file of many multiple files
				FileInfo[] filenames;
				try
				{
					//fail if string does not refer to a directory
					if (!filename.EndsWith("*") && !System.IO.Directory.Exists(filename)) return false;
					//try to resolve filenames from directory
					filenames = DicomFile.ResolveDicomFilenames(filename);
				}
				catch (Exception)
				{
					return false;
				}
				if (filenames.Length == 0)
					return false;
				for (int i = 0; i < filenames.Length; i++)
				{
					try
					{
						if (TPCDicom.File.DicomFile.IsDicomFile(filenames[i].FullName))
							return true;
					}
					catch (Exception) { }
				}
				return false;
			}
		}

		/// <summary>
		/// Create an image data stream from multiple image files
		/// </summary>
		/// <param name="path">Path to the files</param>
		/// <returns>Image data stream</returns>
		public static ImageStream GetMultiFileStream(string path, FileAccess a = FileAccess.ReadWrite)
		{
			return GetMultiFileStream(path, null, a);
		}

		/// <summary>
		/// Create a subimage data stream from multiple image files
		/// </summary>
		/// <param name="path">Path to the files</param>
		/// <param name="limits">Subimage limits</param>
		/// <returns>Image data stream</returns>
		public static ImageStream GetMultiFileStream(string path, IntLimits limits, FileAccess a = FileAccess.ReadWrite)
		{
			DicomFile[] componentFiles;
			DicomHeader combinedHeader;

			// Read headers
			componentFiles = ReadFilesFromPath(path, out combinedHeader).ToArray();

			long[] offsets = new long[componentFiles.Length];
			float[] scaling = new float[componentFiles.Length];
			float[] intercepts = new float[componentFiles.Length];
			string[] fileNames = new string[componentFiles.Length];

			for (int i = 0; i < componentFiles.Length; i++)
			{
				offsets[i] = componentFiles[i].DicomHeader.stream_data_position + 12;
				scaling[i] = componentFiles[i].DicomHeader.scalefactor;
				intercepts[i] = componentFiles[i].DicomHeader.scaleintercept;
				fileNames[i] = componentFiles[i].filename;
			}

			// Calculate planelengths
			int planeLength = combinedHeader.Dim.DimX * combinedHeader.Dim.DimY;
			int byteLength = planeLength * ImageFile.BytesPerPixel(combinedHeader.Datatype);
			long[] dataLengths = new long[offsets.Length];
			for (int i = 0; i < dataLengths.Length; i++) dataLengths[i] = byteLength;

			// Create multifile image stream
			MultiFileStream mfs = new MultiFileStream(fileNames, offsets, dataLengths, false);
			IntLimits lim;
			if (limits is IntLimits) lim = limits;
			else lim = combinedHeader.Dim;
			ImageStream imgs = new ImageStream(mfs, combinedHeader.Dim, lim, 0, combinedHeader.Datatype);

			// Set scaling factors for all planes
			imgs.SetScaling(scaling, intercepts, Limits.PLANES);

			// Set orientation
			imgs.StreamOrientation = combinedHeader.Orientation;

			return imgs;
		}

		#endregion

		#region ImageFile interface

		/// <summary>
		/// File header
		/// </summary>
		public override ImageHeader Header
		{
			get
			{
				return imgHeader;
			}
			set
			{
				imgHeader = new DicomHeader(value);
			}
		}

		/// <summary>
		/// Reads a Dicom file
		/// </summary>
		public override void ReadFile()
		{
			//read one or multiple DICOM files 
			if (filename.EndsWith("*"))
			{
				DicomFile df = DicomFile.ReadMultiple(filename);
				this.Header = df.Header;
				this.image = df.image;
			}
			else
			{
				ReadSingle();
			}
		}

		/// <summary>
		/// Writes single frame into file. If frame number is larger than 
		/// current number of frames in file, then file size is grown.
		/// </summary>
		/// <param name="image">image tacs that is written</param>
		/// <param name="frame_No">frame number that is written [1..no frames+1]</param>
		public override void WriteFrame(ref Image image, int frame_No)
		{
			throw new NotImplementedException();
		}

		/// <summary>
		/// Gets pixel tacs from image. It is strongly suggested that the file header is read 
		/// first in order to fill proper header fields that might be needed for reading.
		/// </summary>
		/// <param name="data">target array</param>
		/// <param name="offset">offset where copying begins to write tacs</param>
		/// <param name="stream">input stream that is expected to be open and stays open</param>
		/// <param name="dim">input tacs dimensions</param>
		/// <param name="datatype">tacs type to be read</param>
		/// <param name="position">file position where reading is started (at beginning of tag (7FE0,0010))</param>
		/// <param name="scale">scale factor</param>
		/// <param name="intercept">intercept</param>
		public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale = 1.0f, float intercept = 0.0f)
		{
			if (offset + dim.GetProduct() > data.DataLength)
				throw new TPCInvalidArgumentsException("Invalid parameters: offset(" + offset + ")+length(" + dim.GetProduct() + ")>= tacs.Length(" + data.DataLength + ")");

			//read tacs from previously acquired positions
			byte[] bytes = null;
			int location = offset;

			try
			{
				stream.Seek(DicomHeader.stream_data_position, SeekOrigin.Begin);
				// tag       (4 bytes)
				//Tag tag = new Tag(stream, dicomheader.transfersyntax);
				// VR        (2 bytes)
				//ValueRepresentation vr = ValueRepresentation.LoadFrom(stream, tag);
				// reserved (2 bytes)
				// length    (4 bytes)
				stream.Seek((long)12, SeekOrigin.Current);
				//read rest of stream as tacs
				bytes = new byte[stream.Length - stream.Position];
				stream.Read(bytes, 0, bytes.Length);
			}
			catch (Exception e)
			{
				throw new TPCDicomFileException("Exception [" + e + "] while reading image tacs of [" + filename + "]");
			}

			//store tacs into image
			if (bytes.Length == 0) throw new TPCDicomFileException("0 bytes in pixel tacs");
			int bytesPerPixel = ImageFile.BytesPerPixel(datatype);
			byte[] value = new byte[bytesPerPixel];

			GetValue getVal;

			switch (datatype)
			{
				case DataType.BIT8_U:
					getVal = delegate(int i) { return bytes[i]; };
					break;
				case DataType.BIT8_S:
					getVal = delegate(int i) { return (sbyte)bytes[i]; };
					break;
				case DataType.BIT16_U:
					if (DicomHeader.transfersyntax.IsLittleEndian == DicomHeader.transfersyntax.IsMachineLittleEndian)
					{
						getVal = delegate(int i)
						{
							Buffer.BlockCopy(bytes, i, value, 0, bytesPerPixel);
							return BitConverter.ToUInt16(value, 0);
						};
					}
					else
					{
						getVal = delegate(int i)
						{
							value[1] = bytes[i];
							value[0] = bytes[i + 1];
							return BitConverter.ToUInt16(value, 0);
						};
					}
					break;
				case DataType.BIT16_S:
					if (DicomHeader.transfersyntax.IsLittleEndian == DicomHeader.transfersyntax.IsMachineLittleEndian)
					{
						getVal = delegate(int i)
						{
							Buffer.BlockCopy(bytes, i, value, 0, bytesPerPixel);
							return BitConverter.ToInt16(value, 0);
						};
					}
					else
					{
						getVal = delegate(int i)
						{
							value[1] = bytes[i];
							value[0] = bytes[i + 1];
							return BitConverter.ToInt16(value, 0);
						};
					}
					break;
				default:
					throw new TPCDicomFileException("Unsupported tacs format:" + Header.Datatype);
			}
			for (int j = 0; j < bytes.Length && location < data.DataLength; j += bytesPerPixel)
			{
				data[location++] = getVal(j) * scale + intercept;
			}
		}

		/// <summary>
		/// Read header from current path.
		/// </summary>
		/// <returns>Image header</returns>
		public override ImageHeader ReadHeader()
		{
			ImageHeader h;

			//read one or multiple DICOM files 
			if (filename.EndsWith("*"))
			{
				DicomHeader dh;
				ReadFilesFromPath(filename, out dh);
				h = dh;
			}
			else
			{
				h = ReadSingleHeader();
			}
			return h;
		}

		/// <summary>
		/// Reads only header tacs from file
		/// </summary>
		/// <returns>Header information in file</returns>
		private ImageHeader ReadSingleHeader()
		{
			TPCDicom.File.AcrNemaFile file = null;

			//set end condition tag as pixel tacs to skip reading it
			Sequence.SkipDataReadTags = new Tag[] { new Tag("(7FE0,0010)") };

			try
			{
				if (TPCDicom.File.DicomFile.IsDicomFile(filename))
					file = new TPCDicom.File.DicomFile(filename, false);
				else if (TPCDicom.File.AcrNemaFile.IsAcrNemaFile(filename))
					file = new TPCDicom.File.AcrNemaFile(filename, false);
				else
					Console.Error.WriteLine("File " + filename + " is neither a DICOM nor an ACR-NEMA file.");
			}
			catch (Exception dicomFileException)
			{
				throw new TPCDicomFileException("Failed to read DICOM file " + filename + ":" + dicomFileException);
			}
			finally
			{
				//take of end condition
				Sequence.SkipDataReadTags = null;
			}

			// Save the transfer syntax
			TransferSyntax tsyntax = new TransferSyntax();
			tsyntax.LoadFrom(file.GetJointDataSets());
			DicomHeader.transfersyntax = tsyntax;

			Sequence sequence = file.GetJointDataSets().GetJointSubsequences();
			Header.Siz.SizeZ = 1.0f;
			//nested depth
			int nested = 0;
			//group tag used to track last grouping tag in 0054
			Tag groupTag = new Tag("(0000,0000)");
			//true if value is already set, prevents overwriting
			bool tag_0028_0103_set = false;

			int dimx = 1;
			int dimy = 1;
			int dimz = 1;
			int dimt = 1;

			foreach (DataElement d in sequence)
			{
				//handle special fields that mark nested fields
				if (d.Tag.Equals("(FFFE,E000)"))
				{
					nested++;
					continue;
				}
				else if (d.Tag.Equals("(FFFE,E0DD)"))
				{
					nested--;
					continue;
				}
				//add field into general header
				try
				{
					// Add only recognized entries and discard private tags
					DataElementDictionaryEntry entry = DicomFile.dataElementDictionary.GetDictionaryEntry(d.Tag);
					if (entry is DataElementDictionaryEntry)
					{
						if (!Header.Contains(entry.Description) && entry.Description != "PixelData")
						{
							if (d.Value is Value)
							{
								// Save multivalues as arrays and others as single values
								if (d.Value.Count == 1)
								{
									Header[entry.Description] = d.Value[0];
								}
								else
								{
									Header[entry.Description] = d.Value.ToArray();
								}
							}
						}
					}
				}
				catch (TPCGeneralHeaderException)
				{
					//ignore multiple fields, just saving the first occurrence
				}
				catch (TPCDicom.DicomException e)
				{
					//do not add null reference_times if it does exist
					throw new TPCDicomFileError("TPCDicom.DicomException:" + e);
				}

				// If header defines this as a dynamic image, change the header type
				if (d.Tag.Equals("(0054,1000)"))
				{
					string s = d.Value.ToString().ToUpperInvariant();
					Header.IsDynamic = s.Contains("DYNAMIC");
				}

					//resolve SOP instanceUID for sorting planes
				else if (d.Tag.Equals("(0008,0018)"))
					Header.Description = d.Value.ToArray()[0].ToString();
				else if (d.Tag.Equals("(0008,0060)"))
				{
					try
					{
						Header.Modality = (ImageModality)System.Enum.Parse(typeof(ImageModality), "M_" + d.Value.ToArray()[0].ToString());
					}
					catch (Exception)
					{
						if (d.Value.ToArray()[0].ToString().Equals("NUCMED"))
							Header.Modality = ImageModality.M_NM;
						else if (d.Value.ToArray()[0].ToString().StartsWith("SPECT"))
							Header.Modality = ImageModality.M_PX;
						else if (d.Value.ToArray()[0].ToString().StartsWith("PET"))
							Header.Modality = ImageModality.M_PT;
						else if (d.Value.ToArray()[0].ToString().StartsWith("RT"))
							Header.Modality = ImageModality.M_RT;
						else
							Header.Modality = ImageModality.Unknown;
					}
				}
				else if (d.Tag.Equals("(0008,1030)"))
					Header.Description = d.Value.ToArray()[0].ToString();
				//reads field for isotope, affects conversion to DynamicImageHeader if needed
				else if (nested > 0 && d.Tag.Equals("(0008,0104)") && groupTag.Equals("(0054,0300)"))
				{
					Header.Isotope = Isotope.CreateIsotope(d.Value.ToArray()[0].ToString());
				}
				else if (d.Tag.Equals("(0010,0010)"))
				{
					string dicomName = d.Value.ToArray()[0].ToString();
					// remove trailing \0 padding, if present
					// remove Dicom separators (^)
					Header.PatientName = new PersonName(dicomName.Replace('^', ' '));
				}
				else if (d.Tag.Equals("(0010,0020)"))
				{
					Header.PatientID = d.Value.ToArray()[0].ToString();
				}
				//radiopharmaceutical name
				else if (nested == 0 && d.Tag.Equals("(0018,0031)"))
				{
					Header.Radiopharma = d.Value.ToArray()[0].ToString();
				}
				else if (d.Tag.Equals("(0018,0050)"))
				{
					try
					{
						Header.Siz.SizeZ = Convert.ToSingle(d.Value.ToArray()[0]);
					}
					catch (Exception)
					{
						//do not change anything
					}
				}
				//injection time (no date information)
				else if (d.Tag.Equals("(0018,1072)"))
				{
					try
					{
						//set only fields HH:mm:ss for dose start time
						TimeSpan span = (TimeSpan)d.Value.ToArray()[0];
						if (Header.DoseStartTime == null) Header.DoseStartTime = new System.DateTime(0);
						else Header.DoseStartTime = new System.DateTime(Header.DoseStartTime.Year,
																		Header.DoseStartTime.Month,
																		Header.DoseStartTime.Day);
						Header.DoseStartTime = Header.DoseStartTime.Add(span);
					}
					catch (Exception e)
					{
						Console.WriteLine(e);
						Header.DoseStartTime = new System.DateTime(0);
					}
				}

				// Injected dose
				else if (d.Tag.Equals("(0018,1074)"))
				{
					try
					{
						Header.InjectedDose = Convert.ToSingle(d.Value.ToArray()[0]);
					}
					catch (Exception e)
					{
						Console.WriteLine(e);
						Header.InjectedDose = 0.0f;
					}
				}

				//injection date and time
				else if (d.Tag.Equals("(0018,1078)"))
				{
					try
					{
						Header.DoseStartTime = (System.DateTime)d.Value.ToArray()[0];
					}
					catch (Exception e)
					{
						Console.WriteLine(e);
						Header.DoseStartTime = new System.DateTime(0);
					}
				}
				//frame duration time, affects image type to be DynamicImage if found
				else if (d.Tag.Equals("(0018,1242)"))
				{
					DicomHeader.frame_duration = Convert.ToDouble(d.Value.ToArray()[0]);
				}
				//patient orientation
				else if (d.Tag.Equals("(0018,5100)"))
				{
					DicomHeader.dicomOrientation = d.Value.ToArray()[0].ToString().TrimEnd('\0');
				}
				else if (d.Tag.Equals("(0020,0013)"))
				{
					DicomHeader.instance_nr = Convert.ToInt32(d.Value.ToArray()[0]);
				}
				else if (d.Tag.Equals("(0020,0032)"))
				{
					object[] arr = d.Value.ToArray();
					DicomHeader.imagepositions[0] = new Point(Convert.ToSingle(arr[0]), Convert.ToSingle(arr[1]), Convert.ToSingle(arr[2]));
				}
				//image orientation in three pairs of cosines for each axis x,y,z directions
				else if (d.Tag.Equals("(0020,0037)"))
				{
					object[] arr = d.Value.ToArray();
					DicomHeader.ImageOrientations = new float[] { 
                        Convert.ToSingle(arr[0]), Convert.ToSingle(arr[1]), Convert.ToSingle(arr[2]), 
                        Convert.ToSingle(arr[3]), Convert.ToSingle(arr[4]), Convert.ToSingle(arr[5]) };
				}
				else if (d.Tag.Equals("(0020,1041)"))
				{
					DicomHeader.imagepositions[0].Z = Convert.ToSingle(d.Value.ToArray()[0]);
				}
				else if (d.Tag.Equals("(0028,0002)"))
					DicomHeader.samplesperpixel = (UInt16)d.Value.ToArray()[0];
				else if (d.Tag.Equals("(0028,0010)"))
					dimy = (UInt16)d.Value.ToArray()[0];
				else if (d.Tag.Equals("(0028,0011)"))
					dimx = (UInt16)d.Value.ToArray()[0];
				else if (d.Tag.Equals("(0028,0012)"))
					dimz = (UInt16)d.Value.ToArray()[0];
				else if (d.Tag.Equals("(0028,0100)"))
				{
					//get number of bytes allocatted for single pixel
					DicomHeader.bytesperpixel = (int)Math.Ceiling(Convert.ToSingle(d.Value.ToArray()[0]) / 8.0f);
				}
				else if (!tag_0028_0103_set && d.Tag.Equals("(0028,0103)"))
				{
					DicomHeader.signed = ((UInt16)d.Value.ToArray()[0] != 0);
					tag_0028_0103_set = true;
				}
				else if (d.Tag.Equals("(0028,0030)"))
				{
					// PixelSpacing:
					// RowSpacing (distance between rows)
					Header.Siz.SizeY = Convert.ToSingle(d.Value.ToArray()[0]);
					// ColumnSpacing (distance between columns)
					Header.Siz.SizeX = Convert.ToSingle(d.Value.ToArray()[1]);
				}
				else if (d.Tag.Equals("(0028,1052)"))
				{
					DicomHeader.scaleintercept = float.Parse(d.Value.ToArray()[0].ToString());
				}
				else if (d.Tag.Equals("(0028,1053)"))
				{
					DicomHeader.scalefactor = float.Parse(d.Value.ToArray()[0].ToString());
				}
				else if (d.Tag.Equals("(0054,0300)") || d.Tag.Equals("(0054,0304)"))
					groupTag = d.Tag;
				//number of planes
				else if (d.Tag.Equals("(0054,0081)"))
				{
					dimz = (UInt16)d.Value.ToArray()[0];
				}
				//number of frames, overwritten when DICOM plane files are stacked into dynamic image
				else if (d.Tag.Equals("(0054,0101)"))
				{
					dimt = (UInt16)d.Value.ToArray()[0];
				}
				//fill header tacs fields
				else if (d.Tag.Equals("(0054,1001)"))
				{
					Header.Dataunit = DicomToDataUnit(d.Value.ToArray()[0].ToString());
				}
				//frame reference_times time, affects image type to be DynamicImage if found
				else if (d.Tag.Equals("(0054,1300)"))
				{
					DicomHeader.frame_start_time = Convert.ToDouble(d.Value.ToArray()[0]);
				}
				//resolve bytes per pixel if header fields did not express it already
				else if (DicomHeader.bytesperpixel == 0 && d.Tag.Equals("(7FE0,0000)"))
				{
					if (DicomHeader.Width * DicomHeader.Height > 0)
						DicomHeader.bytesperpixel = (int)Math.Ceiling((Convert.ToDouble(d.Value.ToArray()[0]) - 12.0) / (DicomHeader.Width * DicomHeader.Height));
				}
				//tacs tag position in DICOM file for later fast access
				else if (d.Tag.Equals("(7FE0,0010)"))
				{
					DicomHeader.stream_data_position = d.StreamPosition;
				}
			}

			//copy image size to header as well
			Header.Dim = new IntLimits(dimx, dimy, dimz, dimt);

			// Set the frame times also to the header
			Header.SetFrameStart(0, DicomHeader.frame_start_time);
			Header.SetFrameDuration(0, DicomHeader.frame_duration);

			//resolve tacs type into ImageHeader
			switch (DicomHeader.bytesperpixel)
			{
				case 1:
					if (DicomHeader.signed) Header.Datatype = DataType.BIT8_S;
					else
					{
						//check for RGB data
						if (DicomHeader.samplesperpixel == 3)
							Header.Datatype = DataType.COLRGB;
						else
							Header.Datatype = DataType.BIT8_U;
					}
					break;
				case 2:
					if (DicomHeader.signed) Header.Datatype = DataType.BIT16_S;
					else Header.Datatype = DataType.BIT16_U;
					break;
				case 4:
					if (DicomHeader.signed) Header.Datatype = DataType.BIT32_S;
					else Header.Datatype = DataType.BIT32_U;
					break;
				default:
					//do not throw exception here so that header can be read even when tacs is not able to be read
					//throw new TPCDicomFileException("Unsupported tacs format: bytes=" + dicomheader.bytesperpixel + ",signed=" + dicomheader.signed);
					break;
			}

			return Header;
		}

		/// <summary>
		/// Reads subregion from file.
		/// </summary>
		/// <param name="region">subregion that is read from file</param>
		public override void ReadSubImage(IntLimits region)
		{
			ReadSingle();
			Header.Dim = region;
			image = image.GetSubImage(region);
		}

		/// <summary>
		/// Implementation of ImageFile interface. Writes a single DICOM file. 
		/// </summary>
		/// <param name="image">image tacs that is written</param>
		public override void WriteFile(ref Image image)
		{
			//error message since float tacs writing has produced errors with Vinci software.
			if (Header.Datatype == DataType.FLT32 || Header.Datatype == DataType.FLT64)
				throw new TPCDicomFileException("Writing of float tacs is not supported.");

			//write higher dimension images recursively
			int i = 0;
			IOProcessEventHandler pevent = null;
			//resolve largest effecting dimension
			for (i = image.Dim.Length - 1; i >= IntLimits.PLANES; i--)
				if (image.Dim.GetDimension(i) > 1)
					break;

			//write planes if file has more than single plane
			if (i >= IntLimits.PLANES)
			{
				//resolve total number of planes
				int total_no_planes = image.Dim.GetProduct() / image.Planelength;
				int planelength = image.Planelength;
				int gatelength = image.Planelength;
				int framelength = image.Planelength;
				if (image.Dim.Length >= IntLimits.GATES)
					gatelength = (int)image.Dim.GetDimension(Limits.WIDTH) * (int)image.Dim.GetDimension(Limits.HEIGHT) * (int)image.Dim.GetDimension(Limits.PLANES) * (int)image.Dim.GetDimension(Limits.FRAMES);
				if (image.Dim.Length >= IntLimits.FRAMES)
					framelength = (int)image.Dim.GetDimension(Limits.WIDTH) * (int)image.Dim.GetDimension(Limits.HEIGHT) * (int)image.Dim.GetDimension(Limits.PLANES);
				DicomFile dcmfile;
				IntLimits planelimits;
				int frame_no = 0;
				int gate_no = 0;

				//generate pseudo unique code values for this image series
				Random rand = new Random();
				int seriesNumber = rand.Next(1000000);

				// Flip the image to the default orientation
				Image.Flip(Header.Orientation, DicomHeader.DefaultDicomOrientation, image);

				//write image plane-by-plane                    
				for (i = 0; i < total_no_planes; i++)
				{
					frame_no = (i * planelength) / framelength;
					gate_no = (i * planelength) / gatelength;
					dcmfile = new DicomFile(filename + "_" + gate_no.ToString("000") + "_" + frame_no.ToString("000") + "_" + i.ToString("00000"));

					planelimits = image.GetPlaneLimits(i % image.Planes);
					if (planelimits.Length > IntLimits.FRAMES)
					{
						planelimits.SetLimits(IntLimits.FRAMES, frame_no, frame_no + 1);
						if (planelimits.Length > IntLimits.GATES)
						{
							planelimits.SetLimits(IntLimits.GATES, gate_no, gate_no + 1);
						}
					}

					// Get the current plane
					dcmfile.image = image.GetSubImage(planelimits);

					// Copy header information
					dcmfile.Header = this.Header;

					// Change header to match single plane data
					dcmfile.Header.IsDynamic = image.IsDynamic;
					dcmfile.Header.FrameDurations = dcmfile.image.GetFrameDurations();
					dcmfile.Header.FrameStartTimes = dcmfile.image.GetFrameStartTimes();
					dcmfile.DicomHeader.slice_start = (int)image.GetFrameStartTime(frame_no);
					dcmfile.DicomHeader.gate_number = gate_no;
					//dcmfile.Header.Dim = dcmfile.image.Dim;

					// Generate new instance numbers
					dcmfile.DicomHeader.series_nr = seriesNumber;
					dcmfile.DicomHeader.instance_nr = i + 1;

					// Set image position
					if (DicomHeader.imagepositions.Length < image.DimZ)
					{
						dcmfile.DicomHeader.imagepositions[0] = new Point(DicomHeader.imagepositions[0].X,
																		  DicomHeader.imagepositions[0].Y,
																		  DicomHeader.imagepositions[0].Z + (i % image.DimZ));
					}
					else
					{
						dcmfile.DicomHeader.imagepositions[0] = DicomHeader.imagepositions[(i % image.DimZ)];
					}

					dcmfile.WriteFile();
					pevent = IOProgress;
					if (pevent != null)
						pevent.Invoke(this, new IOProgressEventArgs((float)Math.Round(100.0f * (float)i / (float)total_no_planes, 0), System.Reflection.MethodBase.GetCurrentMethod()));
				}

				// Flip the image back to the original orientation
				Image.Flip(DicomHeader.DefaultDicomOrientation, Header.Orientation, image);

				return;
			}
			using (FileStream stream = new FileStream(filename, FileMode.Create))
			{
				ResolveScaleFactor(out DicomHeader.scalefactor, out DicomHeader.scaleintercept);
				Write0002(stream);
				Write0008(stream);
				//no effect to CortexID
				//Write0009(stream);
				Write0010(stream);
				//no effect to CortexID
				//Write0011(stream);
				//no effect to CortexID
				Write0013(stream);
				//no effect to CortexID
				Write0018(stream);
				Write0020(stream);
				Write0028(stream);
				//no effect to CortexID
				Write0054(stream);
				Write7FE0(stream);
			}
		}

		#endregion

		#region Other public

		/// <summary>
		/// Write a Dicom image to disk with image data from an ImageStream
		/// </summary>
		/// <param name="datastream">Image data stream</param>
		public void WriteFile(ImageStream datastream)
		{
			//error message since float tacs writing has produced errors with Vinci software.
			if (Header.Datatype == DataType.FLT32 || Header.Datatype == DataType.FLT64)
				throw new TPCDicomFileException("Writing of float tacs is not supported.");

			//write higher dimension images recursively
			int i = 0;
			IOProcessEventHandler pevent = null;
			//resolve largest effecting dimension
			for (i = this.Header.Dim.Length - 1; i >= IntLimits.PLANES; i--)
				if (this.Header.Dim.GetDimension(i) > 1)
					break;
			//write planes if file has more than single plane
			if (i >= IntLimits.PLANES)
			{
				int planeLength = this.Header.Dim.DimX * this.Header.Dim.DimY;
				//resolve total number of planes
				int total_no_planes = this.Header.Dim.GetProduct() / planeLength;
				int planelength = planeLength;
				int gatelength = planeLength;
				int framelength = planeLength;
				if (this.Header.Dim.Length >= IntLimits.GATES)
					gatelength = (int)this.Header.Dim.GetDimension(Limits.WIDTH) * (int)this.Header.Dim.GetDimension(Limits.HEIGHT) * (int)this.Header.Dim.GetDimension(Limits.PLANES) * (int)this.Header.Dim.GetDimension(Limits.FRAMES);
				if (this.Header.Dim.Length >= IntLimits.FRAMES)
					framelength = (int)this.Header.Dim.GetDimension(Limits.WIDTH) * (int)this.Header.Dim.GetDimension(Limits.HEIGHT) * (int)this.Header.Dim.GetDimension(Limits.PLANES);

				//generate pseudo unique code values for this image series
				Random rand = new Random();
				int seriesNumber = rand.Next(1000000);

				// Flip the datastream to the correct orientation
				ImageStream imgs = datastream.Flip(DicomHeader.DefaultDicomOrientation);

				DicomFile dcmfile;
				IntLimits planelimits;
				int frame_no = 0;
				int gate_no = 0;

				//write image plane-by-plane                    
				for (i = 0; i < total_no_planes; i++)
				{
					frame_no = (i * planelength) / framelength;
					gate_no = (i * planelength) / gatelength;
					dcmfile = new DicomFile(filename + "_" + gate_no.ToString("000") + "_" + frame_no.ToString("000") + "_" + i.ToString("00000"));

					planelimits = imgs.Dim.GetDimensionLimits(i % this.Header.Planes, Limits.PLANES);
					if (planelimits.Length > IntLimits.FRAMES)
					{
						planelimits.SetLimits(IntLimits.FRAMES, frame_no, frame_no + 1);
						if (planelimits.Length > IntLimits.GATES)
						{
							planelimits.SetLimits(IntLimits.GATES, gate_no, gate_no + 1);
						}
					}
					dcmfile.DicomHeader.gate_number = gate_no;

					// Copy header information
					dcmfile.Header = this.Header;
					dcmfile.Header.Dim = planelimits;

					// Change header to match single plane data
					dcmfile.DicomHeader.slice_start = (int)Header.GetFrameStart(frame_no);
					dcmfile.Header.SetFrameDuration(0, Header.GetFrameDuration(frame_no));
					dcmfile.Header.SetFrameStart(0, Header.GetFrameStart(frame_no));

					// Generate new instance numbers
					dcmfile.DicomHeader.series_nr = seriesNumber;
					dcmfile.DicomHeader.instance_nr = i + 1;

					// Set image position
					dcmfile.DicomHeader.imagepositions[0] = DicomHeader.imagepositions[i % Header.DimZ];
					dcmfile.WriteFile(imgs);

					pevent = IOProgress;
					if (pevent != null)
						pevent.Invoke(this, new IOProgressEventArgs((float)Math.Round(100.0f * (float)i / (float)total_no_planes, 0), System.Reflection.MethodBase.GetCurrentMethod()));
				}
				return;
			}

			// Read a single plane from the stream
			this.image = new Image(this.Header.Dim.DimX, this.Header.Dim.DimY, this.Header.Dim.DimZ);
			this.image.IsDynamic = this.Header.IsDynamic;
			datastream.ReadImage(this.image);

			// Write the plane to a Dicom file
			this.WriteFile();
		}

		/// <summary>
		/// Reads all DICOM tags and returns pointer to their tacs. File must be closed 
		/// before reading, since file is opened and closed.
		/// </summary>
		/// <remarks>This public method is unsafe. Improper usage may result memory corruption.</remarks>
		/// <returns>tag structure array</returns>
		/// <exception cref="TPCDicomFileException">if tag could not be read</exception>
		public DataElement[] ReadTAGs()
		{
			TPCDicom.File.AcrNemaFile file = null;
			try
			{
				if (TPCDicom.File.DicomFile.IsDicomFile(filename))
					file = new TPCDicom.File.DicomFile(filename, false);
				else if (TPCDicom.File.AcrNemaFile.IsAcrNemaFile(filename))
					file = new TPCDicom.File.AcrNemaFile(filename, false);
				else
					Console.Error.WriteLine("Selected file is wether a " +
						"DICOM nor an ACR-NEMA file.");
			}
			catch (Exception dicomFileException)
			{
				throw new TPCDicomFileException("Failed to read DICOM tags in file " + filename + ":" + dicomFileException);
			}
			Sequence s = file.GetJointDataSets().GetJointSubsequences();
			return s.ToArray();
		}

		/// <summary>
		/// Reads DICOM tag. 
		/// <remarks>Note that all DICOM fields are run through. Use readTAGs if reading multiple tags.</remarks>
		/// </summary>
		/// <param name="group">group number</param>
		/// <param name="element">element number</param>
		/// <returns>tacs element from file</returns>
		/// <exception cref="TPCDicomFileException">if tag could not be read</exception>
		public DataElement ReadTAG(UInt16 group, UInt16 element)
		{
			DataElement[] elements = ReadTAGs();
			foreach (DataElement d in elements)
			{
				if (d.Tag.Equals("(" + group.ToString("0000") + "," + element.ToString("0000") + ")"))
				{
					return d;
				}
			}
			throw new TPCDicomFileException("Tag (" + group.ToString("0000") + "," + element.ToString("0000") + ") not found");
		}

		/// <summary>
		/// Splits dicomfile into DICOM images trough to dimension.
		/// Filename is set to filename _ number .dcm. Result header information is copied from first 
		/// file.
		/// </summary>
		/// <see cref="DicomFile"/>
		/// <param name="dimension">dimension that is splitted (0-based)</param>
		/// <returns>new DicomFile array</returns>
		/// <exception cref="TPCInvalidArgumentsException">if input tacs is inconsistent</exception>
		public DicomFile[] SplitDimension(int dimension)
		{
			DicomFile[] r;

			if (!(image is Image)) throw new TPCInvalidArgumentsException("Image is null.");
			if (!(Header is ImageHeader)) throw new TPCInvalidArgumentsException("Header is null.");
			switch (dimension)
			{
				case Limits.PLANES:
					r = new DicomFile[image.Dim.GetDimension(IntLimits.PLANES)];
					for (int i = 0; i < r.Length; i++)
					{
						r[i] = new DicomFile(filename + "_" + i.ToString("00000"));
						r[i].image = image.GetSubImage(image.GetPlaneLimits(i));
						r[i].image.SetFrameStartTime(0, image.GetFrameStartTime(0));
						r[i].image.SetFrameDuration(0, image.GetFrameDuration(0));
						r[i].DicomHeader.imagepositions[0].Z = i;
						//copy fileinfo
						r[i].DicomHeader.instance_nr = i;
						r[i].DicomHeader.series_nr = DicomHeader.series_nr;
						r[i].Header = Header;
					}
					break;
				case Limits.FRAMES:
					r = new DicomFile[image.Dim.GetDimension(IntLimits.FRAMES)];
					for (int i = 0; i < r.Length; i++)
					{
						r[i] = new DicomFile(filename + "_" + i.ToString("000"));
						r[i].image = image.GetSubImage(image.GetDimensionLimits(i, IntLimits.FRAMES));
						r[i].image.SetFrameStartTime(0, image.GetFrameStartTime(i));
						r[i].image.SetFrameDuration(0, image.GetFrameDuration(i));
						r[i].DicomHeader.slice_start = 0;
						//copy fileinfo
						r[i].DicomHeader.instance_nr = i * image.Planes;
						r[i].DicomHeader.series_nr = DicomHeader.series_nr;
						r[i].Header = Header;
					}
					break;
				case Limits.GATES:
					r = new DicomFile[image.Dim.GetDimension(IntLimits.GATES)];
					for (int i = 0; i < r.Length; i++)
					{
						r[i] = new DicomFile(filename + "_" + i.ToString("000"));
						r[i].image = image.GetSubImage(image.GetDimensionLimits(i, IntLimits.GATES));
						r[i].image.SetFrameStartTime(0, image.GetFrameStartTime(i));
						r[i].image.SetFrameDuration(0, image.GetFrameDuration(i));
						r[i].DicomHeader.gate_number = 0;
						//copy fileinfo
						r[i].DicomHeader.instance_nr = i * image.Planes;
						r[i].DicomHeader.series_nr = DicomHeader.series_nr;
						r[i].Header = Header;
					}
					break;
				default:
					throw new TPCInvalidArgumentsException("Unsupported dimension for splitting:" + dimension);
			}
			return r;
		}

		/// <summary>
		/// Get an image stream for this image
		/// </summary>
		/// <param name="limits">Subimage limits</param>
		/// <returns>Image stream</returns>
		public ImageStream GetStream(IntLimits limits, FileAccess a = FileAccess.ReadWrite)
		{
			//read one or multiple DICOM files 
			if (filename.EndsWith("*"))
			{
				return GetMultiFileStream(filename, limits);
			}
			else
			{
				long offset = this.DicomHeader.stream_data_position + 12;

				ImageStream imgs = new ImageStream(this.filename, this.Header.Dim, limits, (int)offset, this.Header.Datatype, a);
				imgs.SetScaling(this.DicomHeader.scalefactor);
				imgs.StreamOrientation = Header.Orientation;
				return imgs;
			}
		}

		/// <summary>
		/// Get an image stream for this image
		/// </summary>
		/// <returns>Image stream</returns>
		public ImageStream GetStream(FileAccess a = FileAccess.ReadWrite)
		{
			return this.GetStream(this.Header.Dim, a);
		}

		#endregion

		#region Protected/private

		private static readonly IFormatProvider format = System.Globalization.CultureInfo.InvariantCulture;

		/// <summary>
		///     Returns the entire DICOM pixel tacs as array of byte arrays.
		/// </summary>
		/// <remarks>
		///     All non-byte arrays are transcoded into byte arrays. If a DICOM
		///     pixel tacs element is not a DICOM sequence of items, an array
		///     with a single byte array entry will be returned.
		/// </remarks>
		protected static byte[] ToBytesArray(DataElement data)
		{
			byte[] bytesArray;
			if (data.Value[0] is ushort[])
				bytesArray = ByteConvert.ToBytes(
					(ushort[])data.Value[0]);
			else
				bytesArray = (byte[])data.Value[0];
			return bytesArray;
		}

		/// <summary>
		/// Writes group 0x0002 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0002(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;

			// write the empty preamable 
			buf = new byte[128];
			stream.Write(buf, 0, buf.Length);

			// write the signature 
			buf = ToBytesArray("DICM".ToCharArray());
			stream.Write(buf, 0, 4);

			// group 0x0002 
			WriteElement(stream, 0x0002, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			buf = new byte[] { 0x01, 0x00 };
			WriteElement(stream, 0x0002, 0x0001, buf);

			switch (Header.Modality)
			{
				case ImageModality.M_PT:
					//Positron Emission Tomography Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.128".ToCharArray());
					break;
				case ImageModality.M_CT:
					//CT Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.2".ToCharArray());
					break;
				case ImageModality.M_MR:
					//CT Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.4".ToCharArray());
					break;
				default:
					//Nuclear Medicine Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.20".ToCharArray());
					break;
			}
			WriteElement(stream, 0x0002, 0x0002, buf);

			// Media Storage SOP Instance UID = SOP Instance UID
			buf = ToBytesArray(DicomHeader.SOPInstanceUID.ToCharArray());
			WriteElement(stream, 0x0002, 0x0003, buf);

			// transfer syntax 
			if (BitConverter.IsLittleEndian)
			{
				// Raw tacs, Implicit VR, Little Endian 
				buf = ToBytesArray("1.2.840.10008.1.2.1".ToCharArray());
				// JPEG compression, Lossy JPEG
				// 1.2.840.10008.1.2.4.50 
				// 1.2.840.10008.1.2.4.64 
				// JPEG compression, Lossless JPEG
				// 1.2.840.10008.1.2.4.65 
				// 1.2.840.10008.1.2.4.70 
				// Lossless Run Length Encoding 
				// 1.2.840.10008.1.2.5 
			}
			else
			{
				// Raw tacs, Explicit VR, Big Endian 
				buf = ToBytesArray("1.2.840.10008.1.2.2".ToCharArray());
			}
			WriteElement(stream, 0x0002, 0x0010, buf);

			//buf = ToBytesArray("1.2.840.113619.6.218".ToCharArray());
			//WriteElement(stream, 0x0002, 0x0012, buf);
			//string str = String.Empty;
			//while (str.Length + stream.Position - start_pos < 192 - 8)
			//	str += " ";
			//buf = ToBytesArray(str.ToCharArray());
			//WriteElement(stream, 0x0002, 0x0013, buf);
			//buf = toBytesArray("ADW3".ToCharArray());
			//WriteElement(stream, 0x0002, 0x0016, buf);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0008 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0008(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;

			//generate artificial studyID from patient name 
			int studyID = (int)(Math.Abs(Header.PatientName.GetHashCode()) % 100000);

			WriteElement(stream, 0x0008, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			buf = ToBytesArray(string.Format("ISO_IR 100").ToCharArray());
			WriteElement(stream, 0x0008, 0x0005, buf);
			buf = ToBytesArray(string.Format(@"DERIVED\PRIMARY").ToCharArray());
			WriteElement(stream, 0x0008, 0x0008, buf);
			buf = ToBytesArray(string.Format(@"20080326").ToCharArray());
			WriteElement(stream, 0x0008, 0x0012, buf);
			buf = ToBytesArray(string.Format(@"092143.000").ToCharArray());
			WriteElement(stream, 0x0008, 0x0013, buf);
			buf = ToBytesArray(string.Format(@"1.2.840.113619.1.99.120").ToCharArray());
			WriteElement(stream, 0x0008, 0x0014, buf);

			switch (Header.Modality)
			{
				case ImageModality.M_PT:
					//Positron Emission Tomography Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.128".ToCharArray());
					break;
				case ImageModality.M_CT:
					//CT Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.2".ToCharArray());
					break;
				case ImageModality.M_MR:
					//CT Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.4".ToCharArray());
					break;
				default:
					//Nuclear Medicine Image Storage
					buf = ToBytesArray("1.2.840.10008.5.1.4.1.1.20".ToCharArray());
					break;
			}
			WriteElement(stream, 0x0008, 0x0016, buf);
			buf = ToBytesArray(DicomHeader.SOPInstanceUID.ToCharArray());
			WriteElement(stream, 0x0008, 0x0018, buf);

			// Use the DoseStartTime date value as the default for StudyDate.
			System.DateTime dt;
			if (Header.Contains("StudyDate"))
			{
				if (Header["StudyDate"] is System.DateTime)
					dt = (System.DateTime)(Header["StudyDate"]);
				else if (!System.DateTime.TryParse(Header["StudyDate"].ToString(), out dt))
					dt = Header.DoseStartTime;
			}
			else dt = Header.DoseStartTime;
			buf = ToBytesArray(dt.ToString("yyyyMMdd"));
			WriteElement(stream, 0x0008, 0x0020, buf);

			// Use StudyDate as the SeriesDate, if not otherwise specified. Fall back to DoseStartTime, if parsing fails.
			if (Header.Contains("SeriesDate"))
			{
				if (Header["SeriesDate"] is System.DateTime)
					dt = (System.DateTime)(Header["SeriesDate"]);
				else if (!System.DateTime.TryParse(Header["SeriesDate"].ToString(), out dt))
					dt = Header.DoseStartTime;
				buf = ToBytesArray(dt.ToString("yyyyMMdd"));
			}
			WriteElement(stream, 0x0008, 0x0021, buf);

			// Use SeriesDate as the AcquisitionDate, if not otherwise specified. Fall back to DoseStartTime, if parsing fails.
			if (Header.Contains("AcquisitionDate"))
			{
				if (Header["AcquisitionDate"] is System.DateTime)
					dt = (System.DateTime)(Header["AcquisitionDate"]);
				else if (System.DateTime.TryParse(Header["AcquisitionDate"].ToString(), out dt))
					dt = Header.DoseStartTime;
				buf = ToBytesArray(dt.ToString("yyyyMMdd"));
			}
			WriteElement(stream, 0x0008, 0x0022, buf);

			// Use the AcquisitionDate as the ContentDate
			WriteElement(stream, 0x0008, 0x0023, buf);

			// Use DoseStartTime as the default StudyTime, if not otherwise specified. Fall back to DoseStartTime, if parsing fails.
			if (Header.Contains(@"StudyTime"))
			{
				if (Header["StudyTime"] is System.DateTime)
					dt = (System.DateTime)(Header["StudyTime"]);
				else if (!System.DateTime.TryParse(Header["StudyTime"].ToString(), out dt))
					dt = Header.DoseStartTime;
			}
			else dt = Header.DoseStartTime;
			buf = ToBytesArray(dt.ToString("hhmmss.fff").ToCharArray());
			WriteElement(stream, 0x0008, 0x0030, buf);

			// Use StudyTime as the default SeriesTime, if not otherwise specified. Fall back to DoseStartTime, if parsing fails.
			if (Header.Contains(@"SeriesTime"))
			{
				if (Header["SeriesTime"] is System.DateTime)
					dt = (System.DateTime)(Header["SeriesTime"]);
				else if (!System.DateTime.TryParse(Header["SeriesTime"].ToString(), out dt))
					dt = Header.DoseStartTime;
			}
			buf = ToBytesArray(dt.ToString("hhmmss.fff").ToCharArray());
			WriteElement(stream, 0x0008, 0x0031, buf);

			// Use SeriesTime + frameStart as the AcquisitionTime, if not otherwise specified
			System.DateTime acqTime;
			if (Header.Contains(@"AcquisitionTime"))
			{
				if (Header["AcquisitionTime"] is System.DateTime)
					acqTime = (System.DateTime)(Header["AcquisitionTime"]);
				else if (!System.DateTime.TryParse(Header["AcquisitionTime"].ToString(), out acqTime))
					acqTime = dt.AddMilliseconds(Header.GetFrameStart(0));
			}
			else acqTime = dt.AddMilliseconds(Header.GetFrameStart(0));
			buf = ToBytesArray(acqTime.ToString("hhmmss.fff").ToCharArray());
			WriteElement(stream, 0x0008, 0x0032, buf);

			// Write the Acquisition time also to the Content Time
			//buf = ToBytesArray(string.Format(@"000000.000").ToCharArray());
			WriteElement(stream, 0x0008, 0x0033, buf);
			//Accession number, at least ClearCanvas uses this field
			if (Header.Contains("AccessionNumber"))
				buf = ToBytesArray(Header["AccessionNumber"].ToString());
			else
				// Use empty value as default; ClearCanvas expects to have the same accession number for all related studies
				buf = ToBytesArray(String.Empty);
			//buf = ToBytesArray(((int)Math.Abs(filename.Substring(0, filename.Length - 14).GetHashCode()) % 1000000).ToString());
			WriteElement(stream, 0x0008, 0x0050, buf);
			#region write short name for modality
			switch (Header.Modality)
			{
				case ImageModality.M_AS:
					buf = ToBytesArray("AS".ToCharArray());
					break;
				case ImageModality.M_AU:
					buf = ToBytesArray("AU".ToCharArray());
					break;
				case ImageModality.M_BI:
					buf = ToBytesArray("BI".ToCharArray());
					break;
				case ImageModality.M_CD:
					buf = ToBytesArray("CD".ToCharArray());
					break;
				case ImageModality.M_CF:
					buf = ToBytesArray("CF".ToCharArray());
					break;
				case ImageModality.M_CP:
					buf = ToBytesArray("CP".ToCharArray());
					break;
				case ImageModality.M_CR:
					buf = ToBytesArray("CR".ToCharArray());
					break;
				case ImageModality.M_CS:
					buf = ToBytesArray("CS".ToCharArray());
					break;
				case ImageModality.M_CT:
					buf = ToBytesArray("CT".ToCharArray());
					break;
				case ImageModality.M_DD:
					buf = ToBytesArray("DD".ToCharArray());
					break;
				case ImageModality.M_DF:
					buf = ToBytesArray("DF".ToCharArray());
					break;
				case ImageModality.M_DG:
					buf = ToBytesArray("DG".ToCharArray());
					break;
				case ImageModality.M_DM:
					buf = ToBytesArray("DM".ToCharArray());
					break;
				case ImageModality.M_DS:
					buf = ToBytesArray("DS".ToCharArray());
					break;
				case ImageModality.M_DX:
					buf = ToBytesArray("DX".ToCharArray());
					break;
				case ImageModality.M_EC:
					buf = ToBytesArray("EC".ToCharArray());
					break;
				case ImageModality.M_ES:
					buf = ToBytesArray("ES".ToCharArray());
					break;
				case ImageModality.M_FA:
					buf = ToBytesArray("FA".ToCharArray());
					break;
				case ImageModality.M_FS:
					buf = ToBytesArray("FS".ToCharArray());
					break;
				case ImageModality.M_GM:
					buf = ToBytesArray("GM".ToCharArray());
					break;
				case ImageModality.M_HC:
					buf = ToBytesArray("HC".ToCharArray());
					break;
				case ImageModality.M_HD:
					buf = ToBytesArray("HD".ToCharArray());
					break;
				case ImageModality.M_IO:
					buf = ToBytesArray("IO".ToCharArray());
					break;
				case ImageModality.M_LP:
					buf = ToBytesArray("LP".ToCharArray());
					break;
				case ImageModality.M_MA:
					buf = ToBytesArray("MA".ToCharArray());
					break;
				case ImageModality.M_MG:
					buf = ToBytesArray("MG".ToCharArray());
					break;
				case ImageModality.M_MR:
					buf = ToBytesArray("MR".ToCharArray());
					break;
				case ImageModality.M_MS:
					buf = ToBytesArray("MS".ToCharArray());
					break;
				case ImageModality.M_NM:
					buf = ToBytesArray("NM".ToCharArray());
					break;
				case ImageModality.M_OT:
					buf = ToBytesArray("OT".ToCharArray());
					break;
				case ImageModality.M_PT:
					buf = ToBytesArray("PT".ToCharArray());
					break;
				case ImageModality.M_PX:
					buf = ToBytesArray("PX".ToCharArray());
					break;
				case ImageModality.M_RF:
					buf = ToBytesArray("RF".ToCharArray());
					break;
				case ImageModality.M_RG:
					buf = ToBytesArray("RG".ToCharArray());
					break;
				case ImageModality.M_RT:
					buf = ToBytesArray("RT".ToCharArray());
					break;
				case ImageModality.M_SM:
					buf = ToBytesArray("SM".ToCharArray());
					break;
				case ImageModality.M_SR:
					buf = ToBytesArray("SR".ToCharArray());
					break;
				case ImageModality.M_ST:
					buf = ToBytesArray("ST".ToCharArray());
					break;
				case ImageModality.M_TG:
					buf = ToBytesArray("TG".ToCharArray());
					break;
				case ImageModality.M_US:
					buf = ToBytesArray("US".ToCharArray());
					break;
				case ImageModality.M_VF:
					buf = ToBytesArray("VF".ToCharArray());
					break;
				case ImageModality.M_XA:
					buf = ToBytesArray("XA".ToCharArray());
					break;
				case ImageModality.M_XC:
					buf = ToBytesArray("XC".ToCharArray());
					break;
				default:
					buf = ToBytesArray("Unknown".ToCharArray());
					break;
			}
			#endregion
			WriteElement(stream, 0x0008, 0x0060, buf);

			buf = ToBytesArray("Unknown".ToCharArray());
			WriteElement(stream, 0x0008, 0x0070, buf);

			// Use institution name, if available
			if (Header.Contains("InstitutionName"))
			{
				buf = ToBytesArray(Header["InstitutionName"].ToString().ToCharArray());
			}
			WriteElement(stream, 0x0008, 0x0080, buf);

			buf = ToBytesArray("Unknown".ToCharArray());
			WriteElement(stream, 0x0008, 0x0081, buf);
			WriteElement(stream, 0x0008, 0x0090, buf);

			//if(header is DynamicImageHeader) {
			//    str = string.Format("{0:g}",Isotope.GetHalflife((header as DynamicImageHeader).Isotope));
			//} else {
			//    str = "0.00";
			//}
			//buf = ToBytesArray(str.ToCharArray());
			//WriteElement(stream, 0x0008, 0x0100, buf);           

			//station name
			buf = ToBytesArray("Unknown".ToCharArray());
			WriteElement(stream, 0x0008, 0x1010, buf);

			buf = ToBytesArray(Header.Description.ToCharArray());
			WriteElement(stream, 0x0008, 0x1030, buf);

			buf = ToBytesArray(Header.Description.ToCharArray());
			WriteElement(stream, 0x0008, 0x103E, buf);

			// Use department name, if available
			if (Header.Contains("InstitutionalDepartmentName"))
			{
				buf = ToBytesArray(Header["InstitutionalDepartmentName"].ToString().ToCharArray());
			}
			else buf = ToBytesArray("Unknown".ToCharArray());
			WriteElement(stream, 0x0008, 0x1040, buf);

			buf = ToBytesArray("Unknown".ToCharArray());
			WriteElement(stream, 0x0008, 0x1060, buf);
			WriteElement(stream, 0x0008, 0x1070, buf);
			WriteElement(stream, 0x0008, 0x1090, buf);

			buf = ToBytesArray(("library version " + Version.GetVersionInformation()).ToCharArray());
			WriteElement(stream, 0x0008, 0x2111, buf);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0009 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0009(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;

			WriteElement(stream, 0x0009, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			buf = ToBytesArray(string.Format("GEMS_PETD_01").ToCharArray());
			WriteElement(stream, 0x0009, 0x0010, buf);
			buf = ToBytesArray(string.Format("GE Advance").ToCharArray());
			WriteElement(stream, 0x0009, 0x1001, buf);
			buf = ToBytesArray(string.Format("TurkuUni250339-0828").ToCharArray());
			WriteElement(stream, 0x0009, 0x1002, buf);
			buf = ToBytesArray(string.Format("02.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1003, buf);
			buf = ToBytesArray(string.Format("06.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1004, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x1005, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1006, buf);
			buf = ToBytesArray(string.Format("1.2.124.113532.193.143.211.18.20080318.160454.8870403").ToCharArray());
			WriteElement(stream, 0x0009, 0x1007, buf);
			buf = ToBytesArray(string.Format("02.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1008, buf);
			buf = ToBytesArray(string.Format("06.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1009, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205923105.104407").ToCharArray());
			WriteElement(stream, 0x0009, 0x100a, buf);
			buf = ToBytesArray(string.Format("02.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x100b, buf);
			buf = ToBytesArray(string.Format("06.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x100c, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x100d, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x100e, buf);
			buf = ToBytesArray(string.Format("3D AIVOT").ToCharArray());
			WriteElement(stream, 0x0009, 0x100f, buf);
			buf = ToBytesArray(string.Format("Turku PET Centre").ToCharArray());
			WriteElement(stream, 0x0009, 0x1010, buf);
			buf = ToBytesArray(string.Format("Advance").ToCharArray());
			WriteElement(stream, 0x0009, 0x1011, buf);
			buf = ToBytesArray(string.Format("GEMS").ToCharArray());
			WriteElement(stream, 0x0009, 0x1012, buf);
			buf = ToBytesArray(string.Format("Orbital Meatal Line").ToCharArray());
			WriteElement(stream, 0x0009, 0x1014, buf);
			buf = ToBytesArray(string.Format("OM").ToCharArray());
			WriteElement(stream, 0x0009, 0x1015, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1016, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1017, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1018, buf);
			buf = ToBytesArray((Int32)(1));
			WriteElement(stream, 0x0009, 0x1019, buf);
			buf = ToBytesArray((Int32)(1));
			WriteElement(stream, 0x0009, 0x101a, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x101b, buf);
			buf = ToBytesArray((Int32)(1));
			WriteElement(stream, 0x0009, 0x101c, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x101d, buf);
			buf = ToBytesArray((Int32)(1));
			WriteElement(stream, 0x0009, 0x101e, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x101f, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1020, buf);
			buf = ToBytesArray((Int32)(1));
			WriteElement(stream, 0x0009, 0x1021, buf);
			buf = ToBytesArray((Int32)(2));
			WriteElement(stream, 0x0009, 0x1022, buf);
			buf = ToBytesArray((Int32)(34));
			WriteElement(stream, 0x0009, 0x1023, buf);
			buf = ToBytesArray((Int32)(1));
			WriteElement(stream, 0x0009, 0x1024, buf);
			buf = ToBytesArray((Int32)(2));
			WriteElement(stream, 0x0009, 0x1025, buf);
			buf = ToBytesArray((Int32)(11));
			WriteElement(stream, 0x0009, 0x1026, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1027, buf);
			buf = ToBytesArray((Int32)(2));
			WriteElement(stream, 0x0009, 0x1028, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1029, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x102a, buf);
			buf = ToBytesArray((Int32)(55));
			WriteElement(stream, 0x0009, 0x102b, buf);
			buf = ToBytesArray((Int32)(153));
			WriteElement(stream, 0x0009, 0x102c, buf);
			buf = ToBytesArray((Int32)(2));
			WriteElement(stream, 0x0009, 0x102d, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x102e, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1034, buf);
			buf = ToBytesArray((Int32)(0));
			WriteElement(stream, 0x0009, 0x1035, buf);
			buf = ToBytesArray(string.Format("PIB").ToCharArray());
			WriteElement(stream, 0x0009, 0x1036, buf);
			buf = ToBytesArray(string.Format("11C-PIB").ToCharArray());
			WriteElement(stream, 0x0009, 0x1037, buf);
			buf = ToBytesArray(575.0f);
			WriteElement(stream, 0x0009, 0x1038, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x1039, buf);
			buf = ToBytesArray(0.0f);
			WriteElement(stream, 0x0009, 0x103a, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x103b, buf);
			buf = ToBytesArray(3.0f);
			WriteElement(stream, 0x0009, 0x103c, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x103d, buf);

			buf = ToBytesArray(string.Format("11C ").ToCharArray());
			WriteElement(stream, 0x0009, 0x103e, buf);
			//buf = ToBytesArray(1218.0f);
			//WriteElement(stream, 0x0009,0x103f, buf);                              
			buf = ToBytesArray((float)1);
			WriteElement(stream, 0x0009, 0x1040, buf);
			buf = ToBytesArray((UInt32)0);
			WriteElement(stream, 0x0009, 0x104d, buf);
			buf = ToBytesArray((Int32)(-12));
			WriteElement(stream, 0x0009, 0x104e, buf);
			buf = ToBytesArray((Int32)11);
			WriteElement(stream, 0x0009, 0x104f, buf);
			buf = ToBytesArray((Int32)(-6));
			WriteElement(stream, 0x0009, 0x1050, buf);
			buf = ToBytesArray((Int32)6);
			WriteElement(stream, 0x0009, 0x1051, buf);
			buf = ToBytesArray((Int32)56);
			WriteElement(stream, 0x0009, 0x1052, buf);
			buf = ToBytesArray((Int32)800);
			WriteElement(stream, 0x0009, 0x1053, buf);
			//upper energy limit
			buf = ToBytesArray((Int32)900);
			WriteElement(stream, 0x0009, 0x1054, buf);
			//low energy limit
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1055, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199808979.324282").ToCharArray());
			WriteElement(stream, 0x0009, 0x1056, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199777303.246822").ToCharArray());
			WriteElement(stream, 0x0009, 0x1057, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205907722.526955").ToCharArray());
			WriteElement(stream, 0x0009, 0x1058, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199862108.530411").ToCharArray());
			WriteElement(stream, 0x0009, 0x1059, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x105a, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205924350.876255").ToCharArray());
			WriteElement(stream, 0x0009, 0x105c, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205923105.104407").ToCharArray());
			WriteElement(stream, 0x0009, 0x105d, buf);
			buf = ToBytesArray(string.Format("1.2.124.113532.193.143.211.18.20080318.160454.8870403").ToCharArray());
			WriteElement(stream, 0x0009, 0x105e, buf);
			buf = ToBytesArray(string.Format("TurkuUni250339-0828").ToCharArray());
			WriteElement(stream, 0x0009, 0x105f, buf);
			buf = ToBytesArray(string.Format("02.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1060, buf);
			buf = ToBytesArray(string.Format("06.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1061, buf);
			buf = ToBytesArray(string.Format("/dos/RMIIVLKO/SKKWAAQQ/NZMZQYSN/AAAAAAAA").ToCharArray());
			WriteElement(stream, 0x0009, 0x1062, buf);
			buf = ToBytesArray((Int32)25212928);
			WriteElement(stream, 0x0009, 0x1063, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1064, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1065, buf);
			buf = ToBytesArray((float)179);
			WriteElement(stream, 0x0009, 0x1066, buf);
			buf = ToBytesArray((float)-129.5);
			WriteElement(stream, 0x0009, 0x1067, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x1068, buf);
			buf = ToBytesArray((Int32)265);
			WriteElement(stream, 0x0009, 0x1069, buf);
			buf = ToBytesArray((float)-11);
			WriteElement(stream, 0x0009, 0x106a, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x106b, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x106c, buf);
			buf = ToBytesArray((Int32)600);
			WriteElement(stream, 0x0009, 0x106d, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1070, buf);
			buf = ToBytesArray(double.Parse("1,68615e+07"));
			WriteElement(stream, 0x0009, 0x1071, buf);
			buf = ToBytesArray(double.Parse("1,66362e+06"));
			WriteElement(stream, 0x0009, 0x1072, buf);
			buf = ToBytesArray((Int32)1);
			WriteElement(stream, 0x0009, 0x1073, buf);
			buf = ToBytesArray((Int32)207);
			WriteElement(stream, 0x0009, 0x1074, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1075, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1076, buf);
			buf = ToBytesArray((Int32)25212928);
			WriteElement(stream, 0x0009, 0x1077, buf);
			buf = ToBytesArray(string.Format("02.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x1079, buf);
			buf = ToBytesArray(string.Format("06.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x107a, buf);
			buf = ToBytesArray(string.Format("20000101100000.00 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x107b, buf);
			buf = ToBytesArray((Int32)4);
			WriteElement(stream, 0x0009, 0x107c, buf);
			buf = ToBytesArray((Int32)2);
			WriteElement(stream, 0x0009, 0x107d, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x107e, buf);
			buf = ToBytesArray(string.Format(@"0\-0\-0 ").ToCharArray());
			WriteElement(stream, 0x0009, 0x107f, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1080, buf);
			buf = new byte[0];
			WriteElement(stream, 0x0009, 0x1082, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1083, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x1084, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1085, buf);
			buf = ToBytesArray((float)1);
			WriteElement(stream, 0x0009, 0x1086, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1087, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1088, buf);
			buf = ToBytesArray((Int32)1);
			WriteElement(stream, 0x0009, 0x108b, buf);
			buf = ToBytesArray((Int32)7);
			WriteElement(stream, 0x0009, 0x108c, buf);
			buf = ToBytesArray((float)0.096);
			WriteElement(stream, 0x0009, 0x108d, buf);
			buf = ToBytesArray((Int32)3);
			WriteElement(stream, 0x0009, 0x108e, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x1090, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x1091, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x1092, buf);
			buf = ToBytesArray((Int32)5);
			WriteElement(stream, 0x0009, 0x1093, buf);
			buf = ToBytesArray((Int32)8);
			WriteElement(stream, 0x0009, 0x1094, buf);
			buf = ToBytesArray((Int32)3);
			WriteElement(stream, 0x0009, 0x1095, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199862108.530411").ToCharArray());
			WriteElement(stream, 0x0009, 0x1096, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205926124.324699").ToCharArray());
			WriteElement(stream, 0x0009, 0x1097, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1199808979.324282").ToCharArray());
			WriteElement(stream, 0x0009, 0x1098, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205907722.526955").ToCharArray());
			WriteElement(stream, 0x0009, 0x1099, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x109a, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x109b, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205925720.436393").ToCharArray());
			WriteElement(stream, 0x0009, 0x109c, buf);
			buf = ToBytesArray((Int32)3);
			WriteElement(stream, 0x0009, 0x109d, buf);
			buf = ToBytesArray((float)4.6875);
			WriteElement(stream, 0x0009, 0x109e, buf);
			buf = ToBytesArray((Int32)1);
			WriteElement(stream, 0x0009, 0x109f, buf);
			buf = ToBytesArray((float)6.4);
			WriteElement(stream, 0x0009, 0x10a0, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x10a1, buf);
			//axial spacing
			buf = ToBytesArray((float)Header.Siz.SizeZ);
			WriteElement(stream, 0x0009, 0x10a2, buf);
			buf = ToBytesArray((Int32)11);
			WriteElement(stream, 0x0009, 0x10a3, buf);
			buf = ToBytesArray(string.Format("02.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x10a4, buf);
			buf = ToBytesArray(string.Format("06.00").ToCharArray());
			WriteElement(stream, 0x0009, 0x10a5, buf);
			buf = ToBytesArray((Int32)(Header.DimZ + 1 - DicomHeader.instance_nr));
			WriteElement(stream, 0x0009, 0x10a6, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x10a9, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x10aa, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x10ab, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x10ac, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205926481.123992").ToCharArray());
			WriteElement(stream, 0x0009, 0x10ad, buf);
			buf = ToBytesArray(string.Format("1.2.840.113619.2.99.120.1205926032.808059").ToCharArray());
			WriteElement(stream, 0x0009, 0x10ae, buf);
			buf = ToBytesArray((float)0);
			WriteElement(stream, 0x0009, 0x10b9, buf);
			buf = ToBytesArray((Int32)3);
			WriteElement(stream, 0x0009, 0x10be, buf);
			buf = ToBytesArray((float)8.6);
			WriteElement(stream, 0x0009, 0x10bf, buf);
			buf = ToBytesArray((Int32)1);
			WriteElement(stream, 0x0009, 0x10c1, buf);
			buf = ToBytesArray((float)4);
			WriteElement(stream, 0x0009, 0x10c2, buf);
			buf = ToBytesArray((float)1.054591);
			WriteElement(stream, 0x0009, 0x10c6, buf);
			buf = ToBytesArray((Int32)0);
			WriteElement(stream, 0x0009, 0x10c7, buf);
			buf = new byte[0];
			WriteElement(stream, 0x0009, 0x10c8, buf);
			buf = new byte[0];
			WriteElement(stream, 0x0009, 0x10c9, buf);
			buf = new byte[0];
			WriteElement(stream, 0x0009, 0x10ca, buf);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0010 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0010(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;

			WriteElement(stream, 0x0010, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			string dicomFormatName = MakeDicomName(Header.PatientName);
			buf = ToBytesArray(dicomFormatName);
			WriteElement(stream, 0x0010, 0x0010, buf);
			if (ImageHeader.RemoveControlCharacters(Header.PatientID).Trim().Length > 0)
				buf = ToBytesArray(ImageHeader.RemoveControlCharacters(Header.PatientID).Trim().ToCharArray());
			else
				buf = ToBytesArray(String.Empty.ToCharArray());
			WriteElement(stream, 0x0010, 0x0020, buf);

			if (Header.Contains("PatientsBirthDate"))
			{
				System.DateTime dt;
				if (Header["PatientsBirthDate"] is System.DateTime)
				{
					dt = (System.DateTime)(Header["PatientsBirthDate"]);
					buf = ToBytesArray(dt.ToString("yyyyMMdd"));
				}
				else if (System.DateTime.TryParse(Header["PatientsBirthDate"].ToString(), out dt))
				{
					buf = ToBytesArray(dt.ToString("yyyyMMdd"));
				}
			}
			else
				buf = ToBytesArray("20000101".ToCharArray());
			WriteElement(stream, 0x0010, 0x0030, buf);

			if (Header.Contains(@"PatientsBirthTime"))
			{
				System.DateTime dt;
				if (Header["PatientsBirthTime"] is System.DateTime)
				{
					dt = (System.DateTime)(Header["PatientsBirthTime"]);
					buf = ToBytesArray(dt.ToString("hhmmss.fff"));
				}
				else if (System.DateTime.TryParse(Header["PatientsBirthTime"].ToString(), out dt))
				{
					buf = ToBytesArray(dt.ToString("hhmmss.fff"));
				}
			}
			else
				buf = ToBytesArray(string.Format(@"000000.000").ToCharArray());
			WriteElement(stream, 0x0010, 0x0032, buf);

			if (Header.Contains("PatientsSex"))
				buf = ToBytesArray(Header["PatientsSex"].ToString().ToCharArray());
			else
				buf = ToBytesArray("M".ToCharArray());
			WriteElement(stream, 0x0010, 0x0040, buf);
			if (Header.Contains("PatientsAge"))
				buf = ToBytesArray(Header["PatientsAge"].ToString().ToCharArray());
			else
				buf = ToBytesArray("000Y".ToCharArray());
			WriteElement(stream, 0x0010, 0x1010, buf);
			buf = ToBytesArray("1.00".ToCharArray());
			WriteElement(stream, 0x0010, 0x1020, buf);
			if (Header.Contains("PatientsWeight"))
				buf = ToBytesArray(Header["PatientsWeight"].ToString().ToCharArray());
			else
				buf = ToBytesArray("1".ToCharArray());
			WriteElement(stream, 0x0010, 0x1030, buf);
			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0011 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0011(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;

			WriteElement(stream, 0x0011, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			buf = ToBytesArray("UNKNOWN".ToCharArray());
			WriteElement(stream, 0x0011, 0x0010, buf);
			SequenceWriteData sequence = new SequenceWriteData();
			sequence.SetTag(0x0011, 0x1001);
			NestedDataSetWriteData nested = new NestedDataSetWriteData();
			sequence.Add(nested);
			WriteSequenceRecursive(stream, sequence);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0013 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0013(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;

			WriteElement(stream, 0x0013, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			buf = ToBytesArray("GEMS_PETD_01".ToCharArray());
			WriteElement(stream, 0x0013, 0x0010, buf);
			SequenceWriteData sequence = new SequenceWriteData();
			sequence.SetTag(0x0013, 0x1001);
			NestedDataSetWriteData nested = new NestedDataSetWriteData();
			sequence.Add(nested);
			WriteSequenceRecursive(stream, sequence);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0018 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0018(Stream stream)
		{
			byte[] buf;
			long length_pos = 0;
			long start_pos = 0;
			WriteElement(stream, 0x0018, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;
			//radiopharmaceutical name
			if (Header.IsDynamic)
			{
				buf = ToBytesArray(Header.Radiopharma.ToCharArray());
				WriteElement(stream, 0x0018, 0x0031, buf);
			}
			buf = ToBytesArray(Header.SizeZ.ToString(format).ToCharArray());
			WriteElement(stream, 0x0018, 0x0050, buf);
			buf = ToBytesArray(string.Format("TIME").ToCharArray());
			WriteElement(stream, 0x0018, 0x0071, buf);
			buf = ToBytesArray(string.Format("MANU").ToCharArray());
			WriteElement(stream, 0x0018, 0x0073, buf);
			buf = ToBytesArray(string.Format("0").ToCharArray());
			WriteElement(stream, 0x0018, 0x0074, buf);
			buf = ToBytesArray(string.Format("0").ToCharArray());
			WriteElement(stream, 0x0018, 0x0075, buf);
			string[] versionstrings = System.Reflection.Assembly.GetExecutingAssembly().FullName.Split(',');
			buf = ToBytesArray((versionstrings[0] + " " + versionstrings[1].Substring(9)).ToCharArray());
			WriteElement(stream, 0x0018, 0x1020, buf);
			buf = ToBytesArray(string.Format("").ToCharArray());
			WriteElement(stream, 0x0018, 0x1063, buf);

			System.DateTime doseStart;
			doseStart = Header.DoseStartTime;

			buf = ToBytesArray(doseStart.ToString("HHmmss.fff").ToCharArray());
			WriteElement(stream, 0x0018, 0x1072, buf);

			//dosage in Bq or MBq 
			if (Header.Contains("RadionuclideTotalDose"))
				buf = ToBytesArray(Header["RadionuclideTotalDose"].ToString().ToCharArray());
			else
				buf = ToBytesArray(Header.InjectedDose.ToString().ToCharArray());
			WriteElement(stream, 0x0018, 0x1074, buf);

			// Isotope half life
			if (Header.Contains("RadionuclideHalfLife"))
				buf = ToBytesArray(Header["RadionuclideHalfLife"].ToString().ToCharArray());
			else
				buf = ToBytesArray(Isotope.GetHalflife(Header.Isotope).ToString(format).ToCharArray());
			WriteElement(stream, 0x0018, 0x1075, buf);

			//RadiopharmaceuticalStartDatetime
			buf = ToBytesArray(Header.DoseStartTime.ToString("yyyyMMddHHmmss.ffffff").ToCharArray());
			WriteElement(stream, 0x0018, 0x1078, buf);
			buf = new byte[0];
			WriteElement(stream, 0x0018, 0x1081, buf);
			buf = new byte[0];
			WriteElement(stream, 0x0018, 0x1082, buf);
			buf = ToBytesArray(string.Format("0").ToCharArray());
			WriteElement(stream, 0x0018, 0x1083, buf);
			buf = ToBytesArray(string.Format("0").ToCharArray());
			WriteElement(stream, 0x0018, 0x1084, buf);
			//calculate correct value from voxel size and dimension
			buf = ToBytesArray((((int)Math.Round(Header.DimX * Header.SizeX)).ToString() + " ").ToCharArray());
			WriteElement(stream, 0x0018, 0x1100, buf);
			buf = ToBytesArray(string.Format("-0").ToCharArray());
			WriteElement(stream, 0x0018, 0x1120, buf);
			buf = ToBytesArray(string.Format("CYLINDRICAL RING").ToCharArray());
			WriteElement(stream, 0x0018, 0x1147, buf);
			//calculate value from header dimensions
			buf = ToBytesArray((((int)Math.Round(Header.DimX * Header.SizeX)).ToString() + "\\" +
								 ((int)Math.Round(Header.DimZ * Header.SizeZ)).ToString()).ToCharArray());
			WriteElement(stream, 0x0018, 0x1149, buf);
			buf = ToBytesArray(string.Format("NONE").ToCharArray());
			WriteElement(stream, 0x0018, 0x1181, buf);
			//some default string because it is unlikely that this string is read by the software
			buf = ToBytesArray(string.Format(@"Rad:\ hanning\  0.000000 mm\ Ax:\ rectangle\  0.000000 mm").ToCharArray());
			WriteElement(stream, 0x0018, 0x1210, buf);
			buf = ToBytesArray(((int)Header.GetFrameDuration(0)).ToString().ToCharArray());
			WriteElement(stream, 0x0018, 0x1242, buf);
			buf = ToBytesArray(string.Format(DicomHeader.dicomOrientation).ToCharArray());
			WriteElement(stream, 0x0018, 0x5100, buf);
			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0020 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0020(Stream stream)
		{
			byte[] buf;
			string str;
			long length_pos = 0;
			long start_pos = 0;
			WriteElement(stream, 0x0020, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			//generate artificial patientID from patient name
			string patID = Math.Abs((long)Header.PatientName.GetHashCode()).ToString();

			// Study instance UID
			if (Header.Contains("StudyInstanceUID"))
			{
				buf = ToBytesArray(Header["StudyInstanceUID"].ToString().ToCharArray());
			}
			else
			{
				//generated from 1.2.finland(ISO 3166-1).0.0.0.0
				//buf = ToBytesArray(("1.2.246.0.0.0.0").ToCharArray());
				buf = ToBytesArray((DicomHeader.StudyInstanceUID.ToString()).ToCharArray());
			}
			WriteElement(stream, 0x0020, 0x000D, buf);
			// Series instance UID, effects Vinci
			// Study instance UID
			if (Header.Contains("SeriesInstanceUID"))
			{
				buf = ToBytesArray(Header["SeriesInstanceUID"].ToString().ToCharArray());
			}
			else buf = ToBytesArray((DicomHeader.StudyInstanceUID.ToString() + "." + this.DicomHeader.series_nr).ToCharArray());
			WriteElement(stream, 0x0020, 0x000E, buf);
			// StudyID
			buf = ToBytesArray(patID.ToString().ToCharArray());
			WriteElement(stream, 0x0020, 0x0010, buf);
			//calculate own short hascode, may not be unique within filenames
			buf = ToBytesArray(patID.ToString().ToCharArray());
			WriteElement(stream, 0x0020, 0x0011, buf);

			//modified because Vinci does not convert properly:
			//Vinci sorts files according to ASCII codes, which results
			//missorted slices 
			str = DicomHeader.instance_nr.ToString();
			str.PadLeft(2, '0');
			if (str.Length > 2) str = str.Remove(0, str.Length - 2);
			buf = ToBytesArray(str.ToCharArray());
			WriteElement(stream, 0x0020, 0x0013, buf);

			// image pos patient 
			try
			{
				buf = ToBytesArray((DicomHeader.imagepositions[0].X.ToString("E8", format) + "\\" +
							 DicomHeader.imagepositions[0].Y.ToString("E8", format) + "\\" +
							 DicomHeader.imagepositions[0].Z.ToString("E8", format)).ToCharArray());
			}
			catch
			{
				buf = ToBytesArray((
					  (-Header.Siz.SizeX / 2.0f).ToString("E8", format) + "\\" +
					  (-Header.Siz.SizeY / 2.0f).ToString("E8", format) + "\\" +
					  (Header.Siz.SizeZ * DicomHeader.instance_nr).ToString("E8", format)
					).ToCharArray());
			}
			WriteElement(stream, 0x0020, 0x0032, buf);

			// image orient patient 
			try
			{
				buf = ToBytesArray(
					(DicomHeader.ImageOrientations[0].ToString("E8", format) + "\\" +
					DicomHeader.ImageOrientations[1].ToString("E8", format) + "\\" +
					DicomHeader.ImageOrientations[2].ToString("E8", format) + "\\" +
					DicomHeader.ImageOrientations[3].ToString("E8", format) + "\\" +
					DicomHeader.ImageOrientations[4].ToString("E8", format) + "\\" +
					DicomHeader.ImageOrientations[5].ToString("E8", format)).ToCharArray());
			}
			catch
			{
				buf = ToBytesArray("1.00\\0.00\\0.00\\0.00\\1.00\\0.00 ".ToCharArray());
			}
			WriteElement(stream, 0x0020, 0x0037, buf);
			// FrameOfReferenceUID
			buf = ToBytesArray(("777.777.0.0.0.0.123").ToCharArray());
			//buf = ToBytesArray(string.Format("777.777.0.0.0.{0:D}.{1:D}", dicomheader.instance_nr, dicomheader.series_nr).ToCharArray());
			WriteElement(stream, 0x0020, 0x0052, buf);

			buf = ToBytesArray("Orbital Meatal Line".ToCharArray());
			WriteElement(stream, 0x0020, 0x1040, buf);
			//slice location in z-axis
			try
			{
				buf = ToBytesArray((DicomHeader.imagepositions[0].Z.ToString("E8", format)).ToCharArray());
			}
			catch
			{
				buf = ToBytesArray((DicomHeader.instance_nr * Header.Siz.SizeZ).ToString("E8", format).ToCharArray());
			}
			WriteElement(stream, 0x0020, 0x1041, buf);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0028 of DICOM header
		/// </summary>
		/// <param name="stream">stream</param>
		protected virtual void Write0028(Stream stream)
		{
			byte[] buf;
			string str;

			long length_pos = 0;
			long start_pos = 0;
			WriteElement(stream, 0x0028, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;


			//samples per pixel
			buf = BitConverter.GetBytes((UInt16)1);
			WriteElement(stream, 0x0028, 0x0002, buf);
			//Photometric Interpretation. Options MONOCHROME1, MONOCHROME2, PALETTE COLOR and RBG
			buf = ToBytesArray("MONOCHROME2".ToCharArray());
			WriteElement(stream, 0x0028, 0x0004, buf);

			//number of frames
			if (Header.Dim.Frames > 1)
			{
				str = image.Frames.ToString();
				str = str.PadLeft(2, '0');
				if (str.Length > 2) str.Substring(str.Length - 2, 2);
				buf = ToBytesArray(str.ToCharArray());
				WriteElement(stream, 0x0028, 0x0008, buf);
			}
			// Do not write the number of frames in static images
			/* else
			{
				buf = ToBytesArray("01".ToCharArray());
			}*/
			//WriteElement(stream, 0x0028, 0x0008, buf);

			#region frame increment pointer not written currently
			/*
            switch (header.modality)
            {
                case ImageModality.M_PT:
                    switch (this.dicomheader.acquisitiontype)
                    {
                        case AcquisitionType.ACQUISITION_DYNAMIC:
                            buf = new byte[]{0x00, 0x54, 0x00, 0x80, 0x00, 0x54, 0x01, 0x00};
                            break;
                        case AcquisitionType.ACQUISITION_GATED:
                        case AcquisitionType.ACQUISITION_GSPECT:
                        case AcquisitionType.ACQUISITION_STATIC:
                        case AcquisitionType.ACQUISITION_TOMO:
                        case AcquisitionType.ACQUISITION_UNKNOWN:
                        default:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00 };
                            break;
                    }
                    break;
                case ImageModality.M_MR:
                default: // default to NM modality 
                    switch (this.dicomheader.acquisitiontype)
                    {
                        case AcquisitionType.ACQUISITION_DYNAMIC:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00 };
                            break;
                        case AcquisitionType.ACQUISITION_TOMO:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x10, 
                                               0x00, 0x54, 0x00, 0x20, 
                                               0x00, 0x54, 0x00, 0x30, 
                                               0x00, 0x54, 0x01, 0x00 }; 
                            break;
                        case AcquisitionType.ACQUISITION_GATED:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x10, 
                                               0x00, 0x54, 0x00, 0x20, 
                                               0x00, 0x54, 0x00, 0x60, 
                                               0x00, 0x54, 0x00, 0x70 }; 
                            break;
                        case AcquisitionType.ACQUISITION_GSPECT:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x60, 
                                               0x00, 0x54, 0x00, 0x70, 
                                               0x00, 0x54, 0x00, 0x80 };
                            break;
                        case AcquisitionType.ACQUISITION_UNKNOWN:
                        case AcquisitionType.ACQUISITION_STATIC:
                            buf = new byte[] { 0x00, 0x54, 0x00, 0x10, 0x00, 0x54, 0x00, 0x20 };
                            break;
                    }
                    break;
            }
            WriteElement(stream, 0x0028, 0x0009, buf);
             */
			#endregion

			//rows
			buf = BitConverter.GetBytes((UInt16)Header.Dim.Height);
			WriteElement(stream, 0x0028, 0x0010, buf);
			//columns
			buf = BitConverter.GetBytes((UInt16)Header.Dim.Width);
			WriteElement(stream, 0x0028, 0x0011, buf);
			//planes
			buf = BitConverter.GetBytes((UInt16)Header.Dim.Planes);
			WriteElement(stream, 0x0028, 0x0012, buf);
			// pixel spacing, (required by Vinci 2.37.0)
			// PixelSpacing : RowSpacing '\' ColumnSpacing 
			str = (Header.SizeY.ToString(format) + "\\" + Header.SizeX.ToString("E8", format));
			if (str.Length % 2 != 0) str += " ";
			buf = ToBytesArray(str.ToCharArray());
			WriteElement(stream, 0x0028, 0x0030, buf);
			//Corrections applied to the image
			buf = ToBytesArray("UNKNOWN".ToCharArray());
			WriteElement(stream, 0x0028, 0x0051, buf);

			switch (Header.Datatype)
			{
				case DataType.BIT8_U:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)8));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)8));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)7));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
					break;
				case DataType.BIT8_S:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)8));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)8));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)7));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
					break;
				case DataType.SUNI2:
				case DataType.BIT16_S:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)16));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)16));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)15));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
					break;
				case DataType.BIT16_U:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)16));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)16));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)15));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
					break;
				case DataType.BIT32_S:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)32));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)32));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)31));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
					break;
				case DataType.BIT32_U:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)32));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)32));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)31));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
					break;
				case DataType.BIT64_S:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)64));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)64));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)63));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)1));
					break;
				case DataType.BIT64_U:
					//bits allocated, bits stored, high bit
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)64));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)64));
					WriteElement(stream, 0x0028, 0x0102, BitConverter.GetBytes((UInt16)63));
					WriteElement(stream, 0x0028, 0x0103, ToBytesArray((UInt16)0));
					break;
				case DataType.FLT32:
					//bits allocated, bits stored
					WriteElement(stream, 0x0028, 0x0100, BitConverter.GetBytes((UInt16)32));
					WriteElement(stream, 0x0028, 0x0101, BitConverter.GetBytes((UInt16)32));
					break;
				default:
					throw new TPCDicomFileException("Unsupported write tacs type:" + Header.Datatype);
			}

			//buf = ToBytesArray(Convert.ToInt16(dicomheader.imageMinimum));
			//WriteElement(stream, 0x0028, 0x0106, buf);
			//buf = ToBytesArray(Convert.ToInt16(dicomheader.imageMaximum));
			//WriteElement(stream, 0x0028, 0x0107, buf);

			//one or more of these fields are critical for CortexID in ADW workstation

			// Window size; set according to float values
			float windowCenter = 0.0f;
			//buf = ToBytesArray("730.30639648438 ".ToCharArray());
			buf = ToBytesArray(windowCenter.ToString("E8", format).ToCharArray());
			WriteElement(stream, 0x0028, 0x1050, buf);

			float windowWidth = Single.MaxValue;
			//buf = ToBytesArray("2104.6850585938 ".ToCharArray());
			buf = ToBytesArray(windowWidth.ToString("E8", format).ToCharArray());
			WriteElement(stream, 0x0028, 0x1051, buf);

			// Scaling factors
			str = DicomHeader.scaleintercept.ToString("E8", format);
			if (str.Length % 2 != 0) str += " ";
			buf = ToBytesArray(str.ToCharArray());
			WriteElement(stream, 0x0028, 0x1052, buf);
			str = DicomHeader.scalefactor.ToString("E8", format);
			if (str.Length % 2 != 0) str += " ";
			buf = ToBytesArray(str.ToCharArray());
			WriteElement(stream, 0x0028, 0x1053, buf);

			// Unit
			buf = ToBytesArray("US".ToCharArray());
			WriteElement(stream, 0x0028, 0x1054, buf);

			// Lossy compression used (00 == not, 01 == compression used)
			buf = ToBytesArray("00".ToCharArray());
			WriteElement(stream, 0x0028, 0x2110, buf);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x0054 of DICOM header
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected virtual void Write0054(Stream stream)
		{
			byte[] buf;
			string str;
			long length_pos = 0;
			long start_pos = 0;
			SequenceWriteData sequence;
			NestedDataSetWriteData nested;
			SequenceWriteData sequence2;
			NestedDataSetWriteData nested2;

			WriteElement(stream, 0x0054, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			sequence = new SequenceWriteData();
			sequence.SetTag(0x0054, 0x0013);
			nested = new NestedDataSetWriteData();
			buf = ToBytesArray("000000000000300".ToCharArray());
			nested.Add(CreateDataElement(0x0054, 0x0014, buf));
			buf = ToBytesArray("000000000000650".ToCharArray());
			nested.Add(CreateDataElement(0x0054, 0x0015, buf));
			sequence.Add(nested);
			WriteSequenceRecursive(stream, sequence);

			sequence = new SequenceWriteData();
			sequence.SetTag(0x0054, 0x0016);
			nested = new NestedDataSetWriteData();
			buf = ToBytesArray("UNKNOWN".ToCharArray());
			nested.Add(CreateDataElement(0x0018, 0x0031, buf));
			buf = ToBytesArray("0".ToCharArray());
			nested.Add(CreateDataElement(0x0018, 0x1071, buf));
			System.DateTime dosetime = Header.DoseStartTime;
			str = dosetime.ToString("HHmmss.fff");
			buf = ToBytesArray(str.ToCharArray());
			nested.Add(CreateDataElement(0x0018, 0x1072, buf));
			str = (Header.InjectedDose * 1000000).ToString("E8", format);
			buf = ToBytesArray(str.ToCharArray());
			nested.Add(CreateDataElement(0x0018, 0x1074, buf));

			// Isotope half life
			if (Header.Contains("RadionuclideHalfLife"))
			{
				buf = ToBytesArray(Header["RadionuclideHalfLife"].ToString().ToCharArray());
				nested.Add(CreateDataElement(0x0018, 0x1075, buf));
			}
			else if (Header.Contains("HalfLife"))
			{
				buf = ToBytesArray(Header["HalfLife"].ToString().ToCharArray());
				nested.Add(CreateDataElement(0x0018, 0x1075, buf));
			}
			else
			{
				buf = ToBytesArray(Isotope.GetHalflife(Header.Isotope).ToString(format).ToCharArray());
				nested.Add(CreateDataElement(0x0018, 0x1075, buf));
			}
			buf = ToBytesArray("1".ToCharArray());
			nested.Add(CreateDataElement(0x0018, 0x1076, buf));
			sequence2 = new SequenceWriteData();
			sequence2.SetTag(0x0054, 0x0300);
			nested2 = new NestedDataSetWriteData();
			buf = ToBytesArray("Y-X0000".ToCharArray());
			nested2.Add(CreateDataElement(0x0008, 0x0100, buf));
			buf = ToBytesArray("00SDM".ToCharArray());
			nested2.Add(CreateDataElement(0x0008, 0x0102, buf));
			buf = ToBytesArray(Header.Isotope.ToString());
			nested2.Add(CreateDataElement(0x0008, 0x0104, buf));
			sequence2.Add(nested2);
			nested.Add(sequence2);
			sequence2 = new SequenceWriteData();
			sequence2.SetTag(0x0054, 0x0304);
			nested2 = new NestedDataSetWriteData();
			buf = ToBytesArray("00SDM".ToCharArray());
			nested2.Add(CreateDataElement(0x0008, 0x0102, buf));
			buf = ToBytesArray("UNKNOWN".ToCharArray());
			nested2.Add(CreateDataElement(0x0008, 0x0104, buf));
			sequence2.Add(nested2);
			nested.Add(sequence2);
			sequence.Add(nested);
			WriteSequenceRecursive(stream, sequence);

			buf = ToBytesArray((UInt16)Header.DimZ);
			WriteElement(stream, 0x0054, 0x0081, buf);

			// Write the number of frames only if the image is dynamic
			// Dicom standard expects the tag (0054,0101) to be null if image is static
			if (Header.IsDynamic)
			{
				buf = ToBytesArray((UInt16)Header.Dim.GetDimension(IntLimits.FRAMES));
				WriteElement(stream, 0x0054, 0x0101, buf);
			}

			buf = ToBytesArray("NONE".ToCharArray());
			WriteElement(stream, 0x0054, 0x0202, buf);
			sequence = new SequenceWriteData();
			sequence.SetTag(0x0054, 0x0410);
			nested = new NestedDataSetWriteData();
			sequence.Add(nested);
			WriteSequenceRecursive(stream, sequence);
			sequence = new SequenceWriteData();
			sequence.SetTag(0x0054, 0x0414);
			nested = new NestedDataSetWriteData();
			sequence.Add(nested);
			WriteSequenceRecursive(stream, sequence);

			// In addition to this field, see also (0054,0101).
			if (Header.IsDynamic)
				buf = ToBytesArray(@"DYNAMIC\IMAGE".ToCharArray());
			else
				buf = ToBytesArray(@"STATIC\IMAGE".ToCharArray());
			WriteElement(stream, 0x0054, 0x1000, buf);

			// Override the header.Dataunit with a string from general header to enable
			// correct units for parametric images.
			if (Header.Contains("Units"))
				buf = ToBytesArray(Header["Units"].ToString().ToCharArray());
			else
				buf = ToBytesArray(DataUnitToDicom(Header.Dataunit).ToCharArray());
			WriteElement(stream, 0x0054, 0x1001, buf);
			buf = ToBytesArray("UNKNOWN".ToCharArray());
			WriteElement(stream, 0x0054, 0x1002, buf);
			buf = ToBytesArray("UNKNOWN".ToCharArray());
			WriteElement(stream, 0x0054, 0x1100, buf);

			// Set the decay correction, if it is specified.
			if (Header.Contains("DecayCorrection"))
				buf = ToBytesArray(Header["DecayCorrection"].ToString().ToCharArray());
			else
				buf = ToBytesArray("NONE".ToCharArray());
			WriteElement(stream, 0x0054, 0x1102, buf);

			buf = ToBytesArray("UNKNOWN".ToCharArray());
			WriteElement(stream, 0x0054, 0x1103, buf);
			buf = ToBytesArray("0".ToCharArray());
			WriteElement(stream, 0x0054, 0x1104, buf);
			buf = ToBytesArray("UNKNOWN".ToCharArray());
			WriteElement(stream, 0x0054, 0x1105, buf);
			//buf = ToBytesArray(@"1\2".ToCharArray());
			//WriteElement(stream, 0x0054, 0x1201, buf);
			//buf = ToBytesArray("1".ToCharArray());
			//WriteElement(stream, 0x0054, 0x1202, buf);
			//buf = ToBytesArray("12".ToCharArray());
			//WriteElement(stream, 0x0054, 0x1210, buf);
			buf = ToBytesArray(Header.GetFrameStart(0).ToString().ToCharArray());
			WriteElement(stream, 0x0054, 0x1300, buf);
			buf = ToBytesArray("1".ToCharArray());
			WriteElement(stream, 0x0054, 0x1320, buf);
			if (Header.Contains("DecayFactor"))
				buf = ToBytesArray(Convert.ToSingle(Header["DecayFactor"]).ToString(System.Globalization.CultureInfo.InvariantCulture).ToCharArray());
			else
				buf = ToBytesArray("1.0".ToCharArray());
			WriteElement(stream, 0x0054, 0x1321, buf);
			buf = ToBytesArray("1.0".ToCharArray());
			WriteElement(stream, 0x0054, 0x1322, buf);
			buf = ToBytesArray("1.0".ToCharArray());
			WriteElement(stream, 0x0054, 0x1324, buf);
			buf = ToBytesArray((UInt16)DicomHeader.instance_nr);
			WriteElement(stream, 0x0054, 0x1330, buf);

			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
		}

		/// <summary>
		/// Writes group 0x7FE0 of DICOM header (pixel data related, including the data itself)
		/// </summary>
		/// <param name="stream">writable stream</param>
		protected void Write7FE0(Stream stream)
		{
			Tag tag;

			long length_pos = 0;
			long start_pos = 0;

			WriteElement(stream, 0x7FE0, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
			length_pos = stream.Position - 4;
			start_pos = stream.Position;

			//tacs
			tag = new Tag(0x7FE0, 0x0010);
			tag.TransferSyntax = TransferSyntax.Default;
			BinaryWriter binarystream = new BinaryWriter(stream);
			int bytes = image.DataLength * ImageFile.Type2Bytes(Header.Datatype);
			//write tag
			tag.SaveTo(stream);
			// "OW"
			binarystream.Write("OW".ToCharArray(), 0, 2);
			binarystream.Write(new byte[2] { 0x00, 0x00 }, 0, 2);
			//tacs element length
			binarystream.Write(BitConverter.GetBytes((UInt32)bytes));

			WritePixeldata(binarystream, tag.TransferSyntax);
			WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
			binarystream.Flush();
		}

		/// <summary>
		/// Write pixel data to stream using a specified transfersyntax
		/// </summary>
		/// <param name="binarystream"></param>
		/// <param name="ts"></param>
		protected void WritePixeldata(BinaryWriter binarystream, TransferSyntax ts)
		{
			byte[] buf;

			ValueWriter vw;
			switch (Header.Datatype)
			{
				case DataType.BIT8_U:
					vw = delegate(float f)
					{
						byte uint8value = Convert.ToByte(f);
						buf = new byte[] { uint8value };
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT8_S:
					vw = delegate(float f)
					{
						sbyte int8value = Convert.ToSByte(f);
						buf = new byte[] { (byte)int8value };
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT16_S:
					vw = delegate(float f)
					{
						Int16 int16value = Convert.ToInt16(f);
						buf = BitConverter.GetBytes(int16value);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT16_U:
					vw = delegate(float f)
					{
						UInt16 uint16value = Convert.ToUInt16(f);
						buf = BitConverter.GetBytes(uint16value);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT32_S:
					vw = delegate(float f)
					{
						Int32 int32value = Convert.ToInt32(f);
						buf = BitConverter.GetBytes(int32value);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT32_U:
					vw = delegate(float f)
					{
						UInt32 uint32value = Convert.ToUInt32(f);
						buf = BitConverter.GetBytes(uint32value);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT64_S:
					vw = delegate(float f)
					{
						Int64 int64value = Convert.ToInt64(f);
						buf = BitConverter.GetBytes(int64value);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.BIT64_U:
					vw = delegate(float f)
					{
						UInt64 uint64value = Convert.ToUInt64(f);
						buf = BitConverter.GetBytes(uint64value);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				case DataType.FLT32:
					vw = delegate(float f)
					{
						buf = BitConverter.GetBytes(f);
						binarystream.Write(ts.CorrectByteOrdering(buf));
					};
					break;
				default:
					throw new TPCDicomFileException("Unsupported write tacs type:" + Header.Datatype);
			}

			// If scale  is 0, all data values in this image are constant value (= dicomheader.scaleintercept)
			// Voxel values don't matter, write zeros
			if (DicomHeader.scalefactor == 0.0f)
			{
				for (int j = 0; j < image.DataLength; j++)
					vw(0.0f);
			}
			// In other cases, scale the voxel values
			else if (DicomHeader.scaleintercept == 0.0f)
			{
				float scale = DicomHeader.scalefactor;
				for (int j = 0; j < image.DataLength; j++)
					vw(image[j] / scale);
			}
			else
			{
				float scale = DicomHeader.scalefactor;
				float intercept = DicomHeader.scaleintercept;
				for (int j = 0; j < image.DataLength; j++)
					vw((image[j] - intercept) / scale);
			}
		}

		/// <summary>
		/// Sorting according to:
		/// group name
		/// frame start tacs if files have dynamic images
		/// plane z positions
		/// </summary>
		/// <param name="x">first evaluated file</param>
		/// <param name="y">second evaluated file</param>
		/// <returns>{-1, 0, 1} according to order {x &lt; y, x == y, x > y}</returns>
		public int CompareTo(DicomFile y)
		{
			return DicomHeader.CompareTo(y.DicomHeader);
		}

		/// <summary>
		/// Calculates optimal scale factor for curent datatype.
		/// </summary>
		private void ResolveScaleFactor(out float slope, out float intercept)
		{
			slope = 1.0f;
			intercept = 0.0f;

			ImageFile.DataType dt = Header.Datatype;
			if (dt != DataType.FLT32 && dt != DataType.FLT64)
			{
				float[] minmax = minmax = image.GetMinMax();
				float min = minmax[0];
				float max = minmax[1];

				// if min == max, all image values are the same
				if (min == max)
				{
					slope = 0.0f;
					intercept = min;
				}
				// else, calculate optimal datatype use
				else
				{
					double typeMax = MaxValue(dt);
					double typeMin = MinValue(dt);

					// Calculate slope and intercept values
					// Scale to fill the available datatype range
					slope = (float)((max - min) / (typeMax - typeMin));
					intercept = (float)((min - max * typeMin / typeMax) / (1.0 - typeMin / typeMax));
				}
			}
		}

		/// <summary>
		/// Implementation of ImageFile interface. Reads a single DICOM file.
		/// </summary>
		/// <exception cref="TPCDicomFileException">if file was not opened properly</exception>
		/// <exception cref="TPCUnrecognizedFileFormatException">if other error occurred</exception>
		private void ReadSingle()
		{
			//read header information
			this.Header = ReadHeader();

			//read plane image
			image = new Image(Header.DimX, Header.DimY, 1);
			this.image.IsDynamic = Header.IsDynamic;

			GetPixelData(ref image, 0, filename, new IntLimits(Header.DimX, Header.DimY), Header.Datatype, DicomHeader.stream_data_position, DicomHeader.scalefactor, DicomHeader.scaleintercept);

			image.SetFrameStartTime(0, DicomHeader.frame_start_time);
			image.SetFrameDuration(0, DicomHeader.frame_duration);
		}

		/// <summary>
		/// Convert a TPClib data unit to a Dicom data unit string
		/// </summary>
		/// <param name="u">Data unit</param>
		/// <returns>Dicom data unit string</returns>
		private static string DataUnitToDicom(Unit u)
		{
			if (u == DataUnit.BecquerelsPerMillilitre) return @"BQML";
			if (u == DataUnit.Count) return @"CNTS";
			if (u == DataUnit.OnePerSecond) return @"CPS";
			if (u == DataUnit.MillilitresPerGramPerMinutes) return @"MLMING";
			if (u == DataUnit.MillilitresPerMillilitreMinutes) return @"MLMINML";
			if (u == DataUnit.MillilitresPerMillilitre) return @"MLML";
			if (u == DataUnit.MolesPerLitre) return @"UMOLML";
			return u.UnitString;
		}

		/// <summary>
		/// Convert Dicom data unit to a TPClib enumeration
		/// </summary>
		/// <param name="s">Dicom data unit</param>
		/// <returns>Corresponding TPClib data unit</returns>
		private static DataUnit DicomToDataUnit(string s)
		{
			return DataUnit.Parse(s.Trim(new char[] { '\0', ' ' }));
		}

		/// <summary>
		/// Create a dicom format person name.
		/// </summary>
		/// <param name="p">Name to convert</param>
		/// <returns>Dicom name</returns>
		public static string MakeDicomName(PersonName p)
		{
			return p.FullName.Replace(' ', '^');
		}

		#endregion
	}
}
