/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;
using System.Collections.Generic;

namespace TPClib.Image
{
	/// <summary>
	/// Class representing Ecat7 file.
	/// </summary>
	public class Ecat7File : ImageFile, IStreamReader
	{
		#region Header classes

		/// <summary>
		/// General upper class for all subheaders.
		/// </summary>
		public class Ecat7_subheader : IComparable
		{
			/// <summary>
			/// Total duration of current frame (in msec)
			/// </summary>
			public int frame_duration = 0;
			/// <summary>
			/// Frame start time (offset from first frame, in msec)
			/// </summary>
			public int frame_start_time = 0;
			/// <summary> 
			/// If tacs type is integer, this factor is used to 
			/// convert to float values
			/// </summary>
			public float scale_factor = 0.0f;

			/// <summary>
			/// Compare method for sorting headers by frame start time
			/// </summary>
			/// <param name="obj">Object to compare</param>
			/// <returns>-1 if this precedes obj, 0 if equal, 1 if this follows obj</returns>
			public int CompareTo(object obj)
			{
				if (obj is Ecat7_subheader)
				{
					return this.frame_start_time.CompareTo((obj as Ecat7_subheader).frame_start_time);
				}
				else throw new ArgumentException("Not a subheader");
			}
		}

		/// <summary>
		/// Subheader for Matrix Image Files
		/// </summary>
		public class Ecat7_imageheader : Ecat7_subheader
		{
			/// <summary>
			/// Enumerated type (short)
			/// </summary>
			public data_type_e data_type = data_type_e.Unknown_Matrix_Data_Type;
			/// <summary>
			/// Number of dimensions
			/// </summary>
			public short num_dimensions = 0;
			/// <summary>
			/// Dimension along x axis
			/// </summary>
			public short x_dimension = 0;
			/// <summary>
			/// Dimension along y axis 
			/// </summary>
			public short y_dimension = 0;
			/// <summary>
			/// Dimension along z axis 
			/// </summary>
			public short z_dimension = 0;
			/// <summary>
			/// Offset in x axis for recon target (in cm) 
			/// </summary>
			public float x_offset = 0.0f;
			/// <summary>
			/// Offset in y axis for recon target (in cm.) 
			/// </summary>
			public float y_offset = 0.0f;
			/// <summary>
			/// Offset in z axis for recon target (in cm.) 
			/// </summary>
			public float z_offset = 0.0f;
			/// <summary>
			/// Reconstruction magnification factor 
			/// </summary>
			public float recon_zoom = 0.0f;
			/// <summary>
			/// Image minimum pixel value 
			/// </summary>
			public short image_min = 0;
			/// <summary>
			/// Image maximum pixel value 
			/// </summary>
			public short image_max = 0;
			/// <summary>
			/// X dimension pixel size (cm) 
			/// </summary>
			public float x_pixel_size = 0.0f;
			/// <summary>
			/// Y dimension pixel size (cm) 
			/// </summary>
			public float y_pixel_size = 0.0f;
			/// <summary>
			/// Z dimension pixel size (cm) 
			/// </summary>
			public float z_pixel_size = 0.0f;
			/// <summary>
			/// filter code (short)
			/// </summary>
			public filter_code_e filter_code = filter_code_e.all_pass;
			/// <summary>
			/// Resolution in the x dimension (in cm) 
			/// </summary>
			public float x_resolution = 0.0f;
			/// <summary>
			/// Resolution in the y dimension (in cm) 
			/// </summary>
			public float y_resolution = 0.0f;
			/// <summary>
			/// Resolution in the z dimension (in cm) 
			/// </summary>
			public float z_resolution = 0.0f;
			/// <summary>
			/// Number R elements from sinogram 
			/// </summary>
			public float num_r_elements = 0.0f;
			/// <summary>
			/// Nr of angles from sinogram 
			/// </summary>
			public float num_angles = 0.0f;
			/// <summary>
			/// Rotation in the xy plane (in degrees). Use righthand coordinate system for rotation angle sign. 
			/// </summary>
			public float z_rotation_angle = 0.0f;
			/// <summary>
			/// Isotope decay compensation applied to tacs 
			/// </summary>
			public float decay_corr_fctr = 0.0f;
			/// <summary>
			/// Bit mask (0=Not Processed, 1=Normalized, 2=Measured Attenuation Correction,
			/// 4=Calculated Attenuation Correction, 8=X smoothing, 16=Y smoothing, 32=Z smoothing,
			/// 64=2D scatter correction, 128=3D scatter correction, 256=Arc correction, 512=Decay
			/// correction, 1024=Online compression) 
			/// </summary>
			public uint processing_code = 0;
			/// <summary>
			/// Gate duration (in msec) 
			/// </summary>
			public int gate_duration = 0;
			/// <summary>
			/// R wave offset (For phase sliced studies, average, in msec) 
			/// </summary>
			public int r_wave_offset = 0;
			/// <summary>
			/// Number of accepted beats for this gate 
			/// </summary>
			public int num_accepted_beats = 0;
			/// <summary>
			/// Cutoff frequency 
			/// </summary>
			public float filter_cutoff_frequency = 0.0f;
			/// <summary>
			/// Do not use 
			/// </summary>
			public float filter_resolution = 0.0f;
			/// <summary>
			/// Do not use 
			/// </summary>
			public float filter_ramp_slope = 0.0f;
			/// <summary>
			/// Do not use 
			/// </summary>
			public short filter_order = 0;
			/// <summary>
			/// Do not use 
			/// </summary>
			public float filter_scatter_fraction = 0.0f;
			/// <summary>
			/// Do not use 
			/// </summary>
			public float filter_scatter_slope = 0.0f;
			/// <summary>
			/// Free format ASCII 
			/// </summary>
			public char[] annotation = new char[40];
			/// <summary>
			/// Matrix transformation element (1,1) 
			/// </summary>
			public float mt_1_1 = 0.0f;
			/// <summary>
			/// Matrix transformation element (1,2) 
			/// </summary>
			public float mt_1_2 = 0.0f;
			/// <summary>
			/// Matrix transformation element (1,3) 
			/// </summary>
			public float mt_1_3 = 0.0f;
			/// <summary>
			/// Matrix transformation element (2,1) 
			/// </summary>
			public float mt_2_1 = 0.0f;
			/// <summary>
			/// Matrix transformation element (2,2) 
			/// </summary>
			public float mt_2_2 = 0.0f;
			/// <summary>
			/// Matrix transformation element (2,3) 
			/// </summary>
			public float mt_2_3 = 0.0f;
			/// <summary>
			/// Matrix transformation element (3,1) 
			/// </summary>
			public float mt_3_1 = 0.0f;
			/// <summary>
			/// Matrix transformation element (3,2) 
			/// </summary>
			public float mt_3_2 = 0.0f;
			/// <summary>
			/// Matrix transformation element (3,3) 
			/// </summary>
			public float mt_3_3 = 0.0f;
			/// <summary>
			/// Reconstruction filter cutoff 
			/// </summary>
			public float rfilter_cutoff = 0.0f;
			/// <summary>
			/// Reconstruction filter resolution  
			/// </summary>
			public float rfilter_resolution = 0.0f;
			/// <summary>
			/// Reconstruction filter code 
			/// </summary>
			public short rfilter_code = 0;
			/// <summary>
			/// Reconstruction filter order 
			/// </summary>
			public short rfilter_order = 0;
			/// <summary>
			/// 
			/// </summary>
			public float zfilter_cutoff = 0.0f;
			/// <summary>
			/// 
			/// </summary>
			public float zfilter_resolution = 0.0f;
			/// <summary>
			/// 
			/// </summary>
			public short zfilter_code = 0;
			/// <summary>
			/// 
			/// </summary>
			public short zfilter_order = 0;
			/// <summary>
			/// Matrix transformation element (1,4) 
			/// </summary>
			public float mt_1_4 = 0.0f;
			/// <summary>
			/// Matrix transformation element (2,4) 
			/// </summary>
			public float mt_2_4 = 0.0f;
			/// <summary>
			/// Matrix transformation element (3,4) 
			/// </summary>
			public float mt_3_4 = 0.0f;
			/// <summary>
			/// Scatter type (short)
			/// </summary>
			public scatter_type_e scatter_type = scatter_type_e.None;
			/// <summary>
			/// Reconstruction type (short)
			/// </summary>
			public recon_type_e recon_type = recon_type_e.Filtered_backprojection;
			/// <summary>
			/// Number of views used to reconstruct the tacs 
			/// </summary>
			public short recon_views = 0;
			/// <summary>
			/// CTI Reserved space (174 bytes) 
			/// </summary>
			public short[] fill_cti = new short[87];
			/// <summary>
			/// User Reserved space (100 bytes) Note: Use highest bytes first 
			/// </summary>
			public short[] fill_user = new short[48];
			/// <summary>
			/// Filtering type code
			/// </summary>
			public enum filter_code_e
			{
				/// <summary>
				/// No filtering
				/// </summary>
				all_pass = 0,
				/// <summary>
				/// Ramp filtering
				/// </summary>
				ramp,
				/// <summary>
				/// Butterworth filtering
				/// </summary>
				Butterworth,
				/// <summary>
				/// Hanning filter
				/// </summary>
				Hanning,
				/// <summary>
				/// Hamming filter
				/// </summary>
				Hamming,
				/// <summary>
				/// Perzen filter
				/// </summary>
				Parzen,
				/// <summary>
				/// Shepp filter
				/// </summary>
				Shepp,
				/// <summary>
				/// Butterworth filter of order 2
				/// </summary>
				Butterworth_order_2,
				/// <summary>
				/// Gaussian filter
				/// </summary>
				Gaussian,
				/// <summary>
				/// Median filter
				/// </summary>
				Median,
				/// <summary>
				/// Boxcar filter
				/// </summary>
				Boxcar
			}
			/// <summary>
			/// Scattering type
			/// </summary>
			public enum scatter_type_e
			{
				/// <summary>
				/// No scatter
				/// </summary>
				None = 0,
				/// <summary>
				/// Deconvolution scatter
				/// </summary>
				Deconvolution,
				/// <summary>
				/// Simulated scatter
				/// </summary>
				Simulated,
				/// <summary>
				/// Dual energy scatter
				/// </summary>
				Dual_Energy
			}
			/// <summary>
			/// Reconstruction type
			/// </summary>
			public enum recon_type_e
			{
				/// <summary>
				/// Filtered backprojection 
				/// </summary>
				Filtered_backprojection = 0,
				/// <summary>
				/// Forward projection
				/// </summary>
				Forward_projection_3D_PROMIS,
				/// <summary>
				/// 3D Ramp
				/// </summary>
				Ramp_3D,
				/// <summary>
				/// 3D FAVOR
				/// </summary>
				FAVOR_3D,
				/// <summary>
				/// SSRB
				/// </summary>
				SSRB,
				/// <summary>
				/// Multi-slice rebinning
				/// </summary>
				Multi_slice_rebinning,
				/// <summary>
				/// FORE
				/// </summary>
				FORE
			}
			/// <summary>
			/// Converts tacs interpretation type into string
			/// </summary>
			/// <param name="type">tacs type</param>
			/// <returns>string representation of type</returns>
			public static string ToString(data_type_e type)
			{
				switch (type)
				{
					case data_type_e.Unknown_Matrix_Data_Type: return "unknown";
					case data_type_e.Byte_Data: return "byte";
					case data_type_e.VAX_Ix2: return "VAX 2 byte integer";
					case data_type_e.VAX_Ix4: return "VAX 4 byte integer";
					case data_type_e.VAX_Rx4: return "VAX 4 byte float";
					case data_type_e.IEEE_Float: return "IEEE 4 byte float";
					case data_type_e.Sun_short: return "SUN 2 byte integer";
					case data_type_e.Sun_long: return "SUN 4 byte integer";
				}
				throw new TPCEcat7FileException("internal error: unknown enumerate");
			}
			/// <summary>
			/// Data interpretation type.
			/// </summary>
			public enum data_type_e
			{
				/// <summary>
				/// Uknown
				/// </summary>
				Unknown_Matrix_Data_Type = 0,
				/// <summary>
				/// Byte tacs
				/// </summary>
				Byte_Data = 1,
				/// <summary>
				/// 2-byte VAX
				/// </summary>
				VAX_Ix2 = 2,
				/// <summary>
				/// 4-byte VAX
				/// </summary>
				VAX_Ix4 = 3,
				/// <summary>
				/// 4-byte realvalue VAX
				/// </summary>
				VAX_Rx4 = 4,
				/// <summary>
				/// 32-bit IEEE float
				/// </summary>
				IEEE_Float = 5,
				/// <summary>
				/// 16-bit Sun
				/// </summary>
				Sun_short = 6,
				/// <summary>
				/// 32-bit Sun
				/// </summary>
				Sun_long = 7
			}
			/// <summary>
			/// Converts ecat7 imageheader datatype to general tacs type
			/// </summary>
			/// <param name="type">converted datatype</param>
			/// <returns>datatype as Ecat7 format specific enumerator</returns>
			public static data_type_e convertFrom(DataType type)
			{
				switch (type)
				{
					case DataType.BIT16_S:
					case DataType.BIT16_U:
						return data_type_e.Sun_short;
					case DataType.BIT8_S:
					case DataType.BIT8_U:
						return data_type_e.Byte_Data;
					case DataType.VAXI16:
						return data_type_e.VAX_Ix2;
					case DataType.VAXI32:
						return data_type_e.VAX_Ix4;
					case DataType.VAXFL32:
						return data_type_e.VAX_Rx4;
					case DataType.FLT32:
						return data_type_e.IEEE_Float;
					case DataType.SUNI2:
						return data_type_e.Sun_short;
					case DataType.BIT32_S:
					case DataType.BIT32_U:
					case DataType.SUNI4:
						return data_type_e.Sun_long;
				}
				return data_type_e.Unknown_Matrix_Data_Type;
			}
			/// <summary>
			/// Converts ecat7 imageheader datatype to general tacs type
			/// </summary>
			/// <param name="type">Ecat7 format specific enumerator</param>
			/// <returns>general datatype enumerator</returns>
			public static DataType convertFrom(data_type_e type)
			{
				switch (type)
				{
					case data_type_e.Unknown_Matrix_Data_Type:
						//guessing..
						return DataType.BIT16_S;
					case data_type_e.Byte_Data:
						return DataType.BIT8_S;
					case data_type_e.VAX_Ix2:
						return DataType.VAXI16;
					case data_type_e.VAX_Ix4:
						return DataType.VAXI32;
					case data_type_e.VAX_Rx4:
						return DataType.VAXFL32;
					case data_type_e.IEEE_Float:
						return DataType.FLT32;
					case data_type_e.Sun_short:
						return DataType.BIT16_S;
					case data_type_e.Sun_long:
						return DataType.BIT32_S;
				}
				return DataType.BIT16_S;
			}
			/// <summary>
			/// Constructs imageheader with default values
			/// </summary>
			public Ecat7_imageheader()
			{
				scale_factor = 1.0f;
				x_resolution = 0.0f;
				y_resolution = 0.0f;
				z_resolution = 0.0f;
				recon_zoom = 0.0f;
				frame_start_time = 1000;
				frame_duration = 0;
			}
		}

		/// <summary>
		/// Subheader for 3D Matrix Scan Files
		/// </summary>
		public sealed class Ecat7_scanheader : Ecat7_subheader
		{
			/// <summary>
			/// Resolves tacs type
			/// </summary>
			/// <returns>datatype</returns>
			public DataType resolveDataType()
			{
				switch (data_type)
				{
					case data_type_e.ByteData: return DataType.BIT8_S;
					case data_type_e.SunShortt: return DataType.BIT16_S;
				}
				return DataType.BIT16_S;
			}
			/// <summary>
			/// Data type.
			/// </summary>
			public enum data_type_e
			{
				/// <summary>
				/// Byte tacs
				/// </summary>
				ByteData,
				/// <summary>
				/// Sun short
				/// </summary>
				SunShortt
			}
			/// <summary>
			/// Enumerated type.
			/// </summary>
			public data_type_e data_type;
			/// <summary>
			/// Number of Dimensions
			/// </summary>
			public short num_dimensions;
			/// <summary>
			/// Total elemenst collected (r dimension )
			/// </summary>
			public short num_r_elements;
			/// <summary>
			/// Total views collected (theta dimension)
			/// </summary>
			public short num_angles;
			/// <summary> 
			/// Designates processing applied to scan tacs 
			/// (Bit encoded, Bit 0 - Norm, Bit 1 - Atten, Bit 2 -Smooth)
			/// </summary>
			public short corrections_applied;
			/// <summary> 
			/// Total elements collected (z dimension) For 3D scans
			/// </summary>
			public short[] num_z_elements = new short[64];
			/// <summary> 
			/// Max ring difference (d dimension) in this frame 
			/// </summary>
			public short ring_difference;
			/// <summary> 
			/// Data storage order (r theta zd or rz theta d)
			/// </summary>
			public short storage_order;
			/// <summary> 
			/// Axial compression code or factor, generally 
			/// referred to as SPAN
			/// </summary>
			public short axial_compression;
			/// <summary> 
			/// Resolution in r dimension (in cm) 
			/// </summary>
			public float x_resolution;
			/// <summary> 
			/// Resolution in Theta dimension (in radians) 
			/// </summary>
			public float v_resolution;
			/// <summary> 
			/// Resolution in z dimension (in cm) 
			/// </summary>
			public float z_resolution;
			/// <summary> 
			/// Not Used
			/// </summary>
			public float w_resolution;
			/// <summary>
			/// RESERVED for gating 
			/// </summary>
			public short[] fill_gate = new short[6];
			/// <summary> 
			/// Gating segment length (msec, Average time if 
			/// phased gates are used)
			/// </summary>
			public int gate_duration;
			/// <summary> 
			/// Time from start of first gate (Average, in msec.)
			/// </summary>
			public int r_wave_offset;
			/// <summary> 
			/// Number of accepted beats for this gate
			/// </summary>
			public int num_accepted_beats;
			/// <summary> 
			/// Minimum value in sinogram if tacs is in integer form
			/// </summary>
			public short scan_min;
			/// <summary> 
			/// Maximum value in sinogram if tacs is in integer form
			/// </summary>
			public short scan_max;
			/// <summary> 
			/// Total prompts collected in this frame/gate
			/// </summary>
			public int prompts;
			/// <summary> 
			/// Total delays collected in this frame/gate
			/// </summary>
			public int delayed;
			/// <summary> 
			/// Total multiples collected in this frame/gate
			/// </summary>
			public int multiples;
			/// <summary> 
			/// Total net trues (prompts�randoms)
			/// </summary>
			public int net_trues;
			/// <summary> 
			/// Mean value of loss-corrected singles
			/// </summary>
			public float tot_avg_cor;
			/// <summary> 
			/// Mean value of singles (not loss corrected)
			/// </summary>
			public float tot_avg_uncor;
			/// <summary> 
			/// Measured coincidence rate (from IPCP)
			/// </summary>
			public int total_coin_rate;
			/// <summary> 
			/// Dead-time correction factor applied to the 
			/// sinogram
			/// </summary>
			public float deadtime_correction_factor;
			/// <summary>
			/// CTI Reserved space (180 bytes)
			/// </summary>
			public short[] fill_cti = new short[90];
			/// <summary> 
			/// User Reserved space (100 bytes) Note: Use 
			/// highest bytes first
			/// </summary>
			public short[] fill_user = new short[50];
			/// <summary> 
			/// Total uncorrected singles from each bucket
			/// </summary>
			public float[] uncor_singles = new float[128];
		}

		/// <summary>
		/// Subheader for Matrix Polar Map Files
		/// </summary>
		public class Ecat7_polmapheader : Ecat7_subheader
		{
			/// <summary>
			/// Resolves tacs type
			/// </summary>
			/// <returns>datatype</returns>
			public DataType resolveDataType()
			{
				switch (data_type)
				{
					case data_type_e.DTYPE_BYTES: return DataType.BIT8_S;
					case data_type_e._I2: return DataType.BIT16_S;
					case data_type_e._I4: return DataType.FLT32;
				}
				return DataType.BIT16_S;
			}
			/// <summary>
			/// Enumerated type (DTYPE_BYTES, _I2,_I4)
			/// </summary>
			public enum data_type_e
			{
				/// <summary>
				/// Byte tacs
				/// </summary>
				DTYPE_BYTES,
				/// <summary>
				/// 2-byte tacs
				/// </summary>
				_I2,
				/// <summary>
				/// 4-byte tacs
				/// </summary>
				_I4
			}
			/// <summary>
			/// Data type (short)
			/// </summary>
			public data_type_e data_type;
			/// <summary>
			/// Enumerated Type (Always 0 for now; denotes the version of the PM structure)
			/// </summary>
			public short polar_map_type;
			/// <summary>
			/// Number of rings in this polar map
			/// </summary>
			public short num_rings;
			/// <summary>
			/// Number of sectors in each ring for up to 32 rings 
			/// (1, 9, 18, or 32 sectors normally)
			/// </summary>
			public short[] sectors_per_ring = new short[32];
			/// <summary>
			/// Fractional distance along the long axis from base to apex
			/// </summary>
			public float[] ring_position = new float[32];
			/// <summary>
			/// Ring angle relative to long axis(90 degrees along 
			/// cylinder, decreasing to 0 at the apex)
			/// </summary>
			public short[] ring_angle = new short[32];
			/// <summary>
			/// START angle for rings (Always 258 degrees,
			/// defines Polar Map�s 0)
			/// </summary>
			public short start_angle;
			/// <summary>
			/// x, y, z location of long axis base end (in pixels)
			/// </summary>
			public short[] long_axis_left = new short[3];
			/// <summary>
			/// x, y, z location of long axis apex end (in pixels)
			/// </summary>
			public short[] long_axis_right = new short[3];
			/// <summary>
			/// Enumerated type (0 - Not available, 1 - Present)
			/// </summary>
			public short position_data;
			/// <summary>
			/// Minimum pixel value in this polar map
			/// </summary>
			public short image_min;
			/// <summary>
			/// Maximum pixel value in this polar map
			/// </summary>
			public short image_max;
			/// <summary>
			/// Pixel size (in cubic cm, represents voxels)
			/// </summary>
			public float pixel_size;
			/// <summary>
			/// Bit Encoded (1- Map type (0 = Sector Analysis, 1 = Volumetric), 
			/// 2 - Threshold Applied, 
			/// 3 - Summed Map, 
			/// 4 - Subtracted Map, 
			/// 5 - Product of two maps, 
			/// 6 - Ratio of two maps, 
			/// 7 - Bias, 
			/// 8 - Multiplier, 
			/// 9 - Transform, 
			/// 10 - Polar Map calculational protocol used)
			/// </summary>
			public short processing_code;
			/// <summary>
			/// Enumerated Type (0 - Default (see main header),
			/// 1 - Normalized, 2 - Mean, 3 - Std. Deviation from Mean)
			/// </summary>
			public enum quant_units_e
			{
				/// <summary>
				/// Default
				/// </summary>
				Default = 0,
				/// <summary>
				/// Normalized
				/// </summary>
				Normalized = 1,
				/// <summary>
				/// Mean
				/// </summary>
				Mean = 2,
				/// <summary>
				/// Standard deviation from mean
				/// </summary>
				Std_Deviation_from_Mean = 3
			}
			/// <summary>
			/// Quantitation units (short)
			/// </summary>
			public quant_units_e quant_units;
			/// <summary>
			/// Label for polar map display
			/// </summary>
			public char[] annotation = new char[40];
			/// <summary>
			/// Gate duration (in msec)
			/// </summary>
			public int gate_duration;
			/// <summary>
			/// R wave offset (Average, in msec)
			/// </summary>
			public int r_wave_offset;
			/// <summary>
			/// Number of accepted beats for this gate
			/// </summary>
			public int num_accepted_beats;
			/// <summary>
			/// Polar Map protocol used to generate this polar map
			/// </summary>
			public char[] polar_map_protocol = new char[20];
			/// <summary>
			/// Database name used for polar map comparison
			/// </summary>
			public char[] database_name = new char[30];
			/// <summary>
			/// Reserved for future CTI use (54 bytes)
			/// </summary>
			public short[] fill_cti = new short[27];
			/// <summary>
			/// User reserved space (54 bytes) Note: Use highest bytes first
			/// </summary>
			public short[] fill_user = new short[27];
		}

		/// <summary>
		/// Ecat7 matrix directory header. Entry 1 in matrix directory list. A double-linked list of 
		/// directory list blocks.
		/// </summary>
		protected struct ecat7_MatHdr
		{
			/// <summary>
			/// Number of free entries in list [0..31]
			/// </summary>
			public int nr_free;
			/// <summary>
			/// block number (1-based) of next directory list
			/// </summary>
			public int next_blk;
			/// <summary>
			/// block number (1-based) of previous directory list
			/// </summary>
			public int prev_blk;
			/// <summary>
			/// Number of used entries in list [0..31]
			/// </summary>
			public int nr_used;
			/// <summary>
			/// Constructs matrix header. 
			/// <remarks>nr_free + nr_used should be 31</remarks>
			/// </summary>
			/// <param name="nr_free">number of free items</param>
			/// <param name="next_blk">next block number</param>
			/// <param name="prev_blk">previous block number</param>
			/// <param name="nr_used">number of used items</param>
			public ecat7_MatHdr(int nr_free, int next_blk, int prev_blk, int nr_used)
			{
				this.nr_free = nr_free;
				this.next_blk = next_blk;
				this.prev_blk = prev_blk;
				this.nr_used = nr_used;
			}
			/// <summary>
			/// Returns a String that represents the current Object.
			/// </summary>
			/// <returns>A String that represents the current Object.</returns>
			public override string ToString()
			{
				return "ecat7_MatHdr[nr_free=" + nr_free + " next_blk=" + next_blk + " prev_blk=" + prev_blk + " nr_used=" + nr_used + "]";
			}
		}

		/// <summary>
		/// Ecat7 matrix directory entry. Entries 1-31 in directory list.
		/// </summary>
		public class Ecat7_MatDir : IComparable
		{
			/// <summary>
			/// Matrix identifier (int)
			/// </summary>
			public Ecat7_Matval id;
			/// <summary>
			/// Matrix subheader record's block number.
			/// <see cref="MatBLKSIZE"/>
			/// </summary>
			public int strtblk;
			/// <summary>
			/// Last record's block number for matrix tacs block.
			/// <see cref="MatBLKSIZE"/>
			/// </summary>
			public int endblk;
			/// <summary>
			/// Matrix status (int)
			/// </summary>
			public status_e status;
			/// <summary>
			/// Constructs matrix directory item
			/// </summary>
			/// <param name="matval">matrix value</param>
			/// <param name="strtblk">tacs start block number</param>
			/// <param name="endblk">tacs end block number</param>
			/// <param name="status">tacs status</param>
			public Ecat7_MatDir(Ecat7_Matval matval, int strtblk, int endblk, status_e status)
			{
				this.id = matval;
				this.strtblk = strtblk;
				this.endblk = endblk;
				this.status = status;
			}
			/// <summary>
			/// Constructs matrix directory item
			/// </summary>
			/// <param name="id">matrix value code number</param>
			/// <param name="strtblk">tacs start block number</param>
			/// <param name="endblk">tacs end block number</param>
			/// <param name="status">tacs status</param>
			public Ecat7_MatDir(int id, int strtblk, int endblk, status_e status)
			{
				this.id = new Ecat7_Matval(id);
				this.strtblk = strtblk;
				this.endblk = endblk;
				this.status = status;
			}
			/// <summary>
			/// Status enumerator. EXISTS = access read/write, DELETED = no access
			/// </summary>
			public enum status_e
			{
				/// <summary>
				/// Dat does not exist
				/// </summary>
				DELETED = 0,
				/// <summary>
				/// Data is not yet written. No need to use unless
				/// in multithreaded program.
				/// </summary>
				DATE_NOT_YET_WRITTEN = 2,
				/// <summary>
				/// Data exists
				/// </summary>
				EXISTS = 1
			}
			/// <summary>
			/// Returns a String that represents the current Object.
			/// </summary>
			/// <returns>A String that represents the current Object.</returns>
			public override string ToString()
			{
				return "ecat7_MatDir[id=" + id + " strtblk=" + strtblk + " endblk=" + endblk + " status=" + status + "]";
			}

			/// <summary>
			/// Compare method for sorting matrices
			/// </summary>
			/// <param name="obj">Object to compare</param>
			/// <returns>-1 if this precedes obj, 0 if equal, 1 if this follows obj</returns>
			public int CompareTo(object obj)
			{
				if (obj is Ecat7_MatDir)
				{
					// Deleted matrices to the end
					if (this.status == status_e.DELETED && (obj as Ecat7_MatDir).status != status_e.DELETED) return 1;
					if (this.status != status_e.DELETED && (obj as Ecat7_MatDir).status == status_e.DELETED) return -1;
					return this.id.CompareTo((obj as Ecat7_MatDir).id);
				}
				else throw new ArgumentException("Not a matrix directory entry");
			}
		}

		/// <summary>
		/// Class representing coded matrix id value that holds various matrix information.
		/// </summary>
		public class Ecat7_Matval : IComparable
		{
			/// <summary>
			/// matrix id value containing various coded matrix information
			/// </summary>
			public int matrix_id;
			/// <summary>
			/// frame number [0..65536]
			/// </summary>
			public int frame { get { return (int)(matrix_id & 0x1FF); } }
			/// <summary>
			/// plane number [0..65536]
			/// </summary>
			public int plane { get { return (int)(((matrix_id >> 16) & 0xFF) + ((matrix_id >> 1) & 0x300)); } }
			/// <summary>
			/// gate number [0..64]
			/// </summary>
			public int gate { get { return (int)((matrix_id >> 24) & 0x3F); } }
			/// <summary>
			/// tacs number [0..1]
			/// </summary>
			public int data { get { return (int)(((matrix_id >> 30) & 0x3) + ((matrix_id >> 9) & 0x4)); } }
			/// <summary>
			/// bed number [0..16]
			/// </summary>
			public int bed { get { return (int)((matrix_id >> 12) & 0xF); } }
			/// <summary>
			/// Converts matrix-value.
			/// </summary>
			/// <param name="frame">frame number [0..65536]</param>
			/// <param name="plane">plane number [0..65536]</param>
			/// <param name="gate">[0..64]</param>
			/// <param name="data">[0..1]</param>
			/// <param name="bed">[0..16]</param>
			/// <returns>matrix id value</returns>
			public static int ecat7_val_to_id(int frame, int plane, int gate, int data, int bed)
			{
				return ((bed & 0xF) << 12) |    /* bed */
						(frame & 0x1FF) |        /* frame */
						((gate & 0x3F) << 24) |  /* gate */
						((plane & 0xFF) << 16) | /* plane low */
						((plane & 0x300) << 1) | /* plane high */
						((data & 0x3) << 30) |   /* tacs low */
						((data & 0x4) << 9);      /* tacs high */
			}
			/// <summary>
			/// Constructs matrix-value.
			/// </summary>
			/// <param name="matrix_id">matrix id value</param>
			/// <exception cref="TPCInvalidArgumentsException">if parameter would result out of bounds values</exception>
			public Ecat7_Matval(int matrix_id)
			{
				this.matrix_id = matrix_id;
				if (frame < 0 || frame > 65536) throw new TPCInvalidArgumentsException("frame value would be out of bounds");
				if (plane < 0 || plane > 65536) throw new TPCInvalidArgumentsException("plane value would be out of bounds");
				if (gate < 0 || gate > 65536) throw new TPCInvalidArgumentsException("gate value would be out of bounds");
				if (data < 0 || data > 65536) throw new TPCInvalidArgumentsException("tacs value would be out of bounds");
				if (bed < 0 || bed > 65536) throw new TPCInvalidArgumentsException("bed value would be out of bounds");
			}
			/// <summary>
			/// Constructs matrix-value.
			/// </summary>
			/// <param name="frame">frame number [0..65536]</param>
			/// <param name="plane">plane number [0..65536]</param>
			/// <param name="gate">[0..64]</param>
			/// <param name="data">[0..1]</param>
			/// <param name="bed">[0..16]</param>
			/// <exception cref="TPCInvalidArgumentsException">if parameter is out of bounds</exception>
			public Ecat7_Matval(int frame, int plane, int gate, int data, int bed)
			{
				if (frame < 0 || frame > 65536) throw new TPCInvalidArgumentsException("frame value out of bounds");
				if (plane < 0 || plane > 65536) throw new TPCInvalidArgumentsException("plane value out of bounds");
				if (gate < 0 || gate > 65536) throw new TPCInvalidArgumentsException("gate value out of bounds");
				if (data < 0 || data > 65536) throw new TPCInvalidArgumentsException("tacs value out of bounds");
				if (bed < 0 || bed > 65536) throw new TPCInvalidArgumentsException("bed value out of bounds");
				matrix_id = ecat7_val_to_id(frame, plane, gate, data, bed);
			}
			/// <summary>
			/// Returns a String that represents the current Object.
			/// </summary>
			/// <returns>A String that represents the current Object.</returns>
			public override string ToString()
			{
				return "Matval[frame=" + frame + " plane=" + plane + " gate=" + gate + " tacs=" + data + " bed=" + bed + "]";
			}

			/// <summary>
			/// Compare method for sorting matrix ids
			/// </summary>
			/// <param name="obj">Object to compare</param>
			/// <returns>-1 if this precedes obj, 0 if equal, 1 if this follows obj</returns>
			public int CompareTo(object obj)
			{
				if (obj is Ecat7_Matval)
				{
					Ecat7_Matval o = (obj as Ecat7_Matval);
					int comp = 0;
					comp = this.bed.CompareTo(o.bed);
					if (comp != 0) return comp;
					comp = this.gate.CompareTo(o.gate);
					if (comp != 0) return comp;
					comp = this.frame.CompareTo(o.frame);
					if (comp != 0) return comp;
					comp = this.plane.CompareTo(o.plane);
					return comp;
				}
				else throw new ArgumentException("Not a matrix id");
			}
		}

		#endregion

		#region Public fields/properties

		/// <summary>
		/// Event that is sent when reading of file has progressed.
		/// </summary>
		public static event IOProcessEventHandler IOProgress;

		/// <summary>
		/// One or more subheaders. Header type depends on image type. Note that this 
		/// array may or may not contain null references, depending image matrix tacs in last read Ecat7 
		/// file.
		/// </summary>
		public Ecat7_subheader[] subheaders;

		/// <summary>
		/// Matrix list that was last read from file. Read-only from outside of class.
		/// </summary>
		public Ecat7_MatDir[] matrixlist
		{
			get { return mlist; }
		}

		/// <summary>
		/// Static matrix block size
		/// </summary>
		public const int MatBLKSIZE = 512;

		/// <summary>
		/// Ecat 7 magic number
		/// </summary>
		public const string ecat7V_MAGICNR = "MATRIX72v";

		/// <summary>
		/// Ecat 7 magic number
		/// </summary>
		public const string ecat7S_MAGICNR = "MATRIX7011";

		/// <summary>
		/// Shortcut for sample distance. Applicable only if there is mainheader or proper subheader.
		/// </summary>
		public float sampledistance
		{
			get
			{
				if (subheaders.Length > 0 &&
					(subheaders[0] is Ecat7_scanheader) &&
					(subheaders[0] as Ecat7_scanheader).x_resolution > 0.0)
				{
					return 10.0f * (subheaders[0] as Ecat7_scanheader).x_resolution;
				}
				else
				{
					return 10.0f * EcatHeader.bin_size;
				}
			}
		}

		#endregion

		#region Private/protected fields

		/// <summary>
		/// Matrix list that was last read from file.
		/// </summary>
		protected Ecat7_MatDir[] mlist;

		/// <summary> 
		/// Number of processed slices in I/O operation. 
		/// Used to send correct progress events between reading of each slice. 
		/// </summary> 
		private int processed_slices = 0;

		/// <summary> 
		/// Number of processed slices in I/O operation. 
		/// Used to send correct progress events between reading of each slice. 
		/// </summary> 
		private int total_number_of_slices_to_process = 0;

		#endregion

		#region Private/protected methods

		private delegate byte[] ValueWriter(double f);

		private ValueWriter GetWriter(DataType dt)
		{
			switch (dt)
			{
				case DataType.BIT8_S:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToSByte(f)); };
				case DataType.BIT8_U:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToByte(f)); };
				case DataType.BIT16_S:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToInt16(f)); };
				case DataType.BIT16_U:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToUInt16(f)); };
				case DataType.BIT32_S:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToInt32(f)); };
				case DataType.BIT32_U:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToUInt32(f)); };
				case DataType.BIT64_S:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToInt64(f)); };
				case DataType.BIT64_U:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToUInt64(f)); };
				case DataType.FLT32:
					return delegate(double f) { return BitConverter.GetBytes(f); };
				case DataType.FLT64:
					return delegate(double f) { return BitConverter.GetBytes(Convert.ToDouble(f)); };
				default:
					throw new TPCEcat7FileException("Unsupported write tacs type:" + imgHeader.Datatype);
			}
		}

		/// <summary>
		/// Read ECAT 7.x 3D scan headers. Null is placed where matrix does not exist.
		/// </summary>
		/// <param name="filename">file that is read</param>
		/// <returns>array of subheaders</returns>
		/// <exception cref="TPCEcat7FileException">if error occurred while reading file</exception>
		public static Ecat7_scanheader[] Ecat7ReadScanheaders(string filename)
		{
			if (!CheckFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");
			Ecat7_MatDir[] mlist = Ecat7ReadMatlist(filename);
			return Ecat7ReadScanheaders(mlist, filename);
		}

		/// <summary>
		/// Read ECAT 7.x 3D scan headers. Null is placed where matrix does not exist.
		/// </summary>
		/// <param name="matdir">array of matrix directory entries</param>
		/// <param name="filename">Ecat7 file that is read</param>
		/// <returns>array of subheaders</returns>
		protected static Ecat7_scanheader[] Ecat7ReadScanheaders(Ecat7_MatDir[] matdir, string filename)
		{
			List<Ecat7_scanheader> scanheader_list = new List<Ecat7_scanheader>();
			for (int i = 0; i < matdir.Length; i++)
			{
				if (matdir[i].status == Ecat7_MatDir.status_e.EXISTS)
					scanheader_list.Add(Ecat7ReadScanheader(matdir[i], filename));
				else
					scanheader_list.Add(null);

			}
			return scanheader_list.ToArray();
		}

		/// <summary>
		/// Read ECAT 7.x 3D scan header
		/// </summary>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="filename">full path to file that is read</param>
		/// <returns>Ecat7 scan header that contains header tacs from file</returns>
		protected static Ecat7_scanheader Ecat7ReadScanheader(Ecat7_MatDir matdir, string filename)
		{
			byte[] buf = new byte[2 * MatBLKSIZE];
			int blk = matdir.strtblk;
			Ecat7_scanheader h = new Ecat7_scanheader();
			using (FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				/* Seek the subheader block */
				filestream.Seek((blk - 1) * MatBLKSIZE, SeekOrigin.Begin);
				if (filestream.Position != (blk - 1) * MatBLKSIZE)
				{
					throw new TPCEcat7FileException("Invalid stream position after seek with " + matdir + "," + filename + ".");
				}
				/* Read the header block */
				if (filestream.Read(buf, 0, 2 * MatBLKSIZE) < 2 * MatBLKSIZE)
				{
					throw new TPCEcat7FileException("Failed to read all scan header tacs with " + matdir + "," + filename + ".");
				}

				/* Copy the header fields and swap if necessary */
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 0, 2); h.data_type = (Ecat7_scanheader.data_type_e)BitConverter.ToInt16(buf, 0);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 2, 2); h.num_dimensions = BitConverter.ToInt16(buf, 2);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 4, 2); h.num_r_elements = BitConverter.ToInt16(buf, 4);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 6, 2); h.num_angles = BitConverter.ToInt16(buf, 6);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 8, 2); h.corrections_applied = BitConverter.ToInt16(buf, 8);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 10, 64 * 2); h.num_z_elements = ToShortArray(buf, 10, 64);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 138, 2); h.ring_difference = BitConverter.ToInt16(buf, 138);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 140, 2); h.storage_order = BitConverter.ToInt16(buf, 140);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 142, 2); h.axial_compression = BitConverter.ToInt16(buf, 142);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 144, 4); h.x_resolution = BitConverter.ToSingle(buf, 144);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 148, 4); h.v_resolution = BitConverter.ToSingle(buf, 148);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 152, 4); h.z_resolution = BitConverter.ToSingle(buf, 152);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 156, 4); h.w_resolution = BitConverter.ToSingle(buf, 156);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 160, 6 * 2); h.fill_gate = ToShortArray(buf, 160, 6);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 172, 4); h.gate_duration = BitConverter.ToInt32(buf, 172);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 176, 4); h.r_wave_offset = BitConverter.ToInt32(buf, 176);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 180, 4); h.num_accepted_beats = BitConverter.ToInt32(buf, 180);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 184, 4); h.scale_factor = BitConverter.ToSingle(buf, 184);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 188, 2); h.scan_min = BitConverter.ToInt16(buf, 188);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 190, 2); h.scan_max = BitConverter.ToInt16(buf, 190);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 192, 4); h.prompts = BitConverter.ToInt32(buf, 192);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 196, 4); h.delayed = BitConverter.ToInt32(buf, 196);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 200, 4); h.multiples = BitConverter.ToInt32(buf, 200);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 204, 4); h.net_trues = BitConverter.ToInt32(buf, 204);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 208, 4); h.tot_avg_cor = BitConverter.ToSingle(buf, 208);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 212, 4); h.tot_avg_uncor = BitConverter.ToSingle(buf, 212);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 216, 4); h.total_coin_rate = BitConverter.ToInt32(buf, 216);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 220, 4); h.frame_start_time = BitConverter.ToInt32(buf, 220);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 224, 4); h.frame_duration = BitConverter.ToInt32(buf, 224);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 228, 4); h.deadtime_correction_factor = BitConverter.ToSingle(buf, 228);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 232, 90 * 2); h.fill_cti = ToShortArray(buf, 232, 90);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 412, 50 * 2); h.fill_user = ToShortArray(buf, 412, 50);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 512, 128 * 4); h.uncor_singles = ToFloatArray(buf, 512, 128);
			}
			return h;
		}

		/// <summary>
		/// Read ECAT matrix list.
		/// </summary>
		/// <returns>matrix list from file</returns>
		protected static Ecat7_MatDir[] Ecat7ReadMatlist(string filename)
		{
			int i;
			//buffer having directory list tacs
			byte[] buf = new byte[MatBLKSIZE];
			//matrix entry list
			List<Ecat7_MatDir> mlist = new List<Ecat7_MatDir>();
			//filestream
			using (FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				//start from block 1
				int blk = 1;
				//directory list header
				ecat7_MatHdr dirhdr;

				// seek the first directory list block
				try
				{
					filestream.Seek(blk * MatBLKSIZE, SeekOrigin.Begin);
				}
				catch (Exception e)
				{
					throw new TPCEcat7FileException("Failed to seek matrix list beginning:" + e.Message);
				}

				// read until first directory list is going to be the next directory list
				do
				{
					//read the directory list into buffer
					if (filestream.Read(buf, 0, MatBLKSIZE) < MatBLKSIZE) throw new TPCEcat7FileException("Failed to read full matrix block.");
					if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 0, buf.Length);

					/* directory header */
					dirhdr = new ecat7_MatHdr(	BitConverter.ToInt32(buf, 0),
												BitConverter.ToInt32(buf, 4),
												BitConverter.ToInt32(buf, 8),
												BitConverter.ToInt32(buf, 12));
					/* matrix lists */
					for (i = 16; i < MatBLKSIZE; i += 16)
					{
						mlist.Add(new Ecat7_MatDir(	BitConverter.ToInt32(buf, i),
													BitConverter.ToInt32(buf, i + 4),
													BitConverter.ToInt32(buf, i + 8),
													(Ecat7_MatDir.status_e)BitConverter.ToInt32(buf, i + 12)));
					}
					//seek next directory list block
					filestream.Seek((dirhdr.next_blk - 1) * MatBLKSIZE, SeekOrigin.Begin);
				} while (dirhdr.next_blk != 2);
			}
			return mlist.ToArray();
		}

		/// <summary>
		/// Writes ECAT matrix list tacs into file. 
		/// </summary>
		/// <param name="filename">file name</param>
		/// <param name="list">matrix list, length must be power of 31 (prefilled with possible empty lines)</param>
		/// <returns>matrix list from file</returns>
		protected static void ecat7WriteMatlist(string filename, Ecat7_MatDir[] list)
		{
			//byte location in block
			int i = 16;
			//matrix list index number
			int j = 0;
			//buffer having directory list tacs
			byte[] buf = new byte[MatBLKSIZE];
			//current block number
			int blk = 1;

			if (list.Length % 31 != 0)
			{
				throw new TPCEcat7FileException("Number of matrixes must be power of 31.");
			}

			//filestream
			using (FileStream filestream = new FileStream(filename, FileMode.Open))
			{
				//directory list header
				ecat7_MatHdr dirhdr = new ecat7_MatHdr();
				dirhdr.nr_used = 0;
				dirhdr.nr_free = 31;
				dirhdr.next_blk = 1;
				dirhdr.prev_blk = 0;

				//write directory list entries
				for (j = 0; j < list.Length; j++)
				{
					/* add matrix item */
					Array.Copy(BitConverter.GetBytes(list[j].id.matrix_id), 0, buf, i, 4);
					Array.Copy(BitConverter.GetBytes(list[j].strtblk), 0, buf, i + 4, 4);
					Array.Copy(BitConverter.GetBytes(list[j].endblk), 0, buf, i + 8, 4);
					Array.Copy(BitConverter.GetBytes((int)list[j].status), 0, buf, i + 12, 4);
					if (list[j].status == Ecat7_MatDir.status_e.EXISTS)
					{
						dirhdr.nr_free--;
						dirhdr.nr_used++;
					}
					i += 16;
					/* add 'directory header' and write block into file when buffer is full */
					if (i == MatBLKSIZE)
					{
						//set location of next buffer to current buffer if this is the end of matrix list
						if (j == list.Length - 1)
						{
							dirhdr.next_blk = 2;
							dirhdr.prev_blk = 2;
						}
						else
						{
							dirhdr.next_blk = list[j].endblk + 1;
						}
						//set matrixlist header into start of buffer
						Array.Copy(BitConverter.GetBytes(dirhdr.nr_free), 0, buf, 0, 4);
						Array.Copy(BitConverter.GetBytes(dirhdr.next_blk), 0, buf, 4, 4);
						Array.Copy(BitConverter.GetBytes(dirhdr.prev_blk), 0, buf, 8, 4);
						Array.Copy(BitConverter.GetBytes(dirhdr.nr_used), 0, buf, 12, 4);
						//write the directory list from buffer
						if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 0, buf.Length);
						try
						{
							filestream.Seek(blk * MatBLKSIZE, SeekOrigin.Begin);
							filestream.Write(buf, 0, MatBLKSIZE);
							//reset nunber of used and unused list indexes and set previsous and current block number
							dirhdr.nr_used = 0;
							dirhdr.prev_blk = blk;
							blk = dirhdr.next_blk - 1;
							dirhdr.nr_free = 31;
						}
						catch (Exception e)
						{
							throw new TPCEcat7FileException("Failed to write matrix block:" + e.Message);
						}
						i = 16;
					}
				}
				/* commented because making linking according to standard might cause endless loops 
				 * in some programs reading Ecat7
				//write linkage between first and last matrix list
				try
				{
					//write previous block for first item (== 0)
					Array.Copy(BitConverter.GetBytes(blk), 0, buf, 0, 4);
					filestream.Seek(1 * MatBLKSIZE + 8, SeekOrigin.Begin);
					if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 0, 4);
					filestream.Write(buf, 0, 4);
					//write next block for last item (== 2)
					Array.Copy(BitConverter.GetBytes(2), 0, buf, 0, 4);
					filestream.Seek(blk * MatBLKSIZE + 4, SeekOrigin.Begin);
					if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 0, 4);
					filestream.Write(buf, 0, 4);

					filestream.Seek(1 * MatBLKSIZE, SeekOrigin.Begin);
					filestream.Read(buf, 0, 16);
					if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 0, 16);
					// 'directory header'
					dirhdr = new ecat7_MatHdr(BitConverter.ToInt32(buf, 0),
												BitConverter.ToInt32(buf, 4),
												BitConverter.ToInt32(buf, 8),
												BitConverter.ToInt32(buf, 12));
				}
				catch (Exception e)
				{
					throw new TPCEcat7FileException("Failed to seek matrix list beginning:" + e.Message);
				}
				*/
			}
		}

		/// <summary>
		/// Prints character array to standard output.
		/// </summary>
		/// <param name="title">title of line</param>
		/// <param name="array">char tacs in array</param>
		private static void PrintCharArray(string title, char[] array)
		{
			Console.Write(title);
			for (int i = 0; i < array.Length; i++) Console.Write("{0:s}", array[i]);
			Console.WriteLine();
		}

		/// <summary>
		/// Read ECAT7 3D sinogram matrix tacs.
		/// Note: tacs is converted to floats with scale_factor in the scan matrix header.
		/// Note: tacs is not calibrated with ecat_calibration_factor in main header.
		/// Note: tacs is not multiplied with deadtime_correction_factor.
		/// </summary>
		/// <param name="stream">open input stream</param>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="h">scanheader of matrix</param>
		/// <returns>matrix tacs that is read, allways dynamic</returns>
		protected Image Ecat7ReadScanMatrix(Stream stream, Ecat7_MatDir matdir, Ecat7_scanheader h)
		{
			int i, blockNr, trueblockNr, pxlNr, dimz;
			byte[] buf;
			Image volume;

			/* Read subheader */
			pxlNr = h.num_r_elements * h.num_angles;
			for (i = dimz = 0; i < 64; i++)
				dimz += h.num_z_elements[i];
			pxlNr *= dimz;
			if (pxlNr <= 0)
			{
				throw new TPCEcat7FileException("Invalid matrix dimensions: [" + h.num_r_elements + ", " + h.num_angles + "]");
			}

			/* Read matrix tacs */
			blockNr = matdir.endblk - (matdir.strtblk + 2);
			if (blockNr < 0) throw new TPCInvalidArgumentsException("not enough space between last_block and first_block");
			trueblockNr = pxlNr * Type2Bytes(h.resolveDataType());
			trueblockNr = (trueblockNr + MatBLKSIZE - 1) / MatBLKSIZE;
			if (blockNr < trueblockNr) trueblockNr = blockNr;

			//read matrix tacs into buffer
			buf = Ecat7ReadMatrixdata(stream, matdir.strtblk + 1, matdir.strtblk + 1 + trueblockNr);

			volume = new Image(new IntLimits(new int[] { h.num_r_elements - 1, h.num_angles - 1, 1, 1, 1, 1 }));

			/* Convert matrix tacs to floats */
			switch (h.resolveDataType())
			{
				case DataType.BIT16_S:
					throw new System.NotImplementedException();
				case DataType.VAXI16:
				case DataType.SUNI2:
					throw new System.NotImplementedException();
				case DataType.VAXI32:
				case DataType.SUNI4:
					throw new System.NotImplementedException();
				case DataType.VAXFL32:
				case DataType.FLT32:
					throw new System.NotImplementedException();
			}
			return volume;
		}

		/// <summary>
		/// Read ECAT7 3D polarmap matrix tacs.
		/// </summary>
		/// <param name="stream">open input stream</param>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="h">polarmapheader of matrix</param>
		/// <returns>matrix tacs that is read, allways dynamic</returns>
		protected Image Ecat7ReadPolmapMatrix(Stream stream, Ecat7_MatDir matdir, Ecat7_polmapheader h)
		{
			int i, blockNr, trueblockNr, pxlNr;
			byte[] buf;
			Image volume;

			/* Read subheader */
			for (i = pxlNr = 0; i < h.num_rings; i++) pxlNr += h.sectors_per_ring[i];
			if (pxlNr <= 0)
			{
				throw new TPCEcat7FileException("Invalid matrix dimensions: no sectors found in rings");
			}

			/* Read matrix tacs */
			blockNr = matdir.endblk - (matdir.strtblk + 1);
			if (blockNr < 0) throw new TPCInvalidArgumentsException("not enough space between last_block and first_block");
			trueblockNr = pxlNr * Type2Bytes(h.resolveDataType());
			trueblockNr = (trueblockNr + MatBLKSIZE - 1) / MatBLKSIZE;
			if (blockNr < trueblockNr) trueblockNr = blockNr;

			//read matrix tacs into buffer
			buf = Ecat7ReadMatrixdata(stream, matdir.strtblk + 1, matdir.strtblk + 1 + trueblockNr);

			volume = new Image(new IntLimits(new int[] { pxlNr - 1, 1, 1, 1, 1, 1 }));

			/* Convert matrix tacs to floats */
			switch (h.resolveDataType())
			{
				case DataType.BIT16_S:
					throw new System.NotImplementedException();
				case DataType.VAXI16:
				case DataType.SUNI2:
					throw new System.NotImplementedException();
				case DataType.VAXI32:
				case DataType.SUNI4:
					throw new System.NotImplementedException();
				case DataType.VAXFL32:
				case DataType.FLT32:
					throw new System.NotImplementedException();
			}
			return volume;
		}

		/// <summary>
		/// Read ECAT 7.x polar map headers. Null is placed where matrix does not exist.
		/// </summary>
		/// <param name="matdir">array of matrix directory entries</param>
		/// <param name="filename">Ecat7 file that is read</param>
		/// <returns>array of subheaders</returns>
		protected static Ecat7_polmapheader[] Ecat7ReadPolmapheaders(Ecat7_MatDir[] matdir, string filename)
		{
			List<Ecat7_polmapheader> polmapheader_list = new List<Ecat7_polmapheader>();
			for (int i = 0; i < matdir.Length; i++)
			{
				if (matdir[i].status == Ecat7_MatDir.status_e.EXISTS)
					polmapheader_list.Add(Ecat7ReadPolmapheader(matdir[i], filename));
				else
					polmapheader_list.Add(null);
			}
			return polmapheader_list.ToArray();
		}

		/// <summary>
		/// Reads ECAT 7.x polar map header
		/// </summary>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="filename">Ecat7 file that is read</param>
		/// <see cref="MatBLKSIZE"/>
		/// <returns>polar map header</returns>
		protected static Ecat7_polmapheader Ecat7ReadPolmapheader(Ecat7_MatDir matdir, string filename)
		{
			byte[] buf = new byte[MatBLKSIZE];
			int blk = matdir.strtblk;
			Ecat7_polmapheader h = new Ecat7_polmapheader();
			using (FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				/* Seek the subheader block */
				filestream.Seek((blk - 1) * MatBLKSIZE, SeekOrigin.Begin);
				if (filestream.Position != (blk - 1) * MatBLKSIZE)
				{
					throw new TPCEcat7FileException("Invalid stream position after seek.");
				}
				/* Read the header block */
				if (filestream.Read(buf, 0, MatBLKSIZE) < MatBLKSIZE)
				{
					throw new TPCEcat7FileException("Failed to read all scan header tacs.");
				}

				/* Copy the header fields and swap if necessary */
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 0, 2); h.data_type = (Ecat7_polmapheader.data_type_e)BitConverter.ToInt16(buf, 0);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 2, 2); h.polar_map_type = BitConverter.ToInt16(buf, 2);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 4, 2); h.num_rings = BitConverter.ToInt16(buf, 4);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 6, 32 * 2); h.sectors_per_ring = ToShortArray(buf, 6, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 70, 32 * 4); h.ring_position = ToFloatArray(buf, 70, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 198, 32 * 2); h.ring_angle = ToShortArray(buf, 198, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 262, 2); h.start_angle = BitConverter.ToInt16(buf, 262);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 264, 3 * 2); h.long_axis_left = ToShortArray(buf, 264, 3);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 270, 3 * 2); h.long_axis_right = ToShortArray(buf, 270, 3);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 276, 2); h.position_data = BitConverter.ToInt16(buf, 276);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 278, 2); h.image_min = BitConverter.ToInt16(buf, 278);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 280, 2); h.image_max = BitConverter.ToInt16(buf, 280);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 282, 4); h.scale_factor = BitConverter.ToSingle(buf, 282);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 286, 4); h.pixel_size = BitConverter.ToSingle(buf, 286);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 290, 4); h.frame_duration = BitConverter.ToInt32(buf, 290);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 294, 4); h.frame_start_time = BitConverter.ToInt32(buf, 294);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 298, 2); h.processing_code = BitConverter.ToInt16(buf, 298);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 300, 2); h.quant_units = (Ecat7_polmapheader.quant_units_e)BitConverter.ToInt16(buf, 300);
				h.annotation = toCharArray(buf, 302, 40);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 342, 4); h.gate_duration = BitConverter.ToInt32(buf, 342);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 346, 4); h.r_wave_offset = BitConverter.ToInt32(buf, 346);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordAndArrayBytes(buf, 350, 4); h.num_accepted_beats = BitConverter.ToInt32(buf, 350);
				h.polar_map_protocol = toCharArray(buf, 354, 20);
				h.database_name = toCharArray(buf, 374, 30);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 404, 27 * 2); h.fill_cti = ToShortArray(buf, 404, 27);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 464, 27 * 2); h.fill_cti = ToShortArray(buf, 464, 27);
			}
			return h;
		}

		/// <summary>
		/// Read ECAT 7.x image headers.
		/// </summary>
		/// <param name="matdir">array of matrix directory entries</param>
		/// <param name="filename">Ecat7 file that is read</param>
		/// <returns>array of subheaders</returns>
		protected static Ecat7_imageheader[] Ecat7ReadImageheaders(Ecat7_MatDir[] matdir, string filename)
		{
			List<Ecat7_imageheader> ihList = new List<Ecat7_imageheader>();
			for (int i = 0; i < matdir.Length; i++)
			{
				if (matdir[i].status == Ecat7_MatDir.status_e.EXISTS)
					ihList.Add(Ecat7ReadImageheader(matdir[i], filename));
				else ihList.Add(null);
			}
			return ihList.ToArray();
		}

		/// <summary>
		/// Read ECAT 7.x image header.
		/// </summary>
		/// <param name="filename">file that is read</param>
		/// <returns>array of subheaders</returns>
		/// <exception cref="TPCEcat7FileException">if error occurred while reading file</exception>
		public static Ecat7_imageheader[] Ecat7ReadImageheaders(string filename)
		{
			if (!CheckFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");
			Ecat7_MatDir[] mlist = Ecat7ReadMatlist(filename);
			return Ecat7ReadImageheaders(mlist, filename);
		}

		/// <summary>
		/// Reads ECAT 7.x image header
		/// </summary>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="filename">Ecat7 file that is read</param>
		/// <see cref="MatBLKSIZE"/>
		/// <returns>Ecat7 image header containing header tacs from file</returns>
		protected static Ecat7_imageheader Ecat7ReadImageheader(Ecat7_MatDir matdir, string filename)
		{
			byte[] buf = new byte[MatBLKSIZE];
			Ecat7_imageheader h = new Ecat7_imageheader();
			using (FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				int blk = matdir.strtblk;

				/* Seek the subheader block */
				filestream.Seek((blk - 1) * MatBLKSIZE, SeekOrigin.Begin);
				if (filestream.Position != (blk - 1) * MatBLKSIZE)
				{
					throw new TPCEcat7FileException("Invalid stream position after seek.");
				}
				/* Read the header block */
				if (filestream.Read(buf, 0, MatBLKSIZE) < MatBLKSIZE)
				{
					throw new TPCEcat7FileException(filestream.Position + " Failed to read all image header tacs.");
				}

				/* Copy the header fields and swap if necessary */
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 0, 2); h.data_type = (Ecat7_imageheader.data_type_e)BitConverter.ToInt16(buf, 0);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 2, 2); h.num_dimensions = BitConverter.ToInt16(buf, 2);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 4, 2); h.x_dimension = BitConverter.ToInt16(buf, 4);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 6, 2); h.y_dimension = BitConverter.ToInt16(buf, 6);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 8, 2); h.z_dimension = BitConverter.ToInt16(buf, 8);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 10, 4); h.x_offset = BitConverter.ToSingle(buf, 10);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 14, 4); h.y_offset = BitConverter.ToSingle(buf, 14);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 18, 4); h.z_offset = BitConverter.ToSingle(buf, 18);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 22, 4); h.recon_zoom = BitConverter.ToSingle(buf, 22);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 26, 4); h.scale_factor = BitConverter.ToSingle(buf, 26);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 30, 2); h.image_min = BitConverter.ToInt16(buf, 30);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 32, 2); h.image_max = BitConverter.ToInt16(buf, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 34, 4); h.x_pixel_size = BitConverter.ToSingle(buf, 34);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 38, 4); h.y_pixel_size = BitConverter.ToSingle(buf, 38);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 42, 4); h.z_pixel_size = BitConverter.ToSingle(buf, 42);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 46, 4); h.frame_duration = BitConverter.ToInt32(buf, 46);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 50, 4); h.frame_start_time = BitConverter.ToInt32(buf, 50);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 54, 2); h.filter_code = (Ecat7_imageheader.filter_code_e)BitConverter.ToInt16(buf, 54);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 56, 4); h.x_resolution = BitConverter.ToSingle(buf, 56);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 60, 4); h.y_resolution = BitConverter.ToSingle(buf, 60);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 64, 4); h.z_resolution = BitConverter.ToSingle(buf, 64);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 68, 4); h.num_r_elements = BitConverter.ToSingle(buf, 68);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 72, 4); h.num_angles = BitConverter.ToSingle(buf, 72);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 76, 4); h.z_rotation_angle = BitConverter.ToSingle(buf, 76);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 80, 4); h.decay_corr_fctr = BitConverter.ToSingle(buf, 80);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 84, 4); h.processing_code = BitConverter.ToUInt32(buf, 84);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 88, 4); h.gate_duration = BitConverter.ToInt32(buf, 88);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 92, 4); h.r_wave_offset = BitConverter.ToInt32(buf, 92);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 96, 4); h.num_accepted_beats = BitConverter.ToInt32(buf, 96);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 100, 4); h.filter_cutoff_frequency = BitConverter.ToSingle(buf, 100);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 104, 4); h.filter_resolution = BitConverter.ToSingle(buf, 104);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 108, 4); h.filter_ramp_slope = BitConverter.ToSingle(buf, 108);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 112, 2); h.filter_order = BitConverter.ToInt16(buf, 112);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 114, 4); h.filter_scatter_fraction = BitConverter.ToSingle(buf, 114);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 118, 4); h.filter_scatter_slope = BitConverter.ToSingle(buf, 118);
				h.annotation = toCharArray(buf, 122, 40);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 162, 4); h.mt_1_1 = BitConverter.ToSingle(buf, 162);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 166, 4); h.mt_1_2 = BitConverter.ToSingle(buf, 166);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 170, 4); h.mt_1_3 = BitConverter.ToSingle(buf, 170);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 174, 4); h.mt_2_1 = BitConverter.ToSingle(buf, 174);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 178, 4); h.mt_2_2 = BitConverter.ToSingle(buf, 178);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 182, 4); h.mt_2_3 = BitConverter.ToSingle(buf, 182);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 186, 4); h.mt_3_1 = BitConverter.ToSingle(buf, 186);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 190, 4); h.mt_3_2 = BitConverter.ToSingle(buf, 190);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 194, 4); h.mt_3_3 = BitConverter.ToSingle(buf, 194);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 198, 4); h.rfilter_cutoff = BitConverter.ToSingle(buf, 198);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 202, 4); h.rfilter_resolution = BitConverter.ToSingle(buf, 202);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 206, 2); h.rfilter_code = BitConverter.ToInt16(buf, 206);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 208, 2); h.rfilter_order = BitConverter.ToInt16(buf, 208);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 210, 4); h.zfilter_cutoff = BitConverter.ToSingle(buf, 210);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 214, 4); h.zfilter_resolution = BitConverter.ToSingle(buf, 214);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 218, 2); h.zfilter_code = BitConverter.ToInt16(buf, 218);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 220, 2); h.zfilter_order = BitConverter.ToInt16(buf, 220);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 222, 4); h.mt_1_4 = BitConverter.ToSingle(buf, 222);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 226, 4); h.mt_2_4 = BitConverter.ToSingle(buf, 226);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 230, 4); h.mt_3_4 = BitConverter.ToSingle(buf, 230);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 234, 2); h.scatter_type = (Ecat7_imageheader.scatter_type_e)BitConverter.ToInt16(buf, 234);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 236, 2); h.recon_type = (Ecat7_imageheader.recon_type_e)BitConverter.ToInt16(buf, 236);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 238, 2); h.recon_views = BitConverter.ToInt16(buf, 238);
				h.fill_cti = ToShortArray(buf, 240, 87);
				h.fill_user = ToShortArray(buf, 414, 48);
			}
			return h;
		}

		/// <summary>
		/// Writes ECAT 7.x image header
		/// </summary>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="h">image header</param>
		/// <param name="filename">Ecat7 file where header is written</param>
		protected static void Ecat7WriteImageheader(Ecat7_MatDir matdir, Ecat7_imageheader h, string filename)
		{
			byte[] buf = new byte[MatBLKSIZE];
			using (FileStream filestream = new FileStream(filename, FileMode.OpenOrCreate))
			{
				/* Seek the proper block */
				try
				{
					filestream.Seek((matdir.strtblk - 1) * MatBLKSIZE, SeekOrigin.Begin);
				}
				catch (Exception e)
				{
					throw new TPCEcat7FileException("Failed to seek position " + (matdir.strtblk - 1) * MatBLKSIZE + " for imageheader writing:" + e.Message);
				}

				/* Copy the header fields and swap if necessary */
				BitConverter.GetBytes((short)h.data_type).CopyTo(buf, 0); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 0, 2);
				BitConverter.GetBytes(h.num_dimensions).CopyTo(buf, 2); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 2, 2);
				BitConverter.GetBytes(h.x_dimension).CopyTo(buf, 4); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 4, 2);
				BitConverter.GetBytes(h.y_dimension).CopyTo(buf, 6); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 6, 2);
				BitConverter.GetBytes(h.z_dimension).CopyTo(buf, 8); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 8, 2);
				BitConverter.GetBytes(h.x_offset).CopyTo(buf, 10); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 10, 4);
				BitConverter.GetBytes(h.y_offset).CopyTo(buf, 14); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 14, 4);
				BitConverter.GetBytes(h.z_offset).CopyTo(buf, 18); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 18, 4);
				BitConverter.GetBytes(h.recon_zoom).CopyTo(buf, 22); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 22, 4);
				BitConverter.GetBytes(h.scale_factor).CopyTo(buf, 26); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 26, 4);
				BitConverter.GetBytes(h.image_min).CopyTo(buf, 30); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 30, 2);
				BitConverter.GetBytes(h.image_max).CopyTo(buf, 32); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 32, 2);
				BitConverter.GetBytes(h.x_pixel_size).CopyTo(buf, 34); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 34, 4);
				BitConverter.GetBytes(h.y_pixel_size).CopyTo(buf, 38); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 38, 4);
				BitConverter.GetBytes(h.z_pixel_size).CopyTo(buf, 42); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 42, 4);
				BitConverter.GetBytes(h.frame_duration).CopyTo(buf, 46); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 46, 4);
				BitConverter.GetBytes(h.frame_start_time).CopyTo(buf, 50); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 50, 4);
				BitConverter.GetBytes((short)h.filter_code).CopyTo(buf, 54); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 54, 2);
				BitConverter.GetBytes(h.x_resolution).CopyTo(buf, 56); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 56, 4);
				BitConverter.GetBytes(h.y_resolution).CopyTo(buf, 60); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 60, 4);
				BitConverter.GetBytes(h.z_resolution).CopyTo(buf, 64); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 64, 4);
				BitConverter.GetBytes(h.num_r_elements).CopyTo(buf, 68); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 68, 4);
				BitConverter.GetBytes(h.num_angles).CopyTo(buf, 72); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 72, 4);
				BitConverter.GetBytes(h.z_rotation_angle).CopyTo(buf, 76); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 76, 4);
				BitConverter.GetBytes(h.decay_corr_fctr).CopyTo(buf, 80); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 80, 4);
				BitConverter.GetBytes(h.processing_code).CopyTo(buf, 84); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 84, 4);
				BitConverter.GetBytes(h.gate_duration).CopyTo(buf, 88); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 88, 4);
				BitConverter.GetBytes(h.r_wave_offset).CopyTo(buf, 92); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 92, 4);
				BitConverter.GetBytes(h.num_accepted_beats).CopyTo(buf, 96); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 96, 4);
				BitConverter.GetBytes(h.filter_cutoff_frequency).CopyTo(buf, 100); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 100, 4);
				BitConverter.GetBytes(h.filter_resolution).CopyTo(buf, 104); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 104, 4);
				BitConverter.GetBytes(h.filter_ramp_slope).CopyTo(buf, 108); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 108, 4);
				BitConverter.GetBytes(h.filter_order).CopyTo(buf, 112); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 112, 2);
				BitConverter.GetBytes(h.filter_scatter_fraction).CopyTo(buf, 114); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 114, 4);
				BitConverter.GetBytes(h.filter_scatter_slope).CopyTo(buf, 118); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 118, 4);
				ToBytesArray(h.annotation).CopyTo(buf, 122);
				BitConverter.GetBytes(h.mt_1_1).CopyTo(buf, 162); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 162, 4);
				BitConverter.GetBytes(h.mt_1_2).CopyTo(buf, 166); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 166, 4);
				BitConverter.GetBytes(h.mt_1_3).CopyTo(buf, 170); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 170, 4);
				BitConverter.GetBytes(h.mt_2_1).CopyTo(buf, 174); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 174, 4);
				BitConverter.GetBytes(h.mt_2_2).CopyTo(buf, 178); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 178, 4);
				BitConverter.GetBytes(h.mt_2_3).CopyTo(buf, 182); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 182, 4);
				BitConverter.GetBytes(h.mt_3_1).CopyTo(buf, 186); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 186, 4);
				BitConverter.GetBytes(h.mt_3_2).CopyTo(buf, 190); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 190, 4);
				BitConverter.GetBytes(h.mt_3_3).CopyTo(buf, 194); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 194, 4);
				BitConverter.GetBytes(h.rfilter_cutoff).CopyTo(buf, 198); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 198, 4);
				BitConverter.GetBytes(h.rfilter_resolution).CopyTo(buf, 202); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 202, 4);
				BitConverter.GetBytes(h.rfilter_code).CopyTo(buf, 206); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 206, 2);
				BitConverter.GetBytes(h.rfilter_order).CopyTo(buf, 208); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 208, 2);
				BitConverter.GetBytes(h.zfilter_cutoff).CopyTo(buf, 210); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 210, 4);
				BitConverter.GetBytes(h.zfilter_resolution).CopyTo(buf, 214); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 214, 4);
				BitConverter.GetBytes(h.zfilter_code).CopyTo(buf, 218); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 218, 2);
				BitConverter.GetBytes(h.zfilter_order).CopyTo(buf, 220); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 220, 2);
				BitConverter.GetBytes(h.mt_1_4).CopyTo(buf, 222); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 222, 4);
				BitConverter.GetBytes(h.mt_2_4).CopyTo(buf, 226); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 226, 4);
				BitConverter.GetBytes(h.mt_3_4).CopyTo(buf, 230); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 230, 4);
				BitConverter.GetBytes((short)h.scatter_type).CopyTo(buf, 234); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 234, 2);
				BitConverter.GetBytes((short)h.recon_type).CopyTo(buf, 236); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 236, 2);
				BitConverter.GetBytes(h.recon_views).CopyTo(buf, 238); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 238, 2);
				ToBytesArray(h.fill_cti).CopyTo(buf, 240);
				ToBytesArray(h.fill_user).CopyTo(buf, 414);
				//Console.WriteLine("writing imageheader: "+buf.Length+" bytes to "+filestream.Position);
				filestream.Write(buf, 0, buf.Length);
			}
		}

		/// <summary>
		/// Write ECAT 7.x matrix tacs to a specified file position.
		/// Data must be represented in current machines byte order, and it is
		/// always saved in big endian byte order.
		/// </summary>
		/// <param name="start_block">block number from where to start writing</param>
		/// <param name="data">tacs in byte array</param>
		/// <see cref="MatBLKSIZE"/>
		protected void Ecat7WriteMatrixdata(int start_block, byte[] data)
		{
			byte[] buf = new byte[MatBLKSIZE];
			int i, blkNr, dataSize, byteNr;
			if (start_block < 1) throw new TPCInvalidArgumentsException("Cannot write matrix tacs before 1st block.");
			if (data.Length < 1) throw new TPCInvalidArgumentsException("No tacs in input array.");
			using (FileStream filestream = new FileStream(filename, FileMode.OpenOrCreate))
			{
				/* block nr taken by all pixels */
				blkNr = (data.Length + MatBLKSIZE - 1) / MatBLKSIZE;
				/* Search the place for writing */
				filestream.Seek((start_block) * MatBLKSIZE, SeekOrigin.Begin);
				if (filestream.Position != (start_block) * MatBLKSIZE)
					throw new TPCEcat7FileException("Failed to seek correct point in file.");
				/* Write blocks one at a time */
				dataSize = data.Length;
				for (i = 0; i < blkNr; i++)
				{
					byteNr = (dataSize < MatBLKSIZE) ? dataSize : MatBLKSIZE;
					Array.Copy(data, i * MatBLKSIZE, buf, 0, byteNr);
					/* Change matrix byte order in little endian platforms */
					BitSwapper.swapBytes(buf, imgHeader.Datatype);
					/* Write block */
					filestream.Write(buf, 0, MatBLKSIZE);
					/* Prepare for the next block */
					dataSize -= byteNr;
					if (dataSize < 0) break;
				} /* next block */
			}
		}

		/// <summary>
		/// Read ECAT7 matrix tacs and convert byte order if necessary. Datatype 
		/// field is used to determine if byte swpping is necessary.
		/// </summary>
		/// <param name="stream">open input stream</param>
		/// <param name="start_block">Matrix subheader record number</param>
		/// <param name="end_block">last block to read</param>
		/// <returns>Ecat7 matrix (intensity value) tacs from this Ecat7 file</returns>
		protected byte[] Ecat7ReadMatrixdata(Stream stream, int start_block, int end_block)
		{
			if (end_block <= start_block) throw new TPCEcat7FileException("end block is not after start block");
			byte[] buf = new byte[MatBLKSIZE * (end_block - start_block)];
			int ret = 0;

			/* Check the arguments */
			if (start_block < 1) throw new TPCInvalidArgumentsException("Matrix subheader record number.");
			/* Seek the first tacs block */
			stream.Seek((start_block) * MatBLKSIZE, SeekOrigin.Begin);
			if (stream.Position != (start_block) * MatBLKSIZE) throw new TPCEcat7FileException("Failed to seek correct point in file.");
			/* Read the tacs blocks */
			try
			{
				ret = stream.Read(buf, 0, MatBLKSIZE * (end_block - start_block));
			}
			catch (Exception e)
			{
				throw new TPCEcat7FileException("Error while reading image tacs:" + e.Message);
			}
			if (ret < MatBLKSIZE * (end_block - start_block))
				throw new TPCEcat7FileException("Failed to read all of " + MatBLKSIZE * (end_block - start_block) + " matrix tacs. Memory usage:" + System.GC.GetTotalMemory(false) + " bytes.");
			BitSwapper.swapBytes(buf, imgHeader.Datatype);
			return buf;
		}

		/// <summary>
		/// Read ECAT7 image matrix tacs.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="offset">offset for writing into image</param>
		/// <param name="stream">open input stream</param>
		/// <param name="matdir">matrix directory entry</param>
		/// <param name="h">imageheader of matrix</param>
		/// <returns>matrix tacs that is read, allways dynamic</returns>
		protected void Ecat7ReadImageMatrix(ref Image image, int offset, Stream stream, Ecat7_MatDir matdir, Ecat7_imageheader h)
		{
			/* Read subheader */
			int pxlNr = h.x_dimension * h.y_dimension;
			if (h.num_dimensions > 2) pxlNr *= h.z_dimension;
			if (pxlNr <= 0)
			{
				throw new TPCEcat7FileException("Invalid matrix dimensions: [" + h.x_dimension + ", " + h.y_dimension + ", " + h.z_dimension + "]");
			}

			/* Read matrix tacs */
			if (matdir.endblk < matdir.strtblk)
				throw new TPCInvalidArgumentsException("last_block number is smaller than first_block number");

			//use smaller value if subheader declares different number of planes than in mainheader
			if (image.DimZ < h.z_dimension) h.z_dimension = (short)image.DimZ;

			GetPixelData(ref image, offset, stream, new IntLimits(h.x_dimension, h.y_dimension, h.z_dimension), Ecat7File.Ecat7_imageheader.convertFrom(h.data_type), matdir.strtblk * MatBLKSIZE, h.scale_factor);
		}

		private void WriteMainHeader(ImageHeader head)
		{
			try { EcatHeader.acquisition_mode = (Ecat7Header.Acquisition_mode_e)head.GetValue("acquisition_mode"); }
			catch (Exception) { }
			try { EcatHeader.acquisition_type = (Ecat7Header.Acquisition_type_e)head.GetValue("acquisition_type"); }
			catch (Exception) { }
			try { EcatHeader.angular_compression = (short)head.GetValue("angular_compression"); }
			catch (Exception) { }
			try { EcatHeader.axial_samp_mode = (short)head.GetValue("axial_samp_mode"); }
			catch (Exception) { }
			try { EcatHeader.bed_elevation = (float)head.GetValue("bed_elevation"); }
			catch (Exception) { }
			try { EcatHeader.bed_position = (float[])head.GetValue("bed_position"); }
			catch (Exception) { }
			try { EcatHeader.bin_size = (float)head.GetValue("bin_size"); }
			catch (Exception) { }
			try { EcatHeader.branching_fraction = (float)head.GetValue("branching_fraction"); }
			catch (Exception) { }
			try { EcatHeader.calibration_units = (short)head.GetValue("calibration_units"); }
			catch (Exception) { }
			try { EcatHeader.calibration_units_label = (short)head.GetValue("calibration_units_label"); }
			catch (Exception) { }
			try { EcatHeader.coin_samp_mode = (short)head.GetValue("coin_samp_mode"); }
			catch (Exception) { }
			try { EcatHeader.compression_code = (Ecat7Header.Compression_code_e)head.GetValue("compression_code"); }
			catch (Exception) { }
			try { EcatHeader.data_units = (char[])head.GetValue("data_units"); }
			catch (Exception) { }
			try { EcatHeader.distance_scanned = (float)head.GetValue("distance_scanned"); }
			catch (Exception) { }
			try { EcatHeader.dosage = (float)head.GetValue("dosage"); }
			catch (Exception) { }
			try { EcatHeader.dose_start_time = (int)head.GetValue("dose_start_time"); }
			catch (Exception) { }
			try { EcatHeader.ecat_calibration_factor = (float)head.GetValue("ecat_calibration_factor"); }
			catch (Exception) { }
			try { EcatHeader.facility_name = (char[])head.GetValue("facility_name"); }
			catch (Exception) { }
			try { EcatHeader.file_type = (Ecat7Header.datatype_e)head.GetValue("file_type"); }
			catch (Exception) { }
			try { EcatHeader.fill = (short[])head.GetValue("fill"); }
			catch (Exception) { }
			try { EcatHeader.gantry_rotation = (float)head.GetValue("gantry_rotation"); }
			catch (Exception) { }
			try { EcatHeader.gantry_tilt = (float)head.GetValue("gantry_tilt"); }
			catch (Exception) { }
			try { EcatHeader.init_bed_position = (float)head.GetValue("init_bed_position"); }
			catch (Exception) { }
			try { EcatHeader.intrinsic_tilt = (float)head.GetValue("intrinsic_tilt"); }
			catch (Exception) { }
			try { EcatHeader.isotope_halflife = (float)head.GetValue("isotope_halflife"); }
			catch (Exception) { }
			try { EcatHeader.isotope_name = (char[])head.GetValue("isotope_name"); }
			catch (Exception) { }
			try { EcatHeader.lwr_sctr_thres = (short)head.GetValue("lwr_sctr_thres"); }
			catch (Exception) { }
			try { EcatHeader.lwr_true_thres = (short)head.GetValue("lwr_true_thres"); }
			catch (Exception) { }
			try { EcatHeader.magic_number = (char[])head.GetValue("magic_number"); }
			catch (Exception) { }
			try { EcatHeader.num_bed_pos = (short)head.GetValue("num_bed_pos"); }
			catch (Exception) { }
			try { EcatHeader.num_frames = (short)head.GetValue("num_frames"); }
			catch (Exception) { }
			try { EcatHeader.num_gates = (short)head.GetValue("num_gates"); }
			catch (Exception) { }
			try { EcatHeader.num_planes = (short)head.GetValue("num_planes"); }
			catch (Exception) { }
			try { EcatHeader.operator_name = (char[])head.GetValue("operator_name"); }
			catch (Exception) { }
			try { EcatHeader.original_file_name = (char[])head.GetValue("original_file_name"); }
			catch (Exception) { }
			try { EcatHeader.patient_age = (float)head.GetValue("patient_age"); }
			catch (Exception) { }
			try { EcatHeader.patient_birth_date = (int)head.GetValue("patient_birth_date"); }
			catch (Exception) { }
			try { EcatHeader.patient_dexterity = (char)head.GetValue("patient_dexterity"); }
			catch (Exception) { }
			try { EcatHeader.patient_height = (float)head.GetValue("patient_height"); }
			catch (Exception) { }
			try { EcatHeader.patient_id = (char[])head.GetValue("patient_id"); }
			catch (Exception) { }
			try { EcatHeader.patient_name = (char[])head.GetValue("patient_name"); }
			catch (Exception) { }
			try { EcatHeader.patient_sex = (TPClib.Image.ImageHeader.Patient_sex_e)head.GetValue("patient_sex"); }
			catch (Exception) { }
			try { EcatHeader.patient_weight = (float)head.GetValue("patient_weight"); }
			catch (Exception) { }
			try { EcatHeader.physician_name = (char[])head.GetValue("physician_name"); }
			catch (Exception) { }
			try { EcatHeader.plane_separation = (float)head.GetValue("plane_separation"); }
			catch (Exception) { }
			try { EcatHeader.radiopharmaceutical = (char[])head.GetValue("radiopharmaceutical"); }
			catch (Exception) { }
			try { EcatHeader.scan_start_time = (int)head.GetValue("scan_start_time"); }
			catch (Exception) { }
			try { EcatHeader.septa_state = (short)head.GetValue("septa_state"); }
			catch (Exception) { }
			try { EcatHeader.serial_number = (char[])head.GetValue("serial_number"); }
			catch (Exception) { }
			try { EcatHeader.study_description = (char[])head.GetValue("study_description"); }
			catch (Exception) { }
			try { EcatHeader.study_type = (char[])head.GetValue("study_type"); }
			catch (Exception) { }
			try { EcatHeader.sw_version = (short)head.GetValue("sw_version"); }
			catch (Exception) { }
			try { EcatHeader.system_type = (short)head.GetValue("system_type"); }
			catch (Exception) { }
			try { EcatHeader.transaxial_fov = (float)head.GetValue("transaxial_fov"); }
			catch (Exception) { }
			try { EcatHeader.transm_source_type = (Ecat7Header.Transm_source_type_e)head.GetValue("transm_source_type"); }
			catch (Exception) { }
			try { EcatHeader.upr_true_thres = (short)head.GetValue("upr_true_thres"); }
			catch (Exception) { }
			try { EcatHeader.user_process_code = (char[])head.GetValue("user_process_code"); }
			catch (Exception) { }
			try { EcatHeader.well_counter_corr_factor = (float)head.GetValue("well_counter_corr_factor"); }
			catch (Exception) { }
			try { EcatHeader.wobble_speed = (short)head.GetValue("wobble_speed"); }
			catch (Exception) { }

			/*** Write main header ***/
			EcatHeader.num_bed_pos = (short)(imgHeader.Dim.Beds > 1 ? imgHeader.Dim.Beds : 0);
			EcatHeader.num_gates = (short)imgHeader.Dim.Gates;
			EcatHeader.num_frames = (short)imgHeader.Dim.Frames;
			EcatHeader.num_planes = (short)imgHeader.Dim.Planes;

			if (EcatHeader.file_type == Ecat7Header.datatype_e.ecat7_UNKNOWN)
				EcatHeader.file_type = EcatHeader.resolveFileType(head.Datatype);

			Ecat7WriteMainheader(filename, EcatHeader);
		}

		#endregion

		#region Public methods

		/// <summary>
		/// Read ECAT 7.x main header.
		/// </summary>
		/// <param name="filename">full path to file that is read</param>
		/// <returns>Ecat7 main header</returns>
		/// <exception cref="TPCEcat7FileException">if error occurred while reading</exception>
		public static Ecat7Header Ecat7ReadMainheader(string filename)
		{
			if (!CheckFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");

			byte[] buf = new byte[MatBLKSIZE];
			string stringbuffer = "";
			Ecat7Header h = new Ecat7Header();
			using (FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{

				/* Seek the first block */
				filestream.Seek(0, SeekOrigin.Begin);
				/* Read the header block */
				filestream.Read(buf, 0, MatBLKSIZE);
				for (int i = 0; i < buf.Length; i++)
				{
					stringbuffer += (char)buf[i];
				}

				/* Copy the header fields and swap if necessary */
				h.magic_number = stringbuffer.ToCharArray(0, 14);
				h.original_file_name = stringbuffer.ToCharArray(14, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 46, 2); h.sw_version = BitConverter.ToInt16(buf, 46);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 48, 2); h.system_type = BitConverter.ToInt16(buf, 48);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 50, 2); h.file_type = (Ecat7Header.datatype_e)BitConverter.ToUInt16(buf, 50);
				h.serial_number = toCharArray(buf, 52, 10);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 62, 4); h.scan_start_time = BitConverter.ToInt32(buf, 62);
				h.isotope_name = toCharArray(buf, 66, 8);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 74, 4); h.isotope_halflife = BitConverter.ToSingle(buf, 74);
				h.radiopharmaceutical = toCharArray(buf, 78, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 110, 4); h.gantry_tilt = BitConverter.ToSingle(buf, 110);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 114, 4); h.gantry_rotation = BitConverter.ToSingle(buf, 114);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 118, 4); h.bed_elevation = BitConverter.ToSingle(buf, 118);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 122, 4); h.intrinsic_tilt = BitConverter.ToSingle(buf, 122);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 126, 4); h.wobble_speed = BitConverter.ToInt16(buf, 126);
				h.transm_source_type = (Ecat7Header.Transm_source_type_e)BitConverter.ToInt16(buf, 128);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 130, 4); h.distance_scanned = BitConverter.ToSingle(buf, 130);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 134, 4); h.transaxial_fov = BitConverter.ToSingle(buf, 134);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 138, 2); h.angular_compression = BitConverter.ToInt16(buf, 138);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 140, 2); h.coin_samp_mode = BitConverter.ToInt16(buf, 140);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 142, 2); h.axial_samp_mode = BitConverter.ToInt16(buf, 142);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 144, 4); h.ecat_calibration_factor = BitConverter.ToSingle(buf, 144);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 148, 2); h.calibration_units = BitConverter.ToInt16(buf, 148);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 150, 2); h.calibration_units_label = BitConverter.ToInt16(buf, 150);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 152, 2); h.compression_code = (Ecat7Header.Compression_code_e)BitConverter.ToInt16(buf, 152);
				h.study_type = toCharArray(buf, 154, 12);
				h.patient_id = toCharArray(buf, 166, 16);
				h.patient_name = toCharArray(buf, 182, 32);
				h.patient_sex = (ImageHeader.Patient_sex_e)BitConverter.ToChar(buf, 214);
				h.patient_dexterity = BitConverter.ToChar(buf, 215);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 216, 4); h.patient_age = BitConverter.ToSingle(buf, 216);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 220, 4); h.patient_height = BitConverter.ToSingle(buf, 220);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 224, 4); h.patient_weight = BitConverter.ToSingle(buf, 224);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 228, 4); h.patient_birth_date = BitConverter.ToInt32(buf, 228);
				h.physician_name = toCharArray(buf, 232, 32);
				h.operator_name = toCharArray(buf, 264, 32);
				h.study_description = toCharArray(buf, 296, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 328, 2); h.acquisition_type = (Ecat7Header.Acquisition_type_e)BitConverter.ToInt16(buf, 328);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 330, 2); h.patient_orientation = (Ecat7Header.Patient_orientation)BitConverter.ToUInt16(buf, 330);
				h.facility_name = toCharArray(buf, 332, 20);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 352, 2); h.num_planes = BitConverter.ToInt16(buf, 352);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 354, 2); h.num_frames = BitConverter.ToInt16(buf, 354);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 356, 2); h.num_gates = BitConverter.ToInt16(buf, 356);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 358, 2); h.num_bed_pos = BitConverter.ToInt16(buf, 358);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 360, 4); h.init_bed_position = BitConverter.ToSingle(buf, 360);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 364, 60); h.bed_position = ToFloatArray(buf, 364, 15);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 424, 4); h.plane_separation = BitConverter.ToSingle(buf, 424);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 428, 2); h.lwr_sctr_thres = BitConverter.ToInt16(buf, 428);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 430, 2); h.lwr_true_thres = BitConverter.ToInt16(buf, 430);
				h.user_process_code = toCharArray(buf, 434, 10);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 444, 2); h.acquisition_mode = (Ecat7Header.Acquisition_mode_e)BitConverter.ToInt16(buf, 444);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 446, 2); h.bin_size = BitConverter.ToSingle(buf, 446);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 450, 2); h.branching_fraction = BitConverter.ToSingle(buf, 450);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 454, 2); h.dose_start_time = BitConverter.ToInt32(buf, 454);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 458, 2); h.dosage = BitConverter.ToSingle(buf, 458);
				if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 462, 2); h.well_counter_corr_factor = BitConverter.ToSingle(buf, 462);
				h.data_units = toCharArray(buf, 466, 32);
				if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 498, 2); h.septa_state = BitConverter.ToInt16(buf, 498);
				h.fill = ToShortArray(buf, 500, 6);
			}
			return h;
		}

		/// <summary>
		/// Write ECAT 7.2 main header.
		/// </summary>
		/// <param name="filename">full path to file that is read</param>
		/// <param name="h">main header to write</param>
		/// <exception cref="TPCEcat7FileException">if error occurred while writing</exception>
		public static void Ecat7WriteMainheader(string filename, Ecat7Header h)
		{
			byte[] buf = new byte[MatBLKSIZE];
			if (h == null) throw new TPCEcat7FileException("header tacs is null");
			using (FileStream filestream = new FileStream(filename, FileMode.OpenOrCreate))
			{
				/* Seek the first block */
				filestream.Seek(0, SeekOrigin.Begin);

				/* Copy the header fields and swap if necessary */
				ToBytesArray(h.magic_number).CopyTo(buf, 0);
				ToBytesArray(h.original_file_name).CopyTo(buf, 14);
				BitConverter.GetBytes(h.sw_version).CopyTo(buf, 46); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 46, 2);
				BitConverter.GetBytes(h.system_type).CopyTo(buf, 48); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 48, 2);
				BitConverter.GetBytes((short)h.file_type).CopyTo(buf, 50); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 50, 2);
				ToBytesArray(h.serial_number).CopyTo(buf, 52);
				BitConverter.GetBytes(h.scan_start_time).CopyTo(buf, 62); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 62, 4);
				ToBytesArray(h.isotope_name).CopyTo(buf, 66);
				BitConverter.GetBytes(h.isotope_halflife).CopyTo(buf, 74); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 74, 4);
				ToBytesArray(h.radiopharmaceutical).CopyTo(buf, 78);
				BitConverter.GetBytes(h.gantry_tilt).CopyTo(buf, 110); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 110, 4);
				BitConverter.GetBytes(h.gantry_rotation).CopyTo(buf, 114); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 114, 4);
				BitConverter.GetBytes(h.bed_elevation).CopyTo(buf, 118); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 118, 4);
				BitConverter.GetBytes(h.intrinsic_tilt).CopyTo(buf, 122); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 122, 4);
				BitConverter.GetBytes(h.wobble_speed).CopyTo(buf, 126); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 126, 2);
				BitConverter.GetBytes((short)h.transm_source_type).CopyTo(buf, 128); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 128, 2);
				BitConverter.GetBytes(h.distance_scanned).CopyTo(buf, 130); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 130, 4);
				BitConverter.GetBytes(h.transaxial_fov).CopyTo(buf, 134); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 134, 4);
				BitConverter.GetBytes(h.angular_compression).CopyTo(buf, 138); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 138, 2);
				BitConverter.GetBytes(h.coin_samp_mode).CopyTo(buf, 140); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 140, 2);
				BitConverter.GetBytes(h.axial_samp_mode).CopyTo(buf, 142); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 142, 2);
				BitConverter.GetBytes(h.ecat_calibration_factor).CopyTo(buf, 144); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 144, 4);
				BitConverter.GetBytes(h.calibration_units).CopyTo(buf, 148); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 148, 2);
				BitConverter.GetBytes(h.calibration_units_label).CopyTo(buf, 150); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 150, 2);
				BitConverter.GetBytes((short)h.compression_code).CopyTo(buf, 152); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 152, 2);
				ToBytesArray(h.study_type).CopyTo(buf, 154);
				ToBytesArray(h.patient_id).CopyTo(buf, 166);
				ToBytesArray(h.patient_name).CopyTo(buf, 182);
				BitConverter.GetBytes((char)h.patient_sex).CopyTo(buf, 214);
				BitConverter.GetBytes(h.patient_dexterity).CopyTo(buf, 215);
				BitConverter.GetBytes(h.patient_age).CopyTo(buf, 216); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 216, 4);
				BitConverter.GetBytes(h.patient_height).CopyTo(buf, 220); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 220, 4);
				BitConverter.GetBytes(h.patient_weight).CopyTo(buf, 224); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 224, 4);
				BitConverter.GetBytes(h.patient_birth_date).CopyTo(buf, 228); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 228, 4);
				ToBytesArray(h.physician_name).CopyTo(buf, 232);
				ToBytesArray(h.operator_name).CopyTo(buf, 264);
				ToBytesArray(h.study_description).CopyTo(buf, 296);
				BitConverter.GetBytes((short)h.acquisition_type).CopyTo(buf, 328); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 328, 2);
				BitConverter.GetBytes((ushort)h.patient_orientation).CopyTo(buf, 330); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 330, 2);
				ToBytesArray(h.facility_name).CopyTo(buf, 332);
				BitConverter.GetBytes(h.num_planes).CopyTo(buf, 352); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 352, 2);
				BitConverter.GetBytes(h.num_frames).CopyTo(buf, 354); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 354, 2);
				BitConverter.GetBytes(h.num_gates).CopyTo(buf, 356); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 356, 2);
				BitConverter.GetBytes(h.num_bed_pos).CopyTo(buf, 358); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 358, 2);
				BitConverter.GetBytes(h.init_bed_position).CopyTo(buf, 360); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 360, 4);
				ToBytesArray(h.bed_position).CopyTo(buf, 364); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 364, 15 * 4);
				BitConverter.GetBytes(h.plane_separation).CopyTo(buf, 424); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 424, 4);
				BitConverter.GetBytes(h.lwr_sctr_thres).CopyTo(buf, 428); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 428, 2);
				BitConverter.GetBytes(h.lwr_true_thres).CopyTo(buf, 430); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 430, 2);
				ToBytesArray(h.user_process_code).CopyTo(buf, 434);
				BitConverter.GetBytes((short)h.acquisition_mode).CopyTo(buf, 444); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 444, 2);
				BitConverter.GetBytes(h.bin_size).CopyTo(buf, 446); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 446, 4);
				BitConverter.GetBytes(h.branching_fraction).CopyTo(buf, 450); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 450, 2);
				BitConverter.GetBytes(h.dose_start_time).CopyTo(buf, 454); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 454, 4);
				BitConverter.GetBytes(h.dosage).CopyTo(buf, 458); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 458, 4);
				BitConverter.GetBytes(h.well_counter_corr_factor).CopyTo(buf, 462); if (BitConverter.IsLittleEndian) BitSwapper.swapWordBytes(buf, 462, 4);
				ToBytesArray(h.data_units).CopyTo(buf, 466);
				BitConverter.GetBytes(h.septa_state).CopyTo(buf, 498); if (BitConverter.IsLittleEndian) BitSwapper.swapArrayBytes(buf, 498, 2);
				ToBytesArray(h.fill).CopyTo(buf, 500);
				filestream.Write(buf, 0, buf.Length);
			}
		}

		/// <summary>
		/// Read ECAT 7.x polar map header. Null is placed where matrix does not exist.
		/// </summary>
		/// <param name="filename">file that is read</param>
		/// <returns>array of subheaders</returns>
		/// <exception cref="TPCEcat7FileException">if error occurred while reading file</exception>
		public static Ecat7_polmapheader[] Ecat7ReadPolmapheaders(string filename)
		{
			if (!CheckFormat(filename)) throw new TPCEcat7FileException("Ecat7 format is not recognized.");
			Ecat7_MatDir[] mlist = Ecat7ReadMatlist(filename);
			return Ecat7ReadPolmapheaders(mlist, filename);
		}

		/// <summary>
		/// Returns string representation of ecat7 file type.
		/// </summary>
		/// <param name="type">ecat7 file type</param>
		/// <returns>string representation of type</returns>
		public static string Ecat7filetype(Ecat7Header.datatype_e type)
		{
			switch (type)
			{
				case Ecat7Header.datatype_e.ecat7_UNKNOWN:
					return "unknown";
				case Ecat7Header.datatype_e.ecat7_2DSCAN:
					return "2D sinogram";
				case Ecat7Header.datatype_e.ecat7_IMAGE16:
					return "image-16-bits";
				case Ecat7Header.datatype_e.ecat7_ATTEN:
					return "attenuation correction";
				case Ecat7Header.datatype_e.ecat7_2DNORM:
					return "2D normalization";
				case Ecat7Header.datatype_e.ecat7_POLARMAP:
					return "polar map";
				case Ecat7Header.datatype_e.ecat7_VOLUME8:
					return "volume 8-bits";
				case Ecat7Header.datatype_e.ecat7_VOLUME16:
					return "volume 16-bits";
				case Ecat7Header.datatype_e.ecat7_PROJ:
					return "projection 8-bits";
				case Ecat7Header.datatype_e.ecat7_PROJ16:
					return "projection 16-bits";
				case Ecat7Header.datatype_e.ecat7_IMAGE8:
					return "image 8-bits";
				case Ecat7Header.datatype_e.ecat7_3DSCAN:
					return "3D sinogram 16-bits";
				case Ecat7Header.datatype_e.ecat7_3DSCAN8:
					return "3D sinogram 8-bits";
				case Ecat7Header.datatype_e.ecat7_3DNORM:
					return "3D normalization";
				case Ecat7Header.datatype_e.ecat7_3DSCANFIT:
					return "3D sinogram fit";
			}
			return "Internal error: unrecognized enumerator value:" + type;
		}

		/// <summary>
		/// Returns string describing the ECAT7 acquisition_type
		/// </summary>
		/// <param name="acquisition_type">acquisition type</param>
		/// <returns>string representation of type</returns>
		public static string Ecat7acquisitiontype(Ecat7Header.Acquisition_type_e acquisition_type)
		{
			switch (acquisition_type)
			{
				case Ecat7Header.Acquisition_type_e.Blank: return "blank";
				case Ecat7Header.Acquisition_type_e.Dynamic_emission: return "dynamic emission";
				case Ecat7Header.Acquisition_type_e.Emission_rectilinear: return "emission rectilinear";
				case Ecat7Header.Acquisition_type_e.Gated_emission: return "gated emission";
				case Ecat7Header.Acquisition_type_e.Static_emission: return "static emission";
				case Ecat7Header.Acquisition_type_e.Transmission: return "transmission";
				case Ecat7Header.Acquisition_type_e.Transmission_rectilinear: return "transmission rectilinear";
				case Ecat7Header.Acquisition_type_e.Undefined: return "undefined";
			}
			return "Internal error: unrecognized enumerator value:" + acquisition_type;
		}

		/// <summary>
		/// Print ECAT 7.x image header contents
		/// </summary>
		/// <param name="h">image header object</param>
		public static void Ecat7PrintImageheader(Ecat7_imageheader h)
		{
			Console.WriteLine("data_type := {0:g} ({1:s})", h.data_type, Ecat7_imageheader.ToString(h.data_type));
			Console.WriteLine("num_dimensions := {0:n}", h.num_dimensions);
			Console.WriteLine("x_dimension := {0:n}", h.x_dimension);
			Console.WriteLine("y_dimension := {0:n}", h.y_dimension);
			Console.WriteLine("z_dimension := {0:n}", h.z_dimension);
			Console.WriteLine("x_offset := {0:g}", h.x_offset);
			Console.WriteLine("y_offset := {0:g}", h.y_offset);
			Console.WriteLine("z_offset := {0:g}", h.z_offset);
			Console.WriteLine("recon_zoom := {0:g}", h.recon_zoom);
			Console.WriteLine("scale_factor := {0:E}", h.scale_factor);
			Console.WriteLine("image_min := {0:g}", h.image_min);
			Console.WriteLine("image_max := {0:g}", h.image_max);
			Console.WriteLine("x_pixel_size := {0:g}", h.x_pixel_size);
			Console.WriteLine("y_pixel_size := {0:g}", h.y_pixel_size);
			Console.WriteLine("z_pixel_size := {0:g}", h.z_pixel_size);
			Console.WriteLine("frame_duration := {0:g}", h.frame_duration);
			Console.WriteLine("frame_start_time := {0:g}", h.frame_start_time);
			Console.WriteLine("filter_code := {0:g}", h.filter_code);
			Console.WriteLine("x_resolution := {0:g}", h.x_resolution);
			Console.WriteLine("y_resolution := {0:g}", h.y_resolution);
			Console.WriteLine("z_resolution := {0:g}", h.z_resolution);
			Console.WriteLine("num_r_elements := {0:g}", h.num_r_elements);
			Console.WriteLine("num_angles := {0:g}", h.num_angles);
			Console.WriteLine("z_rotation_angle := {0:g}", h.z_rotation_angle);
			Console.WriteLine("decay_corr_fctr := {0:g}", h.decay_corr_fctr);
			Console.WriteLine("processing_code := {0:n}", h.processing_code);
			Console.WriteLine("gate_duration := {0:n}", h.gate_duration);
			Console.WriteLine("r_wave_offset := {0:n}", h.r_wave_offset);
			Console.WriteLine("num_accepted_beats := {0:n}", h.num_accepted_beats);
			Console.WriteLine("filter_cutoff_frequency := {0:E}", h.filter_cutoff_frequency);
			Console.WriteLine("filter_resolution := {0:E}", h.filter_resolution);
			Console.WriteLine("filter_ramp_slope := {0:E}", h.filter_ramp_slope);
			Console.WriteLine("filter_order := {0:n}", h.filter_order);
			Console.WriteLine("filter_scatter_fraction := {0:E}", h.filter_scatter_fraction);
			Console.WriteLine("filter_scatter_slope := {0:E}", h.filter_scatter_slope);
			Console.WriteLine("annotation := {0:40:s}", new string(h.annotation));
			Console.WriteLine("mt_1_1 := {0:g}", h.mt_1_1);
			Console.WriteLine("mt_1_2 := {0:g}", h.mt_1_2);
			Console.WriteLine("mt_1_3 := {0:g}", h.mt_1_3);
			Console.WriteLine("mt_2_1 := {0:g}", h.mt_2_1);
			Console.WriteLine("mt_2_2 := {0:g}", h.mt_2_2);
			Console.WriteLine("mt_2_3 := {0:g}", h.mt_2_3);
			Console.WriteLine("mt_3_1 := {0:g}", h.mt_3_1);
			Console.WriteLine("mt_3_2 := {0:g}", h.mt_3_2);
			Console.WriteLine("mt_3_3 := {0:g}", h.mt_3_3);
			Console.WriteLine("rfilter_cutoff := {0:g}", h.rfilter_cutoff);
			Console.WriteLine("rfilter_resolution := {0:g}", h.rfilter_resolution);
			Console.WriteLine("rfilter_code := {0:n}", h.rfilter_code);
			Console.WriteLine("rfilter_order := {0:n}", h.rfilter_order);
			Console.WriteLine("zfilter_cutoff := {0:g}", h.zfilter_cutoff);
			Console.WriteLine("zfilter_resolution := {0:g}", h.zfilter_resolution);
			Console.WriteLine("zfilter_code := {0:n}", h.zfilter_code);
			Console.WriteLine("zfilter_order := {0:n}", h.zfilter_order);
			Console.WriteLine("mt_1_4 := {0:g}", h.mt_1_4);
			Console.WriteLine("mt_2_4 := {0:g}", h.mt_2_4);
			Console.WriteLine("mt_3_4 := {0:g}", h.mt_3_4);
			Console.WriteLine("scatter_type := {0:g}", h.scatter_type);
			Console.WriteLine("recon_type := {0:g}", h.recon_type);
			Console.WriteLine("recon_views := {0:n}", h.recon_views);
			Console.Write("fill_cti :=");
			for (int i = 0; i < 87; i++) Console.Write(" {0:n}", h.fill_cti[i]);
			Console.WriteLine();
			Console.Write("fill_user :=");
			for (int i = 0; i < 48; i++) Console.Write(" {0:n}", h.fill_user[i]);
			Console.WriteLine();
		}

		/// <summary>
		/// Print ECAT 7.x main header contents
		/// </summary>
		/// <param name="h">main header object</param>
		public static void Ecat7PrintMainheader(Ecat7Header h)
		{
			PrintCharArray("magic_number := ", h.magic_number);
			PrintCharArray("original_file_name := ", h.original_file_name);
			Console.WriteLine("sw_version := {0:g}", h.sw_version);
			Console.WriteLine("system_type := {0:g}", h.system_type);
			Console.WriteLine("file_type := {0:g} ({1:s})", h.file_type, Ecat7filetype(h.file_type));
			PrintCharArray("serial_number := ", h.serial_number);
			Console.WriteLine("scan_start_time := {0:r}", EcatToDateTime(h.scan_start_time));
			PrintCharArray("isotope_name := ", h.isotope_name);
			Console.WriteLine("isotope_halflife := {0:e} sec", h.isotope_halflife);
			PrintCharArray("radiopharmaceutical := ", h.radiopharmaceutical);
			Console.WriteLine("gantry_tilt := {0:g}", h.gantry_tilt);
			Console.WriteLine("gantry_rotation := {0:g}", h.gantry_rotation);
			Console.WriteLine("bed_elevation := {0:g}", h.bed_elevation);
			Console.WriteLine("intrinsic_tilt := {0:g}", h.intrinsic_tilt);
			Console.WriteLine("wobble_speed := {0:g}", h.wobble_speed);
			Console.WriteLine("transm_source_type := {0:d}", h.transm_source_type);
			Console.WriteLine("distance_scanned := {0:g}", h.distance_scanned);
			Console.WriteLine("transaxial_fov := {0:g}", h.transaxial_fov);
			Console.WriteLine("angular_compression := {0:g}", h.angular_compression);
			Console.WriteLine("coin_samp_mode := {0:g}", h.coin_samp_mode);
			Console.WriteLine("axial_samp_mode := {0:g}", h.axial_samp_mode);
			Console.WriteLine("ecat_calibration_factor := {0:e}", h.ecat_calibration_factor);
			Console.WriteLine("calibration_units := {0:g}", h.calibration_units);
			Console.WriteLine("calibration_units_label := {0:g}", h.calibration_units_label);
			Console.WriteLine("compression_code := {0:g}", h.compression_code);
			PrintCharArray("study_type := ", h.study_type);
			PrintCharArray("patient_id := ", h.patient_id);
			PrintCharArray("patient_name := ", h.patient_name);
			Console.WriteLine("patient_sex := {0:g}", h.patient_sex);
			Console.WriteLine("patient_dexterity := {0:s}", (h.patient_dexterity != 0) ? h.patient_dexterity : (char)32);
			Console.WriteLine("patient_age := {0:g}", h.patient_age);
			Console.WriteLine("patient_height := {0:g}", h.patient_height);
			Console.WriteLine("patient_weight := {0:g}", h.patient_weight);
			Console.WriteLine("patient_birth_date := {0:r}", EcatToDateTime(h.patient_birth_date));
			PrintCharArray("physician_name := ", h.physician_name);
			PrintCharArray("operator_name := ", h.operator_name);
			PrintCharArray("study_description := ", h.study_description);
			Console.WriteLine("acquisition_type := {0:g} ({1:s})", h.acquisition_type, Ecat7acquisitiontype(h.acquisition_type));
			Console.WriteLine("patient_orientation := {0:g}", h.patient_orientation);
			PrintCharArray("facility_name := ", h.facility_name);
			Console.WriteLine("num_planes := {0:g}", h.num_planes);
			Console.WriteLine("num_frames := {0:g}", h.num_frames);
			Console.WriteLine("num_gates := {0:g}", h.num_gates);
			Console.WriteLine("num_bed_pos := {0:g}", h.num_bed_pos);
			Console.WriteLine("init_bed_position := {0:g}", h.init_bed_position);
			Console.Write("bed_position :=");
			for (int i = 0; i < 15; i++) Console.Write(" {0:g}", h.bed_position[i]);
			Console.WriteLine();
			Console.WriteLine("plane_separation := {0:g} cm", h.plane_separation);
			Console.WriteLine("lwr_sctr_thres := {0:g}", h.lwr_sctr_thres);
			Console.WriteLine("lwr_true_thres := {0:g}", h.lwr_true_thres);
			Console.WriteLine("upr_true_thres := {0:g}", h.upr_true_thres);
			PrintCharArray("user_process_code := ", h.user_process_code);
			Console.WriteLine("acquisition_mode := {0:g}", h.acquisition_mode);
			Console.WriteLine("bin_size := {0:g} cm", h.bin_size);
			Console.WriteLine("branching_fraction := {0:g}", h.branching_fraction);
			Console.WriteLine("dose_start_time := {0:r}", EcatToDateTime(h.dose_start_time));
			Console.WriteLine("dosage := {0:g}", h.dosage);
			Console.WriteLine("well_counter_corr_factor := {0:e}", h.well_counter_corr_factor);
			PrintCharArray("data_units := ", h.data_units);
			Console.WriteLine("septa_state := {0:g}", h.septa_state);
			Console.Write("fill_cti :=");
			for (int i = 0; i < 6; i++) Console.Write(" {0:g}", h.fill[i]);
			Console.WriteLine();
		}

		/// <summary>
		/// Returns image object if it is present.
		/// </summary>
		/// <returns>image tacs</returns>
		public Image GetImage()
		{
			if (image == null) throw new TPCException("Tried to get image which is null.");
			return new Image(image);
		}

		/// <summary>
		/// Updates Ecat7 matrix list to apply current image tacs. 'tacs' field in matrix indentifier 
		/// is not supported in matrix list items (it is always set to zero).
		/// </summary>
		/// <param name="dim">image tacs dimensions</param>
		/// <param name="datatype">tacs type for update operation</param>
		/// <returns>matrix list that is consistent with image tacs</returns>
		public void UpdateMatrixList(IntLimits dim, ImageFile.DataType datatype)
		{
			int product = 1;
			if (dim.Length < 2) throw new TPCEcat7FileException("1-dimensional tacs is not supported by this method.");
			//calculate number of voxels in single image frame (or gate etc.)
			if (dim.Length < 3)
			{
				product = dim[0] * dim[1];
			}
			else
			{
				product = dim[0] * dim[1] * dim[2];
			}
			//Console.WriteLine(image.dim);
			//number of frames
			int frames = 1;
			if (dim.Length > IntLimits.FRAMES) frames = dim.GetDimension(IntLimits.FRAMES);
			//number of gates
			int gates = 1;
			if (dim.Length > IntLimits.GATES) gates = dim.GetDimension(IntLimits.GATES);
			//number of bed positions
			int beds = 1;
			if (dim.Length > IntLimits.BEDS) beds = dim.GetDimension(IntLimits.BEDS);
			//number of blocks in item, resized to match 512 blocks 
			int blocks = (int)Math.Ceiling((double)(product * ImageFile.Type2Bytes(datatype)) / 512.0);
			//create matrix list, resized to match 31 item blocks
			int list_size = (int)(Math.Ceiling((double)(frames * gates * beds) / 31.0) * 31.0);
			//Console.WriteLine(list_size + "=" + frames + "*" + gates + "*" + beds);
			mlist = new Ecat7_MatDir[list_size];
			//set tacs in matrix list
			int blk = 3;
			int list_i = 0;
			for (int frame_i = 1; frame_i <= frames; frame_i++)
			{
				for (int gate_i = 1; gate_i <= gates; gate_i++)
				{
					for (int bed_i = 0; bed_i < beds; bed_i++)
					{
						mlist[list_i] = new Ecat7_MatDir(new Ecat7_Matval(frame_i, 1, gate_i, 0, bed_i), blk, blk + blocks, Ecat7_MatDir.status_e.EXISTS);
						//set tacs for next round
						blk = mlist[list_i].endblk + 1;
						//add space for matrix list header block when item block of 31 items becomes full
						if ((list_i + 1) % 31 == 0) blk++;
						list_i++;
					}
				}
			}
			//fill rest of the list with empty entries
			while (list_i < mlist.Length)
			{
				//set list item
				mlist[list_i] = new Ecat7_MatDir(new Ecat7_Matval(0, 0, 0, 0, 0), 0, 0, Ecat7_MatDir.status_e.DELETED);
				list_i++;
			}
			//Console.WriteLine(image.dim);
			//Console.WriteLine(header.dim);
			for (int i = 0; i < mlist.Length; i++)
			{
				//Console.WriteLine(mlist[i]);
			}
		}

		#endregion

		#region Constructors

		/// <summary>
		/// Constructs Ecat7 file with filename.
		/// </summary>
		/// <param name="filename">full name of Ecat7 file</param>
		/// 
		public Ecat7File(string filename)
		{
			this.filename = filename;
			Header = new Ecat7Header();
			imgHeader.IsDynamic = true;
			//set accepted datatype as default
			imgHeader.Datatype = DataType.BIT16_S;
			subheaders = new TPClib.Image.Ecat7File.Ecat7_subheader[0];
		}

		/// <summary>
		/// Constructs an empty (all-zero) Ecat7 file with filename and dimensions.
		/// </summary>
		/// <param name="filename">full name of Ecat7 file</param>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		public Ecat7File(string filename, int dimx, int dimy, int dimz)
			: this(filename, new int[] { dimx, dimy, dimz })
		{
		}

		/// <summary>
		/// Constructs an empty (all-zero) Ecat7 file with filename and dimensions.
		/// </summary>
		/// <param name="filename">full name of Ecat7 file</param>
		/// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
		public Ecat7File(string filename, int[] dim)
			: this(filename)
		{
			int[] d = new int[4];
			int i = 0;
			for (; i < dim.Length; i++)
				d[i] = dim[i];
			for (; i < d.Length; i++)
				d[i] = 1;
			image = new Image(new IntLimits(dim));
			imgHeader.Dim = image.Dim;
			imgHeader.IsDynamic = true;
			image.IsDynamic = true;
			subheaders = new TPClib.Image.Ecat7File.Ecat7_subheader[d[3]];
		}

		/// <summary>
		/// Constructs Ecat7 file.
		/// </summary>
		/// <remarks>default filetype is 'ecat7_IMAGE16'</remarks>
		/// <param name="filename">full name of Ecat7 file</param>
		/// <param name="image">image tacs</param>
		/// <param name="hdr">image header</param>
		public Ecat7File(string filename, Image image, ImageHeader hdr)
			: this(filename)
		{
			this.image = image;
			EcatHeader.num_planes = (short)image.Planes;
			EcatHeader.num_frames = (short)image.Frames;
			EcatHeader.num_gates = 1;
			EcatHeader.num_bed_pos = 1;

			this.filetype = FileType.Ecat7;
			EcatHeader.file_type = EcatHeader.resolveFileType(hdr.Datatype);

			Header = hdr;
			Header.Datatype = DataType.BIT16_S;
			Header.Dim = image.Dim;
			Header.Dataunit = hdr.Dataunit;

			//create subheaders for gates and frames
			subheaders = new Ecat7_subheader[image.Gates * image.Frames];

			float[] start_times = image.GetFrameStartTimes();
			float[] durations = image.GetFrameDurations();
			Header.FrameDurations = durations;
			Header.FrameStartTimes = start_times;

			for (int i = 0; i < subheaders.Length; i++)
			{
				subheaders[i] = new Ecat7_subheader();
				subheaders[i].scale_factor = 1.0f;
				if (image.IsDynamic && i < start_times.Length)
				{
					subheaders[i].frame_start_time = (int)start_times[i];
					subheaders[i].frame_duration = (int)durations[i];
				}
				else
				{
					subheaders[i].frame_start_time = 1000;
					subheaders[i].frame_duration = 0;
				}
			}
		}

		#endregion

		#region ImageFile interface

		/// <summary>
		/// Main header of file.
		/// </summary>
		public Ecat7Header EcatHeader { get { return imgHeader as Ecat7Header; } }

		/// <summary>
		/// 
		/// </summary>
		public override ImageHeader Header
		{
			get
			{
				return imgHeader;
			}
			set
			{
				imgHeader = new Ecat7Header(value);
			}
		}

		/// <summary>
		/// Implementation of ImageFile interface. Reads a ecat7 file and fills header and image variables.
		/// </summary>
		/// <exception cref="TPCEcat7FileException">if file was not opened properly</exception>
		/// <exception cref="TPCUnrecognizedFileFormatException">if other error occurred</exception>
		public override void ReadFile()
		{
			int i = 0;
			IOProcessEventHandler pevent = null;

			if (!CheckFormat(filename)) throw new TPCEcat7FileException("Fileformat not recognized.");

			//Read header
			imgHeader = ReadHeader();

			// Read matrix list 
			mlist = Ecat7ReadMatlist(filename);

			//create image
			image = new Image(imgHeader.Dim);
			image.IsDynamic = imgHeader.IsDynamic;

			/* Read matrices */
			using (FileStream reader = new FileStream(filename, FileMode.Open, FileAccess.Read))
			{
				processed_slices = 0;
				total_number_of_slices_to_process = 0;
				for (i = 0; i < mlist.Length; i++)
				{
					if (mlist[i].status != Ecat7_MatDir.status_e.DELETED)
						total_number_of_slices_to_process += image.DimZ;
				}
				int framelength = image.DimX * image.DimY * image.DimZ;
				int frame = 0;
				for (i = 0; i < mlist.Length; i++)
				{
					if (mlist[i].status == Ecat7_MatDir.status_e.DELETED)
					{
						continue;
					}
					Image subimage;
					//ecat7 is expected to have frame durations in milliseconds
					image.SetFrameStartTime(frame, subheaders[i].frame_start_time);
					image.SetFrameDuration(frame, subheaders[i].frame_duration);

					/* Read subheader and tacs */
					IntLimits subheader_dim;
					switch (EcatHeader.resolveImageType())
					{
						case Ecat7Header.ecat7_imagetype.IMAGE:
							Ecat7ReadImageMatrix(ref image, framelength * frame, reader, mlist[i], (subheaders[i] as Ecat7_imageheader));
							break;
						case Ecat7Header.ecat7_imagetype.POLARMAP:
							subheader_dim = new IntLimits(new int[] { 0, 0, (int)mlist[i].id.plane, (int)mlist[i].id.frame, (int)mlist[i].id.gate, (int)mlist[i].id.bed });
							subimage = (Image)Ecat7ReadPolmapMatrix(reader, mlist[i], subheaders[i] as Ecat7_polmapheader);
							image.SetSubImage(subheader_dim, ref subimage);
							//set voxel so that it has desired size
							break;
						case Ecat7Header.ecat7_imagetype.RAW:
							subheader_dim = new IntLimits(new int[] { 0, 0, (int)mlist[i].id.plane, (int)mlist[i].id.frame, (int)mlist[i].id.gate, (int)mlist[i].id.bed });
							subimage = (Image)Ecat7ReadScanMatrix(reader, mlist[i], subheaders[i] as Ecat7_scanheader);
							image.SetSubImage(subheader_dim, ref subimage);
							// also, correct for dead-time
							if ((subheaders[i] as Ecat7_scanheader).deadtime_correction_factor > 0.0)
							{
								image.Multiply((subheaders[i] as Ecat7_scanheader).deadtime_correction_factor);
							}
							break;
					}
					frame++;
				}
			}
			/* Calibrate */
			if (EcatHeader.ecat_calibration_factor > 0.0)
			{
				image.Multiply(EcatHeader.ecat_calibration_factor);
			}
			else
			{
				EcatHeader.ecat_calibration_factor = 1.0f;
			}

			pevent = IOProgress;
			if (pevent != null)
				pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
		}

		/// <summary>
		/// Implementation of ImageFile interface. Writes a ecat7 file.
		/// </summary>
		/// <param name="img">image tacs that is written</param>
		/// <exception cref="TPCEcat7FileException">if file was not opened properly</exception>
		/// <exception cref="TPCUnrecognizedFileFormatException">if other error occurred</exception>
		public override void WriteFile(ref Image img)
		{
			Image.Flip(Header.Orientation, Ecat7Header.DefaultOrientation, img);

			WriteMainHeader(this.imgHeader);

			/* Resolve matrix lists */
			UpdateMatrixList(img.Dim, imgHeader.Datatype);
			ecat7WriteMatlist(filename, matrixlist);

			/* write image intensity values */
			int totalFrames = 0;
			int subheaderIndex = 0;

			for (int bed = 0; bed < img.Beds; bed++)
			{
				for (int gate = 0; gate < img.Gates; gate++)
				{
					for (int frame = 0; frame < img.Frames; frame++)
					{
						while (subheaderIndex < subheaders.Length && subheaders[subheaderIndex] == null) subheaderIndex++;
						//add new subheader at the end if it is missing
						if (subheaderIndex == subheaders.Length)
						{
							List<Ecat7_subheader> subheaders_tmp = new List<Ecat7_subheader>(subheaders);
							subheaders_tmp.Add(new Ecat7_subheader());
							subheaders = subheaders_tmp.ToArray();
						}
						subheaderIndex++;

						// Write frame (note that index starts from 1)
						WriteFrame(ref img, ++totalFrames);
					}
				}
			}
			Image.Flip(Ecat7Header.DefaultOrientation, Header.Orientation, img);
		}

		/// <summary>
		/// Checks file format. Tries to read magic number from file.
		/// </summary>
		/// <param name="filename">full path to file</param>
		/// <returns>true if file is an Ecat7 file</returns>
		/// <exception cref="TPCEcat7FileException">if there was a read problem</exception>
		public static bool CheckFormat(string filename)
		{
			char[] bytes = new char[7];
			string stringbuffer = "";
			try
			{
				using (StreamReader fs = new StreamReader(filename, true))
				{
					fs.Read(bytes, 0, 7);
				}
			}
			catch (Exception)
			{
				return false;
			}
			stringbuffer = new string(bytes);
			//compare only 7 first characters
			return stringbuffer.Equals(Ecat7File.ecat7V_MAGICNR.Substring(0, 7));
		}

		/// <summary>
		/// Writes single frame into file. If frame number is larger than 
		/// current number of frames in file, then file size is grown.
		/// </summary>
		/// <param name="img">image tacs that is written</param>
		/// <param name="frame">frame number that is written [1..no frames+1]</param>
		public override void WriteFrame(ref Image img, int frame)
		{
			//exception because other than 16-bit signed tacs is not tested.
			if (imgHeader.Datatype != DataType.BIT16_S && imgHeader.Datatype != DataType.FLT32)
			{
				// throw new TPCEcat7FileException("Writing of other than 16-bit signed tacs is not supported.");
				imgHeader.Datatype = DataType.BIT16_S;
				Console.WriteLine("Warning: Data type changed to 16-bit signed. Writing of other than 16-bit signed tacs or float 32-bit tacs is not supported.");
			}
			Ecat7_imageheader h = new Ecat7_imageheader();
			int datatypeBytes = Type2Bytes(imgHeader.Datatype);

			h.data_type = Ecat7_imageheader.convertFrom(imgHeader.Datatype);
			h.num_dimensions = 3;
			h.x_dimension = (short)imgHeader.DimX;
			h.y_dimension = (short)imgHeader.DimY;
			h.z_dimension = (short)imgHeader.DimZ;
			h.x_pixel_size = imgHeader.SizeX / 10.0f;
			h.y_pixel_size = imgHeader.SizeY / 10.0f;
			h.z_pixel_size = imgHeader.SizeZ / 10.0f;
			h.frame_start_time = (int)imgHeader.GetFrameStart(frame - 1);
			h.frame_duration = (int)imgHeader.GetFrameDuration(frame - 1);

			// Select the correct frame
			IntLimits frameLimits = img.Dim.GetDimensionLimits(frame - 1, Limits.FRAMES);

			// Get frame min and max values
			float[] frameMinMax = img.GetMinMax(frameLimits);
			float frameMin = frameMinMax[0];
			float frameMax = frameMinMax[1];
			float typeMax;

			// Set header min and max values according to the datatype
			switch (imgHeader.Datatype)
			{
				// Signed
				case DataType.BIT8_S:
					h.image_min = SByte.MinValue;
					h.image_max = SByte.MaxValue;
					typeMax = SByte.MaxValue;
					break;
				case DataType.BIT64_S:
				case DataType.BIT32_S:
				case DataType.BIT16_S:
					h.image_min = Int16.MinValue;
					h.image_max = Int16.MaxValue;
					typeMax = Int16.MaxValue;
					break;
				// Unsigned
				case DataType.BIT8_U:
					h.image_min = 0;
					h.image_max = Byte.MaxValue;
					typeMax = Byte.MaxValue;
					break;
				case DataType.BIT64_U:
				case DataType.BIT32_U:
				case DataType.BIT16_U:
					h.image_min = 0;
					h.image_max = Int16.MaxValue;
					typeMax = Int16.MaxValue;
					break;
				default:
					throw new TPCEcat7FileException("Unsupported datatype " + Enum.GetName(typeof(DataType), imgHeader.Datatype));
			}

			// Calculate scaling factor
			double absMax = frameMax > -frameMin ? frameMax : frameMin;
			double scale = absMax / typeMax;


			// If calibration factor > 0, include it in the scaling factor
			if (EcatHeader.ecat_calibration_factor > 0) h.scale_factor = Convert.ToSingle(scale / EcatHeader.ecat_calibration_factor);
			else h.scale_factor = Convert.ToSingle(scale);

			/* write image intensity values */
			byte[] data = new byte[img.Width * img.Height * img.Planes * datatypeBytes];

			// If scale is 0, no need to do anything (all data is 0)
			System.Reflection.MethodBase method = System.Reflection.MethodBase.GetCurrentMethod();
			if (scale != 0.0)
			{
				ValueWriter vw = GetWriter(imgHeader.Datatype);
				byte[] valuedata = new byte[datatypeBytes];
				int framelength = img.Width * img.Height * img.Planes;
				for (int i = 0, j = framelength * (frame - 1); i < framelength; i++, j++)
				{
					double val = (double)img[j] / scale;
					valuedata = vw(val);
					Buffer.BlockCopy(valuedata, 0, data, i * datatypeBytes, datatypeBytes);

					// Send 10 events per frame
					if (IOProgress != null && (j * 10) % framelength == 0)
						IOProgress.Invoke(this, new IOProgressEventArgs(j * 100.0f / framelength, method));
				}
			}

			//write subheader (note: frame indexing starts from 0)
			Ecat7WriteImageheader(matrixlist[frame - 1], h, filename);

			//write data (note: frame indexing starts from 0)
			Ecat7WriteMatrixdata(matrixlist[frame - 1].strtblk, data);

			// Send final event
			if (IOProgress != null)
				IOProgress.Invoke(this, new IOProgressEventArgs(100.0f, method));
		}

		/// <summary>
		/// Reads only header tacs from file.
		/// </summary>
		/// <returns>Header information in file</returns>
		public override ImageHeader ReadHeader()
		{
			if (!CheckFormat(filename)) throw new TPCEcat7FileException("Fileformat not recognized.");

			// Read main header
			try
			{
				Header = Ecat7ReadMainheader(filename);
			}
			catch (Exception e)
			{
				throw new TPCEcat7FileException("Failed to read main header:" + e.Message);
			}

			// Read matrix list 
			mlist = Ecat7ReadMatlist(filename);
			
			// Read subheaders, depending on image type
			int i = 0;
			switch (EcatHeader.resolveImageType())
			{
				case Ecat7Header.ecat7_imagetype.IMAGE:
					subheaders = Ecat7ReadImageheaders(mlist, filename);

					// Set fileFormat
					Header.Datatype = Ecat7File.Ecat7_imageheader.convertFrom((subheaders[i] as Ecat7_imageheader).data_type);

					//resolve image dimensions
					int width = 0;
					int height = 0;
					int planes = 0;
					int frames = 0;
					int gates = this.EcatHeader.num_gates;
					int beds = this.EcatHeader.num_bed_pos;

					for (i = 0; i < mlist.Length; i++)
					{
						if (mlist[i].status != Ecat7_MatDir.status_e.DELETED)
						{
							frames++;

							Ecat7_imageheader ih = subheaders[i] as Ecat7_imageheader;
							if (ih.x_dimension > width) width = ih.x_dimension;
							if (ih.y_dimension > height) height = ih.y_dimension;
							if (ih.z_dimension > planes) planes = ih.z_dimension;
						}
					}
					IntLimits dim = new IntLimits(width, height, planes, frames);
					if (gates > 1) dim.AddLimit(0, gates);
					if (beds > 1) dim.AddLimit(0, beds);
					Header.Dim = dim;

					// Assume dynamic image, if multiple frames and not otherwise defined
					if (EcatHeader.acquisition_type == Ecat7Header.Acquisition_type_e.Undefined && Header.Frames > 1) Header.IsDynamic = true;

					// Voxel size (convert from cm to mm)
					Header.Siz = new Voxel(10.0 * (subheaders[0] as Ecat7_imageheader).x_pixel_size,
										   10.0 * (subheaders[0] as Ecat7_imageheader).y_pixel_size,
										   10.0 * (subheaders[0] as Ecat7_imageheader).z_pixel_size);
					break;

				case Ecat7Header.ecat7_imagetype.POLARMAP:
					subheaders = Ecat7ReadPolmapheaders(mlist, filename);
					while (subheaders[i] == null) i++;
					if (subheaders[i] == null) throw new TPCEcat7FileException("Couldn't find any polmapheader.");
					if ((subheaders[i] as Ecat7_polmapheader).num_rings == 0)
						throw new TPCEcat7FileException("No rings in polar map");
					Header.Siz = new Voxel(0.001 * (subheaders[0] as Ecat7_polmapheader).pixel_size, 1.0f, 1.0f);
					break;

				case Ecat7Header.ecat7_imagetype.RAW:
					subheaders = Ecat7ReadScanheaders(mlist, filename);
					while (subheaders[i] == null) i++;
					if (subheaders[i] == null) throw new TPCEcat7FileException("Couldn't find any scanheader.");
					if ((subheaders[i] as Ecat7_scanheader).num_r_elements == 0)
						throw new TPCEcat7FileException("Scanheader r-dimension is zero");
					if ((subheaders[i] as Ecat7_scanheader).num_angles == 0)
						throw new TPCEcat7FileException("Scanheader angles-dimension is zero");
					break;
			}

			// Set frame timing
			int frame = 0;
			for (i = 0; i < subheaders.Length; i++)
			{
				if (subheaders[i] != null)
				{
					Header.SetFrameStart(frame, subheaders[i].frame_start_time);
					Header.SetFrameDuration(frame, subheaders[i].frame_duration);
					frame++;
				}
			}

			//update all general header information
			Header.Clear();
			Header.Add("acquisition_mode", EcatHeader.acquisition_mode);
			Header.Add("acquisition_type", EcatHeader.acquisition_type);
			Header.Add("angular_compression", EcatHeader.angular_compression);
			Header.Add("axial_samp_mode", EcatHeader.axial_samp_mode);
			Header.Add("bed_elevation", EcatHeader.bed_elevation);
			Header.Add("bed_position", EcatHeader.bed_position);
			Header.Add("bin_size", EcatHeader.bin_size);
			Header.Add("branching_fraction", EcatHeader.branching_fraction);
			Header.Add("calibration_units", EcatHeader.calibration_units);
			Header.Add("calibration_units_label", EcatHeader.calibration_units_label);
			Header.Add("coin_samp_mode", EcatHeader.coin_samp_mode);
			Header.Add("compression_code", EcatHeader.compression_code);
			Header.Add("data_units", EcatHeader.data_units);
			Header.Add("distance_scanned", EcatHeader.distance_scanned);
			Header.Add("dosage", EcatHeader.dosage);
			Header.Add("dose_start_time", EcatHeader.dose_start_time);
			Header.Add("ecat_calibration_factor", EcatHeader.ecat_calibration_factor);
			Header.Add("facility_name", EcatHeader.facility_name);
			Header.Add("file_type", EcatHeader.file_type);
			Header.Add("fill", EcatHeader.fill);
			Header.Add("gantry_rotation", EcatHeader.gantry_rotation);
			Header.Add("gantry_tilt", EcatHeader.gantry_tilt);
			Header.Add("init_bed_position", EcatHeader.init_bed_position);
			Header.Add("intrinsic_tilt", EcatHeader.intrinsic_tilt);
			Header.Add("isotope_halflife", EcatHeader.isotope_halflife);
			Header.Add("isotope_name", EcatHeader.isotope_name);
			Header.Add("lwr_sctr_thres", EcatHeader.lwr_sctr_thres);
			Header.Add("lwr_true_thres", EcatHeader.lwr_true_thres);
			Header.Add("magic_number", EcatHeader.magic_number);
			Header.Add("num_bed_pos", EcatHeader.num_bed_pos);
			Header.Add("num_frames", EcatHeader.num_frames);
			Header.Add("num_gates", EcatHeader.num_gates);
			Header.Add("num_planes", EcatHeader.num_planes);
			Header.Add("operator_name", EcatHeader.operator_name);
			Header.Add("original_file_name", EcatHeader.original_file_name);
			Header.Add("patient_age", EcatHeader.patient_age);
			Header.Add("patient_birth_date", EcatHeader.patient_birth_date);
			Header.Add("patient_dexterity", EcatHeader.patient_dexterity);
			Header.Add("patient_height", EcatHeader.patient_height);
			Header.Add("patient_id", EcatHeader.patient_id);
			Header.Add("patient_name", EcatHeader.patient_name);
			Header.Add("patient_sex", EcatHeader.patient_sex);
			Header.Add("patient_weight", EcatHeader.patient_weight);
			Header.Add("physician_name", EcatHeader.physician_name);
			Header.Add("plane_separation", EcatHeader.plane_separation);
			Header.Add("radiopharmaceutical", EcatHeader.radiopharmaceutical);
			Header.Add("scan_start_time", EcatHeader.scan_start_time);
			Header.Add("septa_state", EcatHeader.septa_state);
			Header.Add("serial_number", EcatHeader.serial_number);
			Header.Add("study_description", EcatHeader.study_description);
			Header.Add("study_type", EcatHeader.study_type);
			Header.Add("sw_version", EcatHeader.sw_version);
			Header.Add("system_type", EcatHeader.system_type);
			Header.Add("transaxial_fov", EcatHeader.transaxial_fov);
			Header.Add("transm_source_type", EcatHeader.transm_source_type);
			Header.Add("upr_true_thres", EcatHeader.upr_true_thres);
			Header.Add("user_process_code", EcatHeader.user_process_code);
			Header.Add("well_counter_corr_factor", EcatHeader.well_counter_corr_factor);
			Header.Add("wobble_speed", EcatHeader.wobble_speed);

			return Header;
		}

		/// <summary>
		/// Gets pixel tacs from image. It is strongly suggested that the file header is read 
		/// first in order to fill proper header fields that might be needed for reading.
		/// </summary>
		/// <param name="data">target array</param>
		/// <param name="offset">offset where copying begins to write tacs</param>
		/// <param name="stream">input stream that is expected to be open and stays open</param>
		/// <param name="dim">input tacs dimensions</param>
		/// <param name="datatype">tacs type to be read</param>
		/// <param name="position">file position where reading is started</param>
		/// <param name="scale">scale factor</param>
		/// <param name="intercept">scale intercept</param>
		public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale, float intercept = 0.0f)
		{
			//create image
			float value;
			int i = 0;
			int datalength = (int)dim.GetProduct();
			IOProcessEventHandler pevent = null;

			//read matrix tacs into buffer
			//byte[] buf = Ecat7ReadMatrixdata(stream, (int)position/MatBLKSIZE, (int)Math.Ceiling((decimal)(position+datalength*ImageFile.BytesPerPixel(datatype))/MatBLKSIZE));
			int start_block = (int)position / MatBLKSIZE;
			int end_block = (int)Math.Ceiling((decimal)(position + datalength * ImageFile.BytesPerPixel(datatype)) / MatBLKSIZE);

			#region read matrix tacs into buffer

			if (end_block <= start_block) throw new TPCEcat7FileException("end block is not after start block");
			byte[] buf = new byte[ImageFile.BytesPerPixel(datatype)];
			int ret = 0;

			/* Check the arguments */
			if (start_block < 1) throw new TPCInvalidArgumentsException("Matrix subheader record number.");
			/* Seek the first tacs block */
			stream.Seek((start_block) * MatBLKSIZE, SeekOrigin.Begin);
			if (stream.Position != (start_block) * MatBLKSIZE) throw new TPCEcat7FileException("Failed to seek correct point in file.");
			/* Read the tacs blocks */
			try
			{
				/* Convert matrix tacs to floats */
				switch (datatype)
				{
					case ImageFile.DataType.BIT8_S:
					case ImageFile.DataType.BIT8_U:
						for (i = 0; i < datalength; i++)
						{
							ret = stream.Read(buf, 0, buf.Length);
							BitSwapper.swapBytes(buf, datatype);
							data.SetValue(offset + i, ((float)buf[i]) * scale);
							if (i % (dim.DimX * dim.DimY) == 0)
							{
								processed_slices++;
								pevent = IOProgress;
								if (pevent != null)
									pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)processed_slices / total_number_of_slices_to_process,
																				System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
							}
						}
						break;
					case ImageFile.DataType.SUNI2:
					case ImageFile.DataType.BIT16_S:
						for (i = 0; i < datalength; i++)
						{
							ret = stream.Read(buf, 0, buf.Length);
							BitSwapper.swapBytes(buf, datatype);
							value = (float)BitConverter.ToInt16(buf, 0);
							data.SetValue(offset + i, value * scale);
							if (i % (dim.DimX * dim.DimY) == 0)
							{
								processed_slices++;
								pevent = IOProgress;
								if (pevent != null)
									pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)processed_slices / total_number_of_slices_to_process,
																				System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
							}
						}
						break;
					case ImageFile.DataType.VAXI16:
					case ImageFile.DataType.BIT16_U:
						for (i = 0; i < datalength; i++)
						{
							ret = stream.Read(buf, 0, buf.Length);
							BitSwapper.swapBytes(buf, datatype);
							value = (float)BitConverter.ToUInt16(buf, 0);
							data.SetValue(offset + i, value * scale);
							if (i % (dim.DimX * dim.DimY) == 0)
							{
								processed_slices++;
								pevent = IOProgress;
								if (pevent != null)
									pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)processed_slices / total_number_of_slices_to_process,
																				System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
							}
						}
						break;
					case ImageFile.DataType.SUNI4:
					case ImageFile.DataType.BIT32_S:
						// Dividing datalength with 4 in for-loop might cause uncorrect result in future, 
						// if this tacs type is used. Then remove "/ 4".
						for (i = 0; i < (datalength) / 4; i++)
						{
							ret = stream.Read(buf, 0, buf.Length);
							BitSwapper.swapBytes(buf, datatype);
							data.SetValue(offset + i, ((float)BitConverter.ToInt32(buf, 0)) * scale);
							if (i % (dim.DimX * dim.DimY) == 0)
							{
								processed_slices++;
								pevent = IOProgress;
								if (pevent != null)
									pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)processed_slices / total_number_of_slices_to_process,
																				System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
							}
						}
						break;
					case ImageFile.DataType.VAXI32:
					case ImageFile.DataType.BIT32_U:
						// Dividing datalength with 4 in for-loop might cause uncorrect result in future, 
						// if this tacs type is used. Then remove "/ 4".
						for (i = 0; i < (datalength) / 4; i++)
						{
							ret = stream.Read(buf, 0, buf.Length);
							BitSwapper.swapBytes(buf, datatype);
							data.SetValue(offset + i, ((float)BitConverter.ToUInt32(buf, 0)) * scale);
							if (i % (dim.DimX * dim.DimY) == 0)
							{
								processed_slices++;
								pevent = IOProgress;
								if (pevent != null)
									pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)processed_slices / total_number_of_slices_to_process,
																				System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
							}
						}
						break;
					case ImageFile.DataType.FLT32:
					case ImageFile.DataType.VAXFL32:
						for (i = 0; i < datalength; i++)
						{
							ret = stream.Read(buf, 0, buf.Length);
							BitSwapper.swapBytes(buf, datatype);
							data.SetValue(offset + i, ((float)BitConverter.ToSingle(buf, 0)) * scale);
							if (i % (dim.DimX * dim.DimY) == 0)
							{
								processed_slices++;
								pevent = IOProgress;
								if (pevent != null)
									pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)processed_slices / total_number_of_slices_to_process,
																				System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
							}
						}
						break;
					default:
						throw new TPCEcat7FileException("Unsupported image datatype:" + datatype);
				}
			}
			catch (Exception e)
			{
				throw new TPCEcat7FileException("Error while reading image tacs:" + e.Message);
			}
			#endregion

		}

		/// <summary>
		/// Reads subregion from file.
		/// </summary>
		/// <param name="region">subregion that is read from file</param>
		public override void ReadSubImage(IntLimits region)
		{
			ReadFile();
			imgHeader.Dim = region;
			image = image.GetSubImage(region);
		}

		#endregion

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public ImageStream GetStream(FileAccess a = FileAccess.ReadWrite)
		{
			return GetStream(this.imgHeader.Dim, a);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="limits"></param>
		/// <returns></returns>
		public ImageStream GetStream(IntLimits limits, FileAccess a = FileAccess.ReadWrite)
		{
			List<long> starts = new List<long>();
			List<long> lengths = new List<long>();
			List<float> slopes = new List<float>();

			int dataBytes = ImageFile.BytesPerPixel(imgHeader.Datatype);
			int frameLength = imgHeader.DimX * imgHeader.DimY * imgHeader.DimZ * dataBytes;

			Ecat7_MatDir[] matrixList = Ecat7ReadMatlist(this.filename);

			// Use calibration factor, if specified
			float cFactor = 1.0f;
			if (EcatHeader.ecat_calibration_factor > 0.0) cFactor = EcatHeader.ecat_calibration_factor;

			// Get frame start positions, lengths and scaling
			for (int i = 0; i < matrixlist.Length; i++)
			{
				if (matrixlist[i].status != Ecat7_MatDir.status_e.DELETED)
				{
					Ecat7_imageheader h = subheaders[i] as Ecat7_imageheader;

					starts.Add(matrixList[i].strtblk * MatBLKSIZE);
					lengths.Add(frameLength);
					slopes.Add(h.scale_factor * cFactor);
				}
			}

			// Create a stream
			FragmentedStream fs = new FragmentedStream(new FileStream(this.filename, FileMode.OpenOrCreate, a), starts.ToArray(), lengths.ToArray());

			// Create an Image stream: use header data type, big endian
			ImageStream imgs = new ImageStream(fs, this.imgHeader.Dim, limits, 0, this.imgHeader.Datatype, false);

			// Set the scaling factors for the stream
			imgs.SetScaling(slopes.ToArray(), Limits.FRAMES);

			imgs.StreamOrientation = Header.Orientation;

			return imgs;
		}

		/// <summary>
		/// Ecat7 base time (1.1.1970)
		/// </summary>
		public static readonly DateTime ECATBASETIME = new DateTime(1970, 1, 1, 0, 0, 0, DateTimeKind.Local);

		/// <summary>
		/// Convert Ecat7 format time value (either in YYYYMMDD format or seconds from basetime) to a date.
		/// </summary>
		/// <param name="t">Ecat7 time value</param>
		/// <returns>Date</returns>
		public static DateTime EcatToDateTime(int t)
		{
			int time = t;
			int days = time % 100;
			time /= 100;
			int months = time % 100;
			int years = time / 100;

			// If parsing the date from YYYYMMDD format fails, fall back to the time in seconds
			// Check for impossible and unlikely dates
			bool timeInSeconds = days < 1 || days > 31 || months < 1 || months > 12 || years > DateTime.Now.Year || years < 1000;

			if (timeInSeconds)
			{
				return ECATBASETIME + new TimeSpan(0, 0, t);
			}
			else return new DateTime(years, months, days, 0, 0, 0, DateTimeKind.Local);
		}

		/// <summary>
		/// Convert a date to seconds from Ecat7 basetime.
		/// </summary>
		/// <param name="dt">Date</param>
		/// <returns>Seconds from basetime</returns>
		public static int DateTimeToEcatSeconds(DateTime dt)
		{
			if (dt > ECATBASETIME) return (int)(dt - ECATBASETIME).TotalSeconds;
			else return 0;
		}

		/// <summary>
		/// Convert a date to an Ecat7 date (YYYYMMDD format).
		/// </summary>
		/// <param name="dt">Date</param>
		/// <returns>Ecat7 date</returns>
		public static int DateTimeToEcatDate(DateTime dt)
		{
			return dt.Day + 100 * dt.Month + 10000 * dt.Year;
		}
	}
}
