/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.IO;
using TPClib.Common;

namespace TPClib.Image
{
	/// <summary>
	/// Class representing imagefile produced by Micro PET scanner
	/// </summary>
	public class InterfileFile : ImageFile, IStreamReader, IComparable<InterfileFile>
	{
		private float CalibrationScaling
		{
			get
			{
				if (this.imgHeader.Contains("calibration factor"))
					return Convert.ToSingle(this.imgHeader["calibration factor"]);
				else
					return 1.0f;
			}
			set { this.imgHeader["calibration factor"] = value; }
		}

		private float GetPlaneCalibration(int pln)
		{
			string headerName = "efficient factor for plane " + pln;
			if (this.imgHeader.Contains(headerName))
				return Convert.ToSingle(this.imgHeader[headerName]);
			else return 1.0f;
		}

		private void SetPlaneCalibration(int pln, float cal)
		{
			string headerName = "efficient factor for plane " + pln;
			this.imgHeader[headerName] = cal;
		}

		private string calibrationFile = String.Empty;

		// Use the invariant culture for formatting
		private static IFormatProvider format = System.Globalization.CultureInfo.InvariantCulture;
		private const string floatFormat = @"F6"; 

		/// <summary>
		/// Event that is sent when reading of file has progressed.
		/// </summary>
		public static event IOProcessEventHandler IOProgress;

		/// <summary>
		/// List of frame subheaders or null. Subheaders are only generated for 
		/// frames in dynamic tacs.
		/// </summary>
		public ImageHeader[] subheaders;

		/// <summary>
		/// Creates and assings the basename to this class.
		/// </summary>
		/// <param name="filename">full path to filename</param>
		public InterfileFile(string filename)
		{
			if (filename.EndsWith(".i"))
				this.filename = filename.Remove(filename.Length - 2);
			else
				this.filename = filename;
			base.filetype = FileType.Interfile;
			base.imgHeader = new ImageHeader();
			base.image = null;
			imgHeader.Dataunit = DataUnit.Unknown;
			imgHeader.Datatype = TPClib.Image.ImageFile.DataType.FLT32;
		}

		/// <summary>
		/// Constructs Interfile file.
		/// </summary>
		/// <param name="filename">full name of Interfile file</param>
		/// <param name="image">image tacs</param>
		/// <param name="hdr">image header</param>
		public InterfileFile(string filename, Image image, ImageHeader hdr)
			: this(filename)
		{
			this.image = image;
			this.imgHeader = new ImageHeader(hdr);

			//create subheaders for gates and frames
			subheaders = new ImageHeader[image.Frames * image.Gates];

			for (int i = 0; i < subheaders.Length; i++)
			{
				subheaders[i] = new ImageHeader();
				if (i < image.DimT)
				{
					subheaders[i].Add("image relative start time", image.GetFrameStartTime(i) / 1000.0);
					subheaders[i].Add("image duration", image.GetFrameDuration(i) / 1000.0);
				}
				else
				{
					subheaders[i].Add("image relative start time", (int)0);
					subheaders[i].Add("image duration", (int)0);
				}
			}
		}

		/// <summary>
		/// Constructs an empty (all-zero) Interfile file with filename and dimensions.
		/// </summary>
		/// <param name="filename">filename (*.img)</param>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		/// <param name="dimt">t-dimension</param>
		public InterfileFile(string filename, int dimx, int dimy, int dimz, int dimt) :
			this(filename, new int[] { dimx, dimy, dimz, dimt })
		{
		}

		/// <summary>
		/// Constructs an empty (all-zero) Interfile file with filename and dimensions.
		/// </summary>
		/// <param name="filename">filename (*.img)</param>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		public InterfileFile(string filename, int dimx, int dimy, int dimz)
			: this(filename, new int[] { dimx, dimy, dimz })
		{
		}

		/// <summary>
		/// Constructs an empty (all-zero) Interfile file with filename and dimensions.
		/// </summary>
		/// <param name="filename">filename (*.img)</param>
		/// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
		public InterfileFile(string filename, int[] dim)
			: this(filename)
		{
			if (filename.EndsWith(".i"))
				this.filename = filename.Remove(filename.Length - 2);
			else
				this.filename = filename;
			filetype = FileType.Interfile;

			image = new Image(dim);
			image.IsDynamic = (dim.Length > IntLimits.FRAMES);
			imgHeader = new ImageHeader(image.IsDynamic);
			imgHeader.Dim = new TPClib.IntLimits(dim);
			imgHeader.Datatype = TPClib.Image.ImageFile.DataType.FLT32;
			imgHeader.Dataunit = DataUnit.Unknown;
		}

		/// <summary>
		/// Set scaling factors for this file
		/// </summary>
		/// <param name="global">Global scaling</param>
		/// <param name="planes">Individual plane scaling</param>
		public void SetScaling(float global, float[] planes)
		{
			for (int i = 0; i < planes.Length; i++)
			{
				SetPlaneCalibration(i, planes[i]);
			}
			CalibrationScaling = global;
		}

		/// <summary>
		/// Set scaling factors for this file
		/// </summary>
		/// <param name="global">Global scaling</param>
		public void SetScaling(float global)
		{
			CalibrationScaling = global;
		}

		/// <summary>
		/// Set the calibration file used for this image
		/// </summary>
		/// <param name="fname">Path to the calibration file</param>
		public void SetCalibrationFile(string fname)
		{
			calibrationFile = fname;
		}

		/// <summary>
		/// Reads subregion from file.
		/// </summary>
		/// <param name="region">subregion that is read from file</param>
		public override void ReadSubImage(IntLimits region)
		{
			ReadSingle();
			imgHeader.Dim = region;
			image = image.GetSubImage(region);
		}

		/// <summary>
		/// Checks file format. 
		/// </summary>
		/// <param name="filename">full path to file</param>
		/// <returns>true if file is a Interfile file</returns>
		/// <exception cref="TPCInterfileFileException">if there was a read problem</exception>
		public static bool CheckFormat(string filename)
		{
			//files filename and filename.hdr must exist
			string hdrname;
			//try to resolve exsiting header file name
			try
			{
				hdrname = ResolveHeaderFileName(filename);
			}
			catch (Exception)
			{
				return false;
			}
			try
			{
				using (StreamReader reader = new StreamReader(new FileStream(hdrname, FileMode.Open)))
				{
					string line = reader.ReadLine();
					if (line.StartsWith("!INTERFILE"))
					{
						return true;
					}
					else
					{
						return false;
					}
				}
			}
			catch (Exception e)
			{
				throw new TPCInterfileFileException("Cannot open file " + filename + ".hdr for reading:" + e);
			}
		}

		/// <summary>
		/// Help routine for ReadFile
		/// </summary>
		/// <returns></returns>
		private delegate float GetValue();

		/// <summary>
		/// Gets pixel tacs from image. It is strongly suggested that the file header is read 
		/// first in order to fill proper header fields that might be needed for reading.
		/// </summary>
		/// <param name="data">target array</param>
		/// <param name="offset">offset where copying begins to write tacs</param>
		/// <param name="stream">input stream that is expected to be open and stays open</param>
		/// <param name="dim">input tacs dimensions</param>
		/// <param name="datatype">tacs type to be read</param>
		/// <param name="position">file position where reading is started (ignored)</param>
		/// <param name="scale">scale factor</param>
		/// <param name="intercept">scale intercept</param>
		public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale, float intercept = 0.0f)
		{
			int datalength = dim.GetProduct();
			if (offset + datalength > data.DataLength)
				throw new TPCInterfileFileException("Input tacs of length " + datalength + " does not fit into image of length " + data.DataLength);
			//initialization for event sending
			IOProcessEventHandler pevent = null;
			System.Reflection.MethodBase method = System.Reflection.MethodBase.GetCurrentMethod();

			// In case of 0 scaling factor, just write zeros
			if (scale == 0.0f || CalibrationScaling == 0.0f)
			{
				long byteLength = datalength * ImageFile.BytesPerPixel(datatype);
				// Seek to the end of data
				stream.Seek(offset + byteLength, SeekOrigin.Begin);
				// Clear data
				Array.Clear(data, offset, datalength);
			}

			else
			{
				int voxels_in_plane = 0;
				//seek position
				try
				{
					stream.Seek(position, SeekOrigin.Begin);
				}
				catch (Exception e)
				{
					throw new TPCInterfileFileException("Failed to seek starting position " + position + " from " + filename + ":" + e);
				}

				//read tacs from position
				BinaryReader br = new BinaryReader(stream);
				GetValue getValue;
				switch (datatype)
				{
					case DataType.BIT16_U:
						getValue = delegate()
						{
							return br.ReadUInt16();
						};
						break;
					case DataType.BIT16_S:
						getValue = delegate()
						{
							return br.ReadInt16();
						};
						break;
					case DataType.FLT32:
						getValue = delegate()
						{
							return br.ReadSingle();
						};
						break;
					default:
						throw new TPCInterfileFileException("Datatype is unsupported for reading with this method type: " + datatype);
				}

				// Calculate the initial scaling factor
				float totalScale = scale;

				// start reading tacs.
				voxels_in_plane = dim.DimX * dim.DimY;
				int currentPlane = 0;
				for (int i = 0; i < datalength; i++)
				{
					//send event after each plane; update the scaling factor
					if (i % voxels_in_plane == 0)
					{
						totalScale = scale * GetPlaneCalibration(currentPlane);
						currentPlane++;

						pevent = IOProgress;
						if (pevent != null)
							pevent.Invoke(this, new IOProgressEventArgs(i * 100.0f / datalength, method));
					}
					// Insert the data
					data[i + offset] = getValue() * totalScale;
				}
			}
			pevent = IOProgress;
			if (pevent != null)
				pevent.Invoke(this, new IOProgressEventArgs(100.0f, method));
		}

		/// <summary>
		/// Resolves basenames of available Interfile files in path
		/// </summary>
		/// <param name="path">full path to the files</param>
		/// <returns>array of files</returns>
		public static FileInfo[] ResolveInterfileBasenames(string path)
		{
			string dirName = Path.GetDirectoryName(path);
			string filePattern = Path.GetFileName(path);

			string[] files = Directory.GetFiles(dirName, filePattern, SearchOption.TopDirectoryOnly);

			List<string> interfiles = new List<string>();
			for (int i = 0; i < files.Length; i++)
			{
				string current = files[i];
				string ext = Path.GetExtension(current).ToLowerInvariant();
				bool isInterfile = false;
				if (ext == @".hdr" || ext == @".h33")
				{
					current = Path.GetFileNameWithoutExtension(current);
					ext = Path.GetExtension(current).ToLowerInvariant();
					isInterfile = true;
				}
				if (ext == @".i" || ext == @".img")
				{
					current = Path.GetFileNameWithoutExtension(current);
					isInterfile = true;
				}
				if (isInterfile && !interfiles.Contains(current)) interfiles.Add(current);
			}

			return Array.ConvertAll<string, FileInfo>(interfiles.ToArray(), delegate(string s) { return new FileInfo(dirName + Path.DirectorySeparatorChar + s); });
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="file"></param>
		/// <returns></returns>
		public static InterfileFile ReadMultiple(string file)
		{
			return ReadMultiple(file, String.Empty);
		}

		/// <summary>
		/// Reads all interfiles files that match criteria into single file. 
		/// All dimensions higher than or equal to 4th are stacked together as frames.
		/// </summary>
		/// <param name="file">full path to the files</param>
		/// <param name="calibrationFile">full path to the calibration file</param>
		/// <returns>array of DICOM files</returns>
		public static InterfileFile ReadMultiple(string file, string calibrationFile)
		{
			//all filenames found in path
			FileInfo[] filenames = ResolveInterfileBasenames(file);

			InterfileFile interfile;
			if (filenames.Length == 1)
			{
				interfile = new InterfileFile(filenames[0].FullName);
				interfile.SetCalibrationFile(calibrationFile);
				interfile.ReadSingle();
			}
			else
			{
				//1st read DICOM holding header information
				InterfileFile file_1st = null;
				//currently read DICOM holding header information
				InterfileFile file_current;
				//DICOM files holding header information
				List<InterfileFile> files = new List<InterfileFile>();
				//Event handler where progress events are sent to
				IOProcessEventHandler pevent = null;

				//gather recognized files
				for (int i = 0; i < filenames.Length; i++)
				{
					try
					{
						if (!InterfileFile.CheckFormat(filenames[i].FullName))
						{
							//just skip the file if format does not match
							continue;
						}
						//read all header fields except the actual image tacs
						file_current = new InterfileFile(filenames[i].FullName);
						file_current.SetCalibrationFile(calibrationFile);
						file_current.ReadHeader();

						pevent = IOProgress;
						if (i % 5 == 0 && pevent != null)
							pevent.Invoke(file_current, new IOProgressEventArgs(i * 50.0f / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
						if (file_1st != null)
						{
							//ensure consistency of tacs
							if (!file_1st.imgHeader.Dim.Equals(file_current.imgHeader.Dim))
								throw new TPCInterfileFileException("Inconsistent dimensions in read Interfile files. Cannot combine into single file.");
						}
						else
						{
							file_1st = file_current;
						}
						files.Add(file_current);
					}
					catch (TPCUnrecognizedFileFormatException)
					{
						// just skip the file 
					}
					catch (TPCException e)
					{
						// print error and skip the file 
						Console.WriteLine("Cannot read DICOM file " + filenames[i].FullName + ":" + e.Message);
					}
				}
				if (files.Count == 0) throw new TPCInterfileFileException("No Interfile files was found.");
				int framelength = file_1st.imgHeader.DimX * file_1st.imgHeader.DimY * file_1st.imgHeader.DimZ;
				if (framelength == 0) throw new TPCInterfileFileException("Zero-dimensioned tacs");

				// Sort frames
				files.Sort();

				interfile = new InterfileFile(file, file_1st.imgHeader.DimX, file_1st.imgHeader.DimY, file_1st.imgHeader.DimZ, files.Count);
				interfile.imgHeader = new ImageHeader(file_1st.imgHeader);
				interfile.imgHeader.Datatype = file_1st.imgHeader.Datatype;
				interfile.imgHeader.Dim = interfile.image.Dim;
				if (file_1st.imgHeader.Contains("Dose Strength (unit)"))
				{
					interfile.imgHeader.Dataunit = DataUnit.Parse(file_1st.imgHeader["Dose Strength (unit)"].ToString());
				}
				if (file_1st.imgHeader.Contains("Dose type"))
				{
					interfile.imgHeader.Isotope = Isotope.CreateIsotope(file_1st.imgHeader["Dose type"].ToString());
				}
				if (file_1st.imgHeader.Contains("Dose Strength (value)"))
				{
					interfile.imgHeader.InjectedDose = Convert.ToSingle(file_1st.imgHeader["Dose Strength (value)"]);
				}

				interfile.subheaders = Array.ConvertAll<InterfileFile, ImageHeader>(files.ToArray(), delegate(InterfileFile iff) { return iff.Header; });

				for (int i = 0; i < interfile.subheaders.Length; i++)
				{
					ImageHeader h = interfile.subheaders[i];
					float frameStart = h.GetFrameStart(0);
					float frameDuration = h.GetFrameDuration(0);

					interfile.imgHeader.SetFrameStart(i, frameStart);
					interfile.image.SetFrameStartTime(i, frameStart);
					interfile.imgHeader.SetFrameDuration(i, frameDuration);
					interfile.image.SetFrameDuration(i, frameDuration);
				}

				// Image location
				int location = 0;
				float scale = 1.0f;
				for (int i = 0; i < files.Count; i++)
				{
					// Read scaling factor for the frame
					scale = ResolveCorrectionFactor(files[i].imgHeader);

					// Read the frame from the file
					files[i].GetPixelData(ref interfile.image, location, files[i].filename + ".i",
										  file_1st.imgHeader.Dim, interfile.imgHeader.Datatype, 0, scale);

					location += interfile.image.Framelength;
					pevent = IOProgress;
					if (pevent != null)
						pevent.Invoke(files[i], new IOProgressEventArgs(50.0f + i * 50.0f / files.Count, System.Reflection.MethodBase.GetCurrentMethod(), "Reading pixel tacs."));
				}
			}
			return interfile;
		}

		/// <summary>
		/// Writes single frame into file. If frame number is larger than 
		/// current number of frames in file, then file size is grown.
		/// </summary>
		/// <remarks>The header information including scaling factor is not updated by this method.</remarks>
		/// <param name="image">image tacs that is written</param>
		/// <param name="frame_No">frame number that is written [1..no frames+1]</param>
		public override void WriteFrame(ref Image image, int frame_No)
		{
			if (frame_No < 1 || frame_No > image.Frames)
				throw new TPCInvalidArgumentsException("Frame number " + frame_No + " is out of bounds " + image.Dim);
			//try to open file for writing
			FileStream stream;
			try
			{
				stream = File.Open(filename + ".i", FileMode.Create);
			}
			catch (Exception e)
			{
				throw new TPCMicroPETFileException("Could not open file [" + filename + ".i" + "] for writing:" + e);
			}
			IntLimits framedim = new IntLimits(image.Dim.DimX, image.Dim.DimY, image.Dim.DimZ);
			int framelength = framedim.GetProduct();
			float scale = 1.0f;
			try
			{
				scale = Convert.ToSingle(imgHeader["quantification factor"]);
			}
			catch { }
			try
			{
				WritePixelData(ref image, framelength * (frame_No - 1), stream, framedim, imgHeader.Datatype, 0, scale);
			}
			catch (Exception e)
			{
				throw new TPCInterfileFileException("Exception while writing frame tacs:" + e);
			}
			finally
			{
				stream.Close();
			}
		}

		/// <summary>
		/// Writes pixel tacs from image. It is strongly suggested that the file header is read 
		/// first in order to fill proper header fields that might be needed for reading.
		/// </summary>
		/// <param name="data">target array</param>
		/// <param name="offset">offset where copying begins to write tacs</param>
		/// <param name="stream">input stream that is expected to be open and stays open</param>
		/// <param name="dim">input tacs dimensions</param>
		/// <param name="datatype">tacs type to be read</param>
		/// <param name="position">file position for writing</param>
		/// <param name="scale">scale factor</param>
		public void WritePixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
		{
			int write_end = offset + dim.GetProduct();
			if (scale == 0)
				throw new TPCInvalidArgumentsException("Scale factor cannot be zero.");
			if (offset < 0 || write_end > data.DataLength)
				throw new TPCInvalidArgumentsException("Write region [" + offset + ".." + write_end + "] is out of bounds " + data.Dim);
			stream.Seek(position, SeekOrigin.Begin);
			BinaryWriter br = new BinaryWriter(stream);
			switch (datatype)
			{
				//16-bit unsigned int
				case DataType.BIT16_U:
					for (int i = offset; i < write_end; i++)
					{
						br.Write((UInt16)(data[i] / scale));
					}
					break;
				//16-bit signed int
				case DataType.BIT16_S:
					for (int i = offset; i < write_end; i++)
					{
						br.Write((Int16)(data[i] / scale));
					}
					break;
				case ImageFile.DataType.FLT32:
					for (int i = offset; i < write_end; i++)
					{
						br.Write(data[i] / scale);
					}
					break;
				default:
					throw new TPCInterfileFileException("Unsupported tacs type in Interfile for writing: " + imgHeader.Datatype);
			}
			br.Flush();
		}

		/// <summary>
		/// Resolves header filename that corresponds to basename. For dynamic image returns 
		/// first header file name.
		/// </summary>
		/// <param name="filename">Interfile basename</param>
		/// <returns>header file name</returns>
		protected static string ResolveHeaderFileName(string filename)
		{
			if(filename.EndsWith(@"*")) filename = ResolveInterfileBasenames(filename)[0].FullName;
			string ext = Path.GetExtension(filename);
			if ((ext == @".hdr" || ext == @".h33") && File.Exists(filename)) return filename;
			if (File.Exists(filename + ".hdr")) return filename + ".hdr";
			if (File.Exists(filename + ".i.hdr")) return filename + ".i.hdr";
			if (File.Exists(filename + ".h33")) return filename + ".h33";
			
			throw new TPCInterfileFileException("Could not resolve header file for basename [" + filename + "]");
		}

		/// <summary>
		/// Resolves the name of the data file.
		/// </summary>
		/// <param name="fn">Interfile basename</param>
		/// <returns>Name of the image data file</returns>
		protected static string ResolveDataFilename(string fn)
		{
			string dataFile = fn;
			if (File.Exists(fn + ".i")) { dataFile = fn + ".i"; }
			else if (File.Exists(fn + ".img")) { dataFile = fn + ".img"; }
			return dataFile;
		}

		/// <summary>
		/// Get a stream to this image's data
		/// </summary>
		/// <param name="limits">Stream window limits</param>
		/// <returns>An image data stream</returns>
		public ImageStream GetStream(IntLimits limits, FileAccess a = FileAccess.ReadWrite)
		{
			if (this.filename.EndsWith(@"*")) return GetMultiFileStream(this.filename, limits, this.calibrationFile);
			else
			{
				ImageStream imgs = new ImageStream(ResolveDataFilename(this.filename), this.imgHeader.Dim, limits, 0, this.imgHeader.Datatype, a);

				int planeSize = this.imgHeader.DimX * this.imgHeader.DimY;

				if (subheaders != null)
				{
					float[] correction = new float[this.imgHeader.Planes];
					int frame = 0;
					for (int plane = 0; plane < correction.Length; plane++)
					{
						correction[plane] = ResolveCorrectionFactor(this.subheaders[frame], plane);
						if (plane % this.imgHeader.Planes == 0) frame++;
					}
					imgs.SetScaling(correction, Limits.PLANES);
				}
				else
				{
					imgs.SetScaling(ResolveCorrectionFactor(this.imgHeader));
				}

				imgs.StreamOrientation = imgHeader.Orientation;

				return imgs;
			}
		}

		/// <summary>
		/// Get a stream to this image's data
		/// </summary>
		/// <returns>An image data stream</returns>
		public ImageStream GetStream(FileAccess a = FileAccess.ReadWrite)
		{
			return GetStream(this.imgHeader.Dim, a);
		}

		/// <summary>
		/// Create an image data stream from multiple image files
		/// </summary>
		/// <param name="path">Path to the files</param>
		/// <param name="calibration">Calibration file</param>
		/// <returns>Image data stream</returns>
		public static ImageStream GetMultiFileStream(string path, string calibration = @"", FileAccess a = FileAccess.ReadWrite)
		{
			string[] fileNames;
			ImageHeader[] subHeaders;
			ImageHeader imgHeader;
			ReadMultipleHeaders(path, out fileNames, out subHeaders, out imgHeader, calibration);

			float[] correction = new float[imgHeader.Planes * imgHeader.Dim.Frames];

			for (int frame = 0; frame < imgHeader.Frames; frame++)
			{
				for (int plane = 0; plane < imgHeader.Planes; plane++)
				{
					correction[frame*imgHeader.Planes + plane] = ResolveCorrectionFactor(subHeaders[frame], plane);
				}
			}

			// Create stream
			MultiFileStream mfs = new MultiFileStream(fileNames);
			ImageStream imgs = new ImageStream(mfs, imgHeader.Dim, imgHeader.Dim, 0, imgHeader.Datatype);

			// Set scaling for each plane
			imgs.SetScaling(correction, Limits.PLANES);

			imgs.StreamOrientation = imgHeader.Orientation;

			return imgs;
		}

		/// <summary>
		/// Create a subimage data stream from multiple image files
		/// </summary>
		/// <param name="path">Path to the files</param>
		/// <param name="limits">Subimage limits</param>
		/// <param name="calibration">Calibration file</param>
		/// <returns>Image data stream</returns>
		public static ImageStream GetMultiFileStream(string path, IntLimits limits, string calibration = @"", FileAccess a = FileAccess.ReadWrite)
		{
			string[] fileNames;
			ImageHeader[] subHeaders;
			ImageHeader imgHeader;
			ReadMultipleHeaders(path, out fileNames, out subHeaders, out imgHeader, calibration);

			ImageStream imgs;

			if (fileNames.Length > 1)
			{
				float[] correction = new float[imgHeader.Planes * imgHeader.Dim.Frames];

				// Read correction factor for each plane
				for (int frame = 0; frame < imgHeader.Frames; frame++)
				{
					for (int plane = 0; plane < imgHeader.Planes; plane++)
					{
						correction[frame * imgHeader.Planes + plane] = ResolveCorrectionFactor(subHeaders[frame], plane);
					}
				}

				// Create stream
				MultiFileStream mfs = new MultiFileStream(fileNames);
				imgs = new ImageStream(mfs, imgHeader.Dim, limits, 0, imgHeader.Datatype);

				// Set scaling for each plane
				imgs.SetScaling(correction, Limits.PLANES);
			}
			else if (fileNames.Length == 1)
			{
				InterfileFile iff = new InterfileFile(fileNames[0]);
				iff.imgHeader = imgHeader;
				imgs = iff.GetStream(limits, a);
			}
			else throw new TPCInterfileFileException("No Interfile files found in the path " + path);

			imgs.StreamOrientation = imgHeader.Orientation;

			return imgs;
		}

		/// <summary>
		/// Reads only header information from file. 
		/// </summary>
		/// <returns>Header information in file</returns>
		public override ImageHeader ReadHeader()
		{
			//read one or multiple DICOM files 
			if (filename.EndsWith("*"))
			{
				string[] fileNames;
				ImageHeader[] subHeaders;
				ImageHeader imgHeader;
				InterfileFile.ReadMultipleHeaders(filename, out fileNames, out subHeaders, out imgHeader, this.calibrationFile);
				this.imgHeader = imgHeader;
				this.subheaders = subHeaders;
				return imgHeader;
			}
			else
			{
				// All floating point numbers in invariant culture format
				IFormatProvider nFormat = System.Globalization.NumberFormatInfo.InvariantInfo;

				// Read the main header file and the optional calibration file
				string hdrname = ResolveHeaderFileName(filename);
				List<string> lines = new List<string>();
				try
				{
					lines.AddRange(File.ReadAllLines(hdrname));
					if (calibrationFile != String.Empty)
					{
						lines.AddRange(File.ReadAllLines(calibrationFile));
					}
				}
				catch (Exception e)
				{
					throw new TPCInterfileFileException("Cannot open file " + filename + ".hdr for reading:" + e);
				}

				string[] splitted;
				foreach (string line in lines)
				{
					//anythign that has assignment operator is not a comment
					if (!line.Contains(":=")) continue;
					// Stop if end of header reached
					if (line.StartsWith("!end of interfile")) break;

					splitted = line.Split(new string[] { ":=" }, StringSplitOptions.RemoveEmptyEntries);
					if (splitted.Length > 0)
						splitted[0] = splitted[0].Trim();
					#region resolve_data_type
					if (splitted.Length > 1)
					{
						try
						{
							splitted[1] = splitted[1].Trim();
							if (splitted[0].Equals("!study date (dd:mm:yryr)"))
							{
								imgHeader.Add(splitted[0], System.DateTime.Parse(splitted[1].Replace(':', '/')));
							}
							else if (splitted[0].Equals("!study time (hh:mm:ss)"))
							{
								System.DateTime study_time = System.DateTime.Parse("01/01/1970 " + splitted[1]);
								imgHeader[splitted[0]] = study_time;
							}
							else if (splitted[0].Equals("Dose type"))
							{
								imgHeader[splitted[0]] = Isotope.CreateIsotope(splitted[1]);
							}
							else if (splitted[0].Equals("dose_start_time"))
							{
								imgHeader[splitted[0]] = System.DateTime.Parse(splitted[1]);
							}
							else if (splitted[0].Equals("Dosage Strength"))
							{
								splitted = splitted[1].Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
								if (splitted.Length == 1)
									imgHeader.Add("Dosage Strength", Convert.ToSingle(splitted[0], nFormat));
								if (splitted.Length == 2)
								{
									//add additional field containing dosage unit
									imgHeader.Add("Dosage Strength (value)", Convert.ToSingle(splitted[0], nFormat));
									imgHeader.Add("Dosage Strength (unit)", DataUnit.Parse(splitted[1]));
								}
							}
							//NOTE:this header field is handled here so that is taken as a string even if it would be 
							//possible to parse it sometimes into integer
							else if (splitted[0].Equals("Patient ID") ||
									 splitted[0].Equals("Patient name") ||
									 splitted[0].Equals("tacs description"))
							{
								imgHeader.Add(splitted[0], splitted[1]);
							}

							// try to parse field
							// 1.: as int
							// 2.: as long
							// 3.: as float
							else
							{
								int intvalue;
								long longvalue;
								float floatvalue;
								if (int.TryParse(splitted[1], System.Globalization.NumberStyles.Integer, nFormat, out intvalue))
								{
									imgHeader.Add(splitted[0], intvalue);
								}
								else if (long.TryParse(splitted[1], System.Globalization.NumberStyles.Integer, nFormat, out longvalue))
								{
									imgHeader.Add(splitted[0], longvalue);
								}
								else if (float.TryParse(splitted[1], System.Globalization.NumberStyles.Float, nFormat, out floatvalue))
								{
									imgHeader.Add(splitted[0], floatvalue);
								}
								else imgHeader.Add(splitted[0], splitted[1]);
							}
						}
						//do nothing, just skip to next header field
						catch (Exception) { }
					}
					#endregion
				}
				//fill ImageHeader structure
				#region fill_ImageHeader_fields
				try { imgHeader.PatientName = new PersonName(imgHeader["Patient name"].ToString()); }
				catch (Exception) { }
				try { imgHeader.PatientID = imgHeader["Patient ID"].ToString().Trim(); }
				catch (Exception) { }
				try { imgHeader.Description = ImageHeader.RemoveControlCharacters(imgHeader["tacs description"].ToString()); }
				catch (Exception) { }
				try
				{
					int dimx = Convert.ToInt32(imgHeader["matrix size [1]"], nFormat);
					int dimy = Convert.ToInt32(imgHeader["matrix size [2]"], nFormat);
					int dimz = Convert.ToInt32(imgHeader["matrix size [3]"], nFormat);
					imgHeader.Dim = new IntLimits(dimx, dimy, dimz);
				}
				catch (Exception) { }
				imgHeader.Siz = new Voxel(1.0, 1.0, 1.0);
				try { imgHeader.Siz.SizeX = Convert.ToSingle(imgHeader["scaling factor (mm/pixel) [1]"], nFormat); }
				catch { }
				try { imgHeader.Siz.SizeY = Convert.ToSingle(imgHeader["scaling factor (mm/pixel) [2]"], nFormat); }
				catch { }
				try { imgHeader.Siz.SizeZ = Convert.ToSingle(imgHeader["scaling factor (mm/pixel) [3]"], nFormat); }
				catch { }

				try
				{
					string format = imgHeader["number format"].ToString();
					int bytesperpixel = (int)imgHeader["number of bytes per pixel"];
					switch (bytesperpixel)
					{
						case 2:
							if (format.Contains("unsigned"))
								imgHeader.Datatype = DataType.BIT16_U;
							else if (format.Contains("signed"))
								imgHeader.Datatype = DataType.BIT16_S;
							break;
						case 4:
							if (format.Contains("float"))
								imgHeader.Datatype = DataType.FLT32;
							break;
						default:
							imgHeader.Datatype = DataType.FLT32;
							break;
					}
				}
				catch (Exception)
				{
					//default datatype
					imgHeader.Datatype = DataType.FLT32;
				}

				// Dynamic image information
				if (imgHeader.Contains("dose_start_time"))
				{
					imgHeader.DoseStartTime = (System.DateTime)imgHeader["dose_start_time"];
				}
				if (imgHeader.Contains("radiopharmaceutical"))
				{
					imgHeader.Radiopharma = ImageHeader.RemoveControlCharacters((string)imgHeader["radiopharmaceutical"]);
				}
				else if (imgHeader.Contains("Radiopharmaceutical"))
				{
					imgHeader.Radiopharma = ImageHeader.RemoveControlCharacters((string)imgHeader["Radiopharmaceutical"]);
				}
				if (imgHeader.Contains("isotope name"))
				{
					imgHeader.Isotope = Isotope.CreateIsotope((string)imgHeader["isotope name"]);
				}
				if (imgHeader.Contains("Dose type"))
				{
					imgHeader.Isotope = (Isotope_enumerator)imgHeader["Dose type"];
				}
				if (imgHeader.Contains("frame"))
				{
					imgHeader.Dim.AddLimit(0, 1);
				}
				if (imgHeader.Contains("image relative start time"))
				{
					imgHeader.IsDynamic = true;
					imgHeader.SetFrameStart(0, 1000 * Convert.ToSingle(imgHeader["image relative start time"], nFormat));
				}
				else if (imgHeader.IsDynamic)
				{
					imgHeader.SetFrameStart(0, 0.0);
					// If there is some dynamic information, but no frame start time, assume static image
					imgHeader.IsDynamic = false;
				}
				if (imgHeader.Contains("image duration"))
				{
					imgHeader.SetFrameDuration(0, 1000 * Convert.ToSingle(imgHeader["image duration"], nFormat));
				}
				else imgHeader.SetFrameDuration(0, 0.0);

				// Resolve data unit
				// If 'calibration factor' is present, units are converted from counts/s to kBq/ml
				if (imgHeader.Contains("calibration factor"))
					imgHeader.Dataunit = new DataUnit(DataUnit.BecquerelsPerMillilitre, Unit.SIFactor.Kilo);
				else
				{
					if (imgHeader.Contains("Units[1]"))
						imgHeader.Dataunit = DataUnit.Parse(imgHeader["Units[1]"].ToString());
					else if (imgHeader.Contains("units[1]"))
						imgHeader.Dataunit = DataUnit.Parse(imgHeader["units[1]"].ToString());
					else
						imgHeader.Dataunit = DataUnit.Unknown;
				}
				//try to quess modality from existence of some fields
				//the find results are 
				if (imgHeader.Contains("!imaging modality"))
				{
					try
					{
						string str = ((string)imgHeader["!imaging modality"]).Trim().ToUpperInvariant();
						if (str.Equals("NUCMED"))
							imgHeader.Modality = ImageModality.M_NM;
						else if (str.StartsWith("CT"))
							imgHeader.Modality = ImageModality.M_CT;
						else if (str.StartsWith("MR"))
							imgHeader.Modality = ImageModality.M_MR;
						else if (str.StartsWith("SPECT"))
							imgHeader.Modality = ImageModality.M_PX;
						else if (str.StartsWith("PET"))
							imgHeader.Modality = ImageModality.M_PT;
						else if (str.StartsWith("RT"))
							imgHeader.Modality = ImageModality.M_RT;
					}
					catch
					{
						//do nothing if modality was not a string
					}
				}
				else
				{
					foreach (KeyValuePair<string, object> a in imgHeader)
					{
						if (a.Key.Contains("PET tacs type"))
						{
							imgHeader.Modality = ImageModality.M_PT;
							break;
						}
						else if (a.Key.Contains("SPECT STUDY"))
						{
							imgHeader.Modality = ImageModality.M_PX;
							break;
						}
					}
				}

				imgHeader.Orientation = GetOrientation(imgHeader);

				#endregion

				return imgHeader;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public override void ReadFile()
		{
			//read one or multiple DICOM files 
			if (filename.EndsWith("*"))
			{
				InterfileFile df = InterfileFile.ReadMultiple(filename, calibrationFile);
				this.imgHeader = df.imgHeader;
				this.image = df.image;
				this.calibrationFile = df.calibrationFile;
				this.subheaders = df.subheaders;
			}
			else
			{
				ReadSingle();
			}
		}

		/// <summary>
		/// Reads the .img and .hdr from 
		/// the file to the tacs structures.
		/// </summary>
		private void ReadSingle()
		{
			//read header
			imgHeader = ReadHeader();

			// create image according to header information
			image = new Image(imgHeader.Dim);
			image.IsDynamic = imgHeader.IsDynamic;
			image.SetFrameDurations(imgHeader.FrameDurations);
			image.SetFrameStartTimes(imgHeader.FrameStartTimes);

			//read pixel tacs
			string imagefilename = ResolveDataFilename(filename);
			float scale = ResolveCorrectionFactor(imgHeader);
			GetPixelData(ref image, 0, imagefilename, imgHeader.Dim, imgHeader.Datatype, 0, scale * CalibrationScaling);
		}

		/// <summary>
		/// Resolve scaling factor
		/// </summary>
		/// <param name="img">Image to be scaled</param>
		/// <returns>Scaling factor for the image</returns>
		protected float ResolveScalingFactor(Image img)
		{
			float scale = 1.0f;
			float min = img.GetMin();
			float max = img.GetMax();
			switch (imgHeader.Datatype)
			{
				case DataType.BIT8_S:
					if (Math.Abs(min) <= Math.Abs(max))
						scale = max / (SByte.MaxValue);
					else
						scale = min / (SByte.MinValue);
					imgHeader["!maximum pixel count"] = SByte.MaxValue;
					break;
				case DataType.BIT8_U:
					scale = max / Byte.MaxValue;
					imgHeader["!maximum pixel count"] = Byte.MaxValue;
					break;
				case DataType.BIT16_S:
					if (Math.Abs(min) <= Math.Abs(max))
						scale = max / (Int16.MaxValue);
					else
						scale = min / (Int16.MinValue);
					imgHeader["!maximum pixel count"] = Int16.MaxValue;
					break;
				case DataType.BIT16_U:
					scale = max / UInt16.MaxValue;
					imgHeader["!maximum pixel count"] = UInt16.MaxValue;
					break;
				case DataType.FLT32:
				case DataType.FLT64:
				case DataType.VAXFL32:
					break;
				default:
					throw new TPCInterfileFileException("Unsupported datatype " + imgHeader.Datatype.ToString() + " for resolving scale factor.");
			}
			return scale;
		}

		/// <summary>
		/// Resolve all image correction factors for a given plane from the header, including:
		/// - frame normalization
		/// - decay correction
		/// - branching factor
		/// - dead time correction
		/// - quantification factor
		/// - calibration factor
		/// - efficient factor for plane
		/// </summary>
		/// <param name="subHdr">Header</param>
		/// <param name="plane">Plane number</param>
		/// <returns>Correction factor</returns>
		protected static float ResolveCorrectionFactor(HeaderFieldList subHdr, int plane)
		{
			float scale = ResolveCorrectionFactor(subHdr);
			string headerName = "efficient factor for plane " + plane;
			if (subHdr.Contains(headerName))
				scale *= Convert.ToSingle(subHdr[headerName]);
			return scale;
		}

		/// <summary>
		/// Resolve all global image correction factors from the header, including:
		/// - frame normalization
		/// - decay correction
		/// - branching factor
		/// - dead time correction
		/// - quantification factor
		/// </summary>
		/// <param name="subhdr">Single frame header</param>
		/// <returns>Correction factor</returns>
		protected static float ResolveCorrectionFactor(HeaderFieldList subhdr)
		{
			double correction = 1.0;

			// Extra corrections for HRRT
			if (subhdr.Contains(@"!originating system") && subhdr[@"!originating system"].ToString().ToUpper().Contains(@"HRRT"))
			{
				// Frame normalization & decay correction
				// All times in seconds
				if (subhdr.Contains("image duration"))
				{
					// normalization (times in seconds)
					double frameDuration = Convert.ToDouble(subhdr["image duration"]);
					correction /= frameDuration;

					// decay correction
					if (subhdr.Contains("isotope halflife"))
					{
						double halfLife = Convert.ToDouble(subhdr["isotope halflife"]);
						double lambda = Math.Log(2) / halfLife;

						// Get frame start time (times in seconds)
						double frameStart = 0.0;
						if (subhdr.Contains("image relative start time")) frameStart = Convert.ToDouble(subhdr["image relative start time"]);

						double frameDecay = lambda * frameDuration;
						correction *= frameDecay * Math.Exp(lambda * frameStart) / (1 - Math.Exp(-frameDecay));
					}
				}

				if (subhdr.Contains("branching factor"))
				{
					correction /= Convert.ToDouble(subhdr["branching factor"]);
				}

				if (subhdr.Contains("Dead time correction factor"))
				{
					correction *= Convert.ToDouble(subhdr["Dead time correction factor"]);
				}

				if (subhdr.Contains("calibration factor"))
				{
					// Calibration factor changes the unit to kBq/ml from counts/s. Scale it to kBq/ml.
					correction *= Convert.ToDouble(subhdr["calibration factor"]);
					correction = DataUnit.ConvertValue(DataUnit.BecquerelsPerMillilitre, new DataUnit(DataUnit.BecquerelsPerMillilitre, Unit.SIFactor.Kilo), correction); 
				}
			}

			// For non-HRRT, use only quantification factor
			if (subhdr.Contains("quantification factor"))
			{
				correction *= Convert.ToDouble(subhdr["quantification factor"]);
			}

			return Convert.ToSingle(correction);
		}

		/// <summary>
		/// Writes Interfile header tacs into curretly selected header file.
		/// </summary>
		protected void WriteHeader()
		{
			using (StreamWriter hdrfile = new StreamWriter(File.Open(filename + ".i.hdr", FileMode.Create)))
			{
				//copy from general header structure
				#region set_general_header_data
				imgHeader["Patient name"] = imgHeader.PatientName;
				imgHeader["Patient ID"] = imgHeader.PatientID;
				imgHeader["tacs description"] = imgHeader.Description;
				imgHeader["matrix size [1]"] = imgHeader.DimX;
				imgHeader["matrix size [2]"] = imgHeader.DimY;
				imgHeader["matrix size [3]"] = imgHeader.DimZ;
				imgHeader["scaling factor (mm/pixel) [1]"] = imgHeader.SizeX;
				imgHeader["scaling factor (mm/pixel) [2]"] = imgHeader.SizeY;
				imgHeader["scaling factor (mm/pixel) [3]"] = imgHeader.SizeZ;
				switch (imgHeader.Modality)
				{
					case ImageModality.M_NM:
						imgHeader["!imaging modality"] = "NUCMED";
						break;
					default:
						imgHeader["!imaging modality"] = imgHeader.Modality.ToString();
						break;
				}
				if (imgHeader.IsDynamic)
				{
					imgHeader["Dose type"] = imgHeader.Isotope;
					imgHeader["dose_start_time"] = imgHeader.DoseStartTime;
					imgHeader["Radiopharmaceutical"] = imgHeader.Radiopharma;
				}
				imgHeader["number of bytes per pixel"] = ImageFile.BytesPerPixel(imgHeader.Datatype);
				switch (imgHeader.Datatype)
				{
					case DataType.FLT32:
						imgHeader["number format"] = "float";
						break;
					case DataType.BIT16_U:
						imgHeader["number format"] = "unsigned short";
						break;
					case DataType.BIT16_S:
						imgHeader["number format"] = "signed short";
						break;
					default:
						imgHeader["number format"] = "unknown";
						break;
				}
				#endregion
				imgHeader["!tacs starting block"] = 0;
				for (int i = 0; i < imgHeader.Dim.Length; i++)
				{
					imgHeader["Units[" + (i + 1) + "]"] = imgHeader.Dataunit.ToString();
				}

				//write header tacs
				try
				{
					hdrfile.WriteLine("!INTERFILE");
					for (int i = 0; i < imgHeader.Count; i++)
					{
						#region special_rules_for_writing
						if (imgHeader[i].Key.Equals("isotope halflife"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("Dosage Strength (value)"))
						{
							string value = (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format);
							string unit = "";
							try
							{
								unit = imgHeader["Dosage Strength (unit)"].ToString();
							}
							catch (Exception e)
							{
								Console.WriteLine(e);
							}
							hdrfile.WriteLine("Dosage Strength := " + value + " " + unit);
						}
						else if (imgHeader[i].Key.Equals("Dosage Strength (unit)"))
						{
							//do nothing here because the field is written with (value)
						}
						else if (imgHeader[i].Key.Equals("!study date (dd:mm:yryr)"))
						{
							System.DateTime study_date = (System.DateTime)imgHeader[i].Value;
							hdrfile.WriteLine(imgHeader[i].Key + " := " + study_date.ToString("dd:MM:yyyy"));
						}
						else if (imgHeader[i].Key.Equals("!study time (hh:mm:ss)"))
						{
							System.DateTime study_time = (System.DateTime)imgHeader[i].Value;
							hdrfile.WriteLine(imgHeader[i].Key + " := " + study_time.ToString("HH:mm:ss"));
						}
						else if (imgHeader[i].Key.Equals("branching factor"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("Dead time correction factor"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("!histogrammer revision"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString("0.0", format));
						}
						else if (imgHeader[i].Key.Equals("scaling factor (mm/pixel) [1]"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("scaling factor (mm/pixel) [2]"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("scaling factor (mm/pixel) [3]"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("decay correction factor"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("decay correction factor2"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToSingle(imgHeader[i].Value)).ToString(floatFormat, format));
						}
						else if (imgHeader[i].Key.Equals("Dose type"))
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + ((Isotope_enumerator)imgHeader[i].Value).ToString());
						}
						#endregion
						else if (imgHeader[i].Value is Char[])
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + new string((char[])imgHeader[i].Value));
						}
						else if (imgHeader[i].Value is Single || imgHeader[i].Value is Double)
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + (Convert.ToDouble(imgHeader[i].Value).ToString(floatFormat, format)));
						}
						else if (imgHeader[i].Value is float[] || imgHeader[i].Value is double[])
						{
							hdrfile.Write(imgHeader[i].Key + " :=");
							foreach (object o in (imgHeader[i].Value as Array))
								hdrfile.Write(" " + Convert.ToDouble(o).ToString(floatFormat, format));
							hdrfile.WriteLine();
						}
						else if (imgHeader[i].Value is Array)
						{
							Array values = imgHeader[i].Value as Array;
							hdrfile.Write(imgHeader[i].Key + " :=");
							foreach (object o in values)
								hdrfile.Write(" " + o.ToString());
							hdrfile.WriteLine();
						}
						// Default : display the string representation
						// do nothing for pixel tacs since it is not header information
						else if (imgHeader[i].Key != "PixelData")
						{
							hdrfile.WriteLine(imgHeader[i].Key + " := " + imgHeader[i].Value.ToString());
						}
					}
				}
				catch (Exception e)
				{
					throw new TPCInterfileFileException("Failed to write header tacs into file " + filename + ".hdr:" + e);
				}
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="imgs"></param>
		public void WriteFile(ImageStream imgs)
		{
			string basename = filename;
			basename = basename.TrimEnd('*');
			for (int i = 0; i < imgs.Dim.Frames; i++)
			{
				string framename = basename;
				if (imgs.Dim.Frames > 1) filename += "_frame" + i;
				InterfileFile file = new InterfileFile(framename);

				string datafile = Path.GetFileName(file.filename);

				file.imgHeader = new ImageHeader(imgHeader);

				// Use subheader information, if available
				if (subheaders != null && subheaders.Length > i)
				{
					file.imgHeader.AddRange(subheaders[i]);
				}

				// Use datatype and dimensions of the stream
				file.imgHeader.Datatype = imgs.DataType.Type;
				file.imgHeader.Dim = imgs.Dim.GetDimensionLimits(i, Limits.FRAMES);

				// Resolve the orientation for the image
				file.imgHeader.Orientation = GetOrientation(imgHeader);

				int imageSize = file.imgHeader.Dim.GetProduct();

				ImageStream.StreamCopy(imgs, file.GetStream(), imageSize);

				//add or rename some fields
				file.imgHeader["name of tacs file"] = datafile + ".i";
				//set frame number
				file.imgHeader["frame"] = i;
				file.imgHeader["image relative start time"] = Convert.ToSingle(this.imgHeader.GetFrameStart(i) / 1000.0);
				file.imgHeader["image duration"] = Convert.ToSingle(this.imgHeader.GetFrameDuration(i) / 1000.0);
				file.imgHeader["quantification factor"] = imgs.ScaleSlope;
				file.WriteHeader();
			}
		}

		/// <summary>
		/// Writes both the header and image tacs to the disk.
		/// </summary>
		/// <param name="img">image tacs that is written</param>
		public override void WriteFile(ref Image img)
		{
			//resolve scaling factor according to datatype
			float scale = ResolveScalingFactor(img);

			if (img.IsDynamic)
			{
				string basename = filename;
				basename = basename.TrimEnd('*');
				for (int i = 0; i < img.Frames; i++)
				{
					InterfileFile file;
					if (img.Frames > 1) file = new InterfileFile(basename + "_frame" + i);
					else file = new InterfileFile(basename);

					string datafile = Path.GetFileName(file.filename);

					file.imgHeader = new ImageHeader(imgHeader);
					file.imgHeader.Dim = img.Dim;

					// Use subheader information, if available
					if (subheaders != null && subheaders.Length > i)
					{
						file.imgHeader.AddRange(subheaders[i]);
					}

					//add or rename some fields
					file.imgHeader["name of tacs file"] = datafile + ".i";
					file.imgHeader["quantification factor"] = scale;
					//set frame number
					file.imgHeader["frame"] = i;
					file.imgHeader["image relative start time"] = Convert.ToInt32(img.GetFrameStartTime(i) / 1000.0);
					file.imgHeader["image duration"] = Convert.ToInt32(img.GetFrameDuration(i) / 1000.0);
					file.WriteHeader();
					file.WriteFrame(ref img, i + 1);
				}
			}

			// Write static image
			else
			{
				this.imgHeader["name of tacs file"] = this.filename + ".i";
				this.imgHeader["quantification factor"] = scale;

				//write header tacs
				WriteHeader();
				//initialization for event sending
				IOProcessEventHandler pevent = null;
				//write image tacs
				string fname = filename;
				//resolve frame length in voxels
				int framelength = (int)img.Dim.GetProduct(IntLimits.PLANES);

				//try to open file for writing
				using (FileStream stream = File.Open(fname + ".i", FileMode.Create))
				{
					//resolve voxels in plane for sending event after writing each plane
					int voxels_in_plane = img.DimX * img.DimY;

					//write tacs according to tacs type
					long position = 0;
					for (int i = 0; i < img.DimZ; i++)
					{
						WritePixelData(ref img, i * voxels_in_plane, stream, new IntLimits(img.DimX, img.DimY), imgHeader.Datatype, position, scale);
						position += voxels_in_plane * ImageFile.BytesPerPixel(imgHeader.Datatype);
						//send event after each plane
						if (i > 0 && i % voxels_in_plane == 0)
						{
							pevent = IOProgress;
							if (pevent != null)
								pevent.Invoke(this, new IOProgressEventArgs(i * 100.0f / img.DataLength, System.Reflection.MethodBase.GetCurrentMethod()));
						}
					}
				}
				pevent = IOProgress;
				if (pevent != null)
					pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
			}
		}

		/// <summary>
		/// Read orientation from the header (default is Neurological (RAI)).
		/// </summary>
		/// <param name="h">Header</param>
		/// <returns>Orientation</returns>
		private static Orientation GetOrientation(ImageHeader h)
		{
			Orientation o = Orientation.Neurological;

			if (h.Contains("patient orientation") && h["patient orientation"].ToString() == "feet in")
				o = o.Flip(false, false, true);
			if (h.Contains("patient rotation") && h["patient rotation"].ToString() == "prone")
				o = o.Flip(false, true, false);
			return o;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="head"></param>
		/// <param name="img"></param>
		private static void FlipImage(ImageHeader head, Image img)
		{
			Orientation o = GetOrientation(head);
			if (o.RightToLeft) Image.FlipX(img);
			if (o.AnteriorToPosterior) Image.FlipY(img);
			if (o.SuperiorToInferior) Image.FlipZ(img);
		}

		/// <summary>
		/// Read headers from all files that are part of the same image
		/// </summary>
		/// <param name="path">Path to the image folder</param>
		/// <param name="files">Names of the files read</param>
		/// <param name="subheaders">Interfile subheaders for the files</param>
		/// <param name="header">Combined image header</param>
		/// <param name="calibration">Calibration file name</param>
		public static void ReadMultipleHeaders(string path, out string[] files, out ImageHeader[] subheaders, out ImageHeader header, string calibration = @"")
		{
			//all filenames found in path
			FileInfo[] filenames = ResolveInterfileBasenames(path);

			filenames = Array.FindAll<FileInfo>(filenames, delegate(FileInfo fi) { return InterfileFile.CheckFormat(fi.FullName); });

			if (filenames.Length == 1)
			{
				string name = filenames[0].FullName;
				InterfileFile interfile;
				interfile = new InterfileFile(name);
				header = interfile.ReadHeader();
				subheaders = interfile.subheaders;
				files = new string[] { name + @".i" };
			}
			else
			{
				// The first file read holding header information
				InterfileFile file_1st = null;

				// Currently read file holding header information
				InterfileFile file_current;

				// List of subheaders
				List<InterfileFile> subfiles = new List<InterfileFile>();

				// Event handler where progress events are sent to
				IOProcessEventHandler pevent = null;

				//gather recognized files
				for (int i = 0; i < filenames.Length; i++)
				{
					//read all header fields except the actual image tacs
					file_current = new InterfileFile(filenames[i].FullName);
					if (calibration != String.Empty) file_current.SetCalibrationFile(calibration);
					file_current.Header = file_current.ReadHeader();

					pevent = IOProgress;
					if (i % 5 == 0 && pevent != null)
						pevent.Invoke(file_current, new IOProgressEventArgs(i * 100.0f / filenames.Length, System.Reflection.MethodBase.GetCurrentMethod()));
					if (file_1st != null)
					{
						//ensure consistency of tacs
						if (!file_1st.imgHeader.Dim.Equals(file_current.imgHeader.Dim))
							throw new TPCInterfileFileException("Inconsistent dimensions in read Interfile files. Cannot combine into single file.");
					}
					else
					{
						file_1st = file_current;
					}

					subfiles.Add(file_current);
				}
				int framelength = file_1st.imgHeader.DimX * file_1st.imgHeader.DimY * file_1st.imgHeader.DimZ;
				if (framelength == 0) throw new TPCInterfileFileException("Zero-dimensioned tacs");

				//sort files according to frame field 
				subfiles.Sort();
				InterfileFile[] fileArr = subfiles.ToArray();

				// Get the frame timing from the subheaders
				float[] frameStarts = new float[subfiles.Count];
				float[] frameDurations = new float[subfiles.Count];
				for (int i = 0; i < fileArr.Length; i++)
				{
					ImageHeader h = fileArr[i].Header;
					frameStarts[i] = h.GetFrameStart(0);
					frameDurations[i] = h.GetFrameDuration(0);
				}

				header = new ImageHeader(file_1st.imgHeader);
				header.Dim = new IntLimits(file_1st.imgHeader.DimX, file_1st.imgHeader.DimY, file_1st.imgHeader.DimZ, subfiles.Count);
				header.FrameStartTimes = frameStarts;
				header.FrameDurations = frameDurations;

				subheaders = Array.ConvertAll<InterfileFile, ImageHeader>(fileArr, delegate(InterfileFile iff) { return iff.Header; });
				files = Array.ConvertAll<InterfileFile, string>(fileArr, delegate(InterfileFile fi) { return fi.filename + @".i"; });
			}
		}

		/// <summary>
		/// Comparison operator. Sorts according to frame order.
		/// </summary>
		/// <param name="obj">Object to compare with</param>
		/// <returns></returns>
		public int CompareTo(object obj)
		{
			if (!(obj is InterfileFile)) throw new Exception();
			else return this.CompareTo(obj as InterfileFile);
		}

		/// <summary>
		/// Comparison operator. Sorts according to frame order.
		/// </summary>
		/// <param name="other">InterfileFile to compare with</param>
		/// <returns>-1 if this InterfileFile precedes the other, 1 if the other precedes this, 0 in other cases</returns>
		public int CompareTo(InterfileFile other)
		{
			if (this.imgHeader.Contains("frame") && other.imgHeader.Contains("frame"))
			{
				int frame_a = Convert.ToInt32(this.imgHeader["frame"]);
				int frame_b = Convert.ToInt32(other.imgHeader["frame"]);
				return frame_a.CompareTo(frame_b);
			}
			else return 0;
		}
	}
}
