/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib;

namespace TPClib.Image
{
    /// <summary>
    /// Represents Micro PET file header tacs. The value types are as they are defined in the header 
    /// except for enumerator codings.
    /// </summary>
    public class MicroPETHeader
    {
        #region enumerator_types
        /// <summary>
        /// Scanner model number
        /// </summary>
        public enum model_number
        {
            #region model_number_values
            /// <summary>
            /// Unknown
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Primate
            /// </summary>
            Primate = 2000,
            /// <summary>
            /// Rodent
            /// </summary>
            Rodent = 2001,
            /// <summary>
            /// microPET2
            /// </summary>
            microPET2 = 2002,
            /// <summary>
            /// Focus_220
            /// </summary>
            Focus_220 = 2500,
            /// <summary>
            /// Focus_120
            /// </summary>
            Focus_120 = 2501,
            /// <summary>
            /// mCAT
            /// </summary>
            mCAT = 3000,
            /// <summary>
            ///  mCATII
            /// </summary>
            mCATII = 3500,
            /// <summary>
            /// mSPECT
            /// </summary>
            mSPECT = 4000,
            /// <summary>
            /// Inveon Dedicated PET
            /// </summary>
            Inveon_Dedicated_PET = 5000,
            /// <summary>
            /// Inveon MM Platform
            /// </summary>
            Inveon_MM_Platform = 5001,
            /// <summary>
            /// MR/PET Head Insert
            /// </summary>
            MR_PET_Head_Insert = 6000,
            /// <summary>
            /// Tuebingen PET MR
            /// </summary>
            Tuebingen_PET_MR = 8000
            #endregion
        }
        /// <summary>
        /// Acquisition scanner modality
        /// </summary>
        public enum scanner_modality
        {
            #region scanner_modality_values
            /// <summary>
            /// Unknown acquisition modality
            /// </summary>
            Unknown = -1,
            /// <summary>
            /// PET acquisition
            /// </summary>
            PET = 0,
            /// <summary>
            /// CT acquisition
            /// </summary>
            CT = 1,
            /// <summary>
            /// SPECT acquisition
            /// </summary>
            SPECT = 2
            #endregion
        }
        /// <summary>
        /// Data file type
        /// </summary>
        public enum Data_file_type
        {
            #region Data_file_type_values
            /// <summary>
            /// Unknown tacs file type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// List mode tacs file
            /// </summary>
            List_mode = 1,
            /// <summary>
            /// Sinogram tacs file
            /// </summary>
            Sinogram = 2,
            /// <summary>
            /// Normalization tacs file 
            /// </summary>
            Normalization = 3,
            /// <summary>
            /// Attenuation correction tacs file
            /// </summary>
            Attenuation = 4,
            /// <summary>
            /// Image tacs file
            /// </summary>
            Image = 5,
            /// <summary>
            /// Blank tacs file
            /// </summary>
            Blank = 6,
            /// <summary>
            /// 8 - Mu map tacs file
            /// </summary>
            Mu_map = 8,
            /// <summary>
            /// Scatter correction tacs file
            /// </summary>
            Scatter_correction = 9,
            /// <summary>
            /// Crystal efficiency tacs
            /// </summary>
            Crystal_efficiency = 10,
            /// <summary>
            /// Crystal interference correction
            /// </summary>
            Crystal_interference_correction = 11,
            /// <summary>
            /// Transaxial geometric correction
            /// </summary>
            Transaxial_geometric_correction = 12,
            /// <summary>
            /// Axial geometric correction
            /// </summary>
            Axial_geometric_correction = 13,
            /// <summary>
            /// CT projection tacs
            /// </summary>
            CT_projection = 14,
            /// <summary>
            /// SPECT raw projection tacs
            /// </summary>
            SPECT_raw_projection = 15,
            /// <summary>
            /// SPECT energy tacs from projections
            /// </summary>
            SPECT_energy_data = 16,
            /// <summary>
            /// SPECT normalization tacs
            /// </summary>
            SPECT_normalization = 17
            #endregion
        }
        /// <summary>
        /// Acquisition mode
        /// </summary>
        public enum Acquisition_mode
        {
            #region Acquisition_mode_values
            /// <summary>
            /// Unknown acquisition mode
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Blank acquisition
            /// </summary>
            Blank = 1,
            /// <summary>
            /// Emission acquisition
            /// </summary>
            Emission = 2,
            /// <summary>
            /// Dynamic acquisition
            /// </summary>
            Dynamic = 3,
            /// <summary>
            /// Gated acquisition
            /// </summary>
            Gated = 4,
            /// <summary>
            /// Continuous bed motion acquisition 
            /// </summary>
            Continuous_bed_motion = 5,
            /// <summary>
            /// Singles transmission acquisition
            /// </summary>
            Singles_transmission = 6,
            /// <summary>
            /// Windowed coincidence transmission acquisition
            /// </summary>
            Windowed_coincidence_transmission = 7,
            /// <summary>
            /// Non-windowed coincidence transmission acquisition
            /// </summary>
            Non_windowed_coincidence_transmission = 8,
            /// <summary>
            /// CT projection acquisition
            /// </summary>
            CT_projection = 9,
            /// <summary>
            /// CT calibration acquisition
            /// </summary>
            CT_calibration = 10,
            /// <summary>
            /// SPECT planar projection acquisition
            /// </summary>
            SPECT_planar_projection = 11,
            /// <summary>
            /// SPECT multi-projection acquisition
            /// </summary>
            SPECT_multi_projection = 12,
            /// <summary>
            /// SPECT calibration acquisition
            /// </summary>
            SPECT_calibration = 13,
            /// <summary>
            /// SPECT normalization acquisition
            /// </summary>
            SPECT_normalization = 14,
            /// <summary>
            /// SPECT detector setup acquisition
            /// </summary>
            SPECT_detector_estup = 15,
            /// <summary>
            /// SPECT scout view acquisition
            /// </summary>
            SPECT_scout_view = 16
            #endregion
        }
        /// <summary>
        /// Bed control 
        /// </summary>
        public enum Bed_Control
        {
            #region Bed_control_values
            /// <summary>
            /// Unknown bed control
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Dedicated PET
            /// </summary>
            Dedicated_PET = 1,
            /// <summary>
            /// microCAT II
            /// </summary>
            microCAT_II = 2,
            /// <summary>
            /// Multimodality bed control
            /// </summary>
            Multimodality = 3,
            /// <summary>
            /// microPET bed control
            /// </summary>
            microPET = 4
            #endregion
        }
        /// <summary>
        /// Bed motion
        /// </summary>
        public enum Bed_motion
        {
            #region Bed_motion_values
            /// <summary>
            /// Static or unknown bed motion
            /// </summary>
            Static_or_unknown = 0,
            /// <summary>
            /// Continuous bed motion
            /// </summary>
            Continuous = 1,
            /// <summary>
            /// Multiple bed positions, i.e. step and shoot
            /// </summary>
            Multiple = 2
            #endregion
        }
        /// <summary>
        /// Pixel tacs type 
        /// </summary>
        public enum Data_type {
            /// <summary>
            /// Unknown tacs type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Byte (8-bits) tacs type
            /// </summary>
            Byte = 1,
            /// <summary>
            /// 2-byte integer - Intel style
            /// </summary>
            Intel_2byte_Int = 2,
            /// <summary>
            /// 4-byte integer - Intel style
            /// </summary>
            Intel_4byte_Int = 3,
            /// <summary>
            /// 4-byte float - Intel style
            /// </summary>
            Intel_4byte_Float = 4,
            /// <summary>
            /// 4-byte float - Sun style
            /// </summary>
            Sun_4byte_Float = 5,
            /// <summary>
            /// 2-byte integer - Sun style
            /// </summary>
            Sun_2byte_Int = 6,
            /// <summary>
            /// 4-byte integer - Sun style
            /// </summary>
            Sun_4byte_Int = 7
        }
        /// <summary>
        /// Registration tacs availability
        /// </summary>
        public enum Registration_data_availability {
            /// <summary>
            /// No registration tacs available
            /// </summary>
            No_registration_data = 0,
            /// <summary>
            /// No registration tacs available
            /// </summary>
            CT_registration_data = 1,
            /// <summary>
            /// No registration tacs available
            /// </summary>
            PET_registration_data = 2
        }
        /// <summary>
        /// Normalization type applied to tacs set
        /// </summary>
        public enum Normalization_type {
            /// <summary>
            /// No normalization applied
            /// </summary>
            No_normalization = 0,
            /// <summary>
            /// Point source inversion
            /// </summary>
            Point_source_inversion = 1,
            /// <summary>
            /// oint source component based
            /// </summary>
            Point_source_component = 2,
            /// <summary>
            /// Cylinder source inversion
            /// </summary>
            Cylinder_source_inversion = 3,
            /// <summary>
            /// Cylinder source component based
            /// </summary>
            Cylinder_source_component = 4,
            /// <summary>
            /// Dark/bright field log normalization (CT)
            /// </summary>
            Darkbright_field = 5,
            /// <summary>
            /// SPECT flood inversion based
            /// </summary>
            SPECT_flood = 6
        }
        /// <summary>
        /// Reconstruction type
        /// </summary>
        public enum Reconstruction_type {
            /// <summary>
            /// Unknown, or no, algorithm type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Filtered Backprojection
            /// </summary>
            FBP = 1,
            /// <summary>
            /// OSEM2d
            /// </summary>
            OSEM2d = 2,
            /// <summary>
            /// OSEM3d
            /// </summary>
            OSEM3d = 3,
            /// <summary>
            /// OSEM3D followed by MAP or FastMAP
            /// </summary>
            OSEM3D_MAP = 6,
            /// <summary>
            /// MAPTR for transmission image
            /// </summary>
            MAPTR = 7,
            /// <summary>
            /// MAP 3D reconstruction
            /// </summary>
            MAP_3D = 8,
            /// <summary>
            /// Feldkamp cone beam
            /// </summary>
            Feldkamp = 9
        }
        /// <summary>
        /// Filter and/or apodizing windows type
        /// </summary>
        public enum Filter_type {
            /// <summary>
            /// No filter
            /// </summary>
            No_filter = 0,
            /// <summary>
            /// Ramp filter (backprojection) or no filter
            /// </summary>
            Ramp = 1,
            /// <summary>
            /// First-order Butterworth window
            /// </summary>
            First_order = 2,
            /// <summary>
            /// Hanning window
            /// </summary>
            Hanning = 3,
            /// <summary>
            /// Hamming window
            /// </summary>
            Hamming = 4,
            /// <summary>
            /// Parzen window
            /// </summary>
            Parzen = 5,
            /// <summary>
            /// Shepp filter
            /// </summary>
            Shepp = 6,
            /// <summary>
            /// Second-order Butterworth window
            /// </summary>
            Second_order = 7
        }
        /// <summary>
        /// Subject orientation
        /// </summary>
        public enum Subject_orientation {
            /// <summary>
            /// Unknown subject orientation
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Feet first, prone
            /// </summary>
            FFP = 1,
            /// <summary>
            /// Head first, prone
            /// </summary>
            HFP = 2,
            /// <summary>
            /// Feet first, supine
            /// </summary>
            FFS = 3,
            /// <summary>
            /// Head first, supine
            /// </summary>
            HFS = 4,
            /// <summary>
            /// Feet first, right
            /// </summary>
            FFR = 5,
            /// <summary>
            /// Head first, right
            /// </summary>
            HFR = 6,
            /// <summary>
            /// Feet first, left
            /// </summary>
            FFL = 7,
            /// <summary>
            /// Head first, left
            /// </summary>
            HFL = 8
        }
        /// <summary>
        /// Deadtime correction type
        /// </summary>
        public enum Deadtime_correction {
            /// <summary>
            /// No deadtime correction applied
            /// </summary>
            No_deadtime = 0,
            /// <summary>
            /// Global estimate based on singles
            /// </summary>
            Global = 1,
            /// <summary>
            /// CMS estimate based on singles
            /// </summary>
            CMS = 2,
            /// <summary>
            /// Global estimate based on running deadtime average
            /// </summary>
            Global_avg = 3,
            /// <summary>
            /// Blank/TX singles estimate (block based)
            /// </summary>
            Black_TX = 4
        }
        /// <summary>
        /// Attenuation type
        /// </summary>
        public enum Attenuation_correction {
            /// <summary>
            /// No attenuation applied
            /// </summary>
            No_attenuation = 0,
            /// <summary>
            /// Point source in windowed TX coincidence
            /// </summary>
            Point_source_windowed_TX = 1,
            /// <summary>
            /// Point source singles based TX
            /// </summary>
            Point_source_singles = 2,
            /// <summary>
            /// Segmented point source in TX coincidence
            /// </summary>
            Segmented_Point_source_TX = 3,
            /// <summary>
            /// Segmented point source singles based TX
            /// </summary>
            Segmented_Point_source_singled = 4,
            /// <summary>
            /// Calculated by geometry
            /// </summary>
            Geometry = 5,
            /// <summary>
            /// Non-positron source singles based TX
            /// </summary>
            Non_positron = 6,
            /// <summary>
            /// Point source in non-windowed TX coincidence
            /// </summary>
            Point_source_nonwindowed_TX = 7
        }
        /// <summary>
        /// Scatter correction type
        /// </summary>
        public enum Scatter_correction {
            /// <summary>
            /// No scatter correction applied
            /// </summary>
            No_correction = 0,
            /// <summary>
            /// Fit of emission tail
            /// </summary>
            Tail = 1,
            /// <summary>
            /// Monte Carlo of emission and transmission tacs
            /// </summary>
            MonteCarlo = 2,
            /// <summary>
            /// Direct calculation from analytical formulas
            /// </summary>
            Direct = 3,
            /// <summary>
            /// Model-based scatter for singles TX
            /// </summary>
            Model_TX = 4,
            /// <summary>
            /// TX off-window windowed coincidence subtraction
            /// </summary>
            Off_window_TX = 5,
            /// <summary>
            /// Singles TX scaled scatter from attenuation subtraction
            /// </summary>
            Singled_TX = 6
        }
        /// <summary>
        /// Calibration units of pixel tacs
        /// </summary>
        public enum Calibration_units {
            /// <summary>
            /// Unknown calibration units
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// nCi/cc
            /// </summary>
            nCi_cc = 1,
            /// <summary>
            /// Bq/cc
            /// </summary>
            Bq_cc = 2
        }
        /// <summary>
        /// Radioactivity dose units
        /// </summary>
        public enum Radioactivity_units {
            /// <summary>
            /// Unknown dose units
            /// </summary>
            Unknown = 0, 
            /// <summary>
            /// millicurie
            /// </summary>
            mCi = 1,
            /// <summary>
            /// megabecquerel
            /// </summary>
            MBq = 2
        }
        /// <summary>
        /// Rebinning type
        /// </summary>
        public enum Rebinning {
            /// <summary>
            /// Unknown, or no algorithm type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Full 3D binning (span and ring difference)
            /// </summary>
            Full3D = 1,
            /// <summary>
            /// Single-Slice Rebinning
            /// </summary>
            SingleSlice = 2,
            /// <summary>
            /// Fourier Rebinning
            /// </summary>
            Fourier = 3
        }
        /// <summary>
        /// TX source type
        /// </summary>
        public enum Source_type {
            /// <summary>
            /// Unknown TX source type
            /// </summary>
            Unknown = 0, 
            /// <summary>
            /// TX point source
            /// </summary>
            Point = 1,
            /// <summary>
            /// TX line source
            /// </summary>
            Line = 2
        }
        /// <summary>
        /// Datarepresentation order type in tacs file.
        /// </summary>
        public enum Data_order {
            /// <summary>
            /// Element/Axis/View/Ring_Diff - view mode (XYZW)
            /// </summary>
            View = 0,
            /// <summary>
            /// Element/View/Axis/Ring_Diff - sinogram mode (XZYW)
            /// </summary>
            Sinogram = 1
        }
        /// <summary>
        /// OSEM 2D recontruction method
        /// </summary>
        public enum OSEM_method {
            /// <summary>
            /// Unweighted osem2d reconstruction
            /// </summary>
            Unweighted = 0,
            /// <summary>
            /// Attenuation weighted osem2d reconstruction
            /// </summary>
            Attenuation_weighted = 1
        }
        /// <summary>
        /// Arc correction type
        /// </summary>
        public enum Arc_correction {
            /// <summary>
            /// Unknown, or no arc correction
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Radial arc correction (1D)
            /// </summary>
            Radial = 1,
            /// <summary>
            /// Resampled radial/angular arc correction (2D)
            /// </summary>
            Resampled = 2
        }
        /// <summary>
        /// Scanner modality configuration number
        /// </summary>
        public enum Modality_configuration
        {
            /// <summary>
            /// 0 - Unknown
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// 2000 - Primate
            /// </summary>
            Primate = 2000,
            /// <summary>
            /// 2001 - Rodent
            /// </summary>
            Rodent = 2001,
            /// <summary>
            /// 2002 - microPET2
            /// </summary>
            microPET2 = 2002,
            /// <summary>
            /// 2500 - Focus_220
            /// </summary>
            Focus_220 = 2500,
            /// <summary>
            /// 2501 - Focus_120
            /// </summary>
            Focus_120 = 2501,
            /// <summary>
            /// 3000 - mCAT
            /// </summary>
            mCAT = 3000,
            /// <summary>
            /// 3500 - mCATII
            /// </summary>
            mCATII = 3500,
            /// <summary>
            /// 3600 - Inveon_MM_Std_CT
            /// </summary>
            Inveon_MM_Std_CT = 3600,
            /// <summary>
            /// 3601 - Inveon_MM_HiRes_Std_CT
            /// </summary>
            Inveon_MM_HiRes_Std_CT = 3601,
            /// <summary>
            /// 3602 - Inveon_MM_Std_LFOV_CT
            /// </summary>
            Inveon_MM_Std_LFOV_CT = 2602,
            /// <summary>
            /// 3603 - Inveon_MM_HiRes_LFOV_CT
            /// </summary>
            Inveon_MM_HiRes_LFOV_CT = 3603,
            /// <summary>
            /// 5000 - Inveon_Dedicated_PET
            /// </summary>
            Inveon_Dedicated_PET = 5000,
            /// <summary>
            /// 6000 - MR_PET_Head_Insert
            /// </summary>
            MR_PET_Head_Insert = 6000,
            /// <summary>
            /// 8000 - Tuebingen_PET_MR
            /// </summary>
            Tuebingen_PET_MR = 8000,
            /// <summary>
            /// 5500 - Inveon_MM_PET
            /// </summary>
            Inveon_MM_PET = 5500
        }
        /// <summary>
        /// Gantry rotation direction
        /// </summary>
        public enum Rotation_direction {
            /// <summary>
            /// Clockwise gantry rotation
            /// </summary>
            Clockwise = 0,
            /// <summary>
            /// Counterclockwise gantry rotation
            /// </summary>
            Counterclockwise = 1
        }
        /// <summary>
        /// Length unit
        /// </summary>
        public enum Length_unit
        {
            /// <summary>
            /// Unknown length units
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// millimeters
            /// </summary>
            mm = 1,
            /// <summary>
            /// centimeters 
            /// </summary>
            cm = 2,
            /// <summary>
            /// inches
            /// </summary>
            inch = 4
        }
        /// <summary>
        /// Weight unit
        /// </summary>
        public enum Weight_unit
        {
            /// <summary>
            /// Unknown weight units
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Grams
            /// </summary>
            g = 1,
            /// <summary>
            /// Ounces
            /// </summary>
            oz = 2,
            /// <summary>
            /// Kilograms
            /// </summary>
            kg = 3,
            /// <summary>
            /// Pounds
            /// </summary>
            lb = 4
        }
        #endregion
        #region header_fields
        /// <summary>
        /// Version of header parameters (float)
        /// </summary>        
        public float version =0;
        /// <summary>
        /// Manufacturer's name (string)
        /// </summary>
        public string manufacturer="";
        /// <summary>
        ///  Scanner model number (integer). default = Unknown
        /// </summary>
        public model_number model = model_number.Unknown;
        /// <summary>
        /// Scanner modality. default = Unknown
        /// </summary>
        public scanner_modality modality = scanner_modality.Unknown;
        /// <summary>
        /// Scanner modality configuration number. default = Unknown
        /// </summary>
        public Modality_configuration modality_configuration = Modality_configuration.Unknown;
        /// <summary>
        /// Institution identification
        /// </summary>
        public string institution = "Undefined institution";
        /// <summary>
        /// Study type/description
        /// </summary>
        public string study = "Undefined study type";
        /// <summary>
        /// Data filename (*.img), possibly including path
        /// </summary>
		public string file_name = string.Empty;
        /// <summary>
        /// Data file type. default = Unknown
        /// </summary>
        public Data_file_type file_type = Data_file_type.Unknown;
        /// <summary>
        /// Acquisition mode. default = Unknown
        /// </summary>
        public Acquisition_mode acquisition_mode = Acquisition_mode.Unknown;
        /// <summary>
        /// Number of frames in tacs file. default = 0
        /// </summary>
        public int total_frames = 0;
        /// <summary>
        /// Number of time frames in tacs file. default = 0
        /// </summary>
        public int time_frames = 0;
        /// <summary>
        /// Bed control. default = Unknown
        /// </summary>
        public Bed_Control bed_control = Bed_Control.Unknown;
        /// <summary>
        /// Bed motion. default = Unknown
        /// </summary>
        public Bed_motion bed_motion = Bed_motion.Static_or_unknown; 
        /// <summary>
        /// Number of bed positions in tacs file. default = 1
        /// </summary>
        public int number_of_bed_positions = 1;
        /// <summary>
        /// Horizontal bed calibration, in microns. deafult = 0.0
        /// </summary>
        public float horizontal_bed_calibration = 0;
        /// <summary>
        /// Vertical bed calibration, in microns. deafult = 0.0
        /// </summary>
        public float vertical_bed_calibration = 0;
        /// <summary>
        /// Transaxial projection bin size, in cm. default = 0.0
        /// </summary>
        public float transaxial_bin_size = 0.0f;
        /// <summary>
        /// Axial plane size, in cm. defaul = 0.0
        /// </summary>
        public float axial_plane_size = 0.0f;
        /// <summary>
        /// Number of detector panels ("rings/CT" are "1"). default = 1
        /// </summary>
        public int number_detector_panels = 1;
        /// <summary>
        /// Pixel tacs type. default = 4-byte float, Intel style
        /// </summary>
        public Data_type data_type = Data_type.Intel_4byte_Float;
        /// <summary>
        /// Number of dimensions in tacs set (integer)
        /// Order from fastest to slowest is XYZW default = 3;
        /// </summary>
        public int number_of_dimensions = 3;
        /// <summary>
        /// Size of X dimension in tacs set
        /// </summary>
        public int x_dimension = 0;
        /// <summary>
        /// Size of Y dimension in tacs set
        /// </summary>
        public int y_dimension = 0;
        /// <summary>
        /// Size of Z dimension in tacs set
        /// </summary>
        public int z_dimension = 0;
        /// <summary>
        /// Acquisition user ID 
        /// </summary>
		public string acquisition_user_id = string.Empty;
        /// <summary>
        /// Histogram user ID 
        /// </summary>
		public string histogram_user_id = string.Empty;
        /// <summary>
        /// Reconstruction user ID
        /// </summary>
		public string reconstruction_user_id = string.Empty;
        /// <summary>
        /// Scatter correction user ID
        /// </summary>
		public string scatter_correction_user_id = string.Empty;
        /// <summary>
        /// Acquisition notes
        /// </summary>
		public string acquisition_notes = string.Empty;
        /// <summary>
        /// Scan start date and time
        /// Format is: Sun Sep 16 01:03:52 1973
        /// </summary>
		public string scan_time = string.Empty;
        /// <summary>
        /// Scan start date and time - GMT-based
        /// Format is: Sun Sep 16 01:03:52 1973
        /// </summary>
		public string gmt_scan_time = string.Empty;
        /// <summary>
        /// X origin of volume, in voxels
        /// </summary>
        public int volume_origin_x = 0;
        /// <summary>
        /// Y origin of volume, in voxels
        /// </summary>
        public int volume_origin_y = 0;
        /// <summary>
        /// Z origin of volume, in voxels
        /// </summary>
        public int volume_origin_z = 0;
        /// <summary>
        /// Registration tacs availability. default = No tacs available
        /// </summary>
        public Registration_data_availability registration_available = Registration_data_availability.No_registration_data;
        /// <summary>
        /// Transformation matrix filename, possibly including path (string)
        ///    NOTE: Filename may contain spaces, therefore the ENTIRE 
        ///          line, up to the EOL, is used after the parameter name.
        /// </summary>
		public string transformation_matrix = string.Empty;
        /// <summary>
        /// Spatial identification for registration
        /// </summary>
		public string spatial_identifier = string.Empty;
        /// <summary>
        /// Normalization type applied to tacs set. deafult = Cylinder source, component based 
        /// </summary>
        public Normalization_type normalization_applied = Normalization_type.Cylinder_source_component;
        /// <summary>
        /// Normalization filename (*.nrm), possibly including path (string)
        ///    NOTE: Filename may contain spaces, therefore the ENTIRE 
        ///          line, up to the EOL, is used after the parameter name. 
        /// </summary>
		public string normalization_filename = string.Empty;
        /// <summary>
        /// Reconstruction algorithm type. default = 2D OSEM
        /// </summary>
        public Reconstruction_type recon_algorithm = Reconstruction_type.OSEM2d;
        /// <summary>
        /// Version of reconstruction program used
        /// </summary>
        public float recon_version = 0.0f;
        /// <summary>
        /// X filter and/or apodizing windows type. default = No filter
        ///   The Ramp and Shepp should ONLY be used for backprojection  
        /// </summary>
        public Filter_type x_filter_type = Filter_type.No_filter;
        /// <summary>
        /// X filter cutoff 
        /// NOTE that a cutoff of 0.5 is the Nyquist point
        ///   i.e 1.0 / (2.0 * sampling).
        /// </summary>
        public float x_filter_cutoff = 0.5f;
        /// <summary>
        /// Y filter and/or apodizing windows type. default = No filter
        ///   The Ramp and Shepp should ONLY be used for backprojection  
        /// </summary>
        public Filter_type y_filter_type = Filter_type.No_filter;
        /// <summary>
        /// Y filter cutoff 
        /// NOTE that a cutoff of 0.5 is the Nyquist point
        ///   i.e 1.0 / (2.0 * sampling).
        /// </summary>
        public float y_filter_cutoff = 0.5f;
        /// <summary>
        /// Z filter and/or apodizing windows type. default = No filter
        ///   The Ramp and Shepp should ONLY be used for backprojection  
        /// </summary>
        public Filter_type z_filter_type = Filter_type.No_filter;
        /// <summary>
        /// Z filter cutoff 
        /// NOTE that a cutoff of 0.5 is the Nyquist point
        ///   i.e 1.0 / (2.0 * sampling).
        /// </summary>
        public float z_filter_cutoff = 0.5f;
        /// <summary>
        /// X offset, in cm, applied to tacs set. default = 0.0
        /// </summary>
        public float x_offset = 0.0f;
        /// <summary>
        /// Y offset, in cm, applied to tacs set. default = 0.0
        /// </summary>
        public float y_offset = 0.0f;
        /// <summary>
        /// Zoom applied to tacs set. default = 1.0 (no zoom)
        /// </summary>
        public float zoom = 1.0f;
        /// <summary>
        /// Reconstructed pixel size, in cm
        ///     NOTE: pixel_size = (((X_crystal_pitch / 2.0) * X_dim) /
        ///           (image_size * zoom)) * (effective_radius / radius)
        /// </summary>
        public float pixel_size = 0.0f;
        /// <summary>
        /// Reconstructed pixel size in X, in mm. default = 1.0
        /// </summary>
        public float pixel_size_x = 1.0f;
        /// <summary>
        /// Reconstructed pixel size in Y, in mm. default = 1.0
        /// </summary>
        public float pixel_size_y = 1.0f;
        /// <summary>
        /// Reconstructed pixel size in Z, in mm. default = 1.0
        /// </summary>
        public float pixel_size_z = 1.0f;
        /// <summary>
        /// Investigator identification
        /// </summary>
		public string investigator_id = string.Empty;
        /// <summary>
        /// Operator identification
        /// </summary>
		public string operator_id = string.Empty;
        /// <summary>
        /// Study identification (e.g. DICOM UID)
        /// </summary>
		public string study_identifier = string.Empty;
        /// <summary>
        /// Injected compound
        /// </summary>
		public string injected_compound = string.Empty;
        /// <summary>
        /// Subject identifier
        /// </summary>
		public string subject_identifier = string.Empty;
        /// <summary>
        /// Subject genus
        /// </summary>
		public string subject_genus = string.Empty;
        /// <summary>
        /// Subject orientation. default = Unknown
        /// </summary>
        public Subject_orientation subject_orientation = Subject_orientation.Unknown;
        /// <summary>
        /// Subject length units. default = Unknown
        /// </summary>
        public Length_unit subject_length_units = Length_unit.Unknown;
        /// <summary>
        /// Subject length in length units. default = 0.0
        /// <see cref="subject_length_units"/>
        /// </summary>
        public float subject_length = 0.0f;
        /// <summary>
        /// Subject weight units. default = Unknown
        /// </summary>
        public Weight_unit subject_weight_units = Weight_unit.Unknown;
        /// <summary>
        /// Subject length in weight units. default = 0.0
        /// <see cref="subject_weight_units"/>
        /// </summary>
        public float subject_weight = 0.0f;
        /// <summary>
        /// Subject phenotype
        /// </summary>
		public string subject_phenotype = string.Empty;
        /// <summary>
        /// Study model
        /// </summary>
		public string study_model = string.Empty;
        /// <summary>
        /// Subject anesthesia
        /// </summary>
		public string anesthesia = string.Empty;
        /// <summary>
        /// Subject analgesia
        /// </summary>
		public string analgesia = string.Empty;
        /// <summary>
        /// Other drugs
        /// </summary>
		public string other_drugs = string.Empty;
        /// <summary>
        /// Food access
        /// </summary>
		public string food_access = string.Empty;
        /// <summary>
        /// Water access
        /// </summary>
		public string water_access = string.Empty;
        /// <summary>
        /// Subject date of birth
        /// </summary>
		public string subject_date_of_birth = string.Empty;
        /// <summary>
        /// Subject age
        /// </summary>
		public string subject_age = string.Empty;
        /// <summary>
        /// Subject sex
        /// </summary>
		public string subject_sex = string.Empty;
        /// <summary>
        /// Subject scan region
        /// </summary>
		public string subject_scan_region = string.Empty;
        /// <summary>
        /// Original acquisition filename (*.cat), DOES NOT INCLUDE path
        ///    NOTE: Filename may contain spaces, therefore the ENTIRE
        ///          line, up to the EOL, is used after the parameter name.
        /// </summary>
		public string acquisition_file_name = string.Empty;
        /// <summary>
        /// Transaxial crystal pitch, in cm
        /// </summary>
        public float transaxial_crystal_pitch = 0.0f;
        /// <summary>
        /// Axial crystal pitch, in cm
        /// </summary>
        public float axial_crystal_pitch = 0.0f;
        /// <summary>
        /// Crystal thickness, in cm
        /// </summary>
        public float crystal_thickness = 0.0f;
        /// <summary>
        /// Depth of interaction, in cm
        /// </summary>
        public float depth_of_interaction = 0.0f;
        /// <summary>
        /// Isotope description (isotope name e.g. C-11)
        /// </summary>
		public string isotope = string.Empty;
        /// <summary>
        /// Isotope half-life, in secs. default = 0.0
        /// </summary>
        public float isotope_half_life = 0.0f;
        /// <summary>
        /// Isotope branching fraction. default = 1.0
        ///   NOTE: Frame scale factor DOES NOT include
        ///         isotope branching fraction.
        /// </summary>
        public float isotope_branching_fraction = 1.0f;
        /// <summary>
        /// Deadtime correction applied to tacs set. default = Global
        /// </summary>
        public Deadtime_correction deadtime_correction_applied = Deadtime_correction.Global;
        /// <summary>
        /// True if decay correction is applied to dataset. default = false
        /// </summary>
        public bool decay_correction_applied = false;
        /// <summary>
        /// Attenuation type that is applied to tacs set. default = No attenuation
        /// </summary>
        public Attenuation_correction attenuation_applied = Attenuation_correction.No_attenuation;
        /// <summary>
        /// Attenuation correction filename (*.atn), possibly including path
        ///    NOTE: Filename may contain spaces, therefore the ENTIRE 
        ///          line, up to the EOL, is used after the parameter name.
        /// </summary>
		public string attenuation_filename = string.Empty;
        /// <summary>
        /// Scatter correction that is applied to tacs set. default = No correction
        /// </summary>
        public Scatter_correction scatter_correction = Scatter_correction.No_correction;
        /// <summary>
        /// Data calibration units. default = Bq/cc
        /// </summary>
        public Calibration_units calibration_units = Calibration_units.Bq_cc;
        /// <summary>
        /// Data calibration factor. default = 1.0
        ///   NOTE: Frame scale factor DOES NOT include
        ///         calibration factor.
        /// </summary>
        public float calibration_factor = 1.0f;
        /// <summary>
        /// Calibration source branching fraction
        ///   NOTE: Frame scale factor DOES NOT include
        ///         calibration source branching fraction.
        /// </summary>
        public float calibration_branching_fraction = 1.0f;
        /// <summary>
        /// Dose units. default = Unknown
        /// </summary>
        public Radioactivity_units dose_units = Radioactivity_units.Unknown;
        /// <summary>
        /// Injected dose. default = 0.0
        /// </summary>
        public float dose = 0.0f;
        /// <summary>
        /// Injection date and time
        ///    Format is: Sun Sep 16 01:03:52 1973
        /// </summary>
		public string injection_time = string.Empty;
        /// <summary>
        /// Injection decay correction factor
        ///   NOTE: Frame scale factor and decay correction factor 
        ///   DO NOT include injection decay correction factor.
        /// </summary>
        public float injection_decay_correction = 1.0f;
        /// <summary>
        /// Pre- and residual activity units. default = MBq
        /// </summary>
        public Radioactivity_units activity_units = Radioactivity_units.MBq;
        /// <summary>
        /// Activity before injection. default = 0.0;
        /// </summary>
        public float activity_before_injection = 0.0f;
        /// <summary>
        /// Activity before injection measurement date and time
        ///    Format is: Sun Sep 16 01:03:52 1973
        /// </summary>
		public string activity_before_injection_time = string.Empty;
        /// <summary>
        /// Residual activity. default = 0.0
        /// </summary>
        public float residual_activity = 0.0f;
        /// <summary>
        /// Residual activity meansurement date and time
        ///    Format is: Sun Sep 16 01:03:52 1973
        /// </summary>
		public string residual_activity_time = string.Empty;
        /// <summary>
        /// Transaxial crystals per block. default = 0
        /// </summary>
        public int transaxial_crystals_per_block = 0;
        /// <summary>
        /// Axial crystals per block. default = 0
        /// </summary>
        public int axial_crystals_per_block = 0;
        /// <summary>
        /// Crystal offset for intrisic rotation. default = 0
        /// </summary>
        public int intrinsic_crystal_offset = 0;
        /// <summary>
        /// Number of transaxil blocks. default = 0
        /// </summary>
        public int transaxial_blocks = 0;
        /// <summary>
        /// Number of axial blocks. default = 0
        /// </summary>
        public int axial_blocks = 0;
        /// <summary>
        /// Ring radius to crystal face, in cm. default = 0.0
        /// </summary>
        public float radius = 0.0f;
        /// <summary>
        /// Radial field-of-view, in cm 
        /// </summary>
        public float radial_fov = 0.0f;
        /// <summary>
        /// Source radius, in cm 
        /// </summary>
        public float src_radius = 0.0f;
        /// <summary>
        /// Source axial cm per revolution, in cm
        /// </summary>
        public float src_cm_per_rev = 0.0f;
        /// <summary>
        /// Source encoder steps per revolution
        /// </summary>
        public int src_steps_per_rev = 0;
        /// <summary>
        /// Default number of projections
        /// </summary>
        public int default_projections = 0;
        /// <summary>
        /// Default number of transaxial angles 
        /// </summary>
        public int default_transaxial_angles = 0;
        /// <summary>
        /// Lower level energy threshold, in KeV
        /// </summary>
        public float lld = 0.0f;
        /// <summary>
        /// Upper level energy threshold, in KeV
        /// </summary>
        public float uld = 0.0f;
        /// <summary>
        /// Coincidence timing window, in nsecs
        /// </summary>
        public float timing_window = 0.0f;
        /// <summary>
        /// Span of tacs set
        /// </summary>
        public int span = 0;
        /// <summary>
        /// Maximum ring difference of tacs set
        /// </summary>
        public int ring_difference = 0;
        /// <summary>
        /// Size of W (time) dimension in tacs set
        /// </summary>
        public int w_dimension = 0;
        /// <summary>
        /// Version of histogram program used
        /// </summary>
        public float histogram_version = 0.0f;
        /// <summary>
        /// Rebinng type. default = Unknown
        /// </summary>
        public Rebinning rebinning_type = Rebinning.Unknown;
        /// <summary>
        /// Version of rebinning program used
        /// </summary>
        public float rebinning_version = 0.0f;
        /// <summary>
        /// Source type. default = Unknown
        /// </summary>
        public Source_type tx_src_type = Source_type.Unknown;
        /// <summary>
        /// Data order. default = Sinogram (XYZW)
        ///  NOTE that ElVwAxRd (XYZW) is the tacs order for images.
        ///    ElVwAxRd for images means that Z and Y are flipped.
        /// </summary>
        public Data_order data_order = Data_order.Sinogram;
        /// <summary>
        /// 2D OSEM recontruction method. default = Unweighted
        /// </summary>
        public OSEM_method osem2d_method = OSEM_method.Unweighted;
        /// <summary>
        /// Number of osem2d subsets
        /// </summary>
        public int osem2d_subsets = 0;
        /// <summary>
        /// Number of osem2d iterations
        /// </summary>
        public int osem2d_iterations = 0;
        /// <summary>
        /// Number of EM iterations after osem2d iterations
        /// </summary>
        public int osem2d_em_iterations = 0;
        /// <summary>
        /// Epsilon value for map regularization
        /// </summary>
        public float osem2d_map_E = 0;
        /// <summary>
        /// Power value for map regularization
        /// </summary>
        public int osem2d_map_P = 0;
        /// <summary>
        /// Large object osem2d x_offset in cm
        /// </summary>
        public float osem2d_x_offset = 0.0f;
        /// <summary>
        /// Large object osem2d y_offset in cm
        /// </summary>
        public float osem2d_y_offset = 0.0f;
        /// <summary>
        /// Large object osem2d zoom
        /// </summary>
        public float osem2d_zoom = 1.0f;
        /// <summary>
        /// Arc correction applied to tacs set. default = Unknown or no correction
        /// </summary>
        public Arc_correction arc_correction_applied = Arc_correction.Unknown;
        /// <summary>
        /// Number of singles rates in subheader (integer)
        ///   NOTE: This normally is the number of blocks.
        /// </summary>
        public int number_of_singles_rates = 0;
        /// <summary>
        /// Subject glucose level
        /// </summary>
		public string subject_glucose_level = string.Empty;
        /// <summary>
        /// Subject glucose level measurement time
        /// </summary>
		public string subject_glucose_level_time = string.Empty;
                #endregion
        /// <summary>
        /// Default constructor.
        /// </summary>
		public MicroPETHeader() { }

		/// <summary>
		/// Constructor. Create a Micropet header based on a general image header.
		/// </summary>
		/// <param name="hdr"></param>
		public MicroPETHeader(ImageHeader hdr)
		{
			switch (hdr.Datatype)
			{
				case ImageFile.DataType.BIT8_S:
				case ImageFile.DataType.BIT8_U:
					data_type = MicroPETHeader.Data_type.Byte;
					break;
				case ImageFile.DataType.BIT16_S:
				case ImageFile.DataType.BIT16_U:
					data_type = MicroPETHeader.Data_type.Intel_2byte_Int;
					break;
				case ImageFile.DataType.SUNI2:
				case ImageFile.DataType.VAXI16:
					data_type = MicroPETHeader.Data_type.Sun_2byte_Int;
					break;
				case ImageFile.DataType.BIT32_S:
				case ImageFile.DataType.BIT32_U:
					data_type = MicroPETHeader.Data_type.Intel_4byte_Int;
					break;
				case ImageFile.DataType.FLT32:
					data_type = MicroPETHeader.Data_type.Intel_4byte_Float;
					break;
				case ImageFile.DataType.SUNI4:
					data_type = MicroPETHeader.Data_type.Sun_4byte_Int;
					break;
				case ImageFile.DataType.VAXFL32:
					data_type = MicroPETHeader.Data_type.Sun_4byte_Float;
					break;
				default:
					data_type = MicroPETHeader.Data_type.Unknown;
					break;
			}
			if (hdr.IsDynamic) acquisition_mode = Acquisition_mode.Dynamic;
			dose_units = MicroPETHeader.Radioactivity_units.Unknown;
			subject_identifier = hdr.PatientID;
			study = hdr.Description;
			x_dimension = hdr.DimX;
			y_dimension = hdr.DimY;
			z_dimension = hdr.DimZ;
			time_frames = hdr.Frames;
			number_of_bed_positions = hdr.Dim.Beds;
			injection_time = hdr.DoseStartTime.ToString("ddd MMM d HH:mm:ss.fff yyyy", System.Globalization.CultureInfo.CreateSpecificCulture("en-US"));
			isotope = hdr.Radiopharma;
			isotope_half_life = (float)TPClib.Isotope.GetHalflife(hdr.Isotope);
			pixel_size_x = hdr.SizeX;
			pixel_size_y = hdr.SizeY;
			pixel_size_z = hdr.SizeZ;
			switch (hdr.Modality)
			{
				case ImageModality.M_CT:
					modality = MicroPETHeader.scanner_modality.CT;
					break;
				case ImageModality.M_PT:
					modality = MicroPETHeader.scanner_modality.PET;
					break;
				case ImageModality.M_ST:
					modality = MicroPETHeader.scanner_modality.SPECT;
					break;
				default:
					modality = MicroPETHeader.scanner_modality.Unknown;
					break;
			}
		}
 
        /// <summary>
        /// Copy constructor.
        /// </summary>
        public MicroPETHeader(MicroPETHeader hdr)
        {
            this.acquisition_file_name = hdr.acquisition_file_name;
            this.acquisition_mode = hdr.acquisition_mode;
            this.acquisition_notes = hdr.acquisition_notes;
            this.acquisition_user_id = hdr.acquisition_user_id;
            this.activity_before_injection = hdr.activity_before_injection;
            this.activity_before_injection_time = hdr.activity_before_injection_time;
            this.activity_units = hdr.activity_units;
            this.analgesia = hdr.analgesia;
            this.anesthesia = hdr.anesthesia;
            this.arc_correction_applied = hdr.arc_correction_applied;
            this.attenuation_applied = hdr.attenuation_applied;
            this.attenuation_filename = hdr.attenuation_filename;
            this.axial_blocks = hdr.axial_blocks;
            this.axial_crystal_pitch = hdr.axial_crystal_pitch;
            this.axial_crystals_per_block = hdr.axial_crystals_per_block;
            this.axial_plane_size = hdr.axial_plane_size;
            this.bed_control = hdr.bed_control;
            this.bed_motion = hdr.bed_motion;
            this.calibration_branching_fraction = hdr.calibration_branching_fraction;
            this.calibration_factor = hdr.calibration_factor;
            this.calibration_units = hdr.calibration_units;
            this.crystal_thickness = hdr.crystal_thickness;
            this.data_order = hdr.data_order;
            this.data_type = hdr.data_type;
            this.deadtime_correction_applied = hdr.deadtime_correction_applied;
            this.decay_correction_applied = hdr.decay_correction_applied;
            this.default_projections = hdr.default_projections;
            this.default_transaxial_angles = hdr.default_transaxial_angles;
            this.depth_of_interaction = hdr.depth_of_interaction;
            this.dose = hdr.dose;
            this.dose_units = hdr.dose_units;
            this.file_name = hdr.file_name;
            this.file_type = hdr.file_type;
            this.food_access = hdr.food_access;
            this.gmt_scan_time = hdr.gmt_scan_time;
            this.histogram_user_id = hdr.histogram_user_id;
            this.histogram_version = hdr.histogram_version;
            this.horizontal_bed_calibration = hdr.horizontal_bed_calibration;
            this.injected_compound = hdr.injected_compound;
            this.injection_decay_correction = hdr.injection_decay_correction;
            this.injection_time = hdr.injection_time;
            this.institution = hdr.institution;
            this.intrinsic_crystal_offset = hdr.intrinsic_crystal_offset;
            this.investigator_id = hdr.investigator_id;
            this.isotope = hdr.isotope;
            this.isotope_branching_fraction = hdr.isotope_branching_fraction;
            this.isotope_half_life = hdr.isotope_half_life;
            this.lld = hdr.lld;
            this.manufacturer = hdr.manufacturer;
            this.modality = hdr.modality;
            this.modality_configuration = hdr.modality_configuration;
            this.model = hdr.model;
            this.normalization_applied = hdr.normalization_applied;
            this.normalization_filename = hdr.normalization_filename;
            this.number_detector_panels = hdr.number_detector_panels;
            this.number_of_bed_positions = hdr.number_of_bed_positions;
            this.number_of_dimensions = hdr.number_of_dimensions;
            this.number_of_singles_rates = hdr.number_of_singles_rates;
            this.operator_id = hdr.operator_id;
            this.osem2d_em_iterations = hdr.osem2d_em_iterations;
            this.osem2d_iterations = hdr.osem2d_iterations;
            this.osem2d_map_E = hdr.osem2d_map_E;
            this.osem2d_map_P = hdr.osem2d_map_P;
            this.osem2d_method = hdr.osem2d_method;
            this.osem2d_subsets = hdr.osem2d_subsets;
            this.osem2d_x_offset = hdr.osem2d_x_offset;
            this.osem2d_y_offset = hdr.osem2d_y_offset;
            this.osem2d_zoom = hdr.osem2d_zoom;
            this.other_drugs = hdr.other_drugs;
            this.pixel_size = hdr.pixel_size;
            this.pixel_size_x = hdr.pixel_size_x;
            this.pixel_size_y = hdr.pixel_size_y;
            this.pixel_size_z = hdr.pixel_size_z;
            this.radial_fov = hdr.radial_fov;
            this.radius = hdr.radius;
            this.rebinning_type = hdr.rebinning_type;
            this.rebinning_version = hdr.rebinning_version;
            this.recon_algorithm = hdr.recon_algorithm;
            this.recon_version = hdr.recon_version;
            this.reconstruction_user_id = hdr.reconstruction_user_id;
            this.registration_available = hdr.registration_available;
            this.residual_activity = hdr.residual_activity;
            this.residual_activity_time = hdr.residual_activity_time;
            this.ring_difference = hdr.ring_difference;
            this.scan_time = hdr.scan_time;
            this.scatter_correction = hdr.scatter_correction;
            this.scatter_correction_user_id = hdr.scatter_correction_user_id;
            this.span = hdr.span;
            this.spatial_identifier = hdr.spatial_identifier;
            this.src_cm_per_rev = hdr.src_cm_per_rev;
            this.src_radius = hdr.src_radius;
            this.src_steps_per_rev = hdr.src_steps_per_rev;
            this.study = hdr.study;
            this.study_identifier = hdr.study_identifier;
            this.study_model = hdr.study_model;
            this.subject_age = hdr.subject_age;
            this.subject_date_of_birth = hdr.subject_date_of_birth;
            this.subject_genus = hdr.subject_genus;
            this.subject_glucose_level = hdr.subject_glucose_level;
            this.subject_glucose_level_time = hdr.subject_glucose_level_time;
            this.subject_identifier = hdr.subject_identifier;
            this.subject_length = hdr.subject_length;
            this.subject_length_units = hdr.subject_length_units;
            this.subject_orientation = hdr.subject_orientation;
            this.subject_phenotype = hdr.subject_phenotype;
            this.subject_scan_region = hdr.subject_scan_region;
            this.subject_sex = hdr.subject_sex;
            this.subject_weight = hdr.subject_weight;
            this.subject_weight_units = hdr.subject_weight_units;
            this.time_frames = hdr.time_frames;
            this.timing_window = hdr.timing_window;
            this.total_frames = hdr.total_frames;
            this.transaxial_bin_size = hdr.transaxial_bin_size;
            this.transaxial_blocks = hdr.transaxial_blocks;
            this.transaxial_crystal_pitch = hdr.transaxial_crystal_pitch;
            this.transaxial_crystals_per_block = hdr.transaxial_crystals_per_block;
            this.transformation_matrix = hdr.transformation_matrix;
            this.tx_src_type = hdr.tx_src_type;
            this.uld = hdr.uld;
            this.w_dimension = hdr.w_dimension;
            this.water_access = hdr.water_access;
            this.version = hdr.version;
            this.vertical_bed_calibration = hdr.vertical_bed_calibration;
            this.volume_origin_x = hdr.volume_origin_x;
            this.volume_origin_y = hdr.volume_origin_y;
            this.volume_origin_z = hdr.volume_origin_z;
            this.x_dimension = hdr.x_dimension;
            this.x_filter_cutoff = hdr.x_filter_cutoff;
            this.x_filter_type = hdr.x_filter_type;
            this.x_offset = hdr.x_offset;
            this.y_dimension = hdr.y_dimension;
            this.y_filter_cutoff = hdr.y_filter_cutoff;
            this.y_filter_type = hdr.y_filter_type;
            this.y_offset = hdr.y_offset;
            this.z_dimension = hdr.z_dimension;
            this.z_filter_cutoff = hdr.z_filter_cutoff;
            this.z_filter_type = hdr.z_filter_type;
            this.zoom = hdr.zoom;
        }
    }
    /// <summary>
    /// CT specific MicroPET header tacs
    /// </summary>
    public class MicroPETHeader_CTspecific: MicroPETHeader
    {
        #region enumerators
        /// <summary>
        /// CT warping type
        /// </summary>
        public enum Warping_type {
            /// <summary>
            /// Unknown warping type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// No warping
            /// </summary>
            No_warping = 1,
            /// <summary>
            /// Bilinear warping
            /// </summary>
            Bilinear = 2,
            /// <summary>
            /// Nearest neighbor warping
            /// </summary>
            Nearest_neighbour = 3
        }
        /// <summary>
        /// CT projection interpolation type
        /// </summary>
        public enum Interpolation_type {
            /// <summary>
            /// Unknown projection interpolation type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Bilinear projection interpolation
            /// </summary>
            Bilinear = 1,
            /// <summary>
            /// Nearest neighbor projection interpolation
            /// </summary>
            Nearest_neighbour = 2
        }
        #endregion
        #region structures
        /// <summary>
        /// CT projection information
        /// </summary>
        public struct Projection {
            /// <summary>
            /// Projection number
            /// </summary>
            public int projection_number;
            /// <summary>
            /// Float tacs for projection
            /// </summary>
            public float data;
        }
        #endregion
        #region field_names
        /// <summary>
        /// Rotation, in degrees, applied to tacs set (not applied to OSEM2d reconstructions)
        /// </summary>
        public float rotation;
        /// <summary>
        /// Z offset, in cm, applied to tacs set. default = 0.0
        /// </summary>
        public float z_offset = 0.0f;
        /// <summary>
        /// Gantry rotation. For CT only.
        /// </summary>
        public bool gantry_rotation;
        /// <summary>
        /// Rotation direction. For CT only.
        /// </summary>
        public Rotation_direction rotation_direction;
        /// <summary>
        /// Rotating gantry starting angle, in degrees. default = 0.0
        /// </summary>
        public float rotating_stage_start_position = 0.0f;
        /// <summary>
        /// Rotating gantry stop angle, in degrees. default = 0.0
        /// </summary>
        public float rotating_stage_stop_position = 0.0f;
        /// <summary>
        /// Number of rotation projections for rotating gantry
        /// </summary>
        public int number_of_projections;
        /// <summary>
        /// Following parameters are used for CT modality only.
        /// </summary>
        public int ct_file_version;
        /// <summary>
        /// Header size of CAT files before dark and light projections.
        /// </summary>
        public int ct_header_size;
        /// <summary>
        /// CT transaxial projection size, in pixels
        /// </summary>
        public int ct_proj_size_transaxial;
        /// <summary>
        /// CT axial projection size, in pixels
        /// </summary>
        public int ct_proj_size_axial;
        /// <summary>
        /// Number to average the dark calibration projections
        /// </summary>
        public int ct_average_dark_projections;
        /// <summary>
        /// Number to average the light calibration projection
        /// </summary>
        public int ct_average_light_projections;
        /// <summary>
        /// Total positions to acquire the light calibration projections
        /// </summary>
        public int ct_light_calibration_projections;
        /// <summary>
        /// Indicates if the positions to acquire light projections are same as scan projection positions
        /// </summary>
        public int ct_dependent_light_calibration_projections;
        /// <summary>
        /// CT X-ray detector offset, in mm
        /// </summary>
        public float ct_xray_detector_offset;
        /// <summary>
        /// CT detector transaxial position, in cm
        /// </summary>
        public float ct_detector_transaxial_position;
        /// <summary>
        /// CT detector uncropped transaxial pixels
        /// </summary>
        public int ct_uncropped_transaxial_pixels;
        /// <summary>
        /// CT detector uncropped axial pixels
        /// </summary>
        public int ct_uncropped_axial_pixels;
        /// <summary>
        /// CT detector cropped transaxial pixels
        /// </summary>
        public int ct_cropped_transaxial_pixels;
        /// <summary>
        /// CT detector cropped axial pixels
        /// </summary>
        public int ct_cropped_axial_pixels;
        /// <summary>
        /// CT X-ray detector pitch, in um
        /// </summary>
        public float ct_xray_detector_pitch;
        /// <summary>
        /// CT horizontal rotation-axis-bed angle, in degrees
        /// </summary>
        public float ct_horiz_rot_axis_bed_angle;
        /// <summary>
        /// CT vertical rotation-axis-bed angle, in degrees
        /// </summary>
        public float ct_vert_rot_axis_bed_angle;
        /// <summary>
        /// CT exposure time, in msecs
        /// </summary>
        public float ct_exposure_time;
        /// <summary>
        /// CT total scan time, in [min,sec]
        /// </summary>
        public int[] ct_scan_time = new int[2];
        /// <summary>
        /// CT warping type
        /// </summary>
        public Warping_type ct_warping;
        /// <summary>
        /// CT defect map filename, possibly including path
        ///    NOTE: Filename may contain spaces, therefore the ENTIRE 
        ///          line, up to the EOL, is used after the parameter name.
        /// </summary>
        public string ct_defect_map_file_name;
        /// <summary>
        /// CT x-ray voltage, in kVp
        /// </summary>
        public float ct_xray_voltage;
        /// <summary>
        /// CT anode current, in uA
        /// </summary>
        public float ct_anode_current;
        /// <summary>
        /// Number of CT calibration exposures
        /// </summary>
        public int ct_calibration_exposures;
        /// <summary>
        /// CT cone angle, in degrees
        /// </summary>
        public float ct_cone_angle;
        /// <summary>
        /// CT projection interpolation type. default = Unknown
        /// </summary>
        public Interpolation_type ct_projection_interpolation = 0;
        /// <summary>
        /// CT source to detector distance, in cm
        /// </summary>
        public float ct_source_to_detector;
        /// <summary>
        /// CT source to center of rotation, in cm
        /// </summary>
        public float ct_source_to_crot;
        /// <summary>
        /// CT vertical detector offset, in pixels
        /// </summary>
        public float ct_detector_vertical_offset;
        /// <summary>
        /// CT detector tilt relative to horizontal axis, in degrees
        /// </summary>
        public float ct_detector_horizontal_tilt;
        /// <summary>
        /// CT detector tilt relative to vertical axis, in degrees
        /// </summary>
        public float ct_detector_vertical_tilt;
        /// <summary>
        /// CT transaxial projection bin factor, in pixels
        /// </summary>
        public int ct_transaxial_bin_factor;
        /// <summary>
        /// CT axial projection bin factor, in pixels
        /// </summary>
        public int ct_axial_bin_factor;
        /// <summary>
        /// CT gating signal used
        /// </summary>
        public int ct_gating;
        /// <summary>
        /// CT Hounsfield scale
        /// </summary>
        public float ct_hounsfield_scale;
        /// <summary>
        /// CT Hounsfield offset
        /// </summary>
        public float ct_hounsfield_offset;
        /// <summary>
        /// CT projection downsample factor
        /// </summary>
        public int ct_proj_downsample_factor;
        /// <summary>
        /// CT first projection used in reconstruction
        /// </summary>
        public int ct_first_recon_proj;
         /// <summary>
        /// CT last projection used in reconstruction
        /// </summary>
        public int ct_last_recon_proj;
        /// <summary>
        /// CT every Nth projection used for reconstruction
        /// </summary>
        public int ct_recon_every_nth_proj;
        /// <summary>
        /// CT attenuation of water, in cm^-1
        /// </summary>
        public float ct_attenuation_water;
        /// <summary>
        /// CT TX rotation offsets: X Y Z, in mm
        /// </summary>
        public float[] ct_tx_rotation_offsets = new float[3];
        /// <summary>
        /// CT TX transaxial offsets: X Y Z, in mm
        /// </summary>
        public float[] ct_tx_transaxial_offsets = new float[3];
        /// <summary>
        /// CT BH correction applied
        ///     0 (FALSE) - CT BH correction has NOT been applied
        ///     !0 (TRUE) - CT BH correction has been applied
        /// </summary>
        public float ct_bh_correction;
        /// <summary>
        /// CT BH correction coefficients
        ///   NOTE: This is ONLY present when BH correction has been applied.
        /// </summary>
        public float[] ct_bh_correction_coefficients = new float[4];
        /// <summary>
        /// CT aluminum filter thickness (mm)
        /// </summary>
        public float ct_aluminum_filter_thickness;
        /// <summary>
        /// Projection array {projection number, acquisition position(degrees)}
        /// </summary>
        public Projection[] projections;
        /// <summary>
        /// CT projection average center offset
        /// </summary>
        public float ct_projection_average_center_offset;
        /// <summary>
        /// CT projection average center offset array {projection number, average center offset(mm)}
        /// </summary>
        public Projection[] ct_projection_center_offsets;
        /// <summary>
        /// CT projection average center offset array {projection number, horizontal bed offset(mm)}
        /// </summary>
        public Projection[] ct_projection_horizontal_bed_offsets;
        #endregion
        /// <summary>
        /// Constructor from superclass.
        /// </summary>
        public MicroPETHeader_CTspecific(MicroPETHeader hdr): base(hdr)
        {
        }
        /// <summary>
        /// Default constructor.
        /// </summary>
        public MicroPETHeader_CTspecific()
        {
        }
    }
    /// <summary>
    /// Micro PET subheader information
    /// </summary>
    public class MicroPETSubHeader
    {
        #region enumerator_types
        /// <summary>
        /// Event type information in this subheader
        /// </summary>
        public enum Event_type {
            /// <summary>
            /// Unknown event type
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Singles
            /// </summary>
            Singles = 1,
            /// <summary>
            /// Prompt events (coincidences)
            /// </summary>
            Prompt = 2,
            /// <summary>
            /// Delay events
            /// </summary>
            Delay = 3,
            /// <summary>
            /// Trues (prompts - delays)
            /// </summary>
            Trues = 4,
            /// <summary>
            /// Energy spectrum tacs
            /// </summary>
            Energy = 5
        }
        #endregion
        /// <summary>
        /// Singles rate information
        /// </summary>
        public struct Singles_rate {
            /// <summary>
            /// Block index number [0..No blocks-1]
            /// </summary>
            public int block_number;
            /// <summary>
            /// singles/sec
            /// </summary>
            public float singles_per_sec;
            /// <summary>
            /// raw_singles/sec
            /// </summary>
            public float raw_singles_per_sec;   
            /// <summary>
            /// xy_stored_singles/sec
            /// </summary>
            public float xy_stored_singles_per_sec;    
            /// <summary>
            /// block_singles_output/sec
            /// </summary>
            public float block_singles_output_per_sec;  
            /// <summary>
            /// raw_block_prompts/sec
            /// </summary>
            public float raw_block_prompts_per_sec;    
            /// <summary>
            /// block_prompts/sec
            /// </summary>
            public float block_prompts_per_sec;    
            /// <summary>
            /// raw_block_delays/sec
            /// </summary>
            public float raw_block_delays_per_sec;   
            /// <summary>
            /// block_delays/sec
            /// </summary>
            public float block_delays_per_sec;
        }
        #region header_fields
        /// <summary>
        /// Frame number [1..No frames]
        /// </summary>
        public int frame = 1;
        /// <summary>
        /// Detector panel - NOTE: "ring" systems are "0" ONLY (integer)
        /// </summary>
        public int detector_panel = -1;
        /// <summary>
        /// Event type. default = True counts
        /// </summary>
        public Event_type event_type =  Event_type.Trues;
        /// <summary>
        /// Energy window - NOTE PET/CT systems are typically "0" ONLY 
        /// </summary>
        public int energy_window = -1;
        /// <summary>
        /// Gate number
        /// </summary>
        public int gate = 0;
        /// <summary>
        /// Bed number
        /// </summary>
        public int bed = 0;
        /// <summary>
        /// Bed offset, in cm
        /// </summary>
        public float bed_offset = 0.0f;
        /// <summary>
        /// Ending horizontal bed offset, in cm 
        /// </summary>
        public float ending_bed_offset = 0.0f;
        /// <summary>
        /// Number of bed passes during frame
        /// </summary>
        public int bed_passes;
        /// <summary>
        /// Vertical bed offset, in cm 
        /// </summary>
        public float vertical_bed_offset = 0.0f;
        /// <summary>
        /// Data file offset to start of tacs ([low_part 0] or [high_part low_part]) 
        /// </summary>
        public int[] data_file_pointer = new int[2];
        /// <summary>
        /// Frame start time, in secs 
        /// </summary>
        public float frame_start = 0;
        /// <summary>
        /// Frame duration, in secs 
        /// </summary>
        public float frame_duration = 0;
        /// <summary>
        /// Scale factor for tacs set 
        /// </summary>
        public float scale_factor = 0.0f;
        /// <summary>
        /// Minimum value in tacs set 
        /// </summary>
        public float minimum = 0.0f;
        /// <summary>
        /// Maximum value in tacs set 
        /// </summary>
        public float maximum = 0.0f;
        /// <summary>
        /// Deadtime correction for tacs set. Used for PET or SPECT modalities.
        ///             NOTE: Scale factor INCLUDES this value.
        /// </summary>
        public float deadtime_correction = 0.0f;
        /// <summary>
        /// Global decay correction applied to tacs set. Used for PET or SPECT modalities.
        ///             NOTE: Scale factor INCLUDES this value.
        /// </summary>
        public float decay_correction = 0.0f;
        /// <summary>
        /// Prompts count for tacs set. Used for PET or SPECT modalities.
        ///   NOTE: The 3 values represent histogrammed prompts,
        ///         unhistogrammed prompts, and prompts at coincidence detection. 
        /// </summary>
        public long[] prompts = new long[3];
        /// <summary>
        /// Prompts countrate per sec before histogramming. Used for PET or SPECT modalities.
        /// </summary>
        public int prompts_rate = 0;
        /// <summary>
        /// Delays count for tacs set. Used for PET modality only.
        ///   NOTE: The 3 values represent histogrammed delays,
        ///         unhistogrammed delays, and delays at coincidence detection.
        /// </summary>
        public long[] delays = new long[3];
        /// <summary>
        /// Trues count for tacs set. Used for PET modality only.
        ///   NOTE: The 3 values represent histogrammed delays,
        ///         unhistogrammed delays, and delays at coincidence detection. 
        /// </summary>
        public long[] trues = new long[3];
        /// <summary>
        /// Delays countrate per sec before histogramming. Used for PET modality only.
        /// </summary>
        public int delays_rate = 0;
        /// <summary>
        /// Singles rates
        /// </summary>
        public Singles_rate[] singles;
        #endregion
    }
}
