/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// NiftiHeader class
	/// </summary>
	public class NiftiHeader : AnalyzeHeader
	{
		#region Nifti header field description

		private static readonly string[] niftiNames =
        {   "sizeof_hdr", "data_type", "db_name", "extents", "session_error", "regular", "dim_info", "dim", "intent_p1", "intent_p2",
            "intent_p3", "intent_code", "datatype", "bitpix", "slice_start", "pixdim", "vox_offset", "scl_slope", "scl_inter", "slice_end",
            "slice_code", "xyzt_units", "cal_max", "cal_min", "slice_duration", "toffset", "glmax", "glmin", "descrip", "aux_file",
            "qform_code", "sform_code", "quatern_b", "quatern_c", "quatern_d", "qoffset_x", "qoffset_y", "qoffset_z", "srow_x", "srow_y",
            "srow_z", "intent_name", "magic" };

		private static readonly HeaderFieldType[] niftiTypes =
        {   HeaderFieldType.INT, HeaderFieldType.STRING, HeaderFieldType.STRING, HeaderFieldType.INT, HeaderFieldType.SHORT,
            HeaderFieldType.CHAR, HeaderFieldType.CHAR, HeaderFieldType.SHORT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT,
            HeaderFieldType.FLOAT, HeaderFieldType.SHORT, HeaderFieldType.SHORT, HeaderFieldType.SHORT, HeaderFieldType.SHORT,
            HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.SHORT,
            HeaderFieldType.CHAR, HeaderFieldType.CHAR, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT,
            HeaderFieldType.FLOAT, HeaderFieldType.INT, HeaderFieldType.INT, HeaderFieldType.STRING, HeaderFieldType.STRING,
            HeaderFieldType.SHORT, HeaderFieldType.SHORT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT,
            HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT, HeaderFieldType.FLOAT,
            HeaderFieldType.FLOAT, HeaderFieldType.STRING, HeaderFieldType.STRING };

		private static readonly int[] niftiLengths =
        {   1, 10, 18, 1, 1, 1, 1, 8, 1, 1,
            1, 1, 1, 1, 1, 8, 1, 1, 1, 1,
            1, 1, 1, 1, 1, 1, 1, 1, 80, 24,
            1, 1, 1, 1, 1, 1, 1, 1, 4, 4,
            4, 16, 4 };

		#endregion

		/// <summary>
		/// Extension block size; extension header length must be
		/// a multiple of EXTENSIONBLOCK
		/// </summary>
		private const int EXTENSIONSIZE = 16;

		/// <summary>
		/// Size of the extension field at the end of the header
		/// Not part of the header, as it may or may not be present
		/// in two part nifti files
		/// </summary>
		private const int EXTENSIONFIELDSIZE = 4;

		/// <summary>
		/// Magic string at the end of header in combined files
		/// </summary>
		private const string COMBINEDMAGIC = @"n+1";

		/// <summary>
		/// Magic string at the end of header data in the two file format
		/// </summary>
		private const string SEPARATEMAGIC = @"ni1";

		/// <summary>
		/// 
		/// </summary>
		public class NiftiExtensionHeader
		{
			/// <summary>
			/// 
			/// </summary>
			public int size;
			
			/// <summary>
			/// 
			/// </summary>
			public int code;
			
			/// <summary>
			/// 
			/// </summary>
			public byte[] data;

			/// <summary>
			/// 
			/// </summary>
			/// <param name="headerSize"></param>
			public NiftiExtensionHeader(int headerSize)
			{
				if (headerSize % EXTENSIONSIZE != 0) throw new TPCException("Nifti extension header size must be multiple of 16 bytes.");
				size = headerSize;
				data = new byte[size - 2 * sizeof(Int32)];
			}
		}

		/// <summary>
		/// 
		/// </summary>
		private NiftiExtensionHeader[] extensions;

		private void Init()
		{
			extensions = new NiftiExtensionHeader[0];

			// Default to a combined file
			this[@"magic"] = COMBINEDMAGIC;
			this[@"vox_offset"] = 352;

			// Set default coordinate system
			// See nifti.h for details.
			this[@"qform_code"] = 1;
			this[@"sform_code"] = 0;
			this[@"quatern_b"] = 0;
			this[@"quatern_c"] = 0;
			this[@"quatern_d"] = 0;

			Orientation = Orientation.RAS;
		}

		/// <summary>
		/// Default constructor.
		/// </summary>
		public NiftiHeader() : base() { Init(); }

		/// <summary>
		/// Copy constructor.
		/// </summary>
		/// <param name="ih">Image header to copy</param>
		public NiftiHeader(ImageHeader ih) : base(ih) { Init(); }

		/// <summary>
		/// Create a header from file
		/// </summary>
		/// <param name="filename">File to read</param>
		public NiftiHeader(string filename) : this() { Read(filename); }

		/// <summary>
		/// Read header info from a file
		/// </summary>
		/// <param name="filename">File to read</param>
		/// <returns>True, if reading was succesfull</returns>
		public bool Read(string filename)
		{
			bool success = false;

			using (FileStream s = new FileStream(filename, FileMode.Open))
			{
				success = Read(s);
			}

			return success;
		}

		/// <summary>
		/// Read header info from a stream
		/// </summary>
		/// <param name="s">Stream to read</param>
		/// <returns>True, if reading was succesful</returns>
		public override bool Read(Stream s)
		{
			// Clear old contents
			this.Clear();

			bool success = false;

			Endianness endian = NiftiFile.CheckEndian(s);
			this.bigEndian = (endian == Endianness.BigEndian);
			BinaryReader br = EndianReader.CreateReader(s, endian);

			long bytesRead = HeaderConstructor.ReadHeader(new HeaderDescription(niftiNames, niftiTypes, niftiLengths), br, this);

			string magic = (string)this[@"magic"];

			// If header is valid, read extension headers; header length should be 348, this value should be recorded in the 'sizeof_hdr',
			// and the 'magic' field should have one of the two valid values.
			if (bytesRead == HEADERSIZE &&
				(int)this[@"sizeof_hdr"] == HEADERSIZE &&
				(magic == COMBINEDMAGIC || magic == SEPARATEMAGIC))
			{
				// Try to read the extension bytes.
				// If 4 more bytes are succesfully read and first byte > 0, there are extension headers.
				byte[] extBytes = new byte[4];
				if (br.Read(extBytes, 0, 4) == 4 && extBytes[0] != 0)
				{
					System.Collections.Generic.List<NiftiExtensionHeader> extensionList = new System.Collections.Generic.List<NiftiExtensionHeader>();
					NiftiExtensionHeader extHeader;
					while (ReadExtensionHeader(s, out extHeader))
					{
						extensionList.Add(extHeader);
					}
					// Add current stream position -> total bytes read
					extensions = extensionList.ToArray();
				}
				success = true;
			}
			return success;
		}

		/// <summary>
		/// Write this header to stream
		/// </summary>
		/// <param name="s">Output stream</param>
		/// <returns>True, if a valid header was written</returns>
		public override bool Write(Stream s)
		{
			bool success = false;

			using (BinaryWriter bw = new BinaryWriter(s))
			{
				// Write the main header
				long bytesWritten = HeaderConstructor.WriteHeader(new HeaderDescription(niftiNames, niftiTypes, niftiLengths), bw, this);

				if (bytesWritten == HEADERSIZE)
				{
					// Write extension headers
					if (extensions.Length > 0)
					{
						// Extensions exist, write non-zero first extension byte
						bw.Write(new byte[] { 1, 0, 0, 0 });
						foreach (NiftiExtensionHeader ext in extensions)
						{
							bw.Write(ext.size);
							bw.Write(ext.code);
							bw.Write(ext.data);
						}
					}
					else
					{
						// No extensions, all extension bytes zero
						bw.Write(new byte[] { 0, 0, 0, 0 });
					}
				}
				success = true;
			}
			return success;
		}

		/// <summary>
		/// Size of this header in bytes
		/// </summary>
		public override int HeaderSize
		{
			get
			{
				int extensionSize = 0;
				foreach (NiftiExtensionHeader neh in extensions)
				{
					extensionSize += neh.size;
				}
				// Size of the entire header is
				// header size + four extension bytes + possible extended headers
				return base.HeaderSize + EXTENSIONFIELDSIZE + extensionSize;
			}
		}

		/// <summary>
		/// Orientation of the image
		/// </summary>
		public override Orientation Orientation
		{
			get
			{
				// Calculate the orientation from the Nifti quaternions (see nifti.h)
				// Rotations not supported, calculate only coordinate flips
				int qform = Convert.ToInt32(this[@"qform_code"]);
				if (qform != 0)
				{
					int qb = Convert.ToInt32(this[@"quatern_b"]);
					int qc = Convert.ToInt32(this[@"quatern_c"]);
					int qd = Convert.ToInt32(this[@"quatern_d"]);

					int qb2 = qb * qb;
					int qc2 = qc * qc;
					int qd2 = qd * qd;
					int a2 = 1 - qb2 - qc2 - qd2;
					// Quaternion a should be >= 0, if not so, return the default orientation
					if (a2 < 0 || a2 > 1) return Orientation.RAS;

					bool l = (a2 + qb2 - qc2 - qd2) < 0;
					bool p = (a2 + qc2 - qb2 - qd2) < 0;
					bool i = (a2 + qd2 - qc2 - qb2) * qform < 0;

					return new Orientation(l, p, i);
				}

				// If qform not set, assume Analyze 7.5 default
				else return Orientation.LAS;
			}
			set
			{
				// Calculate the Nifti quaternions (see nifti.h)
				int qb = !value.RightToLeft && value.AnteriorToPosterior ? 0 : 1;
				int qc = !value.AnteriorToPosterior && value.RightToLeft ? 0 : 1;
				int qd = value.RightToLeft && value.AnteriorToPosterior ? 0 : 1;

				// Set the qform code to -1, if there are 1 or three flips
				int sum = qb + qc + qd;
				int qform = (sum % 2 == 0) ? 1 : -1;

				// Store values to the header
				this[@"qform_code"] = qform;
				this[@"quatern_b"] = qb;
				this[@"quatern_c"] = qc;
				this[@"quatern_d"] = qd;
			}
		}

		/// <summary>
		/// Description of the header bytes
		/// </summary>
		protected override AnalyzeHeader.HeaderDescription ByteDescription
		{
			get { return new HeaderDescription(niftiNames, niftiTypes, niftiLengths); }
		}

		/// <summary>
		/// Read one extension header from a stream.
		/// </summary>
		/// <param name="s">Stream to read.</param>
		/// <param name="extHeader">Header to read into. Value may be null, if this function return false.</param>
		/// <returns>Returns true iff header was succesfully read.</returns>
		private bool ReadExtensionHeader(Stream s, out NiftiExtensionHeader extHeader)
		{
			byte[] buffer = new byte[sizeof(Int32)];

			bool success = false;
			extHeader = null;

			// Read header size into buffer
			if (s.Read(buffer, 0, buffer.Length) == buffer.Length)
			{
				int headerSize = System.BitConverter.ToInt32(buffer, 0);
				// Extended header size must be multiple of 16 bytes
				if (headerSize % EXTENSIONSIZE == 0)
				{
					extHeader = new NiftiExtensionHeader(headerSize);
					// Read extension code and header data
					if (s.Read(buffer, 0, buffer.Length) == buffer.Length &&
						s.Read(extHeader.data, 0, extHeader.data.Length) == extHeader.data.Length)
					{
						extHeader.code = BitConverter.ToInt32(buffer, 0);
						// Everything done, set return status to true
						success = true;
					}
				}
			}
			return success;
		}
	}

	/// <summary>
	/// NiftiFile class
	/// </summary>
	public class NiftiFile : AnalyzeFile
	{
		// Delegates for IO operations
		private delegate float GetValue();
		private delegate void WriteValue(float f);

		/// <summary>
		/// Nifti file extension for combined files
		/// </summary>
		public const string COMBINEDEXTENSION = @".nii";

		/// <summary>
		/// Default constructor
		/// </summary>
		public NiftiFile()
		{
			this.imgHeader = new NiftiHeader();
			this.filetype = FileType.Nifti;
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fn">File name</param>
		public NiftiFile(string fn)
			: this()
		{
			// Use combined file, if not defined by file extension
			string ext = Path.GetExtension(fn);
			if (ext == COMBINEDEXTENSION || ext == HEADEREXTENSION || ext == DATAEXTENSION)
			{
				this.filename = fn;
			}
			else
			{
				this.filename = fn + COMBINEDEXTENSION;
			}
		}

		public NiftiHeader NiftiHeader { get { return Header as NiftiHeader; } }

		public override ImageHeader Header
		{
			get
			{
				return imgHeader;
			}
			set
			{
				imgHeader = new NiftiHeader(value);
			}
		}

		/// <summary>
		/// Read header from this file
		/// </summary>
		/// <returns>Header of this file</returns>
		protected override AnalyzeHeader ReadAnalyzeHeader()
		{
			return new NiftiHeader(ResolveHeaderFilename(this.filename));
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fn">File name</param>
		/// <param name="img">Image</param>
		/// <param name="hdr">Image header</param>
		public NiftiFile(string fn, Image img, ImageHeader hdr)
		{
			// Use combined file, if not defined by file extension
			string ext = Path.GetExtension(fn);
			if (ext == COMBINEDEXTENSION || ext == HEADEREXTENSION || ext == DATAEXTENSION)
			{
				this.filename = fn;
			}
			else
			{
				this.filename = fn + COMBINEDEXTENSION;
			}
			this.imgHeader = new NiftiHeader(hdr);
			this.image = img;
			this.imgHeader.Dim = img.Dim;
			this.filetype = FileType.Nifti;
		}

		/// <summary>
		/// Name of the header file
		/// </summary>
		public override string HeaderFilename
		{
			get
			{
				string ext = Path.GetExtension(this.filename);

				if (ext == COMBINEDEXTENSION)
				{
					return this.filename;
				}
				else
				{
					return base.HeaderFilename;
				}
			}
		}

		/// <summary>
		/// Name of the image data file
		/// </summary>
		public override string DataFilename
		{
			get
			{
				string ext = Path.GetExtension(this.filename);

				if (ext == COMBINEDEXTENSION)
				{
					return this.filename;
				}
				else
				{
					return base.DataFilename;
				}
			}
		}

		/// <summary>
		/// Check Nifti format validness of a file
		/// </summary>
		/// <param name="fn">File to check</param>
		/// <returns>True, if file is in Nifti format</returns>
		public static bool CheckFormat(string fn)
		{
			string filename = ResolveHeaderFilename(fn);

			if (File.Exists(filename))
			{
				using (FileStream fs = new FileStream(filename, FileMode.Open))
				{
					return CheckFormat(fs);
				}
			}
			else return false;
		}

		/// <summary>
		/// Check Nifti format validness of a stream
		/// </summary>
		/// <param name="s">Stream to check</param>
		/// <returns>True, if stream data is in Nifti format</returns>
		public static bool CheckFormat(Stream s)
		{
			NiftiHeader head = new NiftiHeader();
			return head.Read(s);
		}

		/// <summary>
		/// Resolve the header file name
		/// </summary>
		/// <returns>Full name of the header file</returns>
		private static string ResolveHeaderFilename(string fn)
		{
			string ext = Path.GetExtension(fn);

			if (ext == COMBINEDEXTENSION)
			{
				return fn;
			}
			else
			{
				string dir = Path.GetDirectoryName(fn);
				string file = Path.GetFileNameWithoutExtension(fn);

				return (dir + Path.DirectorySeparatorChar + file + HEADEREXTENSION);
			}
		}

		/// <summary>
		/// Resolve the data file name
		/// </summary>
		/// <returns>Full name of the data file</returns>
		private static string ResolveDataFilename(string fn)
		{
			string ext = Path.GetExtension(fn);

			if (ext == COMBINEDEXTENSION)
			{
				return fn;
			}
			else
			{
				string dir = Path.GetDirectoryName(fn);
				string file = Path.GetFileNameWithoutExtension(fn);
				return (dir + Path.DirectorySeparatorChar + file + DATAEXTENSION);
			}
		}

		/// <summary>
		/// Calculate rotation matrix. Handles only combinations of 90 degree rotations around the main axis.
		/// </summary>
		/// <param name="b"></param>
		/// <param name="c"></param>
		/// <param name="d"></param>
		/// <param name="q"></param>
		/// <returns></returns>
		private int[] CalculateRotation(float b, float c, float d, float q)
		{
			int[] r;
			if (b > 0) r = new int[] { 1, -1, -1 };
			else if (c > 0) r = new int[] { -1, 1, -1 };
			else if (d > 0) r = new int[] { -1, -1, 1 };
			else r = new int[] { 1, 1, 1 };

			if (q < 0) r[2] *= -1;

			return r;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="r"></param>
		/// <param name="i"></param>
		/// <param name="j"></param>
		/// <param name="k"></param>
		/// <returns></returns>
		private int CalculateIndex(int[] r, int i, int j, int k)
		{
			int x, y, z;
			x = ((imgHeader.DimX - 1) * (1 - r[0]) + 2 * r[0] * i) / 2;
			y = ((imgHeader.DimY - 1) * (1 - r[1]) + 2 * r[1] * j) / 2;
			z = ((imgHeader.DimZ - 1) * (1 - r[2]) + 2 * r[2] * k) / 2;
			return x + y * imgHeader.DimX + z * imgHeader.DimX * imgHeader.DimY;
		}
	}
}
