/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Image
{
    /// <summary>
    /// Class for reading/writing .sif files with timeframe data.
    /// </summary>
	public class SifFile : TPCFile
    {
		private float[] starts;
		private float[] ends;
		private float[] prompts;
		private float[] randoms;

		/// <summary>
		/// Read/Write progress event
		/// </summary>
		public static event IOProcessEventHandler IOProgress;

		/// <summary>
		/// 
		/// </summary>
		public const string Extension = @".sif";

		/// <summary>
        /// Class for handling sif header lines.
        /// </summary>
        public class SifHeader
        {
			/// <summary>
			/// 
			/// </summary>
            public DateTime start = new DateTime();
            
			/// <summary>
			/// 
			/// </summary>
			public int frames = 0;
            
			/// <summary>
			/// 
			/// </summary>
			public int columns = 0;
            
			/// <summary>
			/// 
			/// </summary>
			public int version = 1;
            
			/// <summary>
			/// 
			/// </summary>
			public string studyID = string.Empty;
            
			/// <summary>
			/// 
			/// </summary>
			public string itope = string.Empty;

			/// <summary>
			/// Create an empty header
			/// </summary>
            public SifHeader() {}

			/// <summary>
			/// Parse a new header from a string
			/// </summary>
			/// <param name="head">Header string</param>
            public SifHeader(string head)
            {
				// Split header string to component fields. Exactly 7 fields expected.
                string[] fields = head.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
                if (fields.Length < 5) throw new TPCException("Invalid header: missing a required field");

				// Combine first two to date+time string
				string datestring = fields[0] + @" " + fields[1];

				// Try to parse date+time
				bool dateValid = DateTime.TryParseExact(
					datestring,
					new string[] { "dd/MM/yyyy HH:mm:ss", "d/M/yyyy HH:mm:ss", "MM/dd/yyyy HH:mm:ss", "M/d/yyyy HH:mm:ss" },
					System.Globalization.DateTimeFormatInfo.InvariantInfo,
					System.Globalization.DateTimeStyles.None,
					out start);

				if (!dateValid) start = new DateTime();

				// Try to parse the header
				if (int.TryParse(fields[2], out frames) &&
					int.TryParse(fields[3], out columns) &&
					int.TryParse(fields[4], out version))
				{
					// Study id and isotope can be anything
					if (fields.Length > 5) studyID = fields[5];
					if (fields.Length > 6) itope = fields[6];
				}
				// If parsing fails, throw an exception
				else throw new TPCException("Invalid header: invalid format");
            }

			/// <summary>
			/// Create a new header
			/// </summary>
			/// <param name="s">Date and time</param>
			/// <param name="f">Frames</param>
			/// <param name="c">Columns (should be 4)</param>
			/// <param name="v">Version (should be 1)</param>
			/// <param name="id">Research ID</param>
			/// <param name="iso">Isotope name</param>
            public SifHeader(DateTime s, int f, int c, int v, string id, string iso)
            {
                start = s;
                frames = f;
                columns = c;
                version = v;
                studyID = id;
                itope = iso;
            }

			/// <summary>
			/// String representation of the header
			/// </summary>
			/// <returns>Header string</returns>
            public override string ToString()
            {
                System.Text.StringBuilder sb = new System.Text.StringBuilder();
				// Date in standard us english format, fields separated by empty space
				sb.Append(start.ToString("dd/MM/yyyy HH:mm:ss", System.Globalization.DateTimeFormatInfo.InvariantInfo));
                sb.Append(' ');
                sb.Append(frames);
                sb.Append(' ');
                sb.Append(columns);
                sb.Append(' ');
                sb.Append(version);
                sb.Append(' ');
                sb.Append(studyID);
                sb.Append(' ');
                sb.Append(itope);

                return sb.ToString();
            }
        };

		/// <summary>
		/// 
		/// </summary>
        public SifHeader header;

		/// <summary>
		/// 
		/// </summary>
		public float[] StartTimes
		{
			get { return starts; }
			set { starts = value; }
		}
        
		/// <summary>
		/// 
		/// </summary>
		public float[] EndTimes
		{
			get { return ends; }
			set { ends = value; }
		}
        
		/// <summary>
		/// 
		/// </summary>
		public float[] Prompts
		{
			get { return prompts; }
			set { prompts = value; }
		}

        /// <summary>
		/// 
		/// </summary>
		public float[] Randoms
		{
			get { return randoms; }
			set { randoms = value; }
		}
		
		/// <summary>
		/// 
		/// </summary>
		public float[] Durations
		{
			get
			{
				float[] dur = new float[starts.Length];
				for (int i = 0; i < dur.Length; i++)
				{
					dur[i] = ends[i] - starts[i];
				}
				return dur;
			}
		}

        /// <summary>
        /// Create empty file
        /// </summary>
		public SifFile(string file = @"", int n = 0)
		{
			header = new SifHeader();
			filename = file;
			ends = new float[n];
			starts = new float[n];
			randoms = new float[n];
			prompts = new float[n];
		}

		/// <summary>
		/// Read sif file data from an image file
		/// </summary>
		/// <param name="imFile">Dynamic image file</param>
		public SifFile(ImageFile imFile)
			: this(imFile.image, imFile.Header)
		{
			string baseName = System.IO.Path.GetFileNameWithoutExtension(imFile.filename);
			string path = System.IO.Path.GetDirectoryName(imFile.filename) + System.IO.Path.DirectorySeparatorChar;
			this.filename = path + baseName + SifFile.Extension;
		}

		/// <summary>
		/// Read sif data from an image header and an image
		/// </summary>
		/// <param name="img">Image</param>
		/// <param name="head">Image header</param>
		public SifFile(Image img, ImageHeader head)
		{
			this.filename = @"";
			CreateHeaders(head);
			CalculateCounts(img);
		}

        /// <summary>
        /// Read from file
        /// </summary>
        /// <param name="file">Filename</param>
        public void ReadFile(string file)
        {
            filename = file;
            ReadFile();
        }

        /// <summary>
        /// Write to file
        /// </summary>
        /// <param name="file">Filename</param>
        public void WriteFile(string file)
        {
            filename = file;
            WriteFile();
        }

        /// <summary>
        /// Read from file
        /// </summary>
        public override void ReadFile()
        {
			// Read the file
            string[] lines = System.IO.File.ReadAllLines(filename);
			// Remove comment lines
			lines = Array.FindAll<string>(lines, delegate(string s) { return !s.Trim().StartsWith(@"#"); });
			// Parse the header from the first line
			if (lines.Length < 1) throw new TPCException("Empty file");
			header = new SifHeader(lines[0]);
            if (header.columns != 4 || header.version != 1) throw new TPCException("Unknown file format");

			// Progress event handler
			IOProcessEventHandler pevent = null;

			// Read data line by line
			System.Collections.Generic.List<float> startList = new System.Collections.Generic.List<float>();
			System.Collections.Generic.List<float> endList = new System.Collections.Generic.List<float>();
			System.Collections.Generic.List<float> randList = new System.Collections.Generic.List<float>();
			System.Collections.Generic.List<float> promptList = new System.Collections.Generic.List<float>();
			float start, end, rand, prompt;

            int linecount = 0;
			// Start parsing from line 1
			for (int i = 1; i < lines.Length; i++)
            {
				// Try to parse data from the line, invalid lines are skipped
                string[] data = lines[i].Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
				if (float.TryParse(data[0], out start) && float.TryParse(data[1], out end) &&
					float.TryParse(data[2], out prompt) && float.TryParse(data[3], out rand))
                {
					// If frame start is later than frame end, throw and exception
                    if (start > end) throw new TPCException("Invalid data");

					// Add data to lists
					// Convert times to milliseconds
					startList.Add(start * 1000.0f);
					endList.Add(end * 1000.0f);
					randList.Add(rand);
					promptList.Add(prompt);
                    linecount++;
					
					// Read event after each line
					pevent = IOProgress;
					if (pevent != null)
						pevent.Invoke(this, new IOProgressEventArgs(100.0f * i / lines.Length, System.Reflection.MethodBase.GetCurrentMethod()));
				}
            }
			// If header frame count and the number of lines do not match, throw an exception
            if (header.frames != linecount) throw new TPCException("Invalid header");
			starts = startList.ToArray();
			ends = endList.ToArray();
			prompts = promptList.ToArray();
			randoms = randList.ToArray();
        }

        /// <summary>
        /// Write to file
        /// </summary>
        public override void WriteFile()
        {
			// Progress event handler
			IOProcessEventHandler pevent = IOProgress;

			// Write to file
			System.IO.File.WriteAllText(filename, this.ToString(), System.Text.Encoding.ASCII);

			if (pevent != null)
				pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
		}

		/// <summary>
		/// Get sif file content in a string
		/// </summary>
		/// <returns>String representation of the file</returns>
		public override string ToString()
		{
			System.Text.StringBuilder sb = new System.Text.StringBuilder();

			// Add header
			sb.AppendLine(header.ToString());

			// Add data, line by line
			float s, e, p, r;
			int maxLength = starts.Length < ends.Length ? ends.Length : starts.Length;
			maxLength = maxLength < prompts.Length ? prompts.Length : maxLength;
			maxLength = maxLength < randoms.Length ? randoms.Length : maxLength;
			for (int i = 0; i < maxLength; i++)
			{
				s = i < starts.Length ? starts[i] : 0.0f;
				e = i < ends.Length ? ends[i] : 0.0f;
				p = i < prompts.Length ? prompts[i] : 0.0f;
				r = i < randoms.Length ? randoms[i] : 0.0f;

				// Construct data line
				// Convert mseconds to seconds
				sb.AppendFormat(
					"{0} {1} {2} {3}",
					Convert.ToInt64(s * 0.001),
					Convert.ToInt64(e * 0.001),
					Convert.ToInt64(p),
					Convert.ToInt64(r));
				sb.AppendLine();
			}
			return sb.ToString();
		}

		/// <summary>
		/// Read sif from an Ecat7 file. If file has raw scan data, sif is read from the headers.
		/// If file has image data, sif data is constructed from the image.
		/// </summary>
		/// <param name="ecatFile">Filename</param>
		/// <returns>Sif file contents</returns>
		public static SifFile ReadEcat(string ecatFile)
		{
			Ecat7File ecat = Ecat7File.ReadFile(ecatFile) as Ecat7File;
			return ReadEcat(ecat);
		}

		/// <summary>
		/// Read sif from an Ecat7 file. If file has raw scan data, sif is read from the headers.
		/// If file has image data, sif data is constructed from the image.
		/// </summary>
		/// <param name="ecatFile">Ecat7 object</param>
		/// <returns>SifFile contents</returns>
		public static SifFile ReadEcat(Ecat7File ecatFile)
		{
			Ecat7Header mainHead = ecatFile.EcatHeader;
			SifFile sFile = new SifFile();

			// Check if this is a scan file
			if (mainHead.file_type == Ecat7Header.datatype_e.ecat7_3DSCAN ||
				mainHead.file_type == Ecat7Header.datatype_e.ecat7_3DSCAN8 ||
				mainHead.file_type == Ecat7Header.datatype_e.ecat7_3DSCANFIT)
			{
				Ecat7File.Ecat7_scanheader[] scanHeads = ecatFile.subheaders as Ecat7File.Ecat7_scanheader[];
				sFile.CreateAll(mainHead, scanHeads);
			}
			// Not a scan file, so it's an image file
			else
			{
				Ecat7File.Ecat7_imageheader[] iHeads = ecatFile.subheaders as Ecat7File.Ecat7_imageheader[];
				sFile.CreateHeaders(mainHead);
				sFile.CalculateCounts(ecatFile.image);
				// Remove decay correction, if needed
				for (int i = 0; i < sFile.prompts.Length; i++)
				{
					if (iHeads[i].decay_corr_fctr > 1.0)
					{
						sFile.prompts[i] /= iHeads[i].decay_corr_fctr;
					}
				}
			}
			// Ensure that framecount matches
			sFile.header.frames = sFile.prompts.Length;

			return sFile;
		}

		/// <summary>
		/// Read sif from dynamic file. Sif data is constructed from the image.
		/// </summary>
		/// <param name="fn">Filename</param>
		/// <returns>Sif file contents</returns>
		public static SifFile ReadDynamic(string fn)
		{
			ImageFile imgf = ImageFile.ReadFile(fn);
			return ReadDynamic(imgf);
		}

		/// <summary>
		/// Read sif from dynamic file. Sif data is constructed from the image.
		/// </summary>
		/// <param name="imgf">Dynamic image file object</param>
		/// <returns>Sif file contents</returns>
		public static SifFile ReadDynamic(ImageFile imgf)
		{
			SifFile sFile = new SifFile();
			sFile.CreateHeaders(imgf.Header);
			sFile.CalculateCounts(imgf.image);
			return sFile;
		}

		/// <summary>
		/// Calculate counts from Image data
		/// </summary>
		/// <param name="img"></param>
		private void CalculateCounts(Image img)
		{
			int frames = img.Frames;

			starts = new float[frames];
			ends = new float[frames];
			randoms = new float[frames];
			prompts = new float[frames];

			for (int i = 0; i < frames; i++)
			{
				// Set frame duration to at least 1
				float fDur = img.GetFrameDuration(i);
				fDur = fDur < 1.0f ? 1.0f : fDur;
				// Set prompts to average true pixel values multiplied
				// with frameduration (mseconds) and scale coefficient 100.
				prompts[i] = img.GetFrame(i).GetMean() * fDur * 100.0f;

				// Times in mseconds.
				starts[i] = img.GetFrameStartTime(i);
				ends[i] = img.GetFrameEndTime(i);
				// Set all randoms to 0
				randoms[i] = 0.0f;
			}
		}

		/// <summary>
		/// Create header from image headers.
		/// </summary>
		/// <param name="head">Image header</param>
		private void CreateHeaders(ImageHeader head)
		{
			header = new SifHeader();
			header.version = 1;
			header.columns = 4;
			header.studyID = head.PatientID;
			header.frames = head.Frames;
			header.start = head.DoseStartTime;
			header.itope = head.Isotope.ToString();
		}

		/// <summary>
		/// Create header from Ecat image headers
		/// </summary>
		/// <param name="mainHead"></param>
		private void CreateHeaders(Ecat7Header mainHead)
		{
			header = new SifHeader();
			header.version = 1;
			header.columns = 4;
			header.frames = mainHead.num_frames;
			header.start = Ecat7File.EcatToDateTime(mainHead.scan_start_time);
			header.itope = mainHead.Isotope.ToString();
			header.studyID = new string(mainHead.study_type).Trim('\0');
		}

		/// <summary>
		/// Create header from Ecat scan headers.
		/// </summary>
		/// <param name="mainHead">Ecat main header</param>
		/// <param name="scanHeads">Ecat scan headers</param>
		private void CreateAll(Ecat7Header mainHead, Ecat7File.Ecat7_scanheader[] scanHeads)
		{
			CreateHeaders(mainHead);
			header.frames = scanHeads.Length;

			Ecat7File.Ecat7_scanheader scanHead;
			starts = new float[header.frames];
			ends = new float[header.frames];
			randoms = new float[header.frames];
			prompts = new float[header.frames];

			for (int i = 0; i < header.frames; i++)
			{
				scanHead = scanHeads[i];
				// Times in ms, convert to s
				starts[i] = scanHead.frame_start_time;
				ends[i] = scanHead.frame_start_time + scanHead.frame_duration;
				prompts[i] = scanHead.prompts;
				randoms[i] = scanHead.delayed;

				if (prompts[i] <= 0) throw new TPCException("Invalid count statistics");
			}
		}
	}
}
