/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.Image
{
    /// <summary>
    /// Generals utility dataFunctions for Image objects.
    /// </summary>
    public sealed class ImageUtils
    {
        /// <summary>
        /// Coordinates of surrounding 3x3x3 coordinates 
        /// </summary>
        public static readonly Point[] surroundings = new Point[26]{
                                                              new Point(-1, 1, 0), new Point( 1, 0, 0), new Point(-1,-1, 0), 
                                                              new Point( 0, 1, 0), new Point( 1,-1, 0), new Point(-1, 0, 0), 
                                                              new Point( 1, 1, 0), new Point( 0,-1, 0), //              ( 0, 0, 0)
                                                              new Point( 0,-1,-1), new Point( 0, 0,-1), new Point( 0, 1,-1), 
                                                              new Point(-1,-1,-1), new Point(-1, 0,-1), new Point(-1, 1,-1), 
                                                              new Point( 1,-1,-1), new Point( 1, 0,-1), new Point( 1, 1,-1), 

                                                              new Point( 0,-1, 1), new Point( 0, 0, 1), new Point( 0, 1, 1), 
                                                              new Point(-1,-1, 1), new Point(-1, 0, 1), new Point(-1, 1, 1), 
                                                              new Point( 1,-1, 1), new Point( 1, 0, 1), new Point( 1, 1, 1)
                                                          };
        /// <summary>
        /// Event handler for utility progress notifications.
        /// </summary>
        /// <param name="sender">object that sent the event</param>
        /// <param name="e">event arguments</param>
        public delegate void ProgressEventHandler(object sender, ProgressEventArgs e);
        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public static event ProgressEventHandler ProgressEvent;
        /// <summary>
        /// Gaussian window maximum length in voxels
        /// </summary>
        public static int gaussianwindowLength = 65;

		/// <summary>
        /// Implements erosion operation.
        /// </summary>
        /// <param name="image">source image</param>
        /// <param name="targetlocation">location in target image</param>
        /// <param name="kernel">spatial 3D-kernel (for values -kernel..kernel etc.)</param>
        /// <param name="parameters">ignored</param>
        /// <returns>original value or 0</returns>
        public static float SpatialErosionOperation(ref Image image, IntPoint targetlocation, int kernel, float[] parameters)
        {
            //return dilation value if value is found n kernel neighbourhood
            for (int z = targetlocation.Z - kernel; z <= targetlocation.Z + kernel; z++)
            {
                for (int y = targetlocation.Y - kernel; y <= targetlocation.Y + kernel; y++)
                {
                    for (int x = targetlocation.X - kernel; x <= targetlocation.X + kernel; x++)
                    {
                        if (image[x, y, z] == 0) return 0.0f;
                    }
                }
            }
            return image[targetlocation];
        }
        /// <summary>
        /// Implements dilation operation.
        /// </summary>
        /// <param name="image">source image</param>
        /// <param name="targetlocation">location in target image</param>
        /// <param name="kernel">spatial 3D-kernel (for values -kernel..kernel etc.)</param>
        /// <param name="dilation">dilation value at index 0</param>
        /// <returns>dilation value or 0</returns>
        public static float SpatialDilationOperation(ref Image image, IntPoint targetlocation, int kernel, float[] dilation)
        {
            //return dilation value if value is found n kernel neighbourhood
            for (int z = targetlocation.Z - kernel; z <= targetlocation.Z + kernel; z++)
            {
                for (int y = targetlocation.Y - kernel; y <= targetlocation.Y + kernel; y++)
                {
                    for (int x = targetlocation.X - kernel; x <= targetlocation.X + kernel; x++)
                    {
                        if (image[x, y, z] != 0) return dilation[0];
                    }
                }
            }
            return 0.0f;
        }
        /// <summary>
        /// Implements image value inversion operation. Note that number of parameters 
        /// is not checked for performance reasons.
        /// </summary>
        /// <param name="value">image value that is inverted</param>
        /// <param name="minmax">array containing image min and max values {min, max}</param>
        /// <returns>inverted value</returns>
        public static float InverseOperation(float value, float[] minmax)
        {
            //1. move values so that overall minimum is at zero
            //2. multiply with -1 so that the tacs is inverted
            //3. move values up so that old maximum location has zero value
            //4. move values so that old minimum is preserved
            return (((value - minmax[0]) * (-1f)) + minmax[1]) + minmax[0];
        }
        /// <summary>
        /// Inverses images intensity values. If minimum value 
        /// is negative, then that negative value will be placed at locations 
        /// where was maximum value. 
        /// </summary>
        /// <param name="image">target image</param>
        public static void Inverse(ref Image image)
        {
            //calculate minimum and maximum
            float[] minmax = image.GetMinMax();

            //perform inversion operation for all image tacs
            performOperation(ref image, InverseOperation, minmax);
        }
        /// <summary>
        /// Implements image value normalization operation. Note that number of parameters 
        /// is not checked for performance reasons. The parameters p are interpreted as follows:
        /// min = min_new-min_old
        /// C = (max_old-min_old)/(max_new-min_new)
        /// </summary>
        /// <param name="value">image value that is inverted</param>
        /// <param name="p">array containing move towards minimum and tacs range multiplication factor {min, C}</param>
        /// <returns>inverted value</returns>
        public static float NormalizationOperation(float value, float[] p)
        {
            //1. multiply with C to new range 
            //2. move values so that old minimum is preserved
            return (value + p[0]) * p[1];
        }

        /// <summary>
        /// Normalizaes image values.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="limits">minimum and maximum, 0 and 1 values are used values are not given</param>
        public static void Normalization(ref Image image, float[] limits)
        {
            //calculate minimum and maximum
            float[] minmax = image.GetMinMax();
            float[] new_minmax = new float[]{0,1};
            //use limit values if they are given
            if(limits.Length > 0) new_minmax[0] = limits[0];
            if(limits.Length > 1) new_minmax[1] = limits[1];
            if (minmax[1] - minmax[0] == 0) return;
            if (new_minmax[1] - new_minmax[0] == 0) throw new TPCInvalidArgumentsException("New tacs range cannot be zero.");
            float C = (new_minmax[1] - new_minmax[0]) / (minmax[1] - minmax[0]);
            float min_d = new_minmax[0] - minmax[0];
            //perform inversion operation for all image tacs
            performOperation(ref image, NormalizationOperation, new float[] { min_d, C });
        }

        /// <summary>
        /// Performs user-defined operation to image.
        /// </summary>
        /// <param name="image"></param>
        /// <param name="operation">operation that is applied to all image tacs</param>
        /// <param name="parameters">additional parameterNames for operation, if any</param>
        public static void performOperation(ref Image image, ImageOperation operation, float[] parameters)
        {
            for (int i = 0; i < image.DataLength; i++)
                image[i] = operation(image[i], parameters);
        }
        /// <summary>
        /// Performs user-defined spatial operation to image.
        /// </summary>
        /// <param name="image"></param>
        /// <param name="operation">operation that is applied to all image tacs</param>
        /// <param name="kernel">spatial 3D-kernel size [-1..1 etc.]</param>
        /// <param name="parameters">additional parameterNames for operation, if any</param>
        public static void perform3DSpatialOperation(ref Image image, 
                                                   SpatialImageOperation operation, 
                                                   int kernel, 
                                                   float[] parameters)
        {
            int x = 0;
            int y = 0;
            int z = 0;
			Image temp = new Image(image.Dim);
            for (z = kernel; z < image.Planes - kernel; z++)
            {
                for (y = kernel; y < image.Height - kernel; y++)
                {
                    for (x = kernel; x < image.Width - kernel; x++)
                    {
                        temp[x, y, z] = operation(ref image, new IntPoint(x,y,z), kernel, parameters);
                    }
                }
            }
            image = temp;
        }
        /// <summary>
        /// Delegate for doing arbitrary operation to each image value
        /// </summary>
        /// <param name="value">image value</param>
        /// <param name="parameters">operation specific parameterNames</param>
        /// <returns>value after operation</returns>
        public delegate float ImageOperation(float value, float[] parameters);
        /// <summary>
        /// Delegate for doing arbitrary operation to each image value
        /// </summary>
        /// <param name="image">image value</param>
        /// <param name="targetlocation">location in target image</param>
        /// <param name="kernel">spatial 3D-kernel (for values -kernel..kernel etc.)</param>
        /// <param name="parameters">operation specific parameterNames</param>
        /// <returns>value after operation</returns>
        public delegate float SpatialImageOperation(ref Image image,
                                                    IntPoint targetlocation,
                                                    int kernel, 
                                                    float[] parameters);

        /// <summary>
        /// 3D Bresenham's algorithm for line voxelization.
        /// 
        /// Ported from Matlab code (which was ported from another code):
        /// 
        /// http://www.mathworks.com/matlabcentral/fx_files/21057/1/bresenham_line3d.m
        /// 
        /// Original articles:
        /// 
        /// B.Pendleton.  line3d - 3D Bresenham's (a 3D line drawing algorithm)
        /// ftp://ftp.isc.org/pub/usenet/comp.sources.unix/volume26/line3d, 1992
        ///
        /// Fischer, J., A. del Rio (2004).  A Fast Method for Applying Rigid
        /// Transformations to Volume Data, WSCG2004 Conference.
        /// http://wscg.zcu.cz/wscg2004/Papers_2004_Short/M19.pdf
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="a">starting point </param>
        /// <param name="b">ending point</param>
        /// <returns>list of masked coordinate points</returns>
        public static List<Point> maskBresenhamLine3D(ref MaskImage mask, Point a, Point b)
        {
            Point P1 = new Point((int)a.X, (int)a.Y, (int)a.Z);
            Point P2 = new Point((int)b.X, (int)b.Y, (int)b.Z);

			Point P2P1 = P2 - P1;
            double d = Math.Max(Math.Max(Math.Abs(P2P1.X) + 1.0, Math.Abs(P2P1.Y) + 1.0), Math.Abs(P2P1.Z) + 1.0);

            double[] X = new double[(int)d];
            double[] Y = new double[(int)d];
            double[] Z = new double[(int)d];

            double x1 = P1[0];
            double y1 = P1[1];
            double z1 = P1[2];

            double x2 = P2[0];
            double y2 = P2[1];
            double z2 = P2[2];

            double dx = x2 - x1;
            double dy = y2 - y1;
            double dz = z2 - z1;

            double ax = Math.Abs(dx) * 2;
            double ay = Math.Abs(dy) * 2;
            double az = Math.Abs(dz) * 2;

            double sx = Math.Sign(dx);
            double sy = Math.Sign(dy);
            double sz = Math.Sign(dz);

            double x = x1;
            double y = y1;
            double z = z1;
            int idx = 1;

            // x dominant
            if (ax >= Math.Max(ay, az))
            {
                double yd = ay - ax / 2;
                double zd = az - ax / 2;

                while (true)
                {
                    X[idx - 1] = x;
                    Y[idx - 1] = y;
                    Z[idx - 1] = z;
                    idx = idx + 1;
                    //end
                    if (x == x2) break;
                    // move along y
                    if (yd >= 0)
                    {
                        y = y + sy;
                        yd = yd - ax;
                    }
                    // move along z
                    if (zd >= 0)
                    {
                        z = z + sz;
                        zd = zd - ax;
                    }
                    // move along x
                    x = x + sx;
                    yd = yd + ay;
                    zd = zd + az;
                }
            }
            // y dominant
            else if (ay >= Math.Max(ax, az))
            {
                double xd = ax - ay / 2;
                double zd = az - ay / 2;

                while (true)
                {
                    X[idx - 1] = x;
                    Y[idx - 1] = y;
                    Z[idx - 1] = z;
                    idx = idx + 1;
                    // end
                    if (y == y2) break;
                    // move along x
                    if (xd >= 0)
                    {
                        x = x + sx;
                        xd = xd - ay;
                    }
                    // move along z
                    if (zd >= 0)
                    {
                        z = z + sz;
                        zd = zd - ay;
                    }
                    // move along y
                    y = y + sy;
                    xd = xd + ax;
                    zd = zd + az;
                }
            }
            // z dominant
            else if (az >= Math.Max(ax, ay))
            {
                double xd = ax - az / 2;
                double yd = ay - az / 2;

                while (true)
                {
                    X[idx - 1] = x;
                    Y[idx - 1] = y;
                    Z[idx - 1] = z;
                    idx = idx + 1;
                    // end
                    if (z == z2) break;
                    // move along x
                    if (xd >= 0)
                    {
                        x = x + sx;
                        xd = xd - az;
                    }
                    // move along y
                    if (yd >= 0)
                    {
                        y = y + sy;
                        yd = yd - az;
                    }
                    // move along z
                    z = z + sz;
                    xd = xd + ax;
                    yd = yd + ay;
                }
            }

            //mask into image and gather masked points
            List<Point> points = new List<Point>();
            for (int i = 0; i < X.Length; i++)
            {
                if (X[i] < 0) continue;
                if (Y[i] < 0) continue;
                if (Z[i] < 0) continue;
                if (X[i] >= mask.DimX) continue;
                if (Y[i] >= mask.DimY) continue;
                if (Z[i] >= mask.DimZ) continue;
                mask[(int)X[i], (int)Y[i], (int)Z[i]] = 1;
                points.Add(new Point(X[i], Y[i], Z[i]));
            }
            return points;
        }

        /// <summary>
        /// One-directional 2D Bresenham's algorithm for line voxelization.
        /// 
        /// Ported from Matlab code (which was ported from another code): Original author: Andrew Diamond
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="a">starting point </param>
        /// <param name="b">ending point</param>
        /// <returns>list of masked coordinate points</returns>
        public static List<Point> maskBresenhamLine2D(ref MaskImage mask, Point a, Point b)
        {
            //starting and ending coordinates on plane
            double Sx = 0;
            double Sy = 0;
            double Ex = 0;
            double Ey = 0;
            //plane direction index for yz, xz and xy planes 
            int plane_i = 0;
            if (a.X == b.X) {
                Sx = (int)a.Y; Sy = (int)a.Z;
                Ex = (int)b.Y; Ey = (int)b.Z;
                plane_i = 0;
            }
            else if (a.Y == b.Y)
            {
                Sx = (int)a.X; Sy = (int)a.Z;
                Ex = (int)b.X; Ey = (int)b.Z;
                plane_i = 1;
            }
            else if (a.Z == b.Z)
            {
                Sx = (int)a.X; Sy = (int)a.Y;
                Ex = (int)b.X; Ey = (int)b.Y;
                plane_i = 2;
            }
            else
                throw new TPCInvalidArgumentsException("Starting and ending points must be at the same plane for 2D masking");

            double Dx = Ex - Sx;
            double Dy = Ey - Sy;

            double [] x_coordinates = new double[(int)(2.0 * Math.Ceiling(Math.Abs(Dx)+Math.Abs(Dy)))];
            double [] y_coordinates = new double[x_coordinates.Length];
            int coord_i=0;
            double D;
            //horizontal incrementation
            double IncH;
            //difference (vertical) incrementation
            double IncD;
            double X;
            double Y;
            //temporary value for swap operation
            double Tmp;
	        if(Math.Abs(Dy) <= Math.Abs(Dx)) {
		        if(Ey >= Sy) {
			        if(Ex >= Sx) {
				        D = 2*Dy - Dx;
				        IncH = 2*Dy;
				        IncD = 2*(Dy - Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sx;
                        y_coordinates[coord_i] = Sy;
				        while(X < Ex) {
					        if(D <= 0.5) {
						        D = D + IncH;
						        X = X + 1;
					        } else {
						        D = D + IncD;
						        X = X + 1;
						        Y = Y + 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = X;
                            y_coordinates[coord_i] = Y;
				        }
                    // Ex < Sx
                    } else { 
				        D = -2*Dy - Dx;
				        IncH = -2*Dy;
				        IncD = 2*(-Dy - Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sx;
                        y_coordinates[coord_i] = Sy;
				        while(X > Ex) {
                            if (D >= 0.5)
                            {
						        D = D + IncH;
						        X = X - 1;
					        } else {
						        D = D + IncD;
						        X = X - 1;
						        Y = Y + 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = X;
                            y_coordinates[coord_i] = Y;
        		        }
			        }
                // Ey < Sy
		        } else {
			        if(Ex >= Sx) {
				        D = 2*Dy + Dx;
				        IncH = 2*Dy;
				        IncD = 2*(Dy + Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sx;
                        y_coordinates[coord_i] = Sy;
				        while(X < Ex) {
                            if (D >= 0.5)
                            {
						        D = D + IncH;
						        X = X + 1;
                            } else {
						        D = D + IncD;
						        X = X + 1;
						        Y = Y - 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = X;
                            y_coordinates[coord_i] = Y;
				        }
                    // Ex < Sx
                    } else {
				        D = -2*Dy + Dx;
				        IncH = -2*Dy;
				        IncD = 2*(-Dy + Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sx;
                        y_coordinates[coord_i] = Sy;
				        while(X > Ex) {
                            if (D <= 0.5)
                            {
						        D = D + IncH;
						        X = X - 1;
					        } else {
						        D = D + IncD;
						        X = X - 1;
						        Y = Y - 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = X;
                            y_coordinates[coord_i] = Y;
        		        }
			        }
		        }
            // abs(Dy) > abs(Dx) 
            } else {

		        Tmp = Ex;
		        Ex = Ey;
		        Ey = Tmp;
		        Tmp = Sx;
		        Sx = Sy;
		        Sy = Tmp;
		        Dx = Ex - Sx;
		        Dy = Ey - Sy;
		        if(Ey >= Sy) { 
			        if(Ex >= Sx) { 
				        D = 2*Dy - Dx;
				        IncH = 2*Dy;
				        IncD = 2*(Dy - Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sy;
                        y_coordinates[coord_i] = Sx;
				        while(X < Ex) {
                            if (D <= 0.5)
                            {
						        D = D + IncH;
						        X = X + 1;
                            } else {
						        D = D + IncD;
						        X = X + 1;
						        Y = Y + 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = Y;
                            y_coordinates[coord_i] = X;
        		        }
                    // Ex < Sx
                    } else {
				        D = -2*Dy - Dx;
				        IncH = -2*Dy;
				        IncD = 2*(-Dy - Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sy;
                        y_coordinates[coord_i] = Sx;
				        while(X > Ex) {
                            if (D >= 0.5)
                            {
						        D = D + IncH;
						        X = X - 1;
                            } else {
						        D = D + IncD;
						        X = X - 1;
						        Y = Y + 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = Y;
                            y_coordinates[coord_i] = X;
        		        }
			        }
                // Ey < Sy
                } else {
			        if(Ex >= Sx) {
				        D = 2*Dy + Dx;
				        IncH = 2*Dy;
				        IncD = 2*(Dy + Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sy;
                        y_coordinates[coord_i] = Sx;
				        while(X < Ex) {
                            if (D >= 0.5)
                            {
						        D = D + IncH;
						        X = X + 1;
					        } else {
						        D = D + IncD;
						        X = X + 1;
						        Y = Y - 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = Y;
                            y_coordinates[coord_i] = X;
        		        }
                    // Ex < Sx
			        } else {
				        D = -2*Dy + Dx;
				        IncH = -2*Dy;
				        IncD = 2*(-Dy + Dx);
				        X = Sx;
				        Y = Sy;
                        x_coordinates[coord_i] = Sy;
                        y_coordinates[coord_i] = Sx;
				        while(X > Ex) {
                            if (D <= 0.5)
                            {
						        D = D + IncH;
						        X = X - 1;
					        } else {
						        D = D + IncD;
						        X = X - 1;
						        Y = Y - 1;
					        }
                            coord_i++;
                            x_coordinates[coord_i] = Y;
                            y_coordinates[coord_i] = X;
        		        }
			        }
		        }
	        }

            //mask into image and gather masked points
            List<Point> points = new List<Point>();
            //masking for yz, xz and xy planes 
            if(plane_i == 0) {
                for (int i = 0; i < coord_i + 1; i++)
                {
                    if (x_coordinates[i] < 0) continue;
                    if (y_coordinates[i] < 0) continue;
                    if (x_coordinates[i] >= mask.DimY) continue;
                    if (y_coordinates[i] >= mask.DimZ) continue;
                    mask[(int)a.X, (int)x_coordinates[i], (int)y_coordinates[i]] = 1;
                    points.Add(new Point((int)a.X, (int)x_coordinates[i], (int)y_coordinates[i]));
                }
            }
            else if(plane_i == 1) {
                for (int i = 0; i < coord_i + 1; i++)
                {
                    if (x_coordinates[i] < 0) continue;
                    if (y_coordinates[i] < 0) continue;
                    if (x_coordinates[i] >= mask.DimX) continue;
                    if (y_coordinates[i] >= mask.DimZ) continue;
                    mask[(int)x_coordinates[i], (int)a.Y, (int)y_coordinates[i]] = 1;
                    points.Add(new Point((int)x_coordinates[i], (int)a.Y, (int)y_coordinates[i]));
                }
            } else {
                for (int i = 0; i < coord_i + 1; i++)
                {
                    if (x_coordinates[i] < 0) continue;
                    if (y_coordinates[i] < 0) continue;
                    if (x_coordinates[i] >= mask.DimX) continue;
                    if (y_coordinates[i] >= mask.DimY) continue;
                    mask[(int)x_coordinates[i], (int)y_coordinates[i], (int)a.Z] = 1;
                    points.Add(new Point((int)x_coordinates[i], (int)y_coordinates[i], (int)a.Z));
                }
            }
            return points;
        }
    }
}
