/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Image.PixelStores;
using TPClib.Image.ValueScales;

namespace TPClib.Image
{
    /// <summary>
    /// This small view type can be used only for reading values
    /// straight from PixelStore without Shape object.
    /// Reading dimension order and direction are free
    /// </summary>
    /// <typeparam name="T">Type of output</typeparam>
    public class PixelArrayView<T> : VectorCuboidView<T> where T : struct, IConvertible, IComparable
    {
        /// <summary>
        /// Creates view from PixelStore using normal dimensions and directions
        /// </summary>
        /// <param name="store"></param>
        public PixelArrayView(IPixelStore store) : base(new ValueScale<T>(), store.GetScanner())
            { init(store, null); }

        /// <summary>
        /// Creates view from PixelStore using normal dimensions and directions
        /// </summary>
		/// <param name="viewScale"></param>
        /// <param name="store"></param>
        public PixelArrayView(ValueScale<T> viewScale, IPixelStore store) : base(viewScale, store.GetScanner())
            { init(store, null); }

        /// <summary>
        /// Creates view from PixelStore using normal dimensions and directions
        /// </summary>
        /// <param name="dimensionInfo">Dimension order and direction. For example -2 1 3 will cause the data to be read in following order: -y(flipped), x, z</param>
        /// <param name="store"></param>
        public PixelArrayView(IPixelStore store, int[] dimensionInfo) : base(new ValueScale<T>(), store.GetScanner())
            { init(store, dimensionInfo); }
   
        /// <summary>
        /// Creates view from PixelStore using normal dimensions and directions
        /// </summary>
		/// <param name="viewScale"></param>
        /// <param name="dimensionInfo">Dimension order and direction. For example -2 1 3
        /// will cause the data to be read in following order: -y(flipped), x, z</param>
        /// <param name="store"></param>
        public PixelArrayView(ValueScale<T> viewScale, IPixelStore store, int[] dimensionInfo) : base(viewScale, store.GetScanner())
            { init(store, dimensionInfo); }

        private void init(IPixelStore store, int[] dimensionInfo)
        {
            double[][] component_vectors = new double[store.Dimensions.Length][];

            if (dimensionInfo == null)
            {
                // If there is no dimension order/flip info available, we use 
                // the usual component vector order (+1, +2, +3, +4, ...)
                dimensionInfo = new int[store.Dimensions.Length];
                for (int i = 0; i < store.Dimensions.Length; i++) {dimensionInfo[i] = i+1;}
            }

            // Creating the new dimensions and component vectors (length 1)
            // for every dimension 
            for (int i = 0; i < store.Dimensions.Length; i++)
            {
                component_vectors[i] = new double[store.Dimensions.Length];
                component_vectors[i][i] = 1.0f;
            }
         
            // New component vectors after rearrange/flip
            double[][] newComponentVectors = new double[store.Dimensions.Length][];
            // Because dimensions can be reordered, dimensions can change
            uint[] newDimensions;
            // Because of flips, the reading origo changes also
            double[] newOrigo;

            // Flipping and reordering the dimensions according to dimensionInfo parameter:
            Utils.FlipDimensions(store.Dimensions, component_vectors, dimensionInfo, out newComponentVectors, out newDimensions, out newOrigo);
            
            // Giving the final component vectors, start origo and
            // dimensions of result IArray to VectorCuboidView
            SetComponentVectors(newComponentVectors, newOrigo, newDimensions);            
        }
    }
}
