/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image.ValueScales;

namespace TPClib.Image
{

    /// <summary>
    /// Interface of objects, that can produce Scanner from itself
    /// (PixelStores, Shapes, Scannable Arrays, etc...)
    /// </summary>
    public interface IScannable
    {
        /// <summary>        
        /// Creates new scanner 
        /// </summary>
        /// <returns>The scanner object</returns>
        IScanner GetScanner();
    }

    /// <summary>
    /// This class provides data reading operations from
    /// n dimensional data object using vectors.
    /// </summary>
    public interface IScanner
    {
        /// <summary>
        /// Moves the scanner to new position
        /// </summary>
        /// <param name="coords">New position in real space</param>
        void SetPosition(double[] coords);

        /// <summary>
        /// Gets vector object for moving scanner position
        /// </summary>
        /// <param name="vector">Move vector</param>
        /// <returns>The move vector</returns>
        IScannerMoveVector GetMoveVector(double[] vector);

		/// <summary>
		/// If identical scan vector is used multiple times during
		/// reading process, this method should be used. This method asks
		/// the source object to initialize optimized reader specialized
		/// to read only to direction of given vector.
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="outputType"></param>
		/// <param name="scanVector">The component vector</param>
		/// <returns>The fastest vector scanner with given vector</returns>
		IReadVector<T> GetReadVector<T>(
            ValueScale<T> outputType,            
            double[] scanVector)
            where T : struct, IConvertible, IComparable;
    }

	public interface IWritableScanner : IScanner
	{
		IWriteVector<T> GetWriteVector<T>(
            ValueScale<T> inputType,            
            double[] scanVector)
            where T : struct, IConvertible, IComparable;
	}

    /// <summary>
    /// Interface for vector for moving reading pointer of scanner
    /// </summary>
    public interface IScannerMoveVector
    {
        /// <summary>
        /// Moves the reading pointer in the source object along with the vector
        /// </summary>
        /// <param name="amount">Amount, how many times the vector is added. -1
        /// is moving backwards the length of the vector</param>
        void Move(int amount);
    }

	/// <summary>
	/// 
	/// </summary>
	/// <typeparam name="T"></typeparam>
	public interface IWriteVector<T>
	{
		/// <summary>
		/// Information about conversion to target type (dataloss, etc) 
		/// </summary>
		ConversionInfo ConversionInfo { get; }

		/// <summary>
		/// Writes values from a buffer while moving the scanner forward.
		/// </summary>
		/// <param name="buff">Input buffer</param>
		/// <param name="index">Starting index in the input buffer</param>
		/// <param name="length">Number of sequences to write. Total number of values written from the buffer is length*vector.</param>
		void Set(T[] buff, uint index, uint length);

		/// <summary>
		/// Gets the value at current scanner position
		/// </summary>
		/// <returns></returns>
		void SetCurrentValue(T t);
	}

    /// <summary>
    /// This is a scanner type that is optimized to scan along
    /// fixed vector
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public interface IReadVector<T>
    {
        /// <summary>
        /// Information about conversion to target type (dataloss, etc) 
        /// </summary>
        ConversionInfo ConversionInfo { get; }

        /// <summary>
        /// Scans the samples along scanline to the given buffer and moves
        /// the reading pointer in the source object
        /// </summary>
		/// <param name="buff">Output buffer</param>
		/// <param name="index">Starting index in result buffer</param>
        /// <param name="length">Length of reading sequence. The total reading vector will be vector*length</param>
        void Scan(T[] buff, uint index, uint length);

        /// <summary>
        /// Gets the value at current scanner position
        /// </summary>
        /// <returns></returns>
        T GetCurrentValue();
    }
}
