/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image.ValueScales;

namespace TPClib.Image
{

    /// <summary>
    /// Base class for most views. This class handles n dimensional cube view
    /// with component vector for every dimension and the dimension length.
    /// (The real space length of the dimension is then component vector * dimensionLength)
    /// The component vectors can point to any location.
    /// 
    /// This class implements fully the IMultiDimensionArray and contains also
    /// some optimization for indexers
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class VectorCuboidView<T> : IMultiDimensionArray<T> where T : struct, IConvertible, IComparable
    {
        private uint[] dimensions;     
        private uint length;
        private IScanner scanner;

        private IScannerMoveVector[] components;
        private CubeCoordinatorCalculator coord_calc;
        private uint readPosition_Whole;
        private uint[] readDimensionPos;
        private uint[] indexerDimensionPos;

        private ValueScale<T> mapping;

		/// <summary>
		/// 
		/// </summary>
        public ValueScale<T> Scale { get { return mapping; } }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="view_scale"></param>
		/// <param name="scanner"></param>
        protected VectorCuboidView(ValueScale<T> view_scale, IScanner scanner)
        {
            this.mapping = view_scale;
            this.scanner = scanner;                        
        }

        // All these variables are ment to speed the reading process
        // when this[int] indexer of IArray is used. Because the scanning operates
        // with scan vectors, indexing is really slow, because the scan
        // pointer must be moved to correct location. Almost all
        // cases when user is calling indexer thousands of times,
        // he is reading the buffer pixel by pixel along with some
        // dimension. Then the index step between reads is constant and
        // we can use the same scan vector as last time, which is fast
        // -----------------------
        private uint nextExpectedLocation = 0; // the next expected reading index
        private IScannerMoveVector lastScanVectorComponent; // the expected scan vector (one that was used last time)
        private int lastScanLength = 0; // The length of last scan vector
        private int indexDelta = 0; // The delta of index change
        private double[] origo;

        private ConversionInfo conversion;
        private IReadVector<T> ScanVector = null;

		/// <summary>
		/// 
		/// </summary>
        public uint Length { get { return length; } }
        
		/// <summary>
		/// 
		/// </summary>
		public uint[] Dimensions { get { return (uint[])dimensions.Clone(); } }
        
		/// <summary>
		/// 
		/// </summary>
		public ConversionInfo Conversion { get { return conversion; } }
		
        /// <summary>
        /// Updates the component vectors of the view
        /// </summary>
        /// <param name="components">Conmponent vector for every dimension (size of one voxel in view)</param>
        /// <param name="origo">The Origo in Real space</param>
        /// <param name="dimensions">Dimensions of the view</param>
        protected void SetComponentVectors(double[][] components, double[] origo, uint[] dimensions)
        {
            this.origo = origo;
            this.dimensions = dimensions;
            this.readDimensionPos = new uint[dimensions.Length];
            this.indexerDimensionPos = new uint[dimensions.Length];
            coord_calc = new CubeCoordinatorCalculator(dimensions);
            length = 1;
            for (int i = 0; i < dimensions.Length; i++) length *= dimensions[i];
            this.components = new IScannerMoveVector[components.Length];
            for (int i = 0; i < components.Length; i++) this.components[i] = scanner.GetMoveVector(components[i]);
            ResetRead();

            // Updating the scan reading vector:
            ScanVector = scanner.GetReadVector<T>(mapping, components[0]);
            conversion = ScanVector.ConversionInfo;
        }


        /// <summary>
        /// Single value indexer. The whole Array can be indexed using this 
        /// </summary>
        /// <param name="index">The index number</param>
        /// <returns>The value at given index</returns>
        public T this[uint index]
        {
            get
            {
                if (index == nextExpectedLocation)
                {
                    // If the index is expected, we can use same component vector as last
                    // time and expect that the next index will follow same pattern:
                    nextExpectedLocation = (uint)(index + indexDelta);
                    lastScanVectorComponent.Move(lastScanLength);
                    return ScanVector.GetCurrentValue();                                            
                }

                // The asked location is different than the estimated, now we 
                // have to calculate the actual coordinates of the index number:
                uint[] coords = coord_calc.CalculatePosition(index);

                // Move scanner to Origo:
                scanner.SetPosition(origo);

                // Checking how much each dimension must be moved
                // and checking how many dimensions are affected
                int numMovingDimensions = 0;
                int delta = 0;
                for (int i = 0; i < coords.Length; i++)
                {
                    delta = (int)(coords[i] - indexerDimensionPos[i]); // how much the current scan position dimension must be moved
                    if (delta != 0)
                    {
                        numMovingDimensions++;
                        lastScanVectorComponent = components[i];
                        indexerDimensionPos[i] = coords[i];
                        lastScanLength = delta;
                    }
                    components[i].Move((int)coords[i]);
                }

                if (numMovingDimensions == 1)
                {
                    // If the move affects only one dimension, we estimate
                    // the next possible read position behave same way.
                    // next estimation = index + deltaNow
                    indexDelta = (int)(index - (nextExpectedLocation - indexDelta));
                    nextExpectedLocation = (uint)(index + indexDelta);
                }
                else
                {
                    // If not, we cannot estimate anything and set the
                    // expected value to same as it is now
                    nextExpectedLocation = index;
                }

                // Finally we return the value at current scan position 
                return ScanVector.GetCurrentValue();
            }
        }


        /// <summary>
        /// Multi dimensional indexer
        /// </summary>
        /// <param name="coords">The coordinates in the array</param>
        /// <returns>Value at the given coordinates</returns>
        public T this[uint[] coords]
        {
            get
            {
                // Moves the scanner to all dimensions, except the dim 0 (x)
                for (int i = 1; i < coords.Length; i++)
                {
                    int delta = (int)(coords[i] - readDimensionPos[i]); // how much the current scan position dimension must be moved
                    if (delta != 0) components[i].Move(delta); // if the dimension keeps same, we are not moving
                }

                // now we can use the fastest move and get function with dim 0
                // that handles the move, but also returns the value at current scan pointer
                components[0].Move((int)(coords[0] - readDimensionPos[0]));
                return ScanVector.GetCurrentValue();
            }
        }

        /// <summary>
        /// Resets the reading pointer to start
        /// </summary>
        public void ResetRead()
        {
            // Moving the scanner to start:
            scanner.SetPosition(origo);
            readPosition_Whole = 0;
            readDimensionPos = new uint[dimensions.Length];
            lastScanVectorComponent = components[0];
        }

        /// <summary>
        /// Reads information from the view array
        /// </summary>
        /// <param name="buffer">Buffer, where the data is read to</param>
        /// <param name="index">Starting index of the destination buffer</param>
        /// <param name="length">Number of values read</param>
        /// <param name="position">Start position of the reading pointer</param>
        public void Read(T[] buffer, uint index, uint length, uint position)
        {
            // Setting the position
            if (position != readPosition_Whole)
            {
                uint[] coord = coord_calc.CalculatePosition(position);

                // Moving the scanner to correct location:
                scanner.SetPosition(origo);
                for (int i = 0; i < coord.Length; i++) if (coord[i] != 0) { components[i].Move((int)coord[i]); }

                readPosition_Whole = position;
                readDimensionPos = coord;
            }

            // Reading:
            Read(buffer, index, length);
        }

        /// <summary>
        /// Reads information from the view array
        /// </summary>
        /// <param name="buffer">Buffer, where the data is read to</param>
        /// <param name="index">Starting index of the destination buffer</param>
        /// <param name="length">Number of values read</param>
        public void Read(T[] buffer, uint index, uint length)
        {

            // First reading is to read remainding of the row:
            readPosition_Whole += length;
            uint xDim = dimensions[0];
            uint next = xDim - readDimensionPos[0];

            // Reading data dimension by dimension:
            while (length > 0)
            {
                // If the length is too short for whole row
                /*if (next > length) { next = length; ScanVector.Scan(index, next); }
                // If the length takes the whole row, we can use faster full vector scan
                else if (next == xDim) { ScanVector.Scan(index, next); }
                else { ScanVector.Scan(index, next); }*/
                ScanVector.Scan(buffer, index, next);

                // Scanning the first dimension (x)
                index += next;

                // Moving the reading pointer further:
                readDimensionPos[0] += next;
                for (int i = 0; i < dimensions.Length; )
                {
                    if (readDimensionPos[i] >= dimensions[i])
                    {
                        readDimensionPos[i] = 0;

                        // All except first dimension need Move Back command:
                        if (i > 0) components[i].Move(-((int)dimensions[i]));

                        i++;
                        if (i >= readDimensionPos.Length) break;

                        readDimensionPos[i]++;
                        components[i].Move(1);
                    }
                    else break;
                }

                length -= next;
                next = xDim;
            }
        }
    }
}
