/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image.ValueScales;

namespace TPClib.Image.PixelStores
{
    /// <summary>
    /// This class wraps already existing huge array inside the pixelstore interface
    /// without allocating any new memory for data
    /// </summary>
    /// <typeparam name="T_Stored">Type of stored data</typeparam>
    public class HugeArrayStore<T_Stored> : IPixelStore where T_Stored : struct, IConvertible, IComparable
    {
        private ValueScale<T_Stored> value_scale;
        private T_Stored[] data;
        private uint[] dimensions;

        /// <summary>
        /// Gets the dimensions of the store
        /// </summary>
        public uint[] Dimensions { get { return (uint[])dimensions.Clone(); } }

        /// <summary>
        /// Creates a Pixel store (copy) from given data
        /// </summary>        
		/// <param name="src_data">Values in Array that are put into the store</param>
        /// <param name="dimensions">Dimensions of the store</param>
        public HugeArrayStore(T_Stored[] src_data, params uint[] dimensions)
        {
            CommonInit(new ValueScale<T_Stored>(), dimensions, src_data);            
        }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="src_data"></param>
		/// <param name="factor"></param>
		/// <param name="intersection"></param>
		/// <param name="dimensions"></param>
        public HugeArrayStore(T_Stored[] src_data, double factor, double intersection, params uint[] dimensions)
        {
            CommonInit(new ValueScale<T_Stored>(factor, intersection), dimensions, src_data);            
        }

		private void CommonInit(ValueScale<T_Stored> valueScale, uint[] dimensions, T_Stored[] src_data)
        {
            this.dimensions = dimensions;
            this.value_scale = (ValueScale<T_Stored>)valueScale.Clone();
            this.data = src_data;
            FindMinMax();
        }

        /// <summary>
        /// Finds minimum and maximum stored value and updates the scale
        /// Value conversion methods will need this information for checking,
        /// can two scales be converted without windowing (low and high cuts)
        /// </summary>
        private void FindMinMax()
        {
            T_Stored min = Utils.readStaticField<T_Stored>("MaxValue");
            T_Stored max = Utils.readStaticField<T_Stored>("MinValue");

            for (int i = 0; i < data.Length; i++)
            {
                if (min.CompareTo(data[i]) > 0) min = data[i];
                if (max.CompareTo(data[i]) < 0) max = data[i];
            }

            this.value_scale.SetMinMax(min, max);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public IScanner GetScanner() { return new MainScanner(this); }
		
        /// <summary>
        /// Scanner contains scanning position coordinates
        /// and methods for creating moving/reading vectors for it
        /// </summary>
        class MainScanner : IScanner
        {
            private HugeArrayStore<T_Stored> store;

            // Scanning position. Moving vectors are modifying this
            private int index;

            /// <summary>
            /// The factors, how much each dimension must be multiplied when calculating
            /// one big array index, Like: dimF[x] = 1, dimF[y] = width, dimF[z] = width*height, etc...
            /// </summary>
            private int[] dimensionFactors;

			/// <summary>
			/// 
			/// </summary>
			/// <param name="src"></param>
            public MainScanner(HugeArrayStore<T_Stored> src)
            {                
                this.store = src;                

                // Calculating the helper lookup table for index calculation:
                int factor = 1;
                dimensionFactors = new int[src.dimensions.Length];                
                for (int i = 0; i < src.dimensions.Length; i++) { dimensionFactors[i] = factor; factor *= (int)src.dimensions[i]; }
            }

			/// <summary>
			/// 
			/// </summary>
			/// <param name="pos"></param>
            public void SetPosition(double[] pos)
            {
                // Calculating the new index:
                index = 0;
                for (int i = 0; i < pos.Length; i++) { index += (int)pos[i] * dimensionFactors[i]; }
            }

			/// <summary>
			/// 
			/// </summary>
			/// <param name="vector"></param>
			/// <returns></returns>
            public IScannerMoveVector GetMoveVector(double[] vector)
            {
                int dim = Utils.FindDimension(vector);
                int sign = Math.Sign(dim);
                return new MoveVector(this, (int)dimensionFactors[Math.Abs(dim)-1] * sign); 
            }

			/// <summary>
			/// 
			/// </summary>
			/// <typeparam name="T_Out"></typeparam>
			/// <param name="outputType"></param>
			/// <param name="scanVector"></param>
			/// <returns></returns>
            public IReadVector<T_Out> GetReadVector<T_Out>(ValueScale<T_Out> outputType,double[] scanVector)
                where T_Out : struct, IConvertible, IComparable
            {
                // PixelStore scanners are special kind of scanners, which accept only vectors
                // aligned with x,y or z axis and length of 1 or -1
                int dim = Utils.FindDimension(scanVector);
                int sign = Math.Sign(dim);
                return new ReadVector<T_Out>(this, outputType, (int)dimensionFactors[Math.Abs(dim)-1] * sign);
            }

            class MoveVector : IScannerMoveVector
            {
                private int constant; // The moving dimension factor (1=x -1=-x  width=y, width*height = z, etc...)
                private MainScanner scanner;

                public MoveVector(MainScanner src, int constant) { this.scanner = src; this.constant = constant; }
                
				public void Move(int amount) { scanner.index += constant * amount; }
            }

            class ReadVector<T_Out> : IReadVector<T_Out> where T_Out : struct, IConvertible, IComparable
            {
                private readonly MainScanner scanner;
                private readonly IConverter<T_Stored, T_Out> converter;
                private T_Stored[] srcData;
                private int constant; // The moving dimension factor (1=x -1=-x  width=y, width*height = z, etc...)
                private ConversionInfo cinfo;
                
				public ConversionInfo ConversionInfo { get { return cinfo; } }

                public ReadVector(MainScanner parent_scanner, ValueScale<T_Out> outputType, int constant)
                {
                    // Creating the value converter
                    ValueScale<T_Stored> scaleInsideStore = (ValueScale<T_Stored>)parent_scanner.store.value_scale.Clone();                    
                    this.converter = (IConverter<T_Stored, T_Out>)scaleInsideStore.GetConverter<T_Out>(outputType);
                    this.cinfo = converter.ConversionInfo;

                    this.scanner = parent_scanner;
                    this.srcData = parent_scanner.store.data;
                    this.constant = constant;
                }
                
				public T_Out GetCurrentValue() { return converter.Convert(srcData[scanner.index]); }

                public void Scan(T_Out[] buff, uint index, uint length)
                {
                    converter.Convert(srcData, (int)scanner.index, buff, (int)index, (int)length, constant);                   
                }
            }
        } // end mainScanner
    }
}
