/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Image.ValueScales;

namespace TPClib.Image.PixelStores
{
	/// <summary>
	/// Class for storing n-dimensional pixel data. Data is saved plane by plane without
	/// compression. This class supports also modification of the pixels inside
	/// the store with indexer+EndEdit
	/// 
	/// Timo and Harri: tehkää tänne paremmat pikselinkirjoitus systeemit jos tarvitsette
	/// </summary>
	/// <typeparam name="T_Stored">Type of stored data</typeparam>
	public class SeparatedPlanesStore<T_Stored> : IEnumerable<T_Stored>, IPixelStore where T_Stored : struct, IConvertible, IComparable
	{
		private ValueScale<T_Stored> value_scale;
		private T_Stored[][] data; // Data is stored as planes
		private uint planeLength;
		private uint[] dimensions;
		private uint length = 0;
		private StoreStatistics<T_Stored> stats;

		/// <summary>
		/// Gets the dimensions of the store
		/// </summary>
		public uint[] Dimensions { get { return (uint[])dimensions.Clone(); } }

		/// <summary>
		/// Gets / Sets the pixel values in the store.
		/// Remember to call EndEdit after you have finished
		/// modifying data in the store
		/// </summary>
		/// <param name="index">Index of the value in the store</param>
		public T_Stored this[uint index]
		{
			get { return data[index / planeLength][index % planeLength]; }
			set
			{
				data[index / planeLength][index % planeLength] = value;
			}
		}

		/// <summary>
		/// Read or write several values to this PixelStore
		/// </summary>
		/// <param name="index">Starting index</param>
		/// <param name="count">Number of values to read or write</param>
		/// <returns></returns>
		public T_Stored[] this[uint index, uint count]
		{
			get
			{
				T_Stored[] copy = new T_Stored[count];

				uint plane = index / planeLength;
				uint valuesInPlane = planeLength - (index % planeLength);
				uint valuesToCopy = count < valuesInPlane ? count : valuesInPlane;
				Array.Copy(data[plane], index, copy, 0, valuesToCopy);
				count -= valuesToCopy;

				while (count > 0)
				{
					plane++;
					valuesToCopy = count < planeLength ? count : planeLength;
					Array.Copy(data[plane], 0, copy, copy.Length - count, valuesToCopy);
					count -= valuesToCopy;
				}
				return copy;
			}
			set
			{
				uint plane = index / planeLength;
				uint valuesInPlane = planeLength - (index % planeLength);
				uint valuesToCopy = count < valuesInPlane ? count : valuesInPlane;
				Array.Copy(value, 0, data[plane], index, valuesToCopy);
				count -= valuesToCopy;

				while (count > 0)
				{
					plane++;
					valuesToCopy = count < planeLength ? count : planeLength;
					Array.Copy(value, value.Length - count, data[plane], 0, valuesToCopy);
					count -= valuesToCopy;
				}
				Update();
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="dimensions"></param>
		public SeparatedPlanesStore(params uint[] dimensions)
		{
			CommonInit(new ValueScale<T_Stored>(), dimensions);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="factor"></param>
		/// <param name="intersection"></param>
		/// <param name="dimensions"></param>
		public SeparatedPlanesStore(double factor, double intersection, params uint[] dimensions)
		{
			CommonInit(new ValueScale<T_Stored>(factor, intersection), dimensions);
		}

		/// <summary>
		/// Creates a Pixel store (copy) from given data
		/// </summary>
		/// <param name="values">Values in IArray that are put into the store</param>
		/// <param name="dimensions">Dimensions of the store</param>
		public SeparatedPlanesStore(IArray<T_Stored> values, params uint[] dimensions)
		{
			CommonInit(new ValueScale<T_Stored>(), dimensions);
			values.ResetRead();
			for (int i = 0; i < data.Length; i++) { values.Read(data[i], 0, planeLength); }
			Update();
		}

		/// <summary>
		/// Creates a Pixel store (copy) from given data
		/// </summary>
		/// <param name="values">Values in IArray that are put into the store</param>
		/// <param name="dimensions">Dimensions of the store</param>
		/// <param name="factor"></param>
		/// <param name="intersection"></param>
		public SeparatedPlanesStore(IArray<T_Stored> values, double factor, double intersection, params uint[] dimensions)
		{
			CommonInit(new ValueScale<T_Stored>(factor, intersection), dimensions);
			values.ResetRead();
			for (int i = 0; i < data.Length; i++) { values.Read(data[i], 0, planeLength); }
			Update();
		}

		/// <summary>
		/// Creates a Pixel store (copy) from given data
		/// </summary>
		/// <param name="src_data"></param>
		/// <param name="dimensions"></param>
		public SeparatedPlanesStore(IList<T_Stored> src_data, params uint[] dimensions)
		{
			CommonInit(new ValueScale<T_Stored>(), dimensions);
			int fullIndex = 0;
			for (int plane = 0; plane < data.Length; plane++)
				for (int w = 0; w < planeLength; w++) { data[plane][w] = src_data[fullIndex++]; }
			Update();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="src_data"></param>
		/// <param name="factor"></param>
		/// <param name="intersection"></param>
		/// <param name="dimensions"></param>
		public SeparatedPlanesStore(IList<T_Stored> src_data, double factor, double intersection, params uint[] dimensions)
		{
			CommonInit(new ValueScale<T_Stored>(factor, intersection), dimensions);
			int fullIndex = 0;
			for (int plane = 0; plane < data.Length; plane++)
				for (int w = 0; w < planeLength; w++) { data[plane][w] = src_data[fullIndex++]; }
			Update();
		}

		public uint Length { get { return length; } }

		public void Update()
		{
			stats.Update();
			this.value_scale.SetMinMax(stats.Min, stats.Max);
		}

		public StoreStatistics<T_Stored> Statistics { get { return stats; } }

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public IWritableScanner GetScanner() { return new MainScanner(this); }

		IScanner IScannable.GetScanner()
		{
			return GetScanner();
		}

		private void CommonInit(ValueScale<T_Stored> valueScale, uint[] dimensions)
		{
			// Setting dimension information
			planeLength = (dimensions.Length > 1) ? dimensions[0] * dimensions[1] : dimensions[0];
			this.dimensions = dimensions;
			this.value_scale = (ValueScale<T_Stored>)valueScale.Clone();

			// Calculating the full length
			length = 1;
			for (int i = 0; i < dimensions.Length; i++) length *= dimensions[i];

			// Creating empty memory for every plane
			data = new T_Stored[length / planeLength][];
			for (int i = 0; i < data.Length; i++) data[i] = new T_Stored[planeLength];

			stats = new StoreStatistics<T_Stored>(this);
		}

		public IEnumerator<T_Stored> GetEnumerator()
		{
			foreach (T_Stored[] tArr in data)
				foreach (T_Stored t in tArr)
					yield return t;
		}

		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return GetEnumerator();
		}

		/// <summary>
		/// Scanner contains current scanning position coordinates
		/// and methods for creating moving and reading vectors for it
		/// </summary>
		class MainScanner : IWritableScanner
		{
			private SeparatedPlanesStore<T_Stored> store;
			private uint width;

			// These two variables are definig the scanning position
			// Moving vectors are modifying these
			private uint plane_index;
			private uint sub_index;

			/// <summary>
			/// The factors, how much each dimension must be multiplied when calculating
			/// one big array index, Like: dimF[x] = 1, dimF[y] = width, dimF[z] = width*height, etc...
			/// (This array is for calculating only plane index)
			/// </summary>
			private uint[] planeDimensionFactors;

			public MainScanner(SeparatedPlanesStore<T_Stored> parentStore)
			{
				this.store = parentStore;
				width = parentStore.dimensions[0];

				// Calculating the helper lookup table for plane index calculation:
				uint factor = 1;
				planeDimensionFactors = new uint[parentStore.dimensions.Length];
				for (int i = 2; i < parentStore.dimensions.Length; i++) { planeDimensionFactors[i] = factor; factor *= parentStore.dimensions[i]; }
			}

			public void SetPosition(double[] pos)
			{
				// Calculating the new sub index:
				sub_index = (uint)pos[0] + (uint)pos[1] * width;

				// Calculating the new plane index:
				plane_index = 0;
				for (int i = 2; i < pos.Length; i++) { plane_index += (uint)pos[i] * planeDimensionFactors[i]; }
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="vector"></param>
			/// <returns></returns>
			public IScannerMoveVector GetMoveVector(double[] vector)
			{
				int dim = Utils.FindDimension(vector);
				int sign = Math.Sign(dim);
				dim = Math.Abs(dim);
				if (dim == 1) return new SubMoveVector(this, sign); // vector is along x
				else if (dim == 2) return new SubMoveVector(this, sign * (int)width); // vector is along y
				else return new PlaneMoveVector(this, (int)planeDimensionFactors[dim - 1] * sign);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <typeparam name="T_Out"></typeparam>
			/// <param name="outputType"></param>
			/// <param name="scanVector"></param>
			/// <returns></returns>
			public IReadVector<T_Out> GetReadVector<T_Out>(ValueScale<T_Out> outputType, double[] scanVector)
				where T_Out : struct, IConvertible, IComparable
			{
				// Setting the latest min and max values
				ValueScale<T_Stored> map = (ValueScale<T_Stored>)this.store.value_scale.Clone();

				// Creating the fastest value conversion method
				IConverter<T_Stored, T_Out> converter = (IConverter<T_Stored, T_Out>)map.GetConverter<T_Out>(outputType);
				ConversionInfo conversionInfo = converter.ConversionInfo;

				int dim = Utils.FindDimension(scanVector);
				int sign = Math.Sign(dim);
				dim = Math.Abs(dim);
				if (dim == 1) return new SubReadVector<T_Out>(this, converter, conversionInfo, sign);
				else if (dim == 2) return new SubReadVector<T_Out>(this, converter, conversionInfo, sign * (int)width);
				else return new PlaneReadVector<T_Out>(this, converter, conversionInfo, (int)planeDimensionFactors[dim - 1] * sign);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <typeparam name="T_Out"></typeparam>
			/// <param name="outputType"></param>
			/// <param name="scanVector"></param>
			/// <returns></returns>
			public IWriteVector<T> GetWriteVector<T>(ValueScale<T> inputType, double[] scanVector) where T : struct, IConvertible, IComparable
			{
				// Setting the latest min and max values
				ValueScale<T_Stored> map = (ValueScale<T_Stored>)this.store.value_scale.Clone();

				// Creating the fastest value conversion method
				IConverter<T, T_Stored> converter = (IConverter<T, T_Stored>)inputType.GetConverter<T_Stored>(map);
				ConversionInfo conversionInfo = converter.ConversionInfo;

				int dim = Utils.FindDimension(scanVector);
				int sign = Math.Sign(dim);
				dim = Math.Abs(dim);
				if (dim == 1) return new WriteVector<T>(this, converter, conversionInfo, sign);
				else if (dim == 2) return new WriteVector<T>(this, converter, conversionInfo, sign * (int)width);
				else return new WriteVector<T>(this, converter, conversionInfo, (int)(planeDimensionFactors[dim - 1] * sign * store.planeLength));
			}

			class WriteVector<T_Out> : IWriteVector<T_Out> where T_Out : struct, IConvertible, IComparable
			{
				private readonly MainScanner scanner;
				private readonly IConverter<T_Out, T_Stored> converter;
				private SeparatedPlanesStore<T_Stored> store;
				private int constant;
				private ConversionInfo cinfo;

				public ConversionInfo ConversionInfo { get { return cinfo; } }

				public WriteVector(MainScanner scanner, IConverter<T_Out, T_Stored> converter, ConversionInfo inf, int constant)
				{
					this.scanner = scanner;
					this.converter = converter;
					this.store = scanner.store;
					this.constant = constant;
					this.cinfo = inf;
				}

				public void SetCurrentValue(T_Out t) { store.data[scanner.plane_index][scanner.sub_index] = converter.Convert(t); }

				public void Set(T_Out[] buff, uint index, uint length)
				{
					uint pos = scanner.plane_index * scanner.store.planeLength + scanner.sub_index;
					for (int i = (int)index; i != index + length; i++, pos = (uint)(pos + constant))
						store[pos] = converter.Convert(buff[i]);
				}
			}

			class PlaneMoveVector : IScannerMoveVector
			{
				private int constant; // The moving dimension factor for planes (1=z -1=-z  numPlanes=frame, etc...)
				private MainScanner scanner;
				public PlaneMoveVector(MainScanner parentScanner, int constant) { this.scanner = parentScanner; this.constant = constant; }
				public void Move(int amount) { scanner.plane_index = (uint)(scanner.plane_index + constant * amount); }
			}

			class SubMoveVector : IScannerMoveVector
			{
				private int constant; // The moving dimension factor inside of slice (1=x -1=-x  width=y)
				private MainScanner scanner;
				public SubMoveVector(MainScanner src, int constant) { this.scanner = src; this.constant = constant; }
				public void Move(int amount) { scanner.sub_index = (uint)(scanner.sub_index + constant * amount); }
			}

			class PlaneReadVector<T_Out> : IReadVector<T_Out> where T_Out : struct, IConvertible, IComparable
			{
				private readonly MainScanner scanner;
				private readonly IConverter<T_Stored, T_Out> converter;
				private T_Stored[][] srcData;
				private int constant; // The moving dimension factor for planes (1=z -1=-z  numPlanes=frame, etc...)
				private ConversionInfo cinfo;
				public ConversionInfo ConversionInfo { get { return cinfo; } }

				public PlaneReadVector(MainScanner scanner, IConverter<T_Stored, T_Out> converter, ConversionInfo inf, int constant)
				{
					this.scanner = scanner;
					this.converter = converter;
					this.srcData = scanner.store.data;
					this.constant = constant;
					this.cinfo = inf;
				}

				public T_Out GetCurrentValue() { return converter.Convert(srcData[scanner.plane_index][scanner.sub_index]); }

				public void Scan(T_Out[] buff, uint index, uint length)
				{
					uint subPos = scanner.sub_index;
					int pos = (int)scanner.plane_index;
					for (int i = (int)index; i != index + length; i++, pos += constant)
						buff[i] = converter.Convert(srcData[pos][subPos]);
				}
			}

			class SubReadVector<T_Out> : IReadVector<T_Out> where T_Out : struct, IConvertible, IComparable
			{
				private readonly MainScanner scanner;
				private readonly IConverter<T_Stored, T_Out> converter;
				private T_Stored[][] srcData;
				private int constant; // The moving dimension factor inside of slice (1=x -1=-x  width=y)
				private ConversionInfo cinfo;

				public ConversionInfo ConversionInfo { get { return cinfo; } }

				public SubReadVector(MainScanner scanner, IConverter<T_Stored, T_Out> converter, ConversionInfo inf, int constant)
				{
					this.scanner = scanner;
					this.converter = converter;
					this.srcData = scanner.store.data;
					this.constant = constant;
					this.cinfo = inf;
				}

				public T_Out GetCurrentValue() { return converter.Convert(srcData[scanner.plane_index][scanner.sub_index]); }

				public void Scan(T_Out[] buff, uint index, uint length)
				{
					converter.Convert(srcData[scanner.plane_index], (int)scanner.sub_index, buff, (int)index, (int)length, constant);
				}
			}

		} // end mainScanner
	}

	public class StoreStatistics<T> where T : struct, IConvertible, IComparable
	{
		private T min = new T();
		private T max = new T();
		private double mean = 0.0;
		private SeparatedPlanesStore<T> store;

		public StoreStatistics(SeparatedPlanesStore<T> str) { store = str; }

		public T Min { get { return min; } }
		public T Max { get { return max; } }
		public double Mean { get { return mean; } }

		public void Update()
		{
			min = max = store[0];
			mean = 0.0;
			foreach (T t in store)
			{
				if (t.CompareTo(min) < 0) min = t;
				if (t.CompareTo(max) > 0) max = t;
				mean += Convert.ToDouble(t);
			}
			mean /= store.Length;
		}
	}
}
