/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace TPClib.Image
{
    /// <summary>
    /// Interface for image file that can have its intensity values read and written by means of data streaming
    /// </summary>
    interface IStreamableImageFile
    {
        /// <summary>
        /// Returns image voxel data stream to this file. Note that the stream 
        /// should be closed after use.
        /// </summary>
        /// <returns>stream object to image data</returns>
        ImageDataReader GetImageReader();
        /// <summary>
        /// Returns image voxel data stream to this file. Note that the stream 
        /// should be closed after use.
        /// </summary>
        /// <returns>stream object to image data</returns>
        ImageDataWriter GetImageWriter();
    }
    /// <summary>
    /// A reader object for image data stream
    /// </summary>
    public abstract class ImageDataReader : System.IO.BinaryReader {
        /// <summary>
        /// Constructor with filename.
        /// </summary>
        /// <param name="filename">name of file that is going to have its voxel data read and its state is appropriate for reading data</param>
        public ImageDataReader(string filename)
            : base(File.Open(filename, FileMode.OpenOrCreate))
        {
        }
        /// <summary>
        /// Reads next voxel value from the current strean and advances the current position of the stream by one voxel.
        /// </summary>
        /// <returns>intensity value at next voxel</returns>
        /// <exception cref="EndOfStreamException">if end of stream has been reached</exception>
        public abstract float ReadVoxel();
    }
    /// <summary>
    /// A reader object for image data stream
    /// </summary>
    public abstract class ImageDataWriter : System.IO.BinaryWriter
    {
        /// <summary>
        /// Constructor with filename.
        /// </summary>
        /// <param name="filename ">name of file that is going to have its voxel data written and its state is appropriate for writing data</param>
        public ImageDataWriter(string filename)
            : base(File.Open(filename, FileMode.OpenOrCreate))
        {
        }
        /// <summary>
        /// Writes a float to the current stream and advances the stream position by one voxel.
        /// </summary>
        /// <param name="value">The float value to write.</param>
        public abstract void WriteVoxel(float value);
    }
}
