/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Image data stream
	/// </summary>
	public class ImageStream : Stream
	{
		#region Private variables

		private enum StreamOperation : byte
		{
			NONE, READ, WRITE
		}

		private StreamOperation pending = StreamOperation.NONE;

		// Underlying file stream
		private Stream fstream;

		// Offset from the start of the underlying file stream
		private int streamOffset;

		// Stream datatype
		private ImageData.DataType dataType;

		// Size of the underlying image
		private IntLimits imageLimits;

		// Size of each dimension in bytes
		private int[] dimensionSize;

		// Limits of the stream window
		private IntLimits windowLimits;
		private int maxDim;
		private int[] windowSize;
		private int[] preBlock;
		private int[] postBlock;

		// Highest dimension with continuous data in the window
		private int continuousDimension;

		// Current internal byteposition inside the stream window
		// Always points to the start of the next unread/unwritten block
		private long windowPosition;

		// Internal buffer
		// bufferIndex will be set at internalBuffer.Length if all bytes from the buffer are already read/written
		private byte[] internalBuffer;
		private int bufferIndex;

		// Maximum internal buffer size
		private const int MAXBUFFER = 0xFFFFF;

		private float[] scaleSlopes = new float[] { 1.0f };

		private float[] scaleIntercepts = new float[] { 0.0f };

		private int scaleDimension;

		private int scaleLength;

		private long streamHeadStart = 0;

		private Orientation streamOrientation = Orientation.RAS;

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="fname">Filename</param>
		/// <param name="lim">Full image dimensions</param>
		/// <param name="winLim">Subimage dimensions</param>
		/// <param name="mode">Filemode</param>
		/// <param name="head">Bytes from the start of the file to the start of the image data</param>
		/// <param name="dt">Image data datatype</param>
		public ImageStream(string fname, IntLimits lim, IntLimits winLim, int head = 0, ImageFile.DataType dt = ImageFile.DataType.BIT16_S, FileAccess a = FileAccess.ReadWrite)
			: this(new FileStream(fname, FileMode.OpenOrCreate, a), lim, winLim, head, dt) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="fname">Filename</param>
		/// <param name="lim">Full image dimensions</param>
		public ImageStream(string fname, IntLimits lim) : this(fname, lim, lim) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="fname">Filename</param>
		/// <param name="header">Image header</param>
		/// <param name="winLim">Subimage dimensions</param>
		/// <param name="head">Bytes from the start of the file to the start of the image data</param>
		public ImageStream(string fname, ImageHeader header, IntLimits winLim, int head = 0) : this(fname, header.Dim, winLim, head, header.Datatype) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="imgf">Image file</param>
		/// <param name="winLim">Subimage dimensions</param>
		/// <param name="head">Bytes from the start of the file to the start of the image data</param>
		public ImageStream(ImageFile imgf, IntLimits winLim, int head = 0) : this(imgf.filename, imgf.Header, winLim, head) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">Stream with image data</param>
		/// <param name="header">Imageheader for the image</param>
		/// <param name="winLim">Subimage dimensions</param>
		/// <param name="head">Bytes from the start of the stream to the start of the image data</param>
		public ImageStream(Stream s, ImageHeader header, IntLimits winLim, int head = 0) : this(s, header.Dim, winLim, head, header.Datatype) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">Stream with image data</param>
		/// <param name="lim">Image dimensions</param>
		public ImageStream(Stream s, IntLimits lim) : this(s, lim, lim) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">Stream with image data</param>
		/// <param name="lim">Image dimensions</param>
		/// <param name="winLim">Subimage dimensions</param>
		/// <param name="head">Bytes from the start of the stream to the start of the image data</param>
		/// <param name="dt">Image data datatype</param>
		/// <param name="littleEndian">Data is little endian</param>
		public ImageStream(Stream s, IntLimits lim, IntLimits winLim, int head = 0, ImageFile.DataType dt = ImageFile.DataType.BIT16_S, bool littleEndian = true)
		{
			if (!lim.Includes(winLim)) throw new ArgumentException("Parts of the requested subimage are outside the full image.");

			imageLimits = lim;
			windowLimits = winLim;
			streamOffset = head;

			dataType = new ImageData.DataType();
			dataType.Type = dt;
			dataType.ByteOrder = littleEndian ? ImageData.ByteOrder.LittleEndian : ImageData.ByteOrder.BigEndian;

			Init();

			fstream = s;
			SkipToFirst();
		}

		#endregion

		#region Stream interface

		/// <summary>
		/// End of stream reached
		/// </summary>
		public bool EndOfStream { get { return (this.Position >= this.Length); } }

		/// <summary>
		/// Stream length
		/// </summary>
		public override long Length
		{
			get { return windowSize[maxDim]; }
		}

		/// <summary>
		/// Current stream position
		/// </summary>
		public override long Position
		{
			get
			{
				// Internal position modified by the number of bytes in the buffer
				long pos = windowPosition;
				if (pending == StreamOperation.READ) pos += bufferIndex - BufferSize;
				else if (pending == StreamOperation.WRITE) pos += bufferIndex;
				return pos;
			}
			set
			{
				Seek(value, SeekOrigin.Begin);
			}
		}

		/// <summary>
		/// Stream can read
		/// </summary>
		public override bool CanRead
		{
			get { return fstream.CanRead; }
		}

		/// <summary>
		/// Stream can write
		/// </summary>
		public override bool CanWrite
		{
			get { return fstream.CanWrite; }
		}

		/// <summary>
		/// Stream can seek
		/// </summary>
		public override bool CanSeek
		{
			get
			{
				return fstream.CanSeek;
			}
		}

		/// <summary>
		/// Stream timeouts
		/// </summary>
		public override bool CanTimeout
		{
			get { return fstream.CanTimeout; }
		}

		/// <summary>
		/// Seek a position in this stream
		/// </summary>
		/// <param name="offset">Position offset from the seek origin</param>
		/// <param name="origin">Base point for the seek offset</param>
		/// <returns>Stream position after seeking</returns>
		public override long Seek(long offset, SeekOrigin origin)
		{
			// Write any pending bytes
			PendingWrite();

			long trueOffset = offset;
			switch (origin)
			{
				case SeekOrigin.Current:
					trueOffset += this.Position;
					break;
				case SeekOrigin.End:
					trueOffset += this.Length;
					break;
				default:
					break;
			}

			// Number of whole blocks in the seek distance
			long blocks = trueOffset / BufferSize;
			long totalSeek = 0;

			// Calculate distance from the start of data
			for (int i = 0; i < blocks; i++)
			{
				totalSeek += BufferSize;
				totalSeek += SkipLength(continuousDimension, totalSeek);
			}

			// Skip to the start of data
			SkipToFirst();

			// Seek the correct block
			fstream.Seek(totalSeek, SeekOrigin.Current);
			windowPosition = blocks * BufferSize;

			// Check if there are any bytes remaining and read a block to the memory if needed
			bufferIndex = (int)(trueOffset - windowPosition);

			if (bufferIndex > 0)
			{
				ReadBlock(internalBuffer, 0);
			}

			return this.Position;
		}

		/// <summary>
		/// Set the stream length
		/// </summary>
		/// <param name="value">New stream length</param>
		public override void SetLength(long value)
		{
			throw new NotSupportedException();
		}

		/// <summary>
		/// Read from the stream. Output buffer is assumed to be at least offset+count bytes long. This is not checked here.
		/// </summary>
		/// <param name="buffer">Output buffer</param>
		/// <param name="offset">Offset from the start of the output buffer</param>
		/// <param name="count">Number of bytes to read from the stream</param>
		/// <returns>Number of bytes actually read</returns>
		public override int Read(byte[] buffer, int offset, int count)
		{
			int byteCount = 0;
			int bytesInBuffer = BufferSize - bufferIndex;

			// Copy all remaining bytes from the read buffer
			if (pending == StreamOperation.READ)
			{
				PendingWrite();

				byteCount = bytesInBuffer > count ? count : bytesInBuffer;
				Buffer.BlockCopy(internalBuffer, bufferIndex, buffer, offset, byteCount);
				bufferIndex += byteCount;
				// If no bytes remain, clear the pending read
				if (bufferIndex == BufferSize)
					pending = StreamOperation.NONE;
			}

			// Read blocks to the output buffer
			while (!EndOfStream && count - byteCount >= BufferSize)
			{
				byteCount += ReadBlock(buffer, offset + byteCount);
			}

			// If partial block is still needed, read a block to the internal buffer and copy a part of that.
			if (!EndOfStream && count > byteCount)
			{
				ReadBlock(internalBuffer, 0);
				bufferIndex = count - byteCount;
				Buffer.BlockCopy(internalBuffer, 0, buffer, offset + byteCount, bufferIndex);
				byteCount = count;
				pending = StreamOperation.READ;
			}

			if (ioHandler != null) ioHandler.Invoke(this, new IOProgressEventArgs(100 * ((float)windowPosition / this.Length), null, null));

			return byteCount;
		}

		/// <summary>
		/// Write to the stream. Input buffer is assumed to have at least offset+count bytes, length is not checked.
		/// </summary>
		/// <param name="buffer">Input buffer</param>
		/// <param name="offset">Offset from the start of the input buffer</param>
		/// <param name="count">Bytes to write from the input buffer</param>
		public override void Write(byte[] buffer, int offset, int count)
		{
			int byteCount = 0;
			int freeBuffer = BufferSize - bufferIndex;

			// If there are bytes in the internal buffer, replace the unused ones and write if full
			if (pending != StreamOperation.NONE)
			{
				// If last operation was not write, skip back one block,
				// so that the internal buffer is written to the correct position
				if (pending == StreamOperation.READ)
				{
					SkipToPrev();
				}

				byteCount = count > freeBuffer ? freeBuffer : count;
				Buffer.BlockCopy(buffer, offset, internalBuffer, bufferIndex, byteCount);

				// If internal buffer is full, write it
				if (freeBuffer == byteCount)
				{
					WriteBlock(internalBuffer, 0);
				}

				bufferIndex += byteCount;
			}

			// Write whole blocks
			while (count - byteCount >= BufferSize)
			{
				WriteBlock(buffer, offset + byteCount);
				byteCount += BufferSize;
			}

			// If there still are unwritten bytes, store them in the internal buffer
			// This will always result in a partially filled internal buffer
			if (count > byteCount)
			{
				bufferIndex = count - byteCount;
				Buffer.BlockCopy(buffer, offset + byteCount, internalBuffer, 0, bufferIndex);
				pending = StreamOperation.WRITE;
			}
			else pending = StreamOperation.NONE;
		}

		/// <summary>
		/// Flush the internal stream buffer. Stream position is moved to the start of the next block.
		/// </summary>
		public override void Flush()
		{
			PendingWrite();

			// Discard the internal buffer contents and move to the next block
			bufferIndex = BufferSize;

			fstream.Flush();
		}

		/// <summary>
		/// Dispose of this stream
		/// </summary>
		/// <param name="disposing">True, if called by user</param>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				fstream.Dispose();
			}
			base.Dispose(disposing);
		}

		#endregion

		#region Public

		/// <summary>
		/// I/O event handler of this stream
		/// </summary>
		public IOProcessEventHandler ioHandler;

		/// <summary>
		/// Data orientation used by this stream
		/// </summary>
		public Orientation StreamOrientation
		{
			get { return streamOrientation; }
			set { streamOrientation = value; }
		}

		/// <summary>
		/// Data scaling slope coefficient for the last data point read
		/// </summary>
		public float ScaleSlope
		{
			get
			{
				int currentIndex = (int)(this.NonWindowedPosition / this.scaleLength);
				if (currentIndex < scaleSlopes.Length) return scaleSlopes[currentIndex];
				else return scaleSlopes[0];
			}
		}

		/// <summary>
		/// Data scaling intercept term for the last data point read
		/// </summary>
		public float ScaleIntercept
		{
			get
			{
				int currentIndex = (int)(this.NonWindowedPosition / this.scaleLength);
				if (currentIndex < scaleIntercepts.Length) return scaleIntercepts[currentIndex];
				else return scaleIntercepts[0];
			}
		}

		/// <summary>
		/// Set the global scaling factors
		/// </summary>
		/// <param name="slope">Slope coefficient</param>
		/// <param name="intercept">Intercept term</param>
		public void SetScaling(float slope, float intercept = 0.0f)
		{
			SetScaling(new float[] { slope }, new float[] { intercept }, this.maxDim);
		}

		/// <summary>
		/// Set a variable scaling for a dimension
		/// </summary>
		/// <param name="slopes">Scaling factors for the given dimension</param>
		/// <param name="scaleDim">Dimension that determines the scaling fractor</param>
		public void SetScaling(float[] slopes, int scaleDim)
		{
			float[] intercepts = new float[slopes.Length];
			SetScaling(slopes, intercepts, scaleDim);
		}

		/// <summary>
		/// Set a variable scaling for a dimension
		/// </summary>
		/// <param name="slopes">Scaling slope factors for the given dimension</param>
		/// <param name="intercepts">Scaling intercept terms for the given dimension</param>
		/// <param name="scaleDim">Dimension which determines the scaling values used</param>
		public void SetScaling(float[] slopes, float[] intercepts, int scaleDim)
		{
			if (intercepts.Length != slopes.Length)
				throw new ArgumentException("Number of scaling constants do not match the dimension size");

			scaleDimension = scaleDim;
			if (scaleDim > 0) scaleLength = dimensionSize[scaleDim - 1];
			else scaleLength = ImageFile.BytesPerPixel(this.DataType.Type);
			scaleSlopes = slopes;
			scaleIntercepts = intercepts;
		}

		/// <summary>
		/// Scale the entire stream
		/// </summary>
		/// <param name="s">Scaling factor</param>
		public void Scale(float s)
		{
			for (int i = 0; i < scaleIntercepts.Length; i++)
			{
				scaleIntercepts[i] *= s;
				scaleSlopes[i] *= s;
			}
		}

		/// <summary>
		/// Size of the internal buffer
		/// </summary>
		public int BufferSize { get { return internalBuffer.Length; } }

		/// <summary>
		/// 
		/// </summary>
		public ImageData.DataType DataType { get { return this.dataType; } }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="source"></param>
		/// <param name="dest"></param>
		/// <param name="values"></param>
		/// <returns></returns>
		public static long StreamCopy(ImageStream source, ImageStream dest, int values)
		{
			// Flip the source to align orientations
			ImageStream s = source.Flip(dest.StreamOrientation);

			if (s.DataType.Type == dest.DataType.Type && s.DataType.ByteOrder == dest.DataType.ByteOrder)
			{
				int bytes = values * ImageData.Bytes(s.DataType.Type);

				int maxBuffer = s.BufferSize > bytes ? bytes : s.BufferSize;

				int bytesRead = 0;
				int bytesToRead;
				byte[] tempBuffer;
				while (bytesRead < bytes)
				{
					bytesToRead = bytes - bytesRead > maxBuffer ? maxBuffer : bytes - bytesRead;
					tempBuffer = new byte[bytesToRead];
					bytesRead += s.Read(tempBuffer, 0, bytesToRead);
					dest.Write(tempBuffer, 0, tempBuffer.Length);
				}
				return bytesRead;
			}
			else
			{
				int i;
				for (i = 0; i < values && !s.EndOfStream; i++)
				{
					dest.WriteValue(s.ReadValue());
				}
				return i * ImageData.Bytes(s.DataType.Type);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public float ReadValue()
		{
			byte[] tempBuffer = new byte[ImageFile.BytesPerPixel(this.DataType.Type)];

			ImageData.BytesToValue reader = ImageData.GetReader(this.DataType);

			// Get the coefficients before reading the data
			float slp = ScaleSlope;
			float inc = ScaleIntercept;

			this.Read(tempBuffer, 0, tempBuffer.Length);

			return slp * Convert.ToSingle(reader(tempBuffer)) + inc;
		}

		/// <summary>
		/// Write a single value to the stream
		/// </summary>
		/// <param name="f">Value to write</param>
		public void WriteValue(float f)
		{
			byte[] tempBuffer;

			ImageData.ValueToBytes writer = ImageData.GetWriter(this.dataType);

			tempBuffer = writer((f - ScaleIntercept) / ScaleSlope);
			this.Write(tempBuffer, 0, tempBuffer.Length);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="slope"></param>
		/// <param name="intercept"></param>
		/// <returns></returns>
		public Image ReadImage(float slope = 1.0f, float intercept = 0.0f)
		{
			Image img = new Image(this.Dim);

			this.ReadImage(img, slope, intercept);

			return img;
		}

		/// <summary>
		/// Read image from the stream and scale the data
		/// </summary>
		/// <param name="img">Output image</param>
		/// <param name="slope">Scaling slope</param>
		/// <param name="intercept">Scaling intercept</param>
		public void ReadImage(Image img, float slope = 1.0f, float intercept = 0.0f)
		{
			byte[] tempBuffer = new byte[ImageFile.BytesPerPixel(this.DataType.Type)];

			ImageData.BytesToValue reader = ImageData.GetReader(this.DataType);

			float slp = slope;
			float inc = intercept;
			for (int i = 0; i < img.DataLength && !this.EndOfStream; i++)
			{
				// Get the coefficients before reading the data
				slp = slope * ScaleSlope;
				inc = intercept + ScaleIntercept;

				this.Read(tempBuffer, 0, tempBuffer.Length);

				img[i] = slp * Convert.ToSingle(reader(tempBuffer)) + inc;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="img"></param>
		/// <param name="slope"></param>
		/// <param name="intercept"></param>
		public void WriteImage(Image img, float slope = 1.0f, float intercept = 0.0f)
		{
			byte[] tempBuffer;

			ImageData.ValueToBytes writer = ImageData.GetWriter(this.dataType);

			float slp = slope;
			float inc = intercept;
			for (int i = 0; i < img.DataLength && !this.EndOfStream; i++)
			{
				slp = 1 / (slope * ScaleSlope);
				inc = intercept + ScaleIntercept;
				tempBuffer = writer((img[i] - inc) * slp);
				this.Write(tempBuffer, 0, tempBuffer.Length);
			}
		}

		/// <summary>
		/// Flip the stream in one dimension
		/// </summary>
		/// <param name="dim">Dimension to flip</param>
		/// <returns>Flipped imagestream</returns>
		public ImageStream Flip(int dim)
		{
			int elementSize = ImageFile.BytesPerPixel(this.DataType.Type);
			if (dim > 0) elementSize = dimensionSize[dim - 1];
			int dimSize = imageLimits.GetDimension(dim);

			FlippedStream flipped = new FlippedStream(fstream, elementSize, dimSize);
			ImageStream imgs = new ImageStream(flipped, this.imageLimits, FlipWindowLimits(dim), 0, this.DataType.Type, this.DataType.ByteOrder == ImageData.ByteOrder.LittleEndian);

			imgs.SetScaling(this.scaleSlopes, this.scaleIntercepts, this.scaleDimension);

			// Flip the scaling factors too, if needed
			if (dim == this.scaleDimension)
			{
				int i = 0;
				while (i + dimSize <= imgs.scaleIntercepts.Length)
				{
					Array.Reverse(imgs.scaleIntercepts, i, dimSize);
					Array.Reverse(imgs.scaleSlopes, i, dimSize);
					i += dimSize;
				}
			}

			if (dim == 0) imgs.StreamOrientation = imgs.StreamOrientation.Flip(true, false, false);
			else if (dim == 1) imgs.StreamOrientation = imgs.StreamOrientation.Flip(false, true, false);
			else if (dim == 2) imgs.StreamOrientation = imgs.StreamOrientation.Flip(false, false, true);

			return imgs;
		}

		/// <summary>
		/// Flip the imagestream in the three first dimensions.
		/// </summary>
		/// <param name="flipX">Flip the left-right axis</param>
		/// <param name="flipY">Flip the anterior-posterior axis</param>
		/// <param name="flipZ">Flip the plane order</param>
		/// <returns>Flipped stream</returns>
		public ImageStream Flip(bool flipX, bool flipY, bool flipZ)
		{
			ImageStream imgs = this;
			if (flipX) imgs = imgs.Flip(0);
			if (flipY) imgs = imgs.Flip(1);
			if (flipZ) imgs = imgs.Flip(2);
			return imgs;
		}

		/// <summary>
		/// Flip the stream to a new orientation
		/// </summary>
		/// <param name="o">New orientation</param>
		/// <returns>Stream in the new orientation</returns>
		public ImageStream Flip(Orientation o)
		{
			return Flip(
				o.RightToLeft ^ StreamOrientation.RightToLeft,
				o.AnteriorToPosterior ^ StreamOrientation.AnteriorToPosterior,
				o.SuperiorToInferior ^ StreamOrientation.SuperiorToInferior);
		}

		/// <summary>
		/// Dimensions of the image data
		/// </summary>
		public IntLimits Dim
		{
			get { return this.windowLimits; }
		}

		#endregion

		#region Private

		/// <summary>
		/// Flip a stream window dimension.
		/// </summary>
		/// <param name="dim">Dimension to flip</param>
		/// <returns>New stream window</returns>
		private IntLimits FlipWindowLimits(int dim)
		{
			IntLimits lim = new IntLimits(windowLimits);
			int dimSum = imageLimits.GetLimit(dim, Limits.Limit.HIGH) + imageLimits.GetLimit(dim, Limits.Limit.LOW);
			int end = dimSum - windowLimits.GetLimit(dim, Limits.Limit.LOW);
			int start = dimSum - windowLimits.GetLimit(dim, Limits.Limit.HIGH);
			lim.SetLimits(dim, start, end);
			return lim;
		}

		/// <summary>
		/// 
		/// </summary>
		private long NonWindowedPosition
		{
			get
			{
				long pos = fstream.Position + bufferIndex - streamOffset;
				if (pending == StreamOperation.READ) pos -= streamHeadStart + BufferSize;
				return pos;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		private void PendingWrite()
		{
			// Write remaining bytes from the buffer, if write was the last operation
			if (pending == StreamOperation.WRITE)
			{
				byte[] tempBuffer = new byte[BufferSize];
				ReadBlock(tempBuffer, 0);
				SkipToPrev();
				Buffer.BlockCopy(internalBuffer, 0, tempBuffer, 0, bufferIndex);
				WriteBlock(tempBuffer, 0);
				internalBuffer = tempBuffer;
			}
		}

		/// <summary>
		/// Initialize stream
		/// </summary>
		private void Init()
		{
			dimensionSize = new int[imageLimits.Length];
			windowSize = new int[imageLimits.Length];
			preBlock = new int[imageLimits.Length];
			postBlock = new int[imageLimits.Length];

			maxDim = imageLimits.Length - 1;

			int size = ImageFile.BytesPerPixel(this.dataType.Type);
			int wSize = size;
			bool continuous = true;

			continuousDimension = 0;
			windowPosition = 0;

			for (int i = 0; i < imageLimits.Length; i++)
			{
				preBlock[i] = (windowLimits.GetLimit(i, Limits.Limit.LOW) - imageLimits.GetLimit(i, Limits.Limit.LOW)) * size;
				postBlock[i] = (imageLimits.GetLimit(i, Limits.Limit.HIGH) - windowLimits.GetLimit(i, Limits.Limit.HIGH)) * size;

				size *= imageLimits[i];
				wSize *= windowLimits[i];

				windowSize[i] = wSize;
				dimensionSize[i] = size;

				// If data up to this dimension has no gaps and windows contents fits into the buffer,
				// set the next dimension as the continuous dimension
				if (continuous && windowSize[i] < MAXBUFFER)
				{
					continuousDimension = i;
				}
				continuous = (size == wSize);
			}

			internalBuffer = new byte[windowSize[continuousDimension]];
		}

		/// <summary>
		/// 
		/// </summary>
		private void SkipToFirst()
		{
			long skipLength = streamOffset;

			for (int i = continuousDimension; i <= maxDim; i++)
			{
				skipLength += preBlock[i];
			}

			fstream.Seek(skipLength, SeekOrigin.Begin);
			windowPosition = 0;
			streamHeadStart = 0;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="dim"></param>
		/// <param name="pos"></param>
		/// <returns></returns>
		private long SkipLength(int dim, long pos)
		{
			long skipLength = 0;
			long currentPosition = pos % windowSize[dim];

			if (currentPosition == 0)
			{
				skipLength += preBlock[dim] + postBlock[dim];
				if (dim < maxDim)
					skipLength += SkipLength(dim + 1, pos);
			}

			return skipLength;
		}

		/// <summary>
		/// 
		/// </summary>
		private void SkipToNext()
		{
			streamHeadStart = SkipLength(continuousDimension, windowPosition);
			fstream.Seek(streamHeadStart, SeekOrigin.Current);
		}

		/// <summary>
		/// 
		/// </summary>
		private void SkipToPrev()
		{
			streamHeadStart = SkipLength(continuousDimension, windowPosition) + windowSize[continuousDimension];
			fstream.Seek(-streamHeadStart, SeekOrigin.Current);
			windowPosition -= windowSize[continuousDimension];
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="buffer"></param>
		/// <param name="offset"></param>
		private void WriteBlock(byte[] buffer, int offset)
		{
			// Write the block to the stream
			fstream.Write(buffer, offset, windowSize[continuousDimension]);
			windowPosition += windowSize[continuousDimension];

			// Skip to the start of next block
			if (!EndOfStream) SkipToNext();
		}

		/// <summary>
		/// Read one datablock from the stream
		/// </summary>
		/// <param name="buffer">Buffer for the block</param>
		/// <param name="offset">Buffer offset</param>
		/// <returns>Number of bytes read</returns>
		private int ReadBlock(byte[] buffer, int offset)
		{
			int bytesRead = 0;

			if (!EndOfStream)
			{
				// Read one block from the stream and skip to the next block
				bytesRead = fstream.Read(buffer, offset, windowSize[continuousDimension]);
				windowPosition += bytesRead;
				if (!EndOfStream) SkipToNext();
			}

			return bytesRead;
		}

		#endregion
	}
}
