/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Image
{
	/// <summary>
	/// Structures and methods related to the various image data types
	/// </summary>
	public static class ImageData
	{
		private static IFormatProvider format = System.Globalization.NumberFormatInfo.InvariantInfo;

		/// <summary>
		/// Data encoding description
		/// </summary>
		public struct DataType
		{
			/// <summary>
			/// Data type
			/// </summary>
			public ImageFile.DataType Type;

			/// <summary>
			/// Byteorder
			/// </summary>
			public ImageData.ByteOrder ByteOrder;

			/// <summary>
			/// Cast to ImageFile.Datatype
			/// </summary>
			/// <param name="dt">Datatype struct</param>
			/// <returns>ImageFile type</returns>
			public static implicit operator ImageFile.DataType(DataType dt) { return dt.Type; }

			/// <summary>
			/// Cast from ImageFile datatype. Little-endianness assumed.
			/// </summary>
			/// <param name="ifdt">ImageFile.DataType</param>
			/// <returns>DataType struct</returns>
			public static implicit operator DataType(ImageFile.DataType ifdt)
			{
				DataType dt;
				dt.Type = ifdt;
				dt.ByteOrder = ByteOrder.LittleEndian;
				return dt;
			}
		}

		/// <summary>
		/// Data byte order
		/// </summary>
		public enum ByteOrder : byte
		{
			/// <summary></summary>
			LittleEndian = 0,
			/// <summary></summary>
			BigEndian = 1,
		}

		/// <summary>
		/// Bytes per pixel in the specified datatype
		/// </summary>
		/// <param name="dt">Datatype</param>
		/// <returns>Number of bytes used by the datatype</returns>
		public static int Bytes(ImageFile.DataType dt)
		{
			switch (dt)
			{
				case ImageFile.DataType.BIT1:
				case ImageFile.DataType.BIT8_S:
				case ImageFile.DataType.BIT8_U:
					return 1;
				case ImageFile.DataType.BIT16_U:
				case ImageFile.DataType.BIT16_S:
				case ImageFile.DataType.SUNI2:
				case ImageFile.DataType.VAXI16:
					return 2;
				case ImageFile.DataType.COLRGB:
					return 3;
				case ImageFile.DataType.BIT32_U:
				case ImageFile.DataType.BIT32_S:
				case ImageFile.DataType.FLT32:
				case ImageFile.DataType.SUNI4:
				case ImageFile.DataType.VAXI32:
				case ImageFile.DataType.VAXFL32:
					return 4;
				case ImageFile.DataType.BIT64_U:
				case ImageFile.DataType.BIT64_S:
				case ImageFile.DataType.FLT64:
				case ImageFile.DataType.ASCII:
					return 8;
				default:
					throw new TPCException("Unsupported datatype");
			}
		}

		/// <summary>
		/// Convert byte data between types
		/// </summary>
		/// <param name="bytes">Input data bytes</param>
		/// <returns>Converted data bytes</returns>
		public delegate byte[] ByteConverter(byte[] bytes);

		/// <summary>
		/// Convert raw byte data to a value
		/// </summary>
		/// <param name="bytes">Data bytes</param>
		/// <returns>Data value</returns>
		public delegate object BytesToValue(byte[] bytes);

		/// <summary>
		/// Convert a value to bytes
		/// </summary>
		/// <param name="val">Value to convert</param>
		/// <returns>Data bytes</returns>
		public delegate byte[] ValueToBytes(object val);

		/// <summary>
		/// Create a converter between datatypes
		/// </summary>
		/// <param name="source">Source data type</param>
		/// <param name="dest">Destination data type</param>
		/// <returns>Converter between the types</returns>
		public static ByteConverter GetConverter(DataType source, DataType dest)
		{
			BytesToValue reader = GetReader(source);
			ValueToBytes writer = GetWriter(dest);

			if (source.Type == dest.Type)
			{
				return PassThrough();
			}
			else
			{
				return delegate(byte[] b) { return writer(reader(b)); };
			}
		}

		/// <summary>
		/// Create a converter between datatypes
		/// </summary>
		/// <param name="source">Source data type</param>
		/// <param name="dest">Destination data type</param>
		/// <param name="slp">Scale slope</param>
		/// <param name="intr">Scale intercept</param>
		/// <returns>Converter between the types</returns>
		public static ByteConverter GetConverter(DataType source, DataType dest, float slp, float intr)
		{
			BytesToValue reader = GetReader(source);
			ValueToBytes writer = GetWriter(dest);

			return delegate(byte[] b) { return writer(slp * (float)reader(b) + intr); };
		}

		/// <summary>
		/// Get a pass through converter, does nothing
		/// </summary>
		/// <returns>A pass through converter</returns>
		private static ByteConverter PassThrough()
		{
			return delegate(byte[] b) { return b; };
		}

		/// <summary>
		/// Byte reversal
		/// </summary>
		/// <returns>Converter reversing the byte order</returns>
		public static ByteConverter ReverseBytes()
		{
			return delegate(byte[] b)
			{
				byte[] reverse = new byte[b.Length];
				for (int i = 0, ri = b.Length - 1; i < b.Length; i++, ri--)
				{
					reverse[ri] = b[i];
				}
				return reverse;
			};
		}

		/// <summary>
		/// Create a value to bytes converter
		/// </summary>
		/// <param name="dt">Datatype</param>
		/// <returns>Converter</returns>
		public static ValueToBytes GetWriter(DataType dt)
		{
			ValueToBytes writer;

			switch (dt.Type)
			{
				case ImageFile.DataType.BIT16_U:
					writer = delegate(object val) { return BitConverter.GetBytes((ushort)val); };
					break;
				case ImageFile.DataType.BIT16_S:
				case ImageFile.DataType.SUNI2:
				case ImageFile.DataType.VAXI16:
					writer = delegate(object val) { return BitConverter.GetBytes((short)val); };
					break;
				case ImageFile.DataType.COLRGB:
				case ImageFile.DataType.BIT32_U:
					writer = delegate(object val) { return BitConverter.GetBytes((uint)val); };
					break;
				case ImageFile.DataType.SUNI4:
				case ImageFile.DataType.VAXI32:
				case ImageFile.DataType.BIT32_S:
					writer = delegate(object val) { return BitConverter.GetBytes((int)val); };
					break;
				case ImageFile.DataType.FLT32:
				case ImageFile.DataType.VAXFL32:
					writer = delegate(object val) { return BitConverter.GetBytes((float)val); };
					break;
				case ImageFile.DataType.BIT64_U:
					writer = delegate(object val) { return BitConverter.GetBytes((ulong)val); };
					break;
				case ImageFile.DataType.ASCII:
				case ImageFile.DataType.BIT64_S:
					writer = delegate(object val) { return BitConverter.GetBytes((long)val); };
					break;
				case ImageFile.DataType.FLT64:
					writer = delegate(object val) { return BitConverter.GetBytes((double)val); };
					break;
				default:
					throw new TPCException("Unsupported datatype");
			}
			if (dt.ByteOrder != ByteOrder.LittleEndian && BitConverter.IsLittleEndian)
			{
				ByteConverter reverse = ReverseBytes();
				return delegate(object o) { return reverse(writer(o)); };
			}
			else return writer;
		}

		/// <summary>
		/// Create a bytes to value converter
		/// </summary>
		/// <param name="dt">Datatype</param>
		/// <returns>Converter</returns>
		public static BytesToValue GetReader(DataType dt)
		{
			BytesToValue reader;
			switch (dt.Type)
			{
				case ImageFile.DataType.BIT16_U:
					reader = delegate(byte[] b) { return BitConverter.ToUInt16(b, 0); };
					break;
				case ImageFile.DataType.BIT16_S:
				case ImageFile.DataType.SUNI2:
				case ImageFile.DataType.VAXI16:
					reader = delegate(byte[] b) { return BitConverter.ToInt16(b, 0); };
					break;
				case ImageFile.DataType.COLRGB:
				case ImageFile.DataType.BIT32_U:
					reader = delegate(byte[] b) { return BitConverter.ToUInt32(b, 0); };
					break;
				case ImageFile.DataType.SUNI4:
				case ImageFile.DataType.VAXI32:
				case ImageFile.DataType.BIT32_S:
					reader = delegate(byte[] b) { return BitConverter.ToUInt32(b, 0); };
					break;
				case ImageFile.DataType.FLT32:
				case ImageFile.DataType.VAXFL32:
					reader = delegate(byte[] b) { return BitConverter.ToSingle(b, 0); };
					break;
				case ImageFile.DataType.BIT64_U:
					reader = delegate(byte[] b) { return BitConverter.ToUInt64(b, 0); };
					break;
				case ImageFile.DataType.ASCII:
				case ImageFile.DataType.BIT64_S:
					reader = delegate(byte[] b) { return BitConverter.ToInt64(b, 0); };
					break;
				case ImageFile.DataType.FLT64:
					reader = delegate(byte[] b) { return BitConverter.ToDouble(b, 0); };
					break;
				default:
					throw new TPCException("Unsupported datatype");
			}
			if (dt.ByteOrder != ByteOrder.LittleEndian && BitConverter.IsLittleEndian)
			{
				ByteConverter reverse = ReverseBytes();
				return delegate(byte[] b) { return reader(reverse(b)); };
			}
			else return reader;
		}
	}
}
