/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Common;
using TPClib;

namespace TPClib.Image
{
    /// <summary>
    /// 4x4 tranformation matrix for image transform operations
    /// </summary>
    public class TransformationMatrix: Matrix
    {
        /// <summary>
        /// Default constructor. Creates 4x4 identity transform matrix.
        /// </summary>
        public TransformationMatrix()
            : base(new double[][] { new double[] { 1, 0, 0, 0 }, 
                                    new double[] { 0, 1, 0, 0 }, 
                                    new double[] { 0, 0, 1, 0 }, 
                                    new double[] { 0, 0, 0, 1 } })
        {
        }
        /// <summary>
        /// Constructs tranformation matrix from tacs.
        /// </summary>
        /// <param name="data">4x4 matrix containing tacs</param>
        public TransformationMatrix(double[][] data): base(data) {
            if (data.GetLength(0) != 4 || data[0].GetLength(0) != 4) 
                throw new TPCException("Input tacs must be 4x4");
            
        }
        /// <summary>
        /// Rotates around x-axis
        /// </summary>
        /// <param name="r">angle in radians</param>
        public TransformationMatrix RotateX(double r)
        {
            return new TransformationMatrix(new Matrix(new double[][] { new double[] { 1, 0, 0, 0 }, 
                                                             new double[] { 0, Math.Cos(r), -Math.Sin(r), 0 }, 
                                                             new double[] { 0, Math.Sin(r), Math.Cos(r), 0 }, 
                                                             new double[] { 0, 0, 0, 1 } }) * (Matrix)this);
        }
        /// <summary>
        /// Rotates around y-axis
        /// </summary>
        /// <param name="r">angle in radians</param>
        public TransformationMatrix RotateY(double r)
        {
            return new TransformationMatrix(new Matrix(new double[][] { new double[] { Math.Cos(r), 0, Math.Sin(r), 0 }, 
                                                             new double[] { 0, 1, 0, 0 }, 
                                                             new double[] { -Math.Sin(r), 0, Math.Cos(r), 0 }, 
                                                             new double[] { 0, 0, 0, 1 } }) * (Matrix)this);
        }
        /// <summary>
        /// Rotates around z-axis
        /// </summary>
        /// <param name="r">angle in radians</param>
        public TransformationMatrix RotateZ(double r)
        {
            return new TransformationMatrix(new Matrix(new double[][] { new double[] { Math.Cos(r), -Math.Sin(r), 0, 0 }, 
                                                             new double[] { Math.Sin(r), Math.Cos(r), 0, 0 }, 
                                                             new double[] { 0, 0, 1, 0 }, 
                                                             new double[] { 0, 0, 0, 1 } }) * (Matrix)this);
        }
        /// <summary>
        /// Translation at axis directions. 0 means no movement.
        /// </summary>
        /// <param name="x">translation along the x-axis</param>
        /// <param name="y">translation along the y-axis</param>
        /// <param name="z">translation along the z-axis</param>
        public TransformationMatrix Translate(double x, double y, double z)
        {
            return new TransformationMatrix(new Matrix(new double[][] { new double[] { 1, 0, 0, x }, 
                                                             new double[] { 0, 1, 0, y }, 
                                                             new double[] { 0, 0, 1, z }, 
                                                             new double[] { 0, 0, 0, 1 } }) * (Matrix)this);
        }
        /// <summary>
        /// Scaling in axis directions. 1 means not scaling.
        /// </summary>
        /// <param name="x">x-direction</param>
        /// <param name="y">y-direction</param>
        /// <param name="z">z-direction</param>
        public TransformationMatrix Scaling(double x, double y, double z)
        {
            return new TransformationMatrix(new Matrix(new double[][] { new double[] { 1 * x, 0, 0, 0 }, 
                                                             new double[] { 0, 1 * y, 0, 0 }, 
                                                             new double[] { 0, 0, 1 * z, 0 }, 
                                                             new double[] { 0, 0, 0, 1 } }) * (Matrix)this);
        }
        /// <summary>
        /// Linear algebraic matrix multiplication.
        /// </summary>
        /// <param name="m1">left-side of the multiplication</param>
        /// <param name="m2">right-side of the multiplication</param>
        /// <exception cref="System.ArgumentException">Matrix inner dimensions must agree.</exception>
        public static TransformationMatrix operator *(TransformationMatrix m1, TransformationMatrix m2)
        {
            return new TransformationMatrix((Matrix)m1 * (Matrix)m2);
        }
        /// <summary>
        /// Calculates the matrix multiplied by vector
        /// </summary>
        /// <param name="v">Vector</param>
        /// <param name="a">Matrix</param>
        /// <returns>Product A*v</returns>
        public static Vector operator *(TransformationMatrix a, Vector v)
        {
            Vector r = new Vector(4);
            r[0] = a[0, 0] * v[0] + a[0, 1] * v[1] + a[0, 2] * v[2] + a[0, 3] * v[3];
            r[1] = a[1, 0] * v[0] + a[1, 1] * v[1] + a[1, 2] * v[2] + a[1, 3] * v[3];
            r[2] = a[2, 0] * v[0] + a[2, 1] * v[1] + a[2, 2] * v[2] + a[2, 3] * v[3];
            r[3] = a[3, 0] * v[0] + a[3, 1] * v[1] + a[3, 2] * v[2] + a[3, 3] * v[3];
            return r;
        }

		/// <summary>
		/// Calculates a rotation matrix from unit quaternions.
		/// </summary>
		/// <param name="a">Quaternion a</param>
		/// <param name="b">Quaternion b</param>
		/// <param name="c">Quaternion c</param>
		/// <param name="d">Quaternion d</param>
		/// <returns></returns>
		public static TransformationMatrix QuaternionRotation(double a, double b, double c, double d)
		{
			double aa = a * a;
			double bb = b * b;
			double cc = c * c;
			double dd = c * c;
			double ab = a * b;
			double ac = a * c;
			double ad = a * d;
			double bc = b * c;
			double bd = b * d;
			double cd = c * d;
			return new TransformationMatrix(
				new double[][] {
					new double[] { aa + bb - cc - dd, 2 * (bc - ad), 2 * (bd + ac), 0.0 },
					new double[] { 2 * (bc + ad), aa + cc - bb - dd, 2 * (cd - ab), 0.0 },
					new double[] { 2 * (bd - ac), 2 * (cd + ab), aa + dd - cc - bb, 0.0 },
					new double[] { 0.0, 0.0, 0.0, 1.0 }
				});
		}

		/// <summary>
		/// Calculates a rotation matrix from unit quaternions, assuming
		/// a*a + b*b + c*c + d*d = 1 and a >= 0
		/// </summary>
		/// <param name="b">Quaternion b</param>
		/// <param name="c">Quaternion c</param>
		/// <param name="d">Quaternion d</param>
		/// <returns></returns>
		public static TransformationMatrix QuaternionRotation(double b, double c, double d)
		{
			return QuaternionRotation(Math.Sqrt(1.0 - (b * b + c * c + d * d)), b, c, d);
		}
    }
}
