/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Reflection;
using System.Collections;

namespace TPClib.Modeling
{
	/// <summary>
	/// Component info class. Provides methods for creating and reading information trees from modeling component objects.
	/// </summary>
	public static class ComponentInfo
	{
		/// <summary>
		/// Info tree class. Info tree contains the structure and values stored in a modeling component.
		/// </summary>
		public class InfoTree : IComparable<InfoTree>, IEnumerable
		{
			private System.Collections.Generic.SortedList<string, InfoTree> subTrees;

			private MemberInfo leafInfo = null;

			private object leafValue = null;

			private ModelingAttribute leafAttribute = null;

			/// <summary>
			/// Value of this node.
			/// </summary>
			public object Value
			{
				get { return leafValue; }
				set { leafValue = value; }
			}

			/// <summary>
			/// Field information for this node.
			/// </summary>
			public MemberInfo Info
			{
				get { return leafInfo; }
			}

			/// <summary>
			/// This nodes modeling attribute.
			/// </summary>
			public ModelingAttribute AttributeInfo
			{
				get { return leafAttribute; }
			}

			/// <summary>
			/// Default constructor. Creates an empty tree.
			/// </summary>
			public InfoTree()
			{
				subTrees = new System.Collections.Generic.SortedList<string, InfoTree>();
			}

			/// <summary>
			/// Constructor. Create a new node from field information.
			/// </summary>
			/// <param name="f">Field information</param>
			public InfoTree(MemberInfo f)
				: this()
			{
				leafInfo = f;
			}

			/// <summary>
			/// Create an info tree from an object, including only components marked with a specific attribute.
			/// </summary>
			/// <typeparam name="T">Type of the modeling attribute</typeparam>
			/// <param name="pr">Modeling component from which to create the tree</param>
			/// <returns>Info tree</returns>
			public static InfoTree CreateTree<T>(IModelingComponent pr) where T : ModelingAttribute
			{
				InfoTree it = new InfoTree();

				object[] attributes = pr.GetType().GetCustomAttributes(typeof(T), true);
				if (attributes.Length > 0)
				{
					it.leafAttribute = attributes[0] as ModelingAttribute;
				}
				it.leafValue = pr;

				FieldInfo[] fi = pr.GetType().GetFields();
				fi = Array.FindAll<FieldInfo>(fi, delegate(FieldInfo f) { return f.GetCustomAttributes(typeof(T), true).Length > 0; });
				PropertyInfo[] pi = pr.GetType().GetProperties();
				pi = Array.FindAll<PropertyInfo>(pi, delegate(PropertyInfo p) { return p.GetCustomAttributes(typeof(T), true).Length > 0; });

				foreach (FieldInfo f in fi)
				{
					object nodeValue = f.GetValue(pr);
					ModelingAttribute ma = f.GetCustomAttributes(typeof(T), true)[0] as ModelingAttribute;
					if (nodeValue is IModelingComponent)
						it.Add(CreateTree<T>(nodeValue as IModelingComponent), f, ma, nodeValue);
					else it.Add(f, ma, nodeValue);
				}

				foreach (PropertyInfo p in pi)
				{
					object nodeValue = p.GetValue(pr, null);
					ModelingAttribute ma = p.GetCustomAttributes(typeof(T), true)[0] as ModelingAttribute;
					if (nodeValue is IModelingComponent)
						it.Add(CreateTree<T>(nodeValue as IModelingComponent), p, ma, nodeValue);
					else it.Add(p, ma, nodeValue);
				}
				return it;
			}

			/// <summary>
			/// Create an info tree from an object. Include all components marked with any modeling attribute.
			/// </summary>
			/// <param name="pr">Modeling component from which to create the tree</param>
			/// <returns>Info tree</returns>
			public static InfoTree CreateTree(IModelingComponent pr)
			{
				return CreateTree<ModelAttribute>(pr);
			}

			/// <summary>
			/// Initialize an object from a previously created info tree. Include only components marked with a specific attribute.
			/// </summary>
			/// <typeparam name="T">Attribute type</typeparam>
			/// <param name="pr">Object to initialize</param>
			public void SetValues<T>(IModelingComponent pr) where T : ModelingAttribute
			{
				foreach (InfoTree subtree in this)
				{
					if (subtree.Info is FieldInfo)
						(subtree.Info as FieldInfo).SetValue(pr, subtree.Value);
					else if (subtree.Info is PropertyInfo)
						(subtree.Info as PropertyInfo).SetValue(pr, subtree.Value, null);

					if (subtree.Value is IModelingComponent)
					{
						subtree.SetValues<T>(subtree.Value as IModelingComponent);
					}
				}
			}

			/// <summary>
			/// Initialize an object from a previously created info tree. Initialize all components marked with any attribute.
			/// </summary>
			/// <param name="pr">Object to initialize</param>
			public void SetValues(IModelingComponent pr)
			{
				SetValues<ModelingAttribute>(pr);
			}

			/// <summary>
			/// Get the branch matching a field name.
			/// </summary>
			/// <param name="fName">Field name</param>
			/// <returns>Branch for the field</returns>
			public InfoTree this[string fName]
			{
				get
				{
					return subTrees[fName];
				}
				set
				{
					subTrees[fName] = value;
				}
			}

			/// <summary>
			/// Enumerator for the tree. Enumerates through the branches.
			/// </summary>
			/// <returns>Branch enumerator</returns>
			public IEnumerator GetEnumerator()
			{
				return subTrees.Values.GetEnumerator();
			}

			/// <summary>
			/// Add a branch to this tree.
			/// </summary>
			/// <param name="pt">Branch to add</param>
			public void Add(InfoTree pt)
			{
				subTrees.Add(pt.Info.Name, pt);
			}

			/// <summary>
			/// Add a branch to this tree and set field information for the branch.
			/// </summary>
			/// <param name="pt">Branch</param>
			/// <param name="fi">Field information</param>
			public void Add(InfoTree pt, MemberInfo fi)
			{
				pt.leafInfo = fi;
				Add(pt);
			}

			/// <summary>
			/// Add a branch to this tree and set the attribute and value for the branch node.
			/// </summary>
			/// <param name="pt">Branch</param>
			/// <param name="ma">Attribute</param>
			/// <param name="v">Value</param>
			public void Add(InfoTree pt, ModelingAttribute ma, object v)
			{
				pt.leafAttribute = ma;
				pt.leafValue = v;
				Add(pt);
			}

			/// <summary>
			/// Add a branch to this tree and set all branch node fields.
			/// </summary>
			/// <param name="pt">Branch</param>
			/// <param name="fi">Field information for the branch</param>
			/// <param name="ma">Attribute for the branch</param>
			/// <param name="v">Branch node value</param>
			public void Add(InfoTree pt, MemberInfo fi, ModelingAttribute ma, object v)
			{
				pt.leafInfo = fi;
				Add(pt, ma, v);
			}

			/// <summary>
			/// Convert a field information object to a tree node.
			/// </summary>
			/// <param name="f">Field information</param>
			/// <returns>Tree node</returns>
			public static implicit operator InfoTree(MemberInfo f)
			{
				return new InfoTree(f);
			}

			/// <summary>
			/// Convert a tree root node to a field information object.
			/// </summary>
			/// <param name="t">Tree</param>
			/// <returns>Field information of the root node</returns>
			public static implicit operator MemberInfo(InfoTree t) { return t.Info; }

			/// <summary>
			/// Compare trees. Trees are sorted according to the field information names.
			/// </summary>
			/// <param name="t">Tree to compare to this tree</param>
			/// <returns>Sort order for the trees. -1, if this tree is before the other, 0 if equal, 1 if the other is before this tree.</returns>
			public int CompareTo(InfoTree t)
			{
				if (this.Info == null) return -1;
				else if (t.Info == null) return 1;
				else return Info.Name.CompareTo(t.Info.Name);
			}

			/// <summary>
			/// String representation of this tree.
			/// </summary>
			/// <returns>String representation</returns>
			public override string ToString()
			{
				return ToString("\t");
			}

			private string ToString(string prefix)
			{
				System.Text.StringBuilder sb = new System.Text.StringBuilder();
				string s = String.Empty;
				if (AttributeInfo != null)
				{
					if (AttributeInfo.Name != null)
					{
						s = AttributeInfo.Name;
					}
					if (AttributeInfo.Description != null && AttributeInfo.Description.Length > 0)
					{
						s += " (" + AttributeInfo.Description + ")";
					}
				}
				if (Value != null)
				{
					s += " = " + Value.ToString();
				}
				else s += " = null";

				sb.AppendLine(s);

				foreach (InfoTree tree in this)
					sb.AppendFormat("{0}{1}", prefix, tree.ToString(prefix + prefix));
				return sb.ToString();
			}
		}

		/// <summary>
		/// Create an info tree from an object. Include all components marked with any modeling attribute.
		/// </summary>
		/// <param name="rp">Modeling component from which to create the tree</param>
		/// <returns>Info tree</returns>
		public static InfoTree GetInfoTree(IModelingComponent rp)
		{
			return GetInfoTree<ModelingAttribute>(rp);
		}

		/// <summary>
		/// Create an info tree from an object, including only components marked with a specific attribute.
		/// </summary>
		/// <typeparam name="T">Type of the modeling attribute</typeparam>
		/// <param name="rp">Modeling component from which to create the tree</param>
		/// <returns>Info tree</returns>
		public static InfoTree GetInfoTree<T>(IModelingComponent rp) where T : ModelingAttribute
		{
			return InfoTree.CreateTree<T>(rp);
		}

		/// <summary>
		/// Initialize an object from a previously created info tree. Initialize all components marked with any attribute.
		/// </summary>
		/// <param name="pr">Object to initialize</param>
		/// <param name="tree">Info tree with initialization values</param>
		public static void SetValues(IModelingComponent pr, InfoTree tree)
		{
			SetValues<ModelingAttribute>(pr, tree);
		}

		/// <summary>
		/// Initialize an object from a previously created info tree. Include only components marked with a specific attribute.
		/// </summary>
		/// <typeparam name="T">Attribute type</typeparam>
		/// <param name="pr">Object to initialize</param>
		/// <param name="tree">Tree with initialization values</param>
		public static void SetValues<T>(IModelingComponent pr, InfoTree tree) where T : ModelingAttribute
		{
			tree.SetValues<T>(pr);
		}

		/// <summary>
		/// Get all the types derived from a base type.
		/// </summary>
		/// <param name="baseType">Base type</param>
		/// <returns></returns>
		public static Type[] GetDerived(Type baseType)
		{
			System.Reflection.Assembly assembly = System.Reflection.Assembly.GetAssembly(typeof(BaseModel));
			Type[] all_types = assembly.GetTypes();

			System.Collections.Generic.List<Type> derived =
				new System.Collections.Generic.List<Type>();
			foreach (Type t in all_types)
			{
				if (t.IsSubclassOf(baseType) && !t.IsAbstract)
				{
					object[] attr = t.GetCustomAttributes(typeof(ModelingAttribute), false);
				}
			}
			return derived.ToArray();
		}
	}
}
