/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Attribute for marking the FitProcess classes
	/// </summary>
	[AttributeUsage(System.AttributeTargets.Class, AllowMultiple = false)]
	public class FitProcessAttribute : ModelingAttribute { }

	/// <summary>
	/// Base class for fit processes
	/// </summary>
	public abstract class SingleFitProcess : IFitProcess, ICloneable
	{
		/// <summary>
		/// Measured data for the fit
		/// </summary>
		[Input(Name = @"Measured data", Description = @"Data to fit to model", Optional = false)]
		public Plot MeasuredData;

		/// <summary>
		/// Initialize object
		/// </summary>
		public abstract void Init();

		/// <summary>
		/// Run the fit process
		/// </summary>
		/// <returns>Result of the fit</returns>
		public abstract ParameterList Run();

		/// <summary>
		/// Clone this process
		/// </summary>
		/// <returns>Clone of the process</returns>
		public object Clone()
		{
			return this.MemberwiseClone();
		}
	}

	/// <summary>
	/// Base class for fit processes
	/// </summary>
	[FitProcess(Name = @"Image fit", Description = @"Fits all voxels of an image to a model")]
	public class ImageFitProcess : IFitProcess, System.Collections.Generic.IEnumerable<SingleFitProcess>
	{
		private int currentIndex = 0;
		private SingleFitProcess[] resultFits;

		/// <summary>
		/// Measured data for the fit
		/// </summary>
		[Input(Name = @"Measured data", Description = @"Data to fit to model", Optional = false)]
		public TPClib.Image.Image InputImage;

		/// <summary>
		/// 
		/// </summary>
		[Input(Name = @"Initial fitprocess", Description = @"", Optional = false)]
		public SingleFitProcess Initial;

		/// <summary>
		/// 
		/// </summary>
		[Output]
		public SingleFitProcess LastResults;

		/// <summary>
		/// 
		/// </summary>
		public void Init()
		{
			currentIndex = 0;
			resultFits = new SingleFitProcess[InputImage.Framelength];
			Initial.Init();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public ParameterList Run() { return this.Run(currentIndex++); }

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public System.Collections.Generic.IEnumerator<SingleFitProcess> GetEnumerator()
		{
			for (int i = 0; i < resultFits.Length; i++)
			{
				if (resultFits[i] != null) this.Run(i);
				yield return resultFits[i];
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return GetEnumerator();
		}

		private Vector Run(int index)
		{
			SingleFitProcess fp = Initial.Clone() as SingleFitProcess;
			fp.MeasuredData = this.InputImage.GetCurve(index);
			Vector result = fp.Run();
			resultFits[index] = fp;
			LastResults = fp;
			return result;
		}
	}

	/// <summary>
	/// Base class for fit processes
	/// </summary>
	[FitProcess(Name = @"Multiple fit", Description = @"Fits multiple data sets to a model")]
	public class MultiFitProcess : IFitProcess, System.Collections.Generic.IEnumerable<SingleFitProcess>
	{
		private int currentIndex = 0;
		private SingleFitProcess[] resultFits;

		/// <summary>
		/// Measured data for the fit
		/// </summary>
		[Input(Name = @"Measured data", Description = @"Data to fit to model", Optional = false)]
		public Plot[] MeasuredData;

		/// <summary>
		/// 
		/// </summary>
		[Input(Name = @"Initial fitprocess", Description = @"", Optional = false)]
		public SingleFitProcess Initial;

		/// <summary>
		/// 
		/// </summary>
		[Output]
		public SingleFitProcess LastResults;

		/// <summary>
		/// 
		/// </summary>
		public void Init()
		{
			currentIndex = 0;
			resultFits = new SingleFitProcess[MeasuredData.Length];
			Initial.Init();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public ParameterList Run() { return this.Run(currentIndex++); }

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public System.Collections.Generic.IEnumerator<SingleFitProcess> GetEnumerator()
		{
			for (int i = 0; i < resultFits.Length; i++)
			{
				if (resultFits[i] != null) this.Run(i);
				yield return resultFits[i];
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return GetEnumerator();
		}

		private Vector Run(int index)
		{
			SingleFitProcess fp = Initial.Clone() as SingleFitProcess;
			fp.MeasuredData = this.MeasuredData[index];
			Vector result = fp.Run();
			resultFits[index] = fp;
			LastResults = fp;
			return result;
		}
	}
}
