/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Fit Kmono model to data.
	/// </summary>
	[FitProcess(Name = @"Kmono fitting", Description = @"Linear regression fit to linearized logarithmic data")]
	public class KmonoFit : SingleFitProcess
	{
		/// <summary>
		/// Model to fit
		/// </summary>
		[Input(Name = @"Model", Optional = true)]
		public Kmono Model = new Kmono();

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			Model.ReferencePoints = MeasuredData.ReferenceValues;
			Model.Init();
		}

		/// <summary>
		/// Fit model parameters to input data
		/// </summary>
		/// <returns>Fitted parameters</returns>
		public override ParameterList Run()
		{
			double[] logs = Array.ConvertAll<double, double>(MeasuredData, delegate(double f) { return (f > 0) ? (double)Math.Log(f) : 0.0f; });
			DataSeries ds = new DataSeries(logs, new DataUnit("log " + MeasuredData.Unit.UnitString));

			Plot logData = new Plot(ds, MeasuredData.ReferenceValues);

			double[] p = LinearRegression.ApproximateParameters(logData.ReferenceValues.ToArray(), logData);
			p[0] = -p[0];
			Model.Slope = p[0];
			Model.Intercept = p[1];
			return p;
		}
	}
}
