/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Base class for simple fitting processes
	/// </summary>
	public abstract class BaseFit : SingleFitProcess
	{
		/// <summary>
		/// Maximum iterations.
		/// </summary>
		public const int MaxIterations = 1000;

		/// <summary>
		/// Error tolerance. Iteration is stopped, if result is within this value.
		/// </summary>
		public const double Tolerance = 1e-5;

		/// <summary>
		/// Metric
		/// </summary>
		[Initialization(Name = "Metric", Description = "Metric used for goodness of fit", Optional = true)]
		public IFitMetric Metric = SumOfSquaresMetric.SumOfSquares;

		/// <summary>
		/// Model
		/// </summary>
		[Initialization(Name = "Model", Description = "This model is fitted to data", Optional = false)]
		public abstract IModel Model { get; }

		/// <summary>
		/// Optimization method
		/// </summary>
		[Initialization(Name = "Optimization", Description = "This method is used to fit the data", Optional = true)]
		public Optimization Optimization = new PowellBrent();

		/// <summary>
		/// Fitted model parameters
		/// </summary>
		[Output(Name = "Result", Description = "Optimal parameters for the fit")]
		public ParameterList Result;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			// Initialize model
			Model.Init();

			// Get model parameters and parameter limits
			ParameterList parameters = Model.GetParameters();
			Optimization.InitialParams = parameters;
			Optimization.SetCeiling(Array.ConvertAll<OptimizationParameter, double>(parameters, delegate(OptimizationParameter p) { return p.Maximum; }));
			Optimization.SetFloor(Array.ConvertAll<OptimizationParameter, double>(parameters, delegate(OptimizationParameter p) { return p.Minimum; }));

			// Initialize optimization
			Optimization.SetTargetFunction(FitFunction);
			Optimization.SetMaxIterations(MaxIterations);
			Optimization.SetConvergence(Tolerance);
			Optimization.Init();
		}

		/// <summary>
		/// Fit the data to the model
		/// </summary>
		/// <returns>Fitted model parameters in a vector</returns>
		public override ParameterList Run()
		{
			Optimization.IterateUntilStop();
			Result = Optimization.GetMinimum();
			return Result;
		}

		private double FitFunction(Vector p)
		{
			Model.SetParameters(p);
			return Metric.Calculate(Model.Simulate(), MeasuredData);
		}
	}
}
