/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Interface for all modeling components.
	/// </summary>
	public interface IModelingComponent { }

	/// <summary>
	/// Interface for models.
	/// </summary>
	public interface IModel : IModelingComponent
	{
		/// <summary>
		/// Initialize the fit process. Must be called before calling Run().
		/// </summary>
		void Init();

		/// <summary>
		/// Run the model simulation.
		/// </summary>
		/// <returns>Result of the simulation</returns>
		DataSeries Simulate();

		/// <summary>
		/// Get all model parameters.
		/// </summary>
		/// <returns>List of model parameters</returns>
		ParameterList GetParameters();

		/// <summary>
		/// Set all model parameters.
		/// </summary>
		/// <param name="p">List of parameters</param>
		void SetParameters(ParameterList p);
	}

	/// <summary>
	/// Interface for fit processes
	/// </summary>
	public interface IFitProcess : IModelingComponent
	{
		/// <summary>
		/// Initialize the fit process. Must be called before calling Run().
		/// </summary>
		void Init();

		/// <summary>
		/// Run the fit process.
		/// </summary>
		/// <returns>Result of the fit</returns>
		ParameterList Run();
	}

	/// <summary>
	/// Interface for fit metrics.
	/// </summary>
	public interface IFitMetric : IModelingComponent
	{
		/// <summary>
		/// Calculate the metric for two value sets.
		/// </summary>
		/// <param name="a">First value set</param>
		/// <param name="b">Second value set</param>
		/// <returns>Metric value for the sets</returns>
		double Calculate(double[] a, double[] b);
	}
}
