/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// general two compartmental model
	///    k1    k3
	/// Ca----C1----C2
	///    k2    k4
	/// </summary>
	[Model(Name = @"CM2T", Description = @"General two compartmental model")]
	public class CM2T : SingleInputCompartmentModel
	{
		/// <summary>
		/// K1
		/// </summary>
		[Parameter(Name = @"K1")]
		public OptimizationParameter K1 = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// K2
		/// </summary>
		[Parameter(Name = @"K2")]
		public OptimizationParameter K2 = new OptimizationParameter(0.5, 0, 5);

		/// <summary>
		/// K3
		/// </summary>
		[Parameter(Name = @"K3")]
		public OptimizationParameter K3 = new OptimizationParameter(0.5, 0, 5);

		/// <summary>
		/// K4
		/// </summary>
		[Parameter(Name = @"K4")]
		public OptimizationParameter K4 = new OptimizationParameter(0.5, 0, 5);

		/// <summary>
		/// C1
		/// </summary>
		[Output(Name = @"C1")]
		public TimeDataSeries C1;

		/// <summary>
		/// C2
		/// </summary>
		[Output(Name = @"C2")]
		public TimeDataSeries C2;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			base.Init();
			C1 = new TimeDataSeries(new double[Samples], Ca.Times);
			C2 = new TimeDataSeries(new double[Samples], Ca.Times);
		}

		/// <summary>
		/// Simulate data with the current model parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			double iC1 = 0;
			double iC2 = 0;

			double iC1_last = 0;
			double iC2_last = 0;
			double iInput = 0;

			double c1_last = 0;
			double c2_last = 0;

			double dt2 = 0;

			for (int i = 0; i < Samples; i++)
			{
				dt2 = HalfDeltaTimes[i];
				iInput = IntegratedInput[i];

				//based on cm3 series
				C1[i] = (K1 * iInput - (K2 + K3 / (1 + dt2 * K4)) * (iC1_last + dt2 * c1_last) + (K4 / (1 + dt2 * K4)) * (iC2_last + dt2 * c2_last)) /
					(1 + dt2 * (K2 + K3 / (1 + dt2 * K4)));
				iC1 = iC1_last + dt2 * (c1_last + C1[i]);

				C2[i] = (K3 * iC1 - (K4 * (iC1_last + dt2 * c2_last))) / (1 + dt2 * K4);
				iC2 = iC2_last + dt2 * (c2_last + C2[i]);

				c1_last = C1[i];
				iC1_last = iC1;
				c2_last = C2[i];
				iC2_last = iC2;

				CPet[i] = C1[i] + C2[i];

			}
			return CPet;
		}
	}

	/// <summary>
	/// General 2-tissue compartmental model
	/// </summary>
	[Model(Name = @"General 2-tissue compartmental model", Description = @"General 2-tissue compartmental model")]
	public class General2TCModelK3 : CM2T
	{
		/// <summary>
		/// Va
		/// </summary>
		[Parameter(Name = @"VA")]
		public OptimizationParameter Va = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			this.K4 = 0;
			this.K4.Hidden = true;
			base.Init();
		}

		/// <summary>
		/// Simulate data with the current model parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			base.Simulate();

			for (int i = 0; i < CPet.Length; i++)
			{
				CPet[i] = (1 - Va) * CPet[i] + Va * Ca[i];
			}
			return CPet;
		}
	}

	/// <summary>
	/// General 2-tissue compartmental model
	/// </summary>
	[Model(Name = @"General 2-tissue compartmental model", Description = @"General 2-tissue compartmental model")]
	public class General2TCModelK4 : CM2T
	{
		/// <summary>
		/// Va
		/// </summary>
		[Parameter(Name = @"VA")]
		public OptimizationParameter va = new OptimizationParameter(0.5, 0, 1);

		// NOTE: uses the CM3T Series model for simulation, *not* CM2T
		private CM3TSeries cm3ts;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			base.Init();
			cm3ts = new CM3TSeries();
			cm3ts.Ca = this.Ca;
			cm3ts.Init();
		}

		/// <summary>
		/// Simulate data with the current model parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			cm3ts.K1 = this.K1;
			cm3ts.K2 = this.K2;
			cm3ts.K3 = this.K3;
			cm3ts.K4 = this.K4;
			cm3ts.K5 = 0;
			cm3ts.K6 = 0;
			cm3ts.Simulate();

			this.C1 = cm3ts.C1;
			this.C2 = cm3ts.C2;

			for (int i = 0; i < CPet.Length; i++)
			{
				CPet[i] = (1 - va) * (C1[i] + C2[i]) + va * Ca[i];
			}
			return CPet;
		}
	}
}
