/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Reflection;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Attribute for model classes
	/// </summary>
	[AttributeUsage(System.AttributeTargets.Class, AllowMultiple = false)]
	public class ModelAttribute : ModelingAttribute
	{
		private ClassCategory classCategory;

		/// <summary>
		/// Model attribute category
		/// </summary>
		public enum ClassCategory : uint
		{
			/// <summary>
			/// No category specified (default)
			/// </summary>
			NONE = 0x0,
			/// <summary>
			/// Test model
			/// </summary>
			TEST = 0x1,
			/// <summary>
			/// Obsolete model
			/// </summary>
			OBSOLETE = 0x2,
			/// <summary>
			/// Carimas model
			/// </summary>
			CARIMAS = 0x4,
			/// <summary>
			/// Brain model
			/// </summary>
			BRAIN = 0x10,
			/// <summary>
			/// Heart model
			/// </summary>
			HEART = 0x20,
			/// <summary>
			/// Universal model
			/// </summary>
			UNIVERSAL = 0xF0,
			/// <summary>
			/// All categories
			/// </summary>
			ALL = 0xFFFFFFFF
		}

		/// <summary>
		/// Get the model attribute category
		/// </summary>
		public ClassCategory Category { get { return classCategory; } set { classCategory = value; } }

		/// <summary>
		/// Check if model attribute belongs to some category/categories.
		/// </summary>
		/// <param name="cc">Category bit field</param>
		/// <returns>True, if this attribute belongs to all the categories</returns>
		public bool CheckCategory(ClassCategory cc) { return (this.Category & cc) == cc; }
	}

	/// <summary>
	/// 
	/// </summary>
	[Model(Name = @"Default model name", Description = @"Default description")]
	public abstract class BaseModel : IModel
	{
		private FieldInfo[] parameterInfo;

		/// <summary>
		/// Generate simulated values with the current model parameters.
		/// </summary>
		/// <returns>Simulated values</returns>
		public abstract DataSeries Simulate();

		/// <summary>
		/// Initialize this model. Must be called before the first call of Simulate().
		/// </summary>
		public virtual void Init()
		{
			// List all fields marked with ParameterAttribute and sort alphabetically
			parameterInfo = this.GetType().GetFields();
			parameterInfo = Array.FindAll<FieldInfo>(parameterInfo, delegate(FieldInfo f) { return f.GetCustomAttributes(typeof(ParameterAttribute), false).Length > 0; });
			Array.Sort<FieldInfo>(parameterInfo, delegate(FieldInfo a, FieldInfo b) { return a.Name.CompareTo(b.Name); });
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="name"></param>
		/// <param name="pvalue"></param>
		/// <returns></returns>
		public bool SetParameter(string name, OptimizationParameter pvalue)
		{
			foreach (FieldInfo fi in parameterInfo)
			{
				if (fi.Name == name)
				{
					fi.SetValue(this, pvalue);
					return true;
				}
			}
			return false;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="name"></param>
		/// <returns></returns>
		public OptimizationParameter GetParameter(string name)
		{
			foreach (FieldInfo fi in parameterInfo)
			{
				object val = fi.GetValue(this);
				if (fi.Name == name && val is OptimizationParameter)
				{
					return (OptimizationParameter)val;
				}
			}
			throw new TPCException("No parameter " + name + " found");
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="p"></param>
		public void SetParameters(ParameterList p)
		{
			for (int i = 0; i < p.Count; i++)
			{
				object val = parameterInfo[i].GetValue(this);
				if ((val is OptimizationParameter) && !((OptimizationParameter)val).Hidden)
					parameterInfo[i].SetValue(this, (OptimizationParameter)(p[i]));
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public ParameterList GetParameters()
		{
			ParameterList pars = new ParameterList(parameterInfo.Length);

			for (int i = 0; i < pars.Count; i++)
			{
				object val = parameterInfo[i].GetValue(this);
				if ((val is OptimizationParameter) && !((OptimizationParameter)val).Hidden)
					pars[i] = (OptimizationParameter)parameterInfo[i].GetValue(this);
			}
			return pars;
		}

		/// <summary>
		/// 
		/// </summary>
		public BaseModel()
		{
			Init();
		}
	}

	/// <summary>
	/// Model related utility functions
	/// </summary>
	public static class ModelUtils
	{
		/// <summary>
		/// Myocardial density, 1.04 g/ml
		/// </summary>
		public const double HeartMuscleDensity = 1.04;

		/// <summary>
		/// Calculate differences between time points and divide by 2. These values are used in the integrals.
		/// </summary>
		/// <param name="t">Vector of time points</param>
		/// <returns>Vector of halved time differences</returns>
		public static double[] HalfTimeDifs(double[] t)
		{
			double[] difs = new double[t.Length];

			for (int i = 1; i < t.Length; i++)
			{
				difs[i] = (t[i] - t[i - 1]) * 0.5f;
			}

			// If first sample time is positive, assume starting point is 0.0;
			// if it's negative, start is set at first sample time
			difs[0] = (t[0] > 0.0f) ? (t[0] * 0.5f) : 0.0f;

			return difs;
		}

		/// <summary>
		/// Calculates the reference_times integral using trapezium rule
		/// </summary>
		/// <param name="refvals">reference_times values of curve</param>
		/// <param name="halfDeltaTimes">corresponding time delta values divided by 2</param>
		/// <returns>Vector of cumulative integral values</returns>
		public static double[] CalcRefIntegral(double[] refvals, double[] halfDeltaTimes)
		{
			if (halfDeltaTimes.Length != refvals.Length) throw new TPCException("Frame count and tacs length don't match");
			double[] refint = new double[refvals.Length];

			// Cumulative integral value
			double cri;

			// Calculate the integral
			cri = halfDeltaTimes[0] * refvals[0];
			refint[0] = cri;
			for (int i = 1; i < refvals.Length; i++)
			{
				cri += (refvals[i] + refvals[i - 1]) * halfDeltaTimes[i];
				refint[i] = cri;
			}
			return refint;
		}

		/// <summary>
		/// Interpolate a single value
		/// </summary>
		/// <param name="xLow">Lower data point</param>
		/// <param name="xHigh">Higher data point</param>
		/// <param name="yLow">Value at lower data point</param>
		/// <param name="yHigh">Value at higher data point</param>
		/// <param name="x">Interpolation point</param>
		/// <returns>Interpolated value at x</returns>
		public static double Interpolate(double xLow, double xHigh, double yLow, double yHigh, double x)
		{
			double y = 0;
			if (x <= xLow) y = yLow;
			else if (x >= xHigh) y = yHigh;
			else
			{
				double length = xHigh - xLow;
				double d1 = x - xLow;
				double d2 = xHigh - x;
				y = (yLow * d1 + yHigh * d2) / length;
			}

			return y;
		}
	}
}
