/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Base class for compartment models that take one input TAC
	/// </summary>
	[Model(Name = @"Compartment model", Description = @"Compartment model")]
	public abstract class SingleInputCompartmentModel : BaseModel
	{
		/// <summary>
		/// Reference data
		/// </summary>
		[Initialization(Name = @"CA")]
		public TimeDataSeries Ca;

		/// <summary>
		/// Simulation output
		/// </summary>
		[Output(Name = @"CPet")]
		public TimeDataSeries CPet;

		/// <summary>
		/// Time increments divided by 2. Used when integrating with the trapezium method.
		/// </summary>
		protected double[] HalfDeltaTimes;

		/// <summary>
		/// Integrated input function
		/// </summary>
		protected DataSeries IntegratedInput;

		/// <summary>
		/// Number of samples simulated
		/// </summary>
		public int Samples
		{
			get { return HalfDeltaTimes.Length; }
		}

		/// <summary>
		/// Initialization
		/// </summary>
		public override void Init()
		{
			HalfDeltaTimes = ModelUtils.HalfTimeDifs(Ca.Times);
			IntegratedInput = ModelUtils.CalcRefIntegral(Ca, HalfDeltaTimes);
			CPet = new TimeDataSeries(new double[Samples], Ca.Times);
		}
	}
}
