/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// TRTM (Transport-limited reference tissue model) model
	/// </summary>
	[Model(Name = @"TRTM", Description = @"Transport-limited reference tissue model")]
	public class TRTMModel : SingleInputCompartmentModel
	{
		/// <summary>
		/// R1
		/// </summary>
		[Parameter(Name = @"R1", Description = @"Ratio K1/K1'", Optional = true)]
		public OptimizationParameter R1 = 0;

		/// <summary>
		/// K2
		/// </summary>
		[Parameter(Name = @"K2", Description = @"Rate constant of the model", Optional = true)]
		public OptimizationParameter K2 = 0;

		/// <summary>
		/// K3
		/// </summary>
		[Parameter(Name = @"K3", Description = @"Rate constant of the model", Optional = true)]
		public OptimizationParameter K3 = 0;

		/// <summary>
		/// Simulates tissue TAC using reference_times tissue compartment model (transport limited in ref region) and reference_times region TAC, at reference_times region TAC tacs.
		/// The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
		/// </summary>
		/// <returns>Vector of simulated values</returns>
		public override DataSeries Simulate()
		{
			double k23 = K2 + K3;

			//Cumulative integral value
			double cti = 0.0;

			double last_val = 0.0;
			double cr, cri, dt2, ci;

			for (int i = 0; i < Samples; i++)
			{
				cr = Ca[i];
				cri = IntegratedInput[i];
				dt2 = HalfDeltaTimes[i];

				ci = (R1 * (cr + K3 * cri) - k23 * (dt2 * last_val + cti)) /
							(1.0 + dt2 * k23);

				cti += dt2 * (last_val + ci);
				last_val = ci;
				CPet[i] = ci;
			}
			return CPet;
		}
	}
}
