/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
    /// <summary>
    /// Implementation of Nelder-Mead optimization algorithm
    /// </summary>
	[Optimization(Name="Nelder-Mead", Description="Simplex Variant")]
    public class NelderMead : Optimization
    {
        /// <summary>
        /// Reflection scale
        /// </summary>
		[Initialization(Name="Alpha", Description="Reflection scale", Optional=true)]
		public OptimizationParameter alpha = 1.0;

        /// <summary>
        /// Expansion scale
        /// </summary>
		[Input(Name = "Gamma", Description = "Expansion scale", Optional = true)]
		public OptimizationParameter gamma = 2.0;

        /// <summary>
        /// Contraction scale
        /// </summary>
		[Input(Name = "Rho", Description = "Contraction scale", Optional = true)]
		public OptimizationParameter rho = 0.5;

        /// <summary>
        /// Shrink scale
        /// </summary>
		[Input(Name = "Theta", Description = "Shrink scale", Optional = true)]
		public OptimizationParameter theta = 0.5;

        /// <summary>
        /// Polytope
        /// </summary>
        private Polytope ptope;

		/// <summary>
		/// Default constructor
		/// </summary>
		public NelderMead() { }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="target_func">Function to optimize</param>
        /// <param name="p">Initial parameters</param>
		public NelderMead(RealFunction target_func, Vector p) :
			base(target_func, p) { }

		/// <summary>
		/// Initialize.
		/// </summary>
		public override void Init()
		{
			MakePolytope(InitialParams, alpha);
		}

		/// <summary>
        /// Single minimization step; called from Optimization.Iterate()
        /// </summary>
        /// <returns>
        /// </returns>
        protected override Vector Step()
        {
            // Calculate the reflection of the worst point
            Vector worst = ptope.GetMaximum();
            Vector candidate = ptope.GetReflection(worst, alpha);
            double fval = ConstrainedTargetFunction(candidate);


            // Is the reflection better than worst point?
            if (ptope.IsBetter(fval))
            {
                // If the reflection is best point so far...
                if (ptope.IsBest(fval))
                {
                    // ... try to find even better point further along the same line
                    Vector newcandidate = worst + gamma * (candidate - worst);
                    double fval_new = ConstrainedTargetFunction(newcandidate);

                    // Keep looking (exponentially further) as long as better points are found
                    while (fval_new < fval)
                    {
                        candidate = newcandidate;
                        fval = fval_new;

                        newcandidate = worst + gamma * (candidate - worst);
                        fval_new = ConstrainedTargetFunction(newcandidate);
                    }
                }
                // Replace the worst point
                ptope.ReplaceWorst(candidate, fval);
            }

            // If the reflection is worse than the worst point so far,
            // try to find a better one along the same line
            else
            {
                Vector newcandidate = ptope.GetReflection(worst, rho);
                double fval_new = ConstrainedTargetFunction(newcandidate);

                // If it indeed is better, use it
                if (fval_new < fval)
                {
                    // Replace the worst point
                    ptope.ReplaceWorst(newcandidate, fval_new);
                }

                // If it is even worse, try to continue with a new polytope
                else
                {
                    MakePolytope(GetMinimum(), theta);
                }
            }

            return ptope.GetMinimum();
        }

        /// <summary>
        /// Constructs a polytope with 'initial' point as one of the vertices,
        /// other vertices are along the unit vectors, 'scale' units away
        /// </summary>
        /// <param name="initial">Initial point</param>
        /// <param name="scale">
        /// Polytope size; polytope with 'scale==1' has
        /// vertices separated from initial point by unit vectors
        /// </param>
        private void MakePolytope(Vector initial, double scale)
        {
            ptope = new Polytope();
            double funcval = ConstrainedTargetFunction(initial);
            ptope.Add(initial, funcval);

            for (int i = 0; i < initial.Length; i++)
            {
                Vector vertex = initial + scale * Vector.Unit(initial.Length, i);
                funcval = ConstrainedTargetFunction(vertex);
                ptope.Add(vertex, funcval);
            }
        }

        /// <summary>
        /// Get the vertices of the current polytope
        /// </summary>
        /// <returns>Array of vectors</returns>
        public Vector[] GetSimplex()
        {
            return ptope.GetVertices();
        }
    }
}
