/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Modeling attribute for optimization classes.
	/// </summary>
	[AttributeUsage(System.AttributeTargets.Class, AllowMultiple = false)]
	public class OptimizationAttribute : ModelingAttribute { }

	/// <summary>
    /// Abstract superclass for optimization methods
    /// </summary>
    public abstract class Optimization : IModelingComponent
	{
		#region Private
		
		/// <summary>
		/// Number of iterations performed so far
		/// </summary>
		private int iterations = 0;

		/// <summary>
		/// Number of target function calls done so far
		/// </summary>
		private int functioncalls = 0;

		/// <summary>
		/// Target function. Not to be directly assigned, use SetTargetFunction instead
		/// </summary>
		private RealFunction targetfunc;

		/// <summary>
		/// List of stop conditions
		/// </summary>
		private List<StopCondition> stopConditions = new List<StopCondition>();

		/// <summary>
		/// List of constraints
		/// </summary>
		private List<Constraint> constraints = new List<Constraint>();

		/// <summary>
		/// Constrained target function. If function parameters are outside the allowed range,
		/// penalty function is applied.
		/// </summary>
		/// <param name="par">Function parameters</param>
		/// <returns>Modified target function value</returns>
		private double ConstrainedFunction(Vector par)
		{
			// Count function calls
			functioncalls++;

			// If there are defined constraints, apply them.
			if (constraints.Count > 0)
			{
				Vector limited = (Vector)par.Clone();
				double sum = 0.0;
				double penalty;

				// Add each constraint dataFunctions square to the original target function
				foreach (Constraint c in constraints)
				{
					penalty = c(ref limited);
					if (penalty > 0.0)
						sum += penalty;
				}

				// Return the modified target function value
				return targetfunc(limited) + sum;
			}
			// If there are no constraints, use original target function
			else return targetfunc(par);
		}

		/// <summary>
		/// Check the stop conditions
		/// </summary>
		/// <returns>True, if at least one stop condition is reached</returns>
		private bool Stop()
		{
			if (stopConditions.Count > 0)
			{
				foreach (StopCondition s in stopConditions)
					if (s(currentParams)) return true;
			}
			return false;
		}

		#endregion

		#region Protected

		/// <summary>
		/// Parameter change in last iteration. Used with convergence stop criteria.
		/// </summary>
		protected Vector delta;

		/// <summary>
		/// The best parameter values found so far
		/// </summary>
		protected Vector currentParams = new Vector();

		/// <summary>
		/// Initial parameter values
		/// </summary>
		protected Vector initialParams;

		/// <summary>
		/// Abstract method to be overriden in optimization method implementations.
		/// Performs a single iteration of the optimization process. Definition of 'single iteration,
		/// is left open, but it must return a single, 'best' point found so far.
		/// </summary>
		/// <returns>parameter values after optimization step</returns>
		protected abstract Vector Step();

		/// <summary>
		/// Creating optimization without initialization values is prohibited. Use constructor with initialization 
		/// values instead.
		/// </summary>
		protected Optimization() { }

		/// <summary>
		/// Constructs an optimization method for use.
		/// </summary>
		/// <param name="targetFunction">optimized target function</param>
		protected Optimization(RealFunction targetFunction) : this(targetFunction, new Vector()) { }

		/// <summary>
		/// Constructs an optimization method for use.
		/// </summary>
		/// <param name="targetFunction">optimized target function</param>
		/// <param name="init">initial parameter values</param>
		protected Optimization(RealFunction targetFunction, Vector init)
		{
			this.targetfunc = targetFunction;
			this.initialParams = (Vector)init.Clone();
		}

		/// <summary>
		/// Constructs an optimization method for use.
		/// </summary>
		/// <param name="targetFunction">optimized target function</param>
		/// <param name="ceilings">parameter ceiling values</param>
		/// <param name="floors">parameter floor values</param>
		/// <param name="init">initial parameter values</param>
		protected Optimization(RealFunction targetFunction, Vector ceilings, Vector floors, Vector init)
		{
			this.targetfunc = targetFunction;
			this.SetCeiling(ceilings);
			this.SetFloor(floors);
			this.initialParams = (Vector)init.Clone();
		}

		#endregion

		#region Public

		/// <summary>
        /// Event handler for Iteration notifications.
        /// </summary>
        /// <param name="sender">object that sent the event</param>
        /// <param name="e">event arguments</param>
        public delegate void IterationEventHandler(object sender, IterationEventArgs e);

		/// <summary>
		/// Stop condition delegate. These are called after each iteration, passing the
		/// current 'best' parameters reached in the optimization as an argument.
		/// </summary>
		/// <param name="v">Current parameter values</param>
		/// <returns>True, if stop condition has been met</returns>
		public delegate bool StopCondition(Vector v);

		/// <summary>
		/// Optimization constraint delegate. These dataFunctions are called before calling the target function
		/// to check that the current parameter values are within limits. The parameter values are passed by reference
		/// and the constraint methods may modify them, typically moving them back within the limits. The
		/// target function is called only with these modified parameter values, but the result obtained is
		/// considered to be the target dataFunctions value at the original, unmodified, point.
		/// </summary>
		/// <param name="v">parameters to check</param>
		/// <returns>Should return negative value if within limits, positive if outside.</returns>
		public delegate double Constraint(ref Vector v);

		/// <summary>
        /// Event that is sent when one iteration of optimization method is passed.
        /// </summary>
        public event IterationEventHandler IterationNotify;

        /// <summary>
        /// For calling the target function outside this class.
        /// Masks the original target function and the modified, constrained function, as a single function
        /// </summary>
        public RealFunction ConstrainedTargetFunction
        {
            get { return this.ConstrainedFunction; }
        }

        /// <summary>
        /// Get/Set optimization starting parameters.
        /// Note: setting new initial parameters between iterations may result in
        /// unpredictable behavior.
        /// </summary>
		public virtual Vector InitialParams
		{
			get { return initialParams;}
			set
			{
				initialParams = (Vector)value.Clone();
				currentParams = (Vector)value.Clone();
			}
		}

		/// <summary>
        /// Number of iterations made so far
        /// </summary>
        public int Iterations
        {
            get { return iterations; }
        }

        /// <summary>
        /// Number of target function calls made so far
        /// </summary>
        public int FunctionCalls
        {
            get { return functioncalls; }
        }

		/// <summary>
		/// Initialization of the optimization process
		/// </summary>
		public abstract void Init();

		/// <summary>
		/// Add new constraint(s)
		/// </summary>
		/// <param name="cns">
        /// Constraint function(s)
		/// </param>
		public void AddConstraint(Constraint[] cns)
		{
            foreach (Constraint c in cns)
            {
                constraints.Add(c);
            }
		}

		/// <summary>
		/// Remove constraint(S)
		/// </summary>
		/// <param name="cns">
        /// Constraint function(s) to remove
		/// </param>
		public void RemoveConstraint(Constraint[] cns)
		{
            foreach (Constraint c in cns)
            {
                constraints.Remove(c);
            }
		}

		/// <summary>
		/// Clear all constraints
		/// </summary>
		public void ClearConstraints()
		{
			constraints.Clear();
		}

		/// <summary>
        /// Add stop condition
        /// </summary>
        /// <param name="stopcondition">Stop condition function</param>
        public void AddStop(StopCondition stopcondition)
        {
            if (!stopConditions.Contains(stopcondition)) stopConditions.Add(stopcondition);
        }

		/// <summary>
		/// Add stop conditions
		/// </summary>
		/// <param name="scs">
        /// Stop condition functions
		/// </param>
		public void AddStop(StopCondition[] scs)
		{
            foreach (StopCondition s in scs)
            {
                if(!stopConditions.Contains(s)) stopConditions.Add(s);
            }
		}

		/// <summary>
		/// Remove stop condition(s)
		/// </summary>
		/// <param name="scs">
        /// Stop condition(s) to remove
		/// </param>
		public void RemoveStop(StopCondition[] scs)
		{
            foreach (StopCondition s in scs)
            {
                stopConditions.RemoveAll(delegate(StopCondition sc) { return s.Equals(sc); });
            }
		}

		/// <summary>
		/// Clear all stop conditions
		/// </summary>
		public void ClearStops()
		{
			stopConditions.Clear();
		}

        /// <summary>
        /// Get all stop conditions
        /// </summary>
        /// <returns>Array of stop condition dataFunctions</returns>
        public StopCondition[] GetStops()
        {
            return stopConditions.ToArray();
        }

        /// <summary>
        /// Get all Constraint dataFunctions
        /// </summary>
        /// <returns>Array of Constraint dataFunctions</returns>
        public Constraint[] GetConstraints()
        {
            return constraints.ToArray();
        }

        /// <summary>
        /// Set target function to optimize
        /// </summary>
        /// <param name="function">Real function</param>
        public void SetTargetFunction(RealFunction function)
		{
			targetfunc = function;
		}

		/// <summary>
		/// Performs a single iteration of optimization.
		/// </summary>
		/// <returns>true, if any supplied stop condition is fulfilled.</returns>
        public bool Iterate()
        {
            // Update current parameters
            Vector newparams = Step();
            delta = currentParams - newparams;
            currentParams = newparams;

            // Update iteration count
            iterations++;

            // notify clients
            if(IterationNotify != null)
                IterationNotify.Invoke(this, new IterationEventArgs(currentParams));

            // Check if stop conditions are fulfilled and return
            return Stop();
        }

		/// <summary>
		/// Runs n iterations of optimization, or until a stop condition is reached
		/// </summary>
		/// <param name="n">
        /// Maximum iterations
		/// </param>
		/// <returns>
        /// True, if stop condition was reached, false otherwise.
		/// </returns>
		public bool Iterate(int n)
		{
			bool stop_reached = false;
			for(int i = 0; i<n; i++)
			{
				stop_reached = Iterate();
				if( stop_reached ) break;
			}
			return stop_reached;
		}
		
		/// <summary>
		/// Iterates until a stop condition is reached
		/// </summary>
		public void IterateUntilStop()
		{
            while (!Iterate())
            {
            }
		}

        /// <summary>
        /// Get the current candidate for minimum
        /// </summary>
        /// <returns>Current minimum</returns>
        public ParameterList GetMinimum()
        {
            return currentParams;
        }

        /// <summary>
        /// Target function value at current minimum
        /// </summary>
        /// <returns>Target function value at minimum</returns>
        public double AtMinimum()
        {
            return targetfunc(GetMinimum());
        }

        /// <summary>
        /// Set upper limit for iterations
        /// </summary>
        /// <param name="n">Upper limit of iterations to run</param>
        public void SetMaxIterations(int n)
        {
            stopConditions.Add(delegate(Vector v) { return this.iterations >= n; });
        }

        /// <summary>
        /// Set a stop condition: optimization stops at first point p,
        /// where TargetFunction(p) is less than 'd'
        /// </summary>
        /// <param name="d">Upper limit for acceptable funtion value at minimum</param>
        public void SetTolerance(double d)
        {
            stopConditions.Add(delegate(Vector v) { return this.targetfunc(v) < d; });
        }

        /// <summary>
        /// Set stop condition: if the parameter vector has changed less than 'd'
        /// (measured in euclidean distance) in one iteration, optimization stops.
        /// </summary>
        /// <param name="d">Minimal change of parameters</param>
        public void SetConvergence(double d)
        {
            if (d < 0) throw new OptimizationException("Can't set negative convergence criterion");
            stopConditions.Add(delegate(Vector c) { return Vector.Norm(delta) < d; });
        }

        /// <summary>
        /// Set constraint: upper limit (inclusive) for a single parameter.
        /// This is a "hard" limit in the sense that it guarantees target function
        /// is not called with parameters outside this limit. Instead target function is
        /// called with value at the limit, plus a penalty function.
        /// The default penalty function rises very rapidly to Double.Max, but is not a "vertical"
        /// wall to enable all optimization methods to work correctly near the limits of the allowed
        /// area. If needed, different limit types can be supplied via delegates to the constraint
        /// mecahnism.
        /// (See the source code for an example)
        /// </summary>
        /// <param name="parameter">Index of constrained parameter</param>
        /// <param name="limit">Upper limit (inclusive)</param>
        public void SetCeiling(int parameter, double limit)
        {
            constraints.Add(
                delegate(ref Vector v)
                {
                    // If dif < 0, v is inside the limits
                    double dif = v[parameter] - limit;

                    // Define a penalty function in two parts.
                    // This penalty function will get values between 0 and Double.Max outside
                    // the limits, but it's derivative is always > 0 to ensure that all methods
                    // work correctly.
                    if (dif > 1.0)
                    {
                        v[parameter] = limit;
                        dif = Double.MaxValue * (1.0 - 1.0 / (dif * dif + 1.0));
                    }
                    else if (dif > 0.0)
                    {
                        v[parameter] = limit;
                        dif = 0.5 * dif * dif * Double.MaxValue;
                    }
                    return dif;
                }
            );
        }

        /// <summary>
        /// Set constraint: upper limits (inclusive) for all parameters.
        /// See SetCeiling(int parameter, double limit) for details.
        /// </summary>
        /// <param name="limits">Vector of upper limits</param>
        public void SetCeiling(Vector limits)
        {
            for (int i = 0; i < limits.Length; i++)
            {
                SetCeiling(i, limits[i]);
            }
        }

        /// <summary>
        /// Set constraint: lower limit (inclusive) for a single parameter.
        /// See SetCeiling(int parameter, double limit) for details.
        /// </summary>
        /// <param name="parameter">Index of constrained parameter</param>
        /// <param name="limit">Lower limit (inclusive)</param>
        public void SetFloor(int parameter, double limit)
        {
            constraints.Add(
                delegate(ref Vector v)
                {
                    double dif = limit - v[parameter];

                    // Define a penalty function in two parts.
                    // Thi penalty function will get values between 0 and Double.Max outside
                    // the limits, but it's derivative is always > 0 to ensure that all methods
                    // work correctly.
                    if (dif > 1.0)
                    {
                        v[parameter] = limit;
                        dif = Double.MaxValue * (1.0 - 1.0 / (dif * dif + 1.0));
                    }
                    else if (dif > 0.0)
                    {
                        v[parameter] = limit;
                        dif = 0.5 * dif * dif * Double.MaxValue;
                    }
                    return dif;
                }
            );
        }

        /// <summary>
        /// Set constraint: lower limits (inclusive) for all parameters.
        /// See SetFloor(int parameter, double limit) for details.
        /// </summary>
        /// <param name="limits">Vector of lower limits</param>
        public void SetFloor(Vector limits)
        {
            for (int i = 0; i < limits.Length; i++)
            {
                SetFloor(i, limits[i]);
            }
        }

        /// <summary>
        /// Set a constraint: allowed parameter range (inclusive)
        /// See SetCeiling(int parameter, double limit) and
        /// SetFloor(int parameter, double limit) for details.
        /// </summary>
        /// <param name="parameter">Index of constrained parameter</param>
        /// <param name="floor">Lower limit</param>
        /// <param name="ceiling">Upper limit</param>
        public void SetRange(int parameter, double floor, double ceiling)
        {
            SetCeiling(parameter, ceiling);
            SetFloor(parameter, floor);
        }

        /// <summary>
        /// Set a constraint: allowed parameter ranges (inclusive)
        /// See SetCeiling(int parameter, double limit) and
        /// SetFloor(int parameter, double limit) for details.
        /// </summary>
        /// <param name="floor">Vector of lower limits</param>
        /// <param name="ceiling">Vector of upper limits</param>
        public void SetRange(Vector floor, Vector ceiling)
        {
            SetCeiling(ceiling);
            SetFloor(floor);
        }

        /// <summary>
        /// Set a constraint: limit parameters to 'radius' (euclidean) distance from 'center'.
        /// This is an example of "soft" limits; parameters may move outside the radius, but
        /// a penalty is added to the target function value.
        /// </summary>
        /// <param name="center">Center point</param>
        /// <param name="radius">Max distance allowed (inclusive)</param>
        public void SetRadius(Vector center, double radius)
        {
            constraints.Add(
                delegate(ref Vector v) {
                    Vector diff = v - center;
                    return radius * radius - diff * diff;
                });
		}

		/// <summary>
		/// Clone this Optimization component
		/// </summary>
		/// <returns></returns>
		public object Clone()
		{
			return this.MemberwiseClone();
		}

		#endregion
	}
}
