/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
    /// <summary>
    /// Direction Set or Powell's Method in Multidimensions.
    /// Minimization routine.
    /// This is the method of choice when you cannot easily
    /// calculate derivatives, and is not necessarily to be sneered at even if you can. 
    /// Although derivatives are not needed, the method does require
    /// a one-dimensional minimization sub-algorithm.
    /// This algorithm utilizes Brent’s method.
    /// Storage is of order N^2.
    /// </summary>
    public class PowellBrent : Optimization
    {
        /// <summary>
        /// if delta is below this, the parameter is fixed
        /// </summary>
		[Input(Name = @"Fixed tolerance", Description = @"If delta is below fixed tolerance, the parameter is fixed for the remainder of the iteration", Optional = true)]
        public double FixedTolerance = 1.0e-20;

        /// <summary>
        /// The minimum difference between last optimum value.
        /// if current iteration and last iteration difference is 
        /// below this, the algorithm stops.
        /// </summary>
		[Input(Name = @"Minimum tolerance", Description = @"If the difference to the previous iteration is smaller than minimum tolerance, the current iteration stops", Optional = true)]
		public double MinimumTolerance = 1.0e-10;

		/// <summary>
		/// Tolerance value used by the Brent line algorithm
		/// </summary>
		[Input(Name = @"Brent tolerance", Description = @"Tolerance value used by the Brent algorithm", Optional = true)]
		public double BrentTolerance = 2.0e-4;

		/// <summary>
		/// Powell delta vector (defines initial search directions)
		/// </summary>
		[Initialization(Name = @"Powell delta", Optional = true)]
		public double[] PowellDelta = null;

		/// <summary>
		/// The direction to go.
		/// </summary>
		private Vector dir;

		/// <summary>
        /// true if parameter index in direction is fixed
        /// </summary>
        private bool[] fiXed;

        /// <summary>
        /// The direction NxN matrix.
        /// </summary>
        private double[,] dirmat;

		/// <summary>
		/// Current optimal point
		/// </summary>
		private Vector p;

		/// <summary>
        /// Previus optimal point
        /// </summary>
        private double[] pprev;

        /// <summary>
        /// Previus previus optimal point.
        /// </summary>
        private double[] avg;

        /// <summary>
        /// Saves function value at initial point. 
        /// </summary>
        private double fvalue;

        /// <summary>
        /// Gets a multivariable funtion as a one dimensional "cut"
        /// </summary>
        private Linemethod linemin;

        /// <summary>
        /// Original initial direction matrix values
        /// </summary>
        private double[] originalDelta;

		/// <summary>
		/// Default constructor.
		/// </summary>
		public PowellBrent() : base() { }

        /// <summary>
        /// Initialization of Powell-Brent optimization.
        /// </summary>
        public override void Init()
        {
			p = (Vector)InitialParams.Clone();

			originalDelta = PowellDelta;

			if (PowellDelta == null)
			{
				PowellDelta = Vector.Fill(initialParams.Length, 0.001);
			}

			int parNr = initialParams.Length;
			this.dir = new double[parNr];
            this.pprev = new double[parNr];
            this.fiXed = new bool[parNr];
            this.dirmat = new double[parNr, parNr];

            // Checks which paratemers are fixed.
            for (int i = 0; i < parNr; i++)
            {
                this.fiXed[i] = (Math.Abs(PowellDelta[i]) < FixedTolerance);
            }

            // Initiates matrix for directions.
            for (int i = 0; i < parNr; i++)
            {
                for (int j = 0; j < parNr; j++)
                {
                    dirmat[i, j] = (i == j) ? PowellDelta[i] : 0.0;
                }
            }
            this.fvalue = ConstrainedTargetFunction(InitialParams);
			this.linemin = new Linemethod(delegate(double t) { return ConstrainedTargetFunction(this.p + t * dir); }, BrentTolerance);

            // Saves the initial point
            pprev = (Vector)InitialParams.Clone();
        }

        /// <summary>
        /// Single minimization step; called from Optimization.Iterate()
        /// </summary>
        /// <returns>Best found parameters.</returns>
        protected override Vector Step()
        {
			// current function value
            double fprev1 = fvalue;
            double fprev2;
            // the best index in direction set 
            int ibig = 0;
            // the best difference
            double newdel, del = 0.0;

            // Loops over all directions (matrix row) in the set.
            // After we know the best (ibig) direction to go.
			for (int row = 0; row < p.Length; row++)
            {
                // Leave the fixed parameters alone.
                if (!fiXed[row])
                {
					for (int col = 0; col < p.Length; col++)
                    {
                        // yes, we don't move along fixed component.
                        dir[col] = fiXed[col] ? 0.0 : dirmat[col, row];
                    }

                    // Store the latest.
                    fprev2 = fvalue;

                    // minimazes along direction and get new minimum value
					fvalue = linemin.LineMinimum(ref p, ref dir);

                    // Stores the best index
                    newdel = Math.Abs(fprev2 - fvalue);
                    if (newdel > del)
                    {
                        del = newdel;
                        ibig = row;
                    }
                }
            }

            // Check if the new direction is better
            double c1 = 2.0 * Math.Abs(fprev1 - fvalue);
            double c2 = MinimumTolerance * (Math.Abs(fprev1) + Math.Abs(fvalue));
			if (c1 > c2)
			{
				// and the average direction moved.
				avg = 2 * p - pprev;
				dir = p - pprev;

				// saves the old starting point
				pprev = p;

				// gets the extrapolated function value.
				fprev2 = ConstrainedTargetFunction(avg);

				// if we found better minimum.
				if (fprev2 < fprev1)
				{
					double sq1 = fprev1 - fvalue - del;
					double sq2 = fprev1 - fprev2;
					double t = 2.0 * (fprev1 - 2.0 * fvalue + fprev2) * sq1 * sq1 - del * sq2 * sq2;

					if (t < 0.0)
					{
						this.fvalue = linemin.LineMinimum(ref p, ref dir);

						// updates the direction matrix 
						// move to the minimum of the new direction,
						// and save the new direction.
						for (int i = 0; i < p.Length; i++)
						{
							dirmat[i, ibig] = dirmat[i, p.Length - 1];
							dirmat[i, p.Length - 1] = dir[i];
						}
					}
				}
			}

            // the next iteration
            return p;
        }
    }
}
