/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.ROI
{
    /// <summary>
    /// Creates ellipse shaped regions of interests. Reference point is located at the center of the ellipse. 
    /// Radius is defined with two vectors. The ellipse is currently defined only in transaxial plane.
    /// </summary>
    public class EllipseROI : ROI, IGeometricalROI
    {
        /// <summary>
        /// Width in units.
        /// </summary>
        /// <seealso cref="ROI.Unit"/>
        private double width = 0;
        /// <summary>
        /// Height in units.
        /// </summary>
        /// <seealso cref="ROI.Unit"/>
        private double height = 0;
        /// <summary>
        /// Angle in radians
        /// </summary>
        private double angle = 0;

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        /// <param name="x">Location x</param>
        /// <param name="y">Location y</param>
		/// <param name="z">Location z</param>
		/// <param name="width">Ellipse width</param>
        /// <param name="height">Ellipse height</param>
        public EllipseROI(double x, double y, double z, double width, double height)
            : base(ROI_Type.Ellipse)
        {
            this.Location = new Point(x,y,z);
            this.width = width;
            this.height = height;
        }

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        public EllipseROI()
            : base(ROI_Type.Ellipse)
        {
        }

        /// <summary>
        /// Creates replica of EllipseROI
        /// </summary>
        public override object Clone()
        {
            EllipseROI r = new EllipseROI();
            r.width = this.width;
            r.height = this.height;
            r.angle = this.angle;
            r.Location = new Point(this.Location);
            r.ROIName = this.ROIName;
            r.Unit = this.Unit;
            return r;
        }

        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Fill is done in Bresenham style algorithm.
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="method">filling method</param>
        public override void Fill(ref TPClib.Image.MaskImage mask, Fill_Method method)
        {
            //resolving number of points according to width and height could be added here
            //currently using 100 steps
            double npoints = 100.0;
            //list of edge coordinates for left and right side
            List<Point> points = new List<Point>();
            // Angle is given by Degree Value
            double sinbeta = Math.Sin(angle);
            double cosbeta = Math.Cos(angle);
            double sinalpha;
            double cosalpha;
            int X;
            int Y;
            int j = 0;
            //angle that is iterated
            double angle_i = 0;
            double step = (2*Math.PI / npoints);
            double PI_times_2 = Math.PI*2.0;
            //resolve points at left and right side of ellipse
            for (angle_i = 0.0; angle_i < PI_times_2; angle_i += step)
            {
                sinalpha = Math.Sin(angle_i);
                cosalpha = Math.Cos(angle_i);
                X = (int)(Location.X + (width * cosalpha * cosbeta - height * sinalpha * sinbeta));
                Y = (int)(Location.Y + (width * cosalpha * sinbeta + height * sinalpha * cosbeta));
                points.Add(new Point(X, Y, Location.Z));
            }
            //sort according to y then according to x
            points.Sort(new Comparison<Point>(delegate(Point a, Point b) {
                if (a.Y < b.Y) return -1;
                else if (a.Y > b.Y) return 1;
                if (a.X < b.X) return -1;
                else if (a.X > b.X) return 1;
                else return 0;
            }));
            //create list of voxels at both sides
            List<Point> l_points = new List<Point>();
            List<Point> r_points = new List<Point>();
            for (int i = 0; i < points.Count; i++)
            {
                //take leftmost and right most points
                j = (int)points[i].Y;
                while (points[i++].Y == j);
            }
            //remove duplicates 
            //mask voxels at scanlines
            int lo;
            int hi;
            for (int i = 0; i < points.Count; i++)
            {
                lo = (int)l_points[i].X + (int)Location.Z * mask.DimX * mask.DimY + (int)l_points[i].Y * mask.DimX;
                hi = (int)r_points[i].X + (int)Location.Z * mask.DimX * mask.DimY + (int)r_points[i].Y * mask.DimX;
                for (j = lo; j <= hi; j++) mask[j] = 1;
            }
        }
        /// <summary>
        /// Gets bounding box of circle
        /// </summary>
        /// <returns>bounds of this ROI</returns>
        public override Limits GetBoundingBox()
        {
            return new Limits(new double[] { this.Location.X - width / 2.0, this.Location.Y - height / 2.0, this.Location.Z },
                              new double[] { this.Location.X + width / 2.0, this.Location.Y + height / 2.0, this.Location.Z });
        }
        /// <summary>
        /// Width of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        double IGeometricalROI.Width
        {
            get
            {
                return width;
            }
        }
        /// <summary>
        /// Height of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        double IGeometricalROI.Height
        {
            get
            {
                return height;
            }
        }
    }
}
