/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Interface for objects that can be masked (a binary mask representation can be given). 
    /// </summary>
    public abstract class Maskable
    {
        /// <summary>
        /// The method which is used when the mask is calculated. If the method is Imadeus or ImageTool,
        /// points are used as in the trace roi. If Shape, the mask area is calculated straight from
        /// shape of current class, for example ellipses or circles arc or rectangles dimensions.
        /// In that case points are not used.
        /// </summary>
        public enum Fill_Method
        {
            /// <summary>
            /// Most ROI classes have possibility to calculate the ROI area straight from shape. For example
            /// CircleROI can calculate the ROI matemathically from x,y and radius. If method is Shape, the
            /// ROI class will use its own method to calculate ROI. This usually ignores the
            /// transforming ( flipping, zooming, rotating )
            /// </summary>
            Shape,
            /// <summary>
            /// The ROI lines will be calculated with Bresenham's algorithm run into both directions. 
            /// Every shape will be converted to points when this method is used.
            /// </summary>
            TwoWayBresenham,
            /// <summary>
            /// The ROI will be calculated with ImageTool masking algorithm. Every shape will be converted to points
            /// when this method is used.
            /// </summary>
            ImageTool
        }
        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Origo is considered to be located at bottom left low corner of mask. 
        /// Two-way Bresenham algorithm is used by default.
        /// </summary>
        /// <param name="maskarray">mask data array</param>
        /// <param name="dim">dimensions for mask data</param>
        /// <param name="method">filling method</param>
        public void Fill(ref byte[] maskarray, IntLimits dim, Fill_Method method)
        {
            if (dim.GetProduct() != maskarray.Length)
                throw new ArgumentException("Dimensions " + dim + " do not match mask length " + maskarray.Length);
            MaskImage maskimage = new MaskImage(dim);
            Fill(ref maskimage, method);
            maskimage.CopyTo(maskarray);
        }
        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Origo is considered to be located at bottom left low corner of mask. 
        /// Two-way Bresenham algorithm is used by default.
        /// </summary>
        /// <param name="maskarray">mask data array</param>
        /// <param name="dim">dimensions for mask data</param>
        public void Fill(ref byte[] maskarray, IntLimits dim)
        {
            if (dim.GetProduct() != maskarray.Length)
                throw new ArgumentException("Dimensions "+dim+" do not match mask length "+maskarray.Length);
            MaskImage maskimage = new MaskImage(dim);
            Fill(ref maskimage);
            maskimage.CopyTo(maskarray);
        }
        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Origo is considered to be located at bottom left low corner of mask. 
        /// Two-way Bresenham algorithm is used by default.
        /// </summary>
        /// <param name="mask">target mask image</param>
        public void Fill(ref MaskImage mask)
        {
            Fill(ref mask, Fill_Method.TwoWayBresenham);
        }
        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Origo is considered to be located at bottom left low corner of mask 
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="method">filling method</param>
        public abstract void Fill(ref MaskImage mask, Fill_Method method);
        /// <summary>
        /// Returns boundings of this maskable item.
        /// </summary>
        /// <returns>lower and upper limits of this maskable item</returns>
        public abstract Limits GetBoundingBox();
    }
}
