/********************************************************************************
*                                                                               *
*  TPCDicom 0.1 Dicom I/O library                                               *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
*  TPCDicom is based on the openDicom# 0.1.1 library.                           *
*                                                                               *
*  *** openDicom license ***                                                    *
*                                                                               *
*  openDICOM.NET openDICOM# 0.1.1                                               *
*                                                                               *
*  openDICOM# provides a library for DICOM related development on Mono.         *
*  Copyright (C) 2006-2007  Albert Gnandt                                       *
*                                                                               *
*  This library is free software; you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License as published by     *
*  the Free Software Foundation; either version 2.1 of the License, or (at      *
*  your option) any later version.                                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public       *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this library; if not, write to the                                *
*  Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA      *
*  02110-1301 USA                                                               *
*                                                                               *
********************************************************************************/

using System;
using TPCDicom.DataStructure;


namespace TPCDicom.Encoding
{

    /// <summary>
    ///     This class represents the specific DICOM VR Floating Point Single (FL).
    /// </summary>
    public sealed class FloatingPointSingle: ValueRepresentation
    {
        /// <summary>
        /// Constructor with tag value
        /// </summary>
        /// <param name="tag">corresponding tag value</param>
        public FloatingPointSingle(Tag tag) : base("FL", tag) { }

        /// <summary>
        /// String description about this object.
        /// </summary>
        /// <returns></returns>
        public override string ToLongString()
        {
            return "Floating Point Single (FL)";
        }

        /// <summary>
        /// Converts byte array. 
        /// </summary>
        /// <param name="bytes">input array</param>
        /// <returns>array of float values containing FL value</returns>
        protected override Array DecodeImproper(byte[] bytes)
        {
            byte[][] multiValue = ToImproperMultiValue(bytes, 4);
            float[] number = new float[multiValue.Length];
            for (int i = 0; i < number.Length; i++)
                number[i] = BitConverter.ToSingle(
                    TransferSyntax.CorrectByteOrdering(multiValue[i]), 0);
            return number;
        }

        /// <summary>
        /// Converts byte array. 
        /// </summary>
        /// <param name="bytes">input array</param>
        /// <returns>array of float values containing FL value</returns>
        protected override Array DecodeProper(byte[] bytes)
        {
            byte[][] multiValue = ToProperMultiValue(bytes, 4);
            if (bytes.Length == 4 * multiValue.Length)
            {
                float[] number = new float[multiValue.Length];
                for (int i = 0; i < number.Length; i++)
                    number[i] = BitConverter.ToSingle(
                        TransferSyntax.CorrectByteOrdering(multiValue[i]), 0);
                return number;
            }
            else
                throw new EncodingException(
                    "A value of multiple 4 bytes is only allowed.", Tag,
                    Name + "/value.Length", bytes.Length.ToString());
        }
    }

}
