/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;

namespace TPClib.Common
{
	/// <summary>
	/// Series of time values
	/// </summary>
	public class TimeSeries : ISeries
	{
		#region Protected/private

		/// <summary>
		/// Start times.
		/// </summary>
		protected double[] startTimes;

		/// <summary>
		/// End times, if present.
		/// </summary>
		protected double[] endTimes = null;

		#endregion

		#region Public

		/// <summary>
		/// Enumeration of supported time units.
		/// </summary>
		public enum TimexUnit : uint
		{
			/// <summary>
			/// Milliseconds
			/// </summary>
			Millisecond = 1,
			/// <summary>
			/// Seconds (1000 ms)
			/// </summary>
			Second = 1000,
			/// <summary>
			/// Minutes (60000 ms)
			/// </summary>
			Minute = 60000,
			/// <summary>
			/// Hours (3600000 ms)
			/// </summary>
			Hour = 3600000
		}

		/// <summary>
		/// Series indexer.
		/// </summary>
		/// <param name="i">Index</param>
		/// <returns>Series element at the index</returns>
		public double this[int i] { get { return this[i, TimexUnit.Millisecond]; } set { this[i, TimexUnit.Millisecond] = value; } }

		/// <summary>
		/// Series indexer.
		/// </summary>
		/// <param name="i">Index</param>
		/// <param name="tu">Time unit of the return value (default is milliseconds)</param>
		/// <returns>Value at the index in the requested time unit</returns>
		public double this[int i, TimexUnit tu = TimexUnit.Millisecond]
		{
			get { return startTimes[i] / (uint)tu; }
			set { startTimes[i] = value * (uint)tu; }
		}

		/// <summary>
		/// Mid time series flag. True, if this series consists of frame mid times.
		/// </summary>
		public bool IsMidtimeSeries { get { return endTimes == null; } }

		/// <summary>
		/// Get all series values in an array.
		/// </summary>
		/// <returns>Array of values in milliseconds</returns>
		public double[] ToArray() { return startTimes; }

		/// <summary>
		/// Get all series values in an array.
		/// </summary>
		/// <param name="tu">Time unit</param>
		/// <returns>Array of values in the specified time unit</returns>
		public double[] GetValues(TimexUnit tu = TimexUnit.Millisecond)
		{
			return Array.ConvertAll<double, double>(startTimes, delegate(double f) { return f / (uint)tu; });
		}

		/// <summary>
		/// Get all series start times in an array.
		/// </summary>
		/// <param name="tu">Time unit</param>
		/// <returns>Array of start times in the specified time unit</returns>
		public double[] GetStartTimes(TimexUnit tu = TimexUnit.Millisecond)
		{
			return GetValues(tu);
		}

		/// <summary>
		/// Get all series end times in an array.
		/// </summary>
		/// <param name="tu">Time unit</param>
		/// <returns>Array of end times in the specified time unit</returns>
		public double[] GetEndTimes(TimexUnit tu = TimexUnit.Millisecond)
		{
			if (this.IsMidtimeSeries) return GetStartTimes();
			else return Array.ConvertAll<double, double>(endTimes, delegate(double f) { return f / (uint)tu; });
		}

		/// <summary>
		/// Get all series mid times in an array.
		/// </summary>
		/// <param name="tu">Time unit</param>
		/// <returns>Array of mid times in the specified time unit</returns>
		public double[] GetMidTimes(TimexUnit tu = TimexUnit.Millisecond)
		{
			double[] midtimes;
			if (this.IsMidtimeSeries) midtimes = GetStartTimes();
			else
			{
				midtimes = new double[this.Count];
				for (int i = 0; i < midtimes.Length; i++)
				{
					midtimes[i] = (startTimes[i] + endTimes[i]) / 2;
				}
			}
			return midtimes;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public static string TimeUnitToString(TimexUnit tu)
		{
			switch (tu)
			{
				case TimexUnit.Hour: return "h";
				case TimexUnit.Millisecond: return "ms";
				case TimexUnit.Minute: return "m";
				case TimexUnit.Second: return "s";
				default:
					return String.Empty;
			}
		}

		/// <summary>
		/// String representation.
		/// </summary>
		/// <returns>String representation of the series using milliseconds</returns>
		public override string ToString()
		{
			return ToString(TimexUnit.Millisecond);
		}

		/// <summary>
		/// String representation.
		/// </summary>
		/// <param name="tu">Timeunit</param>
		/// <returns>String representation of the series using a specified timeunit</returns>
		public string ToString(TimexUnit tu)
		{
			double[] starts = this.GetStartTimes(tu);
			double[] ends = this.GetEndTimes(tu);
			System.Text.StringBuilder sb = new System.Text.StringBuilder();
			sb.Append("[");
			int i = 0;
			while (i < this.Count)
			{
				if (this.IsMidtimeSeries) sb.AppendFormat("{0}", starts[i]);
				else sb.AppendFormat("({0}-{1})", starts[i], ends[i]);
				if (++i < this.Count) sb.Append(";");
			}
			sb.AppendFormat("] ({0})", TimeUnitToString(tu));
			return sb.ToString();
		}

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor. Creates a mid time series.
		/// </summary>
		/// <param name="t">Array of values</param>
		/// <param name="tu">Time unit of the values</param>
		public TimeSeries(double[] t, TimexUnit tu = TimexUnit.Millisecond)
		{
			startTimes = Array.ConvertAll<double, double>(t, delegate(double d) { return d * (uint)tu; });
		}

		/// <summary>
		/// Constructor. Creates a mid time series.
		/// </summary>
		/// <param name="t">Array of values</param>
		/// <param name="tu">Time unit of the values</param>
		public TimeSeries(float[] t, TimexUnit tu = TimexUnit.Millisecond)
		{
			startTimes = Array.ConvertAll<float, double>(t, delegate(float f) { return f * (uint)tu; });
		}

		/// <summary>
		/// Constructor. Creates a series with start and end times.
		/// </summary>
		/// <param name="st">Array of start times</param>
		/// <param name="et">Array of end times</param>
		/// <param name="tu">Time unit of the time values</param>
		public TimeSeries(double[] st, double[] et, TimexUnit tu = TimexUnit.Millisecond)
		{
			startTimes = Array.ConvertAll<double, double>(st, delegate(double f) { return f * (uint)tu; });
			endTimes = Array.ConvertAll<double, double>(et, delegate(double f) { return f * (uint)tu; });
		}

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="t">Array of data values</param>
		public TimeSeries(params double[] t) : this(t, TimexUnit.Millisecond) { }

		#endregion

		#region Type conversions

		/// <summary>
		/// Implicit cast operator to array of floats
		/// </summary>
		/// <param name="ts">Time series</param>
		/// <returns>Array of floats with series times in milliseconds</returns>
		public static implicit operator float[](TimeSeries ts)
		{
			return Array.ConvertAll<double, float>(ts.startTimes, Convert.ToSingle);
		}

		/// <summary>
		/// Implicit cast operator to array of doubles
		/// </summary>
		/// <param name="ts">Time series</param>
		/// <returns>Array of doubles with series times in milliseconds</returns>
		public static implicit operator double[](TimeSeries ts)
		{
			return ts.ToArray();
		}

		/// <summary>
		/// Implicit cast operator to TimeSeries from an array of floats
		/// </summary>
		/// <param name="f">Array of float values, values in milliseconds</param>
		/// <returns>Time series</returns>
		public static implicit operator TimeSeries(float[] f)
		{
			return new TimeSeries(Array.ConvertAll<float, double>(f, Convert.ToDouble));
		}

		/// <summary>
		/// Implicit cast operator to TimeSeries from an array of doubles
		/// </summary>
		/// <param name="d">Array of double values, values in milliseconds</param>
		/// <returns>Time series</returns>
		public static implicit operator TimeSeries(double[] d)
		{
			return new TimeSeries(d);
		}

		#endregion

		#region IList interface

		IEnumerator<double> IEnumerable<double>.GetEnumerator()
		{
			return (startTimes as IEnumerable<double>).GetEnumerator();
		}

		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return startTimes.GetEnumerator();
		}

		/// <summary>
		/// Number of items in this series.
		/// </summary>
		public int Count { get { return startTimes.Length; } }

		/// <summary>
		/// Index of an item in this list.
		/// </summary>
		/// <param name="item">Item to look for</param>
		/// <returns>Index for the item</returns>
		public int IndexOf(double item)
		{
			return Array.IndexOf(startTimes, item);
		}

		/// <summary>
		/// Insert a new item to this list at a specified index.
		/// </summary>
		/// <param name="index">Index</param>
		/// <param name="item">Item to add</param>
		public void Insert(int index, double item)
		{
			startTimes[index] = item;
		}

		/// <summary>
		/// Remove the item at a specified index.
		/// </summary>
		/// <param name="index">Index</param>
		public void RemoveAt(int index)
		{
			(startTimes as IList<double>).RemoveAt(index);
			if (!this.IsMidtimeSeries) (endTimes as IList<double>).RemoveAt(index);
		}

		/// <summary>
		/// Add a new item at the end of the list.
		/// </summary>
		/// <param name="item">Item to add</param>
		public void Add(double item)
		{
			throw new NotImplementedException();
		}

		/// <summary>
		/// Clear this list.
		/// </summary>
		public void Clear()
		{
			throw new NotImplementedException();
		}

		/// <summary>
		/// Check if this list contains an item.
		/// </summary>
		/// <param name="item">Item</param>
		/// <returns>True, if this list contains the item</returns>
		public bool Contains(double item)
		{
			throw new NotImplementedException();
		}

		/// <summary>
		/// Copy all values in this list to an array.
		/// </summary>
		/// <param name="array">Destination array</param>
		/// <param name="arrayIndex">Start index in the array</param>
		public void CopyTo(double[] array, int arrayIndex)
		{
			Array.Copy(startTimes, 0, array, arrayIndex, startTimes.Length);
		}

		/// <summary>
		/// Read only list.
		/// </summary>
		public bool IsReadOnly
		{
			get { return false; }
		}

		/// <summary>
		/// Remove an item in this list.
		/// </summary>
		/// <param name="item">Item to remove</param>
		/// <returns></returns>
		public bool Remove(double item)
		{
			throw new NotImplementedException();
		}

		#endregion
	}

	/// <summary>
	/// Series of data values
	/// </summary>
	public class DataSeries : ISeries
	{
		#region Protected/private

		/// <summary>
		/// Data values.
		/// </summary>
		protected double[] data;

		/// <summary>
		/// Data unit of this series.
		/// </summary>
		protected Unit dataUnit = DataUnit.Unknown;

		#endregion

		#region Public

		/// <summary>
		/// Return all values in this series
		/// </summary>
		public ISeries Values { get { return this; } }

		/// <summary>
		/// Get all series values in an array.
		/// </summary>
		/// <returns>Array of values in the base unit</returns>
		public double[] ToArray() { return this; }

		/// <summary>
		/// Length of this series.
		/// </summary>
		public int Length { get { return data.Length; } }

		/// <summary>
		/// 
		/// </summary>
		public Unit Unit { get { return dataUnit; } }

		/// <summary>
		/// Indexer.
		/// </summary>
		/// <param name="i">Index</param>
		/// <returns>Element at the index</returns>
		public double this[int i] { get { return data[i]; } set { data[i] = value; } }

		/// <summary>
		/// Indexer.
		/// </summary>
		/// <param name="i">Index</param>
		/// <param name="u">Scale unit</param>
		/// <returns>Element at the index scaled with the specified scale unit</returns>
		public double this[int i, Unit u]
		{
			get
			{
				if (u == dataUnit) return data[i];
				else return Unit.ConvertValue(dataUnit, u, data[i]);
			}
			set
			{
				if (u == dataUnit) data[i] = value;
				else data[i] = Unit.ConvertValue(u, dataUnit, value);
			}
		}

		/// <summary>
		/// Get all values in this series in an array
		/// </summary>
		/// <returns>Array of the series values</returns>
		public double[] GetValues() { return GetValues(dataUnit); }

		/// <summary>
		/// Get all values in this series in an array
		/// </summary>
		/// <param name="u">Scale unit</param>
		/// <returns>Array of the series values, scaled to the specified unit</returns>
		public double[] GetValues(Unit u)
		{
			double[] d;
			if (u == dataUnit)
			{
				d = new double[data.Length];
				Array.Copy(data, d, data.Length);
			}
			else
			{
				d = Array.ConvertAll<double, double>(data, delegate(double f) { return Unit.ConvertValue(dataUnit, u, f); });
			}
			return d;
		}

		/// <summary>
		/// String representation.
		/// </summary>
		/// <returns>String representation of the series using milliseconds</returns>
		public override string ToString()
		{
			return ToString(dataUnit);
		}

		/// <summary>
		/// String representation.
		/// </summary>
		/// <param name="u">Timeunit</param>
		/// <returns>String representation of the series using a specified scaling unit</returns>
		public virtual string ToString(Unit u)
		{
			System.Text.StringBuilder sb = new System.Text.StringBuilder();
			sb.Append("[");
			int i = 0;
			while (i < this.Length)
			{
				sb.AppendFormat("{0}", this[i,u]);
				if (++i < this.Length) sb.Append(";");
			}
			sb.Append("]");
			if(u.UnitString != String.Empty) sb.AppendFormat(" ({0})", u.UnitString);
			return sb.ToString();
		}

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">ISeries object containing the data values</param>
		public DataSeries(ISeries s) : this(s.ToArray()) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="d">Array of data values</param>
		/// <param name="u">Base unit for this series</param>
		public DataSeries(double[] d, Unit u)
		{
			dataUnit = u;
			data = new double[d.Length];
			Array.Copy(d, data, d.Length);
		}

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="d">Array of data values</param>
		public DataSeries(params double[] d) : this(d, DataUnit.Unknown) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="n">Number of elements</param>
		public DataSeries(int n) : this(new double[n]) { }

		#endregion

		#region Type conversions

		/// <summary>
		/// Implicit cast operator to an array of doubles.
		/// </summary>
		/// <param name="ds">Data series</param>
		/// <returns>Array of doubles with the series values in the base unit</returns>
		public static implicit operator double[](DataSeries ds)
		{
			return ds.GetValues();
		}

		/// <summary>
		/// Implicit cast operator to an array of floats.
		/// </summary>
		/// <param name="ds">Data series</param>
		/// <returns>Array of floats with the series values in the base unit</returns>
		public static implicit operator float[](DataSeries ds)
		{
			return Array.ConvertAll<double, float>(ds.data, Convert.ToSingle);
		}

		/// <summary>
		/// Implicit cast operator to DataSeries from an array of doubles.
		/// </summary>
		/// <param name="d">Array of doubles, values in base unit</param>
		/// <returns>Data series</returns>
		public static implicit operator DataSeries(double[] d)
		{
			return new DataSeries(d);
		}

		/// <summary>
		/// Implicit cast operator to DataSeries from an array of floats.
		/// </summary>
		/// <param name="f">Array of floats, values in base unit</param>
		/// <returns>Data series</returns>
		public static implicit operator DataSeries(float[] f)
		{
			return new DataSeries(Array.ConvertAll<float,double>(f, Convert.ToDouble));
		}

		#endregion

		#region IList interface

		/// <summary>
		/// Get an enumerator for this list.
		/// </summary>
		/// <returns>Enumerator</returns>
		public IEnumerator<double> GetEnumerator()
		{
			return (data as IList<double>).GetEnumerator();
		}

		/// <summary>
		/// Get an enumerator for this list.
		/// </summary>
		/// <returns>Enumerator</returns>
		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return data.GetEnumerator();
		}

		/// <summary>
		/// Index of an item in this list.
		/// </summary>
		/// <param name="item">Item to look for</param>
		/// <returns>Index for the item</returns>
		public int IndexOf(double item)
		{
			return (data as IList<double>).IndexOf(item);
		}

		/// <summary>
		/// Insert a new item to this list at a specified index.
		/// </summary>
		/// <param name="index">Index</param>
		/// <param name="item">Item to add</param>
		public void Insert(int index, double item)
		{
			(data as IList<double>).Insert(index, item);
		}

		/// <summary>
		/// Remove the item at a specified index.
		/// </summary>
		/// <param name="index">Index</param>
		public void RemoveAt(int index)
		{
			(data as IList<double>).RemoveAt(index);
		}

		/// <summary>
		/// Add a new item at the end of the list.
		/// </summary>
		/// <param name="item">Item to add</param>
		public void Add(double item)
		{
			(data as IList<double>).Add(item);
		}

		/// <summary>
		/// Clear this list.
		/// </summary>
		public void Clear()
		{
			(data as IList<double>).Clear();
		}

		/// <summary>
		/// Check if this list contains an item.
		/// </summary>
		/// <param name="item">Item</param>
		/// <returns>True, if this list contains the item</returns>
		public bool Contains(double item)
		{
			return (data as IList<double>).Contains(item);
		}

		/// <summary>
		/// Copy all values in this list to an array.
		/// </summary>
		/// <param name="array">Destination array</param>
		/// <param name="arrayIndex">Start index in the array</param>
		public void CopyTo(double[] array, int arrayIndex)
		{
			(data as IList<double>).CopyTo(array, arrayIndex);
		}

		/// <summary>
		/// Number of items in this series.
		/// </summary>
		public int Count
		{
			get { return (data as IList<double>).Count; }
		}

		/// <summary>
		/// Read only list.
		/// </summary>
		public bool IsReadOnly
		{
			get { return (data as IList<double>).IsReadOnly; }
		}

		/// <summary>
		/// Remove an item in this list.
		/// </summary>
		/// <param name="item">Item to remove</param>
		/// <returns></returns>
		public bool Remove(double item)
		{
			return (data as IList<double>).Remove(item);
		}

		#endregion
	}

	/// <summary>
	/// A series plotted against another, sorted series (xy-plot)
	/// </summary>
	public class Plot : DataSeries
	{
		/// <summary>
		/// Values at the x-axis.
		/// </summary>
		protected ISeries refPoints;

		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="y">Y-values</param>
		/// <param name="x">X-values</param>
		public Plot(ISeries y, ISeries x)
			: base(y)
		{
			refPoints = x;
		}

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="t">ISeries used as the x-axis values</param>
		public Plot(ISeries t)
			: this(new DataSeries(t.Count), t) { }

		/// <summary>
		/// Copy constructor.
		/// </summary>
		/// <param name="p">Plot to copy</param>
		public Plot(Plot p) : this(p, p.ReferenceValues) { }

		#endregion

		/// <summary>
		/// X-axis values.
		/// </summary>
		public ISeries ReferenceValues { get { return this.refPoints; } }

		/// <summary>
		/// String representation.
		/// </summary>
		/// <param name="u">Timeunit</param>
		/// <returns>String representation of the series using a specified scaling unit</returns>
		public override string ToString(Unit u)
		{
			System.Text.StringBuilder sb = new System.Text.StringBuilder();
			sb.Append("[");
			int i = 0;
			while (i < this.Length)
			{
				sb.AppendFormat("({0} {1})", ReferenceValues[i], this[i, u] );
				if (++i < this.Length) sb.Append(";");
			}
			sb.Append("]");
			if (u.UnitString != String.Empty) sb.AppendFormat(" ({0})", u.UnitString);
			return sb.ToString();
		}

		/// <summary>
		/// Implicit cast operator to a Plot from an array of doubles.
		/// </summary>
		/// <param name="s">Array of doubles used as the x-axis</param>
		/// <returns>Plot with the values as the x-axis</returns>
		public static implicit operator Plot(double[] s)
		{
			return new Plot(s);
		}

		/// <summary>
		/// Implicit cast operator to a Plot from an array of doubles.
		/// </summary>
		/// <param name="s">Array of doubles used as the x-axis</param>
		/// <returns>Plot with the values as the x-axis</returns>
		public static implicit operator Plot(float[] s)
		{
			return new Plot(s);
		}
	}

	/// <summary>
	/// Values plotted against sorted time values (ty-plot)
	/// </summary>
	public class TimeDataSeries : Plot
	{
		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="d">Data points</param>
		/// <param name="t">Time points</param>
		public TimeDataSeries(DataSeries d, TimeSeries t)
			: base(d, t) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="t">Time series used as the t-axis</param>
		public TimeDataSeries(TimeSeries t)
			: this(new DataSeries(t.Count), t) { }

		/// <summary>
		/// Copy constructor.
		/// </summary>
		/// <param name="ts">TimedDataSeries to copy</param>
		public TimeDataSeries(TimeDataSeries ts) : this(ts, ts.Times) { }

		#endregion

		/// <summary>
		/// Time axis values
		/// </summary>
		public TimeSeries Times { get { return (this.ReferenceValues as TimeSeries); } }

		/// <summary>
		/// Implicit cast operator to a TimeSeries.
		/// </summary>
		/// <param name="ds">TimeDataSeries</param>
		/// <returns>Values of the t-axis</returns>
		public static explicit operator TimeSeries(TimeDataSeries ds)
		{
			return new TimeSeries(ds.ReferenceValues.ToArray());
		}
	}
}
