/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Common
{
	public class Matrix : ICloneable, IEquatable<Matrix>
	{
		private double[][] val;

		private int rows;
		private int cols;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="m">Rows</param>
		/// <param name="n">Columns</param>
		public Matrix(int m, int n)
		{
			val = new double[n][];
			for (int i = 0; i < val.Length; i++)
			{
				val[i] = new double[m];
			}
			rows = m;
			cols = n;
		}

		/// <summary>
		/// Constructor. Square matrix, n rows.
		/// </summary>
		/// <param name="n">Rows, columns</param>
		public Matrix(int n) : this(n, n) { }

		public Matrix(params double[][] d) : this(d.Length, d[0].Length)
		{
			for (int i = 0; i < Rows; i++)
			{
				for (int j = 0; j < Columns; j++)
				{
					this[i, j] = d[i][j];
				}
			}
		}

		public Matrix(Matrix m)
			: this(m.Rows, m.Columns)
		{
			rows = m.Rows;
			cols = m.Columns;
			for (int i = 0; i < m.Rows; i++)
			{
				for (int j = 0; j < m.Columns; j++)
				{
					this[i, j] = m[i, j];
				}
			}
		}

		public double this[int i, int j]
		{
			get { return Get(i, j); }
			set { Set(i, j, value); }
		}

		public int Columns { get { return cols; } }

		public int Rows { get { return rows; } }

		public static explicit operator Matrix(Vector v)
		{
			return new Matrix((double[])v);
		}

		public Matrix Add(Matrix a)
		{
			Matrix m = new Matrix(this.Rows, this.Columns);
			for (int i = 0; i < m.Rows; i++)
			{
				for (int j = 0; j < m.Columns; j++)
				{
					m[i, j] = this[i, j] + a[i, j];
				}
			}
			return m;
		}

		public Matrix Substract(Matrix a)
		{
			Matrix m = new Matrix(this.Rows, this.Columns);
			for (int i = 0; i < m.Rows; i++)
			{
				for (int j = 0; j < m.Columns; j++)
				{
					m[i, j] = this[i, j] - a[i, j];
				}
			}
			return m;
		}

		public Matrix Multiply(Matrix a)
		{
			Matrix m = new Matrix(this.Rows, a.Columns);
			Vector r;
			for (int i = 0; i < this.Rows; i++)
			{
				r = this.GetRowVector(i);
				for (int j = 0; j < a.Columns; j++)
				{
					m[i, j] = r * a.GetColumnVector(j);
				}
			}
			return m;
		}

		public static Matrix Transpose(Matrix a)
		{
			Matrix m = new Matrix(a.Columns, a.Rows);
			for (int i = 0; i < m.Rows; i++)
			{
				for (int j = 0; j < m.Columns; j++)
				{
					m[i, j] = a[j, i];
				}
			}
			return m;
		}

		public static Matrix Negate(Matrix a)
		{
			Matrix m = new Matrix(a.Rows, a.Columns);
			for (int i = 0; i < m.Rows; i++)
			{
				for (int j = 0; j < m.Columns; j++)
				{
					m[i, j] = -a[i, j];
				}
			}
			return m;
		}

		public object Clone()
		{
			return new Matrix(this);
		}

		public Vector GetColumnVector(int n)
		{
			double[] d = new double[this.Rows];
			Array.Copy(val[n], d, this.Rows);
			return d;
		}

		public Vector GetRowVector(int n)
		{
			double[] d = new double[this.Columns];
			for (int i = 0; i < this.Columns; i++)
			{
				d[i] = this[n, i];
			}
			return d;
		}

		public double Get(int i, int j) { return val[j][i]; }

		public void Set(int i, int j, double v) { val[j][i] = v; }

		public Matrix Add(double d)
		{
			Matrix m = new Matrix(this.Rows, this.Columns);
			for (int i = 0; i < this.Rows; i++)
			{
				for (int j = 0; j < this.Columns; j++)
				{
					m[i, j] = this[i,j] + d;
				}
			}
			return m;
		}

		public Matrix Substract(double d)
		{
			Matrix m = new Matrix(this.Rows, this.Columns);
			for (int i = 0; i < this.Rows; i++)
			{
				for (int j = 0; j < this.Columns; j++)
				{
					m[i, j] = this[i, j] - d;
				}
			}
			return m;
		}

		public Matrix Multiply(double d)
		{
			Matrix m = new Matrix(this.Rows, this.Columns);
			for (int i = 0; i < this.Rows; i++)
			{
				for (int j = 0; j < this.Columns; j++)
				{
					m[i, j] = this[i, j] * d;
				}
			}
			return m;
		}

		public static Matrix Expand(Matrix a, int n)
		{
			Matrix m = new Matrix(a.Rows + n, a.Columns + n);
			for (int i = 0, mi = n; i < a.Rows; i++, mi++)
			{
				for (int j = 0, mj = n; j < a.Columns; j++, mj++)
				{
					m[mi, mj] = a[i, j];
				}
			}
			for (int i = 0; i < n && i < n; i++)
			{
				m[i, i] = 1.0;
			}
			return m;
		}

		public static Matrix Identity(int n)
		{
			Matrix m = new Matrix(n, n);
			for (int i = 0; i < n; i++)
			{
				m[i, i] = 1.0;
			}
			return m;
		}

		public static Matrix operator *(Matrix a, Matrix b)
		{
			return a.Multiply(b);
		}

		public static Matrix operator *(double b, Matrix a)
		{
			return a * b;
		}

		public static Matrix operator *(Matrix a, double b)
		{
			return a.Multiply(b);
		}

		public static Matrix operator /(Matrix a, double b)
		{
			return a.Multiply(1.0 / b);
		}

		public static Matrix operator +(Matrix a, Matrix b)
		{
			return a.Add(b);
		}

		public static Matrix operator -(Matrix a, Matrix b)
		{
			return a.Substract(b);
		}

		public static Vector operator *(Vector v, Matrix a)
		{
			Vector res = new Vector(a.Columns);
			for (int i = 0; i < res.Length; i++)
			{
				res[i] = v * a.GetColumnVector(i);
			}
			return res;
		}

		public static Vector operator *(Matrix a, Vector v)
		{
			Vector res = new Vector(a.Rows);
			for (int i = 0; i < res.Length; i++)
			{
				res[i] = v * a.GetRowVector(i);
			}
			return res;
		}

		public static implicit operator Matrix(double[][] d)
		{
			return new Matrix(d);
		}

		public static implicit operator double[][](Matrix m)
		{
			double[][] d  = new double[m.Rows][];
			for(int i = 0; i < m.Rows; i++)
			{
				d[i] = new double[m.Columns];
				for (int j = 0; j < m.Columns; j++)
				{
					d[i][j] = m[i, j];
				}
			}
			return d;
		}

		public bool Equals(Matrix other)
		{
			if (Rows != other.Rows || Columns != other.Columns) return false;

			for (int i = 0; i < Rows; i++)
			{
				for (int j = 0; j < Columns; j++)
				{
					if (this[i, j] != other[i, j]) return false;
				}
			}
			return true;
		}

		public override int GetHashCode()
		{
			return val.GetHashCode();
		}

		public override bool Equals(object obj)
		{
			if (obj is Matrix) return this.Equals(obj as Matrix);
			else return false;
		}
	}
}
