/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Common
{
	/// <summary>
	/// Point in 3D space.
	/// </summary>
	public class Point : ICloneable, IEquatable<Point>
	{
		private double x;
		private double y;
		private double z;

		/// <summary>
		/// x-coordinate, (input/output)
		/// </summary>
		public double X { get { return x; } set { x = value; } }

		/// <summary>
		/// y-coordinate, (input/output)
		/// </summary>
		public double Y { get { return y; } set { y = value; } }

		/// <summary>
		/// z-coordinate, (input/output)
		/// </summary>
		public double Z { get { return z; } set { z = value; } }

		/// <summary>
		/// Constructs point at specified location.
		/// </summary>
		/// <param name="x">x-coordinate</param>
		/// <param name="y">y-coordinate</param>
		/// <param name="z">z-coordinate</param>
		public Point(double x, double y, double z)
		{
			this.X = x;
			this.Y = y;
			this.Z = z;
		}

		/// <summary>
		/// Constructs point at specified location.
		/// </summary>
		/// <param name="p">location</param>
		public Point(Point p)
		{
			this.X = p.X;
			this.Y = p.Y;
			this.Z = p.Z;
		}

		/// <summary>
		/// Constructs point at specified location.
		/// </summary>
		/// <param name="p">coordinates in 3-length array {x,y,z}</param>
		public Point(double[] p)
		{
			this.X = p[0];
			this.Y = p[1];
			this.Z = p[2];
		}

		/// <summary>
		/// Constructs point at location (0, 0, 0)
		/// </summary>
		public Point()
		{
			X = 0.0;
			Y = 0.0;
			Z = 0.0;
		}

		/// <summary>
		/// Addition operator
		/// </summary>
		/// <param name="p1">1st addition term point</param>
		/// <param name="p2">2nd addition term point</param>
		public static Point operator +(Point p1, Point p2)
		{
			return new Point(p1.X + p2.X, p1.Y + p2.Y, p1.Z + p2.Z);
		}

		/// <summary>
		/// Decrement operator
		/// </summary>
		/// <param name="p1">minuend point</param>
		/// <param name="p2">subtrahend point</param>
		public static Point operator -(Point p1, Point p2)
		{
			return new Point(p1.X - p2.X, p1.Y - p2.Y, p1.Z - p2.Z);
		}

		/// <summary>
		/// Division operator. Divides each coordinate component with divisor.
		/// </summary>
		/// <param name="p1">divident point</param>
		/// <param name="d">divisor</param>
		public static Point operator /(Point p1, double d)
		{
			return new Point(p1.X / d, p1.Y / d, p1.Z / d);
		}

		/// <summary>
		/// Division operator
		/// </summary>
		/// <param name="p1">divident point</param>
		/// <param name="p2">divisor point</param>
		public static Point operator /(Point p1, Point p2)
		{
			return new Point(p1.X / p2.X, p1.Y / p2.Y, p1.Z / p2.Z);
		}

		/// <summary>
		/// Multiplication operator
		/// </summary>
		/// <param name="p1">1st point that is multiplied</param>
		/// <param name="p2">2nd point that is multiplied</param>
		public static Point operator *(Point p1, Point p2)
		{
			return new Point(p1.X * p2.X, p1.Y * p2.Y, p1.Z * p2.Z);
		}

		/// <summary>
		/// Multiplication operator
		/// </summary>
		/// <param name="p1">1st point that is multiplied</param>
		/// <param name="d">multiplicator</param>
		public static Point operator *(Point p1, double d)
		{
			Point r = new Point();
			r.X = p1.X * d;
			r.Y = p1.Y * d;
			r.Z = p1.Z * d;
			return r;
		}

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>A String that represents the current Object.</returns>
		public override string ToString()
		{
			return "Point[" + X.ToString("0.000").Replace(',', '.') + "," +
							  Y.ToString("0.000").Replace(',', '.') + "," +
							  Z.ToString("0.000").Replace(',', '.') + "]";
		}

		/// <summary>
		/// Equality test.
		/// </summary>
		/// <param name="obj">other object</param>
		/// <returns>true if obj is equal to this object</returns>
		public override bool Equals(object obj)
		{
			if (obj is Point) return this.Equals(obj as Point);
			else return false;
		}

		/// <summary>
		/// Returns hash code.
		/// </summary>
		/// <returns>base class hash code</returns>
		public override int GetHashCode()
		{
			return x.GetHashCode() ^ y.GetHashCode() ^ z.GetHashCode();
		}

		/// <summary>
		/// Calculates euclidean distance
		/// </summary>
		/// <param name="p">reference_times point</param>
		/// <returns>distance between points in 3D space</returns>
		public double Dist(Point p)
		{
			return (this - p).Norm();
		}

		/// <summary>
		/// Calculates cross-product of points
		/// </summary>
		/// <param name="p1">point1</param>
		/// <param name="p2">point2</param>
		/// <returns>cross product point</returns>
		public static Point Cross(Point p1, Point p2)
		{
			Point r = new Point();
			r.X = p1.Y * p2.Z - p1.Z * p2.Y;
			r.Y = -p1.X * p2.Z + p1.Z * p2.X;
			r.Z = p1.X * p2.Y - p1.Y * p2.X;
			return r;
		}

		/// <summary>
		/// Calculates cross-product of points
		/// </summary>
		/// <param name="r">target point</param>
		/// <param name="p1">point1</param>
		/// <param name="p2">point2</param>
		/// <returns>cross product point</returns>
		public static void Cross(out Point r, Point p1, Point p2)
		{
			r = Cross(p1, p2);
		}

		/// <summary>
		/// Calculates dot product
		/// </summary>
		/// <param name="p">reference_times point</param>
		/// <returns>dot product of points</returns>
		public double Dot(Point p)
		{
			return X * p.X + Y * p.Y + Z * p.Z;
		}

		/// <summary>
		/// Calculates dot product
		/// </summary>
		/// <param name="p1">reference_times point</param>
		/// <param name="p2">reference_times point</param>
		/// <returns>dot product of points</returns>
		public static double Dot(Point p1, Point p2)
		{
			return p1.Dot(p2);
		}

		/// <summary>
		/// Calculates the norm of point as vector.
		/// </summary>
		/// <returns>norm of coordinate vector</returns>
		public double Norm()
		{
			return Math.Sqrt(x * x + y * y + z * z);
		}

		/// <summary>
		/// Normalize point. This means that the norm of 
		/// point coordinate becomes 1 (or very close).
		/// </summary>
		public void Normalize()
		{
			double norm = Norm();
			if (norm > 0)
			{
				this.Divide(norm);
			}
			else
			{
				X = 0.0;
				Y = 0.0;
				Z = 0.0;
			}
		}

		/// <summary>
		/// Sets point location
		/// </summary>
		/// <param name="x">x-coordinate</param>
		/// <param name="y">y-coordinate</param>
		/// <param name="z">z-coordinate</param>
		public void Set(int x, int y, int z)
		{
			this.X = x;
			this.Y = y;
			this.Z = z;
		}

		/// <summary>
		/// Normalize point. This means that the norm of 
		/// point coordinate becomes 1 (or very close).
		/// </summary>
		public static Point Normalize(Point p)
		{
			p.Normalize();
			return p;
		}

		/// <summary>
		/// Division with constant.
		/// </summary>
		/// <param name="d">divisor</param>
		public void Divide(double d)
		{
			X /= d;
			Y /= d;
			Z /= d;
		}

		/// <summary>
		/// Sets point location.
		/// </summary>
		/// <param name="p">new location</param>
		public void Set(Point p)
		{
			this.X = p.X;
			this.Y = p.Y;
			this.Z = p.Z;
		}

		/// <summary>
		/// Subtraction operation
		/// </summary>
		/// <param name="r">target point</param>
		/// <param name="p1">subtracted</param>
		/// <param name="p2">subtractor</param>
		public static void Sub(out Point r, Point p1, Point p2)
		{
			r = p1 - p2;
		}

		/// <summary>
		/// Access to coordinate indexes
		/// </summary>
		/// <param name="i">index in [0,1,2]</param>
		/// <returns>coordinate value at wanted index</returns>
		/// <exception cref="TPCException">if coordinate index is not in {1,2,3}</exception>
		public double this[int i]
		{
			get
			{
				if (i == 0) return X;
				else if (i == 1) return Y;
				else if (i == 2) return Z;
				else throw new TPCException("Index out of bounds.");
			}
			set
			{
				if (i == 0) X = value;
				else if (i == 1) Y = value;
				else if (i == 2) Z = value;
				else throw new TPCException("Index out of bounds.");
			}
		}

		/// <summary>
		/// Returns tacs of this point as array
		/// </summary>
		/// <returns>array {x,y,z}</returns>
		public double[] ToArray()
		{
			return new double[] { X, Y, Z };
		}

		/// <summary>
		/// Creates deep copy of this object
		/// </summary>
		/// <returns>cloned object</returns>
		public Object Clone()
		{
			return new Point(this);
		}

		/// <summary>
		/// Equality operator
		/// </summary>
		/// <param name="other"></param>
		/// <returns></returns>
		public bool Equals(Point other)
		{
			return
				this.x.Equals(other.x) &&
				this.y.Equals(other.y) &&
				this.z.Equals(other.z);
		}
	}
}
