/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Common
{
	/// <summary>
	/// Base class for all units
	/// </summary>
	public class Unit : IEquatable<Unit>
	{
		#region Protected

		private bool IsBaseUnit { get { return ReferenceEquals(this.baseUnit, Unit.RootUnit); } }

		private bool IsRootUnit { get { return ReferenceEquals(this.baseUnit, this); } }

		/// <summary>
		/// Conversion factor from the base unit
		/// </summary>
		private readonly double factor = 1.0;

		/// <summary>
		/// Base unit
		/// </summary>
		private readonly Unit baseUnit;

		/// <summary>
		/// Unit string representation
		/// </summary>
		private readonly string unitString = String.Empty;

		/// <summary>
		/// Get the string representation of a SI prefix.
		/// </summary>
		/// <param name="f">SI prefix</param>
		/// <returns>String representation</returns>
		private static string FactorToString(SIFactor f)
		{
			switch (f)
			{
				case SIFactor.Pico: return @"p";
				case SIFactor.Nano: return @"n";
				case SIFactor.Micro: return @"u";
				case SIFactor.Milli: return @"m";
				case SIFactor.Centi: return @"c";
				case SIFactor.Deci: return @"d";
				case SIFactor.Deca: return @"da";
				case SIFactor.Hecto: return @"h";
				case SIFactor.Kilo: return @"k";
				case SIFactor.Mega: return @"M";
				case SIFactor.Giga: return @"G";
				case SIFactor.Tera: return @"T";
				default: return String.Empty;
			}
		}

		/// <summary>
		/// Get the conversion factor for a SI prefix.
		/// </summary>
		/// <param name="f">SI prefix</param>
		/// <returns>Conversion factor</returns>
		private static double FactorToValue(SIFactor f)
		{
			switch (f)
			{
				case SIFactor.Pico: return 1e-12;
				case SIFactor.Nano: return 1e-9;
				case SIFactor.Micro: return 1e-6;
				case SIFactor.Milli: return 1e-3;
				case SIFactor.Centi: return 1e-2;
				case SIFactor.Deci: return 1e-1;
				case SIFactor.Deca: return 1e1;
				case SIFactor.Hecto: return 1e2;
				case SIFactor.Kilo: return 1e3;
				case SIFactor.Mega: return 1e6;
				case SIFactor.Giga: return 1e9;
				case SIFactor.Tera: return 1e12;
				default: return 1;
			}
		}

		private static Unit RootUnit = new Unit();

		#endregion

		#region Constructors

		/// <summary>
		/// Default constructor. Creates the Unit.None object.
		/// </summary>
		private Unit()
		{
			factor = 1.0;
			baseUnit = this;
			unitString = String.Empty;
		}

		protected Unit(string s)
		{
			baseUnit = Unit.RootUnit;
			factor = 1.0;
			unitString = s;
		}

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="bu">Base unit</param>
		/// <param name="f">Factor</param>
		/// <param name="s">Unit name</param>
		protected Unit(Unit bu, double f, string s)
		{
			factor = f * bu.factor;
			if (bu.IsBaseUnit)
			{
				baseUnit = bu;
			}
			else baseUnit = bu.baseUnit;
			unitString = s;
		}

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		/// <param name="s">Unit name</param>
		protected Unit(Unit u, SIFactor f, string s)
			: this(u, FactorToValue(f), FactorToString(f) + s) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		protected Unit(Unit u, SIFactor f)
			: this(u, f, u.UnitString) { }

		protected Unit(Unit u, string s)
			: this(u, 1.0, s) { }

		#endregion

		#region Public

		/// <summary>
		/// The most common SI unit prefixes
		/// </summary>
		public enum SIFactor
		{
			/// <summary>
			/// Pico (1e-12)
			/// </summary>
			Pico,
			/// <summary>
			/// Nano (1e-9)
			/// </summary>
			Nano,
			/// <summary>
			/// Micro (1e-6)
			/// </summary>
			Micro,
			/// <summary>
			/// Milli (1e-3)
			/// </summary>
			Milli,
			/// <summary>
			/// Centi (1e-2)
			/// </summary>
			Centi,
			/// <summary>
			/// Deci (1e-1)
			/// </summary>
			Deci,
			/// <summary>
			/// Deca (1e1)
			/// </summary>
			Deca,
			/// <summary>
			/// Hecto (1e2)
			/// </summary>
			Hecto,
			/// <summary>
			/// Kilo (1e3)
			/// </summary>
			Kilo,
			/// <summary>
			/// Mega (1e6)
			/// </summary>
			Mega,
			/// <summary>
			/// Giga (1e9)
			/// </summary>
			Giga,
			/// <summary>
			/// Tera (1e12)
			/// </summary>
			Tera
		}

		/// <summary>
		/// String representation of this unit.
		/// </summary>
		public string UnitString { get { return unitString; } }

		/// <summary>
		/// Convert value in one unit to another. Exception thrown, if conversion not possible.
		/// </summary>
		/// <param name="from">Original unit</param>
		/// <param name="to">Result unit</param>
		/// <param name="x">Value to convert</param>
		/// <returns>Value in the result unit</returns>
		public static double ConvertValue(Unit from, Unit to, double x)
		{
			if (AreConvertible(from, to)) return x * from.factor / to.factor;
			else throw new TPCException("Invalid conversion from " + "" + " to " + "" + ".");
		}

		/// <summary>
		/// Check if conversion is possible between two units.
		/// </summary>
		/// <param name="a">First unit</param>
		/// <param name="b">Second unit</param>
		/// <returns>True, if values measured in the first unit can be converted to the second</returns>
		public static bool AreConvertible(Unit a, Unit b)
		{
			if (a.IsBaseUnit && b.IsBaseUnit) return false;
			if (a.IsBaseUnit) return b.baseUnit.Equals(a);
			if (b.IsBaseUnit) return a.baseUnit.Equals(b);
			return a.baseUnit.Equals(b.baseUnit);
		}

		/// <summary>
		/// Equality operator. Units are equal, if they have the same base unit and the same factor. Unit name may differ.
		/// </summary>
		/// <param name="other">Comparison unit</param>
		/// <returns>True, if equal</returns>
		public bool Equals(Unit other)
		{
			if (this.factor.Equals(other.factor))
			{
				if (this.IsBaseUnit && other.IsBaseUnit) return this.UnitString.Equals(other.UnitString);
				if (this.IsBaseUnit) return this.Equals(other.baseUnit);
				if (other.IsBaseUnit) return other.Equals(this.baseUnit);
				return this.baseUnit.Equals(other.baseUnit);
			}
			return false;
		}

		/// <summary>
		/// Equality operator. Units are equal, if they have the same base unit and the same factor. Unit name may differ.
		/// </summary>
		/// <param name="obj">Comparison object</param>
		/// <returns>True, if equal</returns>
		public override bool Equals(object obj)
		{
			if (obj is Unit) return Equals(obj as Unit);
			else return false;
		}

		/// <summary>
		/// Hashcode for this instance.
		/// </summary>
		/// <returns>Hashcode</returns>
		public override int GetHashCode()
		{
			return unitString.GetHashCode() ^ factor.GetHashCode();
		}

		/// <summary>
		/// String representation of this unit.
		/// </summary>
		/// <returns>String representation</returns>
		public override string ToString()
		{
			return unitString;
		}

		/// <summary>
		/// Equivalency operators
		/// </summary>
		/// <param name="a">Unit</param>
		/// <param name="b">Unit</param>
		/// <returns>True, if the units have the same base unit and factor</returns>
		public static bool operator ==(Unit a, Unit b)
		{
			return a.Equals(b);
		}

		/// <summary>
		/// Inequality operator
		/// </summary>
		/// <param name="a">Unit</param>
		/// <param name="b">Unit</param>
		/// <returns>True, if units have a different baseunit or factor</returns>
		public static bool operator !=(Unit a, Unit b)
		{
			return !(a.Equals(b));
		}

		#endregion
	}

	/// <summary>
	/// Data unit.
	/// </summary>
	public class DataUnit : Unit
	{
		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		public DataUnit(DataUnit u, SIFactor f)
			: base(u, f) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		/// <param name="s">Unit name</param>
		public DataUnit(DataUnit u, SIFactor f, string s)
			: base(u, f, s) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		/// <param name="s">Unit name</param>
		public DataUnit(DataUnit u, double f, string s)
			: base(u, f, s) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">Unit name</param>
		public DataUnit(string s)
			: base(s) { }

		#endregion

		#region Public

		/// <summary>
		/// Constructs unit from unit's name. A new unit is constructed if
		/// no match to known units is found.
		/// </summary>
		/// <param name="name">Units name</param>
		/// <returns>Data unit</returns>
		public static DataUnit Parse(string name)
		{
			name = name.Trim().ToLowerInvariant();

			if (name.Equals("mlming") ||
				name.Equals("ml/min/g") ||
				name.Equals("ml/g/min") ||
				name.Equals("ml/(min*g)") ||
				name.Equals("ml/(g*min)"))
				return new DataUnit(DataUnit.LitresPerGramPerMinutes, SIFactor.Milli);

			if (name.Equals("ml/(ml*min)") ||
				name.Equals("ml/(min*ml)") ||
				name.Equals("mlminml"))
				return new DataUnit(LitresPerMillilitreMinutes, SIFactor.Milli);

			if (name.Equals("ml/(dl*min)"))
				return DataUnit.MillilitresPerDesilitreMinutes;

			if (name.Equals("bq/ml") ||
				name.Equals("bqml") ||
				name.Equals("bq/cc") ||
				name.Equals("bqcc"))
				return DataUnit.BecquerelsPerMillilitre;

			if (name.Equals("mbq/ml") ||
				name.Equals("mbqml") ||
				name.Equals("mbq/cc") ||
				name.Equals("mbqcc"))
				return new DataUnit(BecquerelsPerMillilitre, SIFactor.Mega);

			if (name.Equals("kbq/ml") ||
				name.Equals("kbqml") ||
				name.Equals("kbq/cc") ||
				name.Equals("kbqcc"))
				return new DataUnit(BecquerelsPerMillilitre, SIFactor.Kilo);

			if (name.Equals("um/ml") ||
				name.Equals("mmol/l") ||
				name.Equals("umolml"))
				return new DataUnit(MolesPerLitre, SIFactor.Milli);

			if (name.Equals("1/sec") ||
				name.Equals("1/s") ||
				name.Equals("/s") ||
				name.Equals("cps"))
				return DataUnit.OnePerSecond;

			if (name.Equals("1/min") ||
				name.Equals("/min"))
				return DataUnit.OnePerMinute;

			if (name.Equals("ml/ml") ||
				name.Equals("mlml"))
				return new DataUnit(LitresPerMillilitre, SIFactor.Milli);

			if (name.Equals("ml/dl"))
				return new DataUnit(LitresPerMillilitre, 1e-5, "ml/dl");

			if (name.Equals("counts") ||
				name.Equals("cnts"))
				return DataUnit.Count;

			if (name.Equals("unknown") ||
				name.Equals("undefined"))
				return DataUnit.Unknown;

			return new DataUnit(name);
		}

		/// <summary>
		/// Create a new unit as a product of two units
		/// </summary>
		/// <param name="a">First unit</param>
		/// <param name="b">Second unit</param>
		/// <returns>Unit representing the product a*b</returns>
		public static DataUnit operator *(DataUnit a, Unit b)
		{
			string aNom, bNom, aDenom, bDenom;

			GetParts(a.UnitString, out aNom, out aDenom);
			GetParts(b.UnitString, out bNom, out bDenom);

			string s = Combine(aNom, bNom, aDenom, bDenom);
			return new DataUnit(s);
		}

		/// <summary>
		/// Create a new unit as a ratio of two units
		/// </summary>
		/// <param name="a">First unit</param>
		/// <param name="b">Second unit</param>
		/// <returns>Unit representing the ratio a/b</returns>
		public static DataUnit operator /(DataUnit a, Unit b)
		{
			string aNom, bNom, aDenom, bDenom;

			GetParts(a.UnitString, out aNom, out aDenom);
			GetParts(b.UnitString, out bNom, out bDenom);

			string s = Combine(aNom, bDenom, aDenom, bNom);
			return new DataUnit(s);
		}

		private static string Combine(string n1, string n2, string d1, string d2)
		{
			string s = n1;
			if (n1 != String.Empty && n2 != String.Empty) s += '*';
			s += n2;
			if (d1 != String.Empty || d2 != String.Empty)
			{
				s += "/";
				if (d1 != String.Empty && d2 != String.Empty)
					s += "(" + d1 + '*' + d2 + ')';
				else s += d1 + d2;
			}
			return s;
		}

		private static void GetParts(string s, out string nom, out string denom)
		{
			nom = String.Empty;
			denom = String.Empty;

			string[] split = s.Split('/');
			if (split.Length > 0) nom = split[0];
			if (split.Length > 1) denom = split[1];
		}

		#endregion

		#region Predefined units

		/// <summary>
		/// Unknown data unit
		/// </summary>
		public readonly static DataUnit Unknown = new DataUnit("unknown");

		// Basic units

		/// <summary>
		/// Pixel
		/// </summary>
		public readonly static DataUnit Pixel = new DataUnit(@"px");

		/// <summary>
		/// Metres
		/// </summary>
		public readonly static DataUnit Metre = new DataUnit(@"m");

		/// <summary>
		/// Counts
		/// </summary>
		public readonly static DataUnit Count = new DataUnit(@"counts");

		/// <summary>
		/// Beqquerels
		/// </summary>
		public readonly static DataUnit Becquerel = new DataUnit(@"Bq");

		/// <summary>
		/// Litres
		/// </summary>
		public readonly static DataUnit Litre = new DataUnit(@"l");

		/// <summary>
		/// Grams
		/// </summary>
		public readonly static DataUnit Gram = new DataUnit(@"g");

		/// <summary>
		/// Moles
		/// </summary>
		public readonly static DataUnit Mole = new DataUnit(@"mol");

		/// <summary>
		/// Counts per second
		/// </summary>
		public readonly static DataUnit OnePerSecond = new DataUnit(@"/s");

		// Common derived units

		/// <summary>
		/// Kilobecquerel
		/// </summary>
		public readonly static DataUnit KiloBecquerel = new DataUnit(Becquerel, SIFactor.Kilo);

		/// <summary>
		/// Megabecquerel
		/// </summary>
		public readonly static DataUnit MegaBecquerel = new DataUnit(Becquerel, SIFactor.Mega);

		/// <summary>
		/// Curie
		/// </summary>
		public readonly static DataUnit Curie = new DataUnit(Becquerel, 3.7e10, "Ci");

		/// <summary>
		/// Millicurie
		/// </summary>
		public readonly static DataUnit MilliCurie = new DataUnit(Curie, SIFactor.Milli);

		/// <summary>
		/// Counts per minute
		/// </summary>
		public readonly static DataUnit OnePerMinute = new DataUnit(OnePerSecond, 1.0f / 60f, @"/min");

		/// <summary>
		/// Millilitres
		/// </summary>
		public readonly static DataUnit Millilitre = new DataUnit(DataUnit.Litre, SIFactor.Milli);

		/// <summary>
		/// Millimetre
		/// </summary>
		public readonly static DataUnit Millimetre = new DataUnit(DataUnit.Metre, SIFactor.Milli);

		/// <summary>
		/// Centimetre
		/// </summary>
		public readonly static DataUnit Centimetre = new DataUnit(DataUnit.Metre, SIFactor.Centi);

		/// <summary>
		/// Inch (25.4 mm)
		/// </summary>
		public readonly static DataUnit Inch = new DataUnit(DataUnit.Millimetre, 25.4, "in");

		/// <summary>
		/// Moles per litre
		/// </summary>
		public readonly static DataUnit MolesPerLitre = Mole / Litre;

		/// <summary>
		/// Bequerels per millilitre
		/// </summary>
		public readonly static DataUnit BecquerelsPerMillilitre = Becquerel / Millilitre;

		/// <summary>
		/// Litres per millilitre
		/// </summary>
		public readonly static DataUnit LitresPerMillilitre = Litre / Millilitre;

		/// <summary>
		/// Millilitres per millilitre
		/// </summary>
		public readonly static DataUnit MillilitresPerMillilitre = new DataUnit(LitresPerMillilitre, SIFactor.Milli);

		/// <summary>
		/// Litres per millilitres per minutes
		/// </summary>
		public readonly static DataUnit LitresPerMillilitreMinutes = LitresPerMillilitre / TimeUnit.Minute;

		/// <summary>
		/// Milliitres per millilitres per minutes
		/// </summary>
		public readonly static DataUnit MillilitresPerMillilitreMinutes = new DataUnit(LitresPerMillilitreMinutes, SIFactor.Milli);

		/// <summary>
		/// Milliitres per desilitres per minutes
		/// </summary>
		public readonly static DataUnit MillilitresPerDesilitreMinutes = new DataUnit(LitresPerMillilitreMinutes, 1e-5, @"ml/(dl*min)");

		/// <summary>
		/// Litres per grams per minutes
		/// </summary>
		public readonly static DataUnit LitresPerGramPerMinutes = (Litre / Gram) / TimeUnit.Minute;

		/// <summary>
		/// Millilitres per grams per minutes
		/// </summary>
		public readonly static DataUnit MillilitresPerGramPerMinutes = new DataUnit(LitresPerGramPerMinutes, SIFactor.Milli);

		/// <summary>
		/// Counts per millilitre
		/// </summary>
		public readonly static DataUnit CountsPerMillilitre = Count / Millilitre;

		#endregion
	}

	/// <summary>
	/// Time unit.
	/// </summary>
	public class TimeUnit : Unit
	{
		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">Time unit name</param>
		protected TimeUnit(string s)
			: base(s) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		public TimeUnit(TimeUnit u, SIFactor f)
			: base(u, f) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		/// <param name="s">Unit name</param>
		public TimeUnit(TimeUnit u, SIFactor f, string s)
			: base(u, f, s) { }

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="u">Base unit</param>
		/// <param name="f">Factor</param>
		/// <param name="s">Unit name</param>
		public TimeUnit(TimeUnit u, double f, string s)
			: base(u, f, s) { }

		#endregion

		#region Predefined time units

		// Default unit

		/// <summary>
		/// Unknown time unit
		/// </summary>
		public readonly static TimeUnit Unknown = new TimeUnit(@"unknown");

		// Common time units

		/// <summary>
		/// Seconds
		/// </summary>
		public readonly static TimeUnit Second = new TimeUnit(@"s");

		/// <summary>
		/// Nanoseconds
		/// </summary>
		public readonly static TimeUnit Nanosecond = new TimeUnit(TimeUnit.Second, Unit.SIFactor.Nano);

		/// <summary>
		/// Microseconds
		/// </summary>
		public readonly static TimeUnit Microsecond = new TimeUnit(TimeUnit.Second, Unit.SIFactor.Micro);

		/// <summary>
		/// Milliseconds
		/// </summary>
		public readonly static TimeUnit Millisecond = new TimeUnit(TimeUnit.Second, Unit.SIFactor.Milli);

		/// <summary>
		/// Minutes
		/// </summary>
		public readonly static TimeUnit Minute = new TimeUnit(Second, 60.0, @"min");

		/// <summary>
		/// Hours
		/// </summary>
		public readonly static TimeUnit Hour = new TimeUnit(Second, 3600.0, @"h");

		#endregion

		/// <summary>
		/// Constructs a timeunit from a name.
		/// </summary>
		/// <param name="name">Time unit name</param>
		/// <returns>Time unit</returns>
		public static TimeUnit Parse(string name)
		{
			name = name.Trim().ToLowerInvariant();

			if (name.Equals("ns") || name.Equals("nsec")) return TimeUnit.Nanosecond;
			if (name.Equals("us") || name.Equals("usec")) return TimeUnit.Microsecond;
			if (name.Equals("ms") || name.Equals("msec")) return TimeUnit.Millisecond;
			if (name.Equals("s") || name.Equals("sec")) return TimeUnit.Second;
			if (name.Equals("min")) return TimeUnit.Minute;
			if (name.Equals("h")) return TimeUnit.Hour;
			if (name.Equals("d")) return new TimeUnit(TimeUnit.Hour, 24, "d");

			return TimeUnit.Unknown;
		}
	}
}
