/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Common
{
	public class Vector : ICloneable, IEquatable<Vector>
	{
		private double[] val;

		public Vector() : this(0) { }

		public Vector(int n)
		{
			val = new double[n];
		}

		public Vector(params double[] d)
		{
			val = d;
		}

		public Vector(params float[] f)
		{
			val = Array.ConvertAll(f, Convert.ToDouble);
		}

		public int Length { get { return val.Length; } }

		public double this[int n]
		{
			get { return val[n]; }
			set { val[n] = value; }
		}

		public static double Norm(Vector v)
		{
			return Math.Sqrt(v * v);
		}

		public static double SquaredNorm(Vector v)
		{
			return v * v;
		}

		public static Vector Sum(Vector[] v)
		{
			Vector sum = v[0];
			for (int i = 1; i < v.Length; i++)
			{
				sum += v[i];
			}
			return sum;
		}

		public static Vector Fill(int n, double d)
		{
			Vector v = new Vector(n);
			for (int i = 0; i < v.Length; i++)
			{
				v[i] = d;
			}
			return v;
		}

		public static Vector Range(Vector v, int start, int length)
		{
			Vector vv = new Vector(length);
			for (int i = start, j = 0; j < length; i++, j++)
			{
				vv[j] = v[i];
			}
			return vv;
		}

		public static Vector Unit(int n, int i)
		{
			Vector v = new Vector(n);
			v[i] = 1.0;
			return v;
		}

		public static Vector ElementProduct(Vector a, Vector b)
		{
			Vector p = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				p[i] = a[i] * b[i];
			}
			return p;
		}

		public static double operator *(Vector a, Vector b)
		{
			double sum = 0.0;
			for (int i = 0; i < a.Length; i++)
			{
				sum += a[i] * b[i];
			}
			return sum;
		}

		public static Vector operator *(double d, Vector a)
		{
			Vector sum = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				sum[i] = a[i] * d;
			}
			return sum;
		}

		public static Vector operator *(Vector a, double d)
		{
			return d * a;
		}

		public static Vector operator /(Vector a, double d)
		{
			Vector sum = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				sum[i] = a[i] / d;
			}
			return sum;
		}

		public static Vector operator +(Vector a, Vector b)
		{
			Vector sum = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				sum[i] = a[i] + b[i];
			}
			return sum;
		}

		public static Vector operator +(Vector a, double b)
		{
			Vector sum = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				sum[i] = a[i] + b;
			}
			return sum;
		}

		public static Vector operator -(Vector a, Vector b)
		{
			Vector sum = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				sum[i] = a[i] - b[i];
			}
			return sum;
		}

		public static Vector operator -(Vector a, double b)
		{
			Vector sum = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				sum[i] = a[i] - b;
			}
			return sum;
		}

		public static Vector operator -(Vector a)
		{
			Vector v = new Vector(a.Length);
			for (int i = 0; i < a.Length; i++)
			{
				v[i] = -a[i];
			}
			return v;
		}

		public static implicit operator Vector(double[] d)
		{
			return new Vector(d);
		}

		public static implicit operator double[](Vector v)
		{
			return v.val;
		}

		public static explicit operator float[](Vector v)
		{
			return Array.ConvertAll(v.val, Convert.ToSingle);
		}

		public static explicit operator Vector(float[] f)
		{
			return new Vector(f);
		}

		public object Clone()
		{
			Vector v = new Vector(this.Length);
			Array.Copy(this.val, v.val, Length);
			return v;
		}

		public static Vector Random(double[] high, double[] low, int seed)
		{
			Random rand = new Random(seed);
			Vector v = new Vector(high.Length);
			double l, r;
			for (int i = 0; i < high.Length; i++)
			{
				l = low[i];
				r = high[i] - l;
				v[i] = l + rand.NextDouble() * r;
			}
			return v;
		}

		public static double Max(Vector v)
		{
			double max = Double.MinValue;
			double d;
			for (int i = 0; i < v.Length; i++)
			{
				d = v[i];
				if (d > max) max = d;
			}
			return max;
		}

		public static double Min(Vector v)
		{
			double min = Double.MaxValue;
			double d;
			for (int i = 0; i < v.Length; i++)
			{
				d = v[i];
				if (d < min) min = d;
			}
			return min;
		}

		public bool Equals(Vector other)
		{
			if (Length != other.Length) return false;
			for (int i = 0; i < Length; i++)
				if (this[i] != other[i]) return false;
			return true;
		}

		public override int GetHashCode()
		{
			return val.GetHashCode();
		}

		public override bool Equals(object obj)
		{
			if (obj is Vector) return this.Equals(obj as Vector);
			else return false;
		}
	}
}
