/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib
{
    /// <summary>
    /// Represents voxel in 3D-space that has a volume.
    /// </summary>
    public class Voxel : IEquatable<Voxel>
    {
        /// <summary>
        /// Holds x-coordinate dimension
        /// </summary>
        public float SizeX { 
            get { return (float)size.X; }
            set { if (value < 0) throw new TPCException("Tried to set negative size for x-dimension."); size.X = value; }
        }
        /// <summary>
        /// Holds y-coordinate dimension
        /// </summary>
        public float SizeY { 
            get { return (float)size.Y; }
            set { if (value < 0) throw new TPCException("Tried to set negative size for y-dimension."); size.Y = value; }
        }
        /// <summary>
        /// Holds z-coordinate dimension
        /// </summary>
        public float SizeZ { 
            get { return (float)size.Z; }
            set { if (value < 0) throw new TPCException("Tried to set negative size for z-dimension."); size.Z = value; }
        }
        /// <summary>
        /// Holds volume of voxel in units
        /// </summary>
        public float Volume { get { return (float)(size.X * size.Y * size.Z); } }
        /// <summary>
        /// Point holding voxel dimension information.
        /// </summary>
        private Point size = new Point();
        /// <summary>
        /// Constructs voxel from another voxel
        /// </summary>
        /// <param name="v">voxel</param>
        public Voxel(Voxel v)
        {
            size = new Point(v.SizeX, v.SizeY, v.SizeZ);
        }
        /// <summary>
        /// Constructs voxel with desired size
        /// </summary>
        /// <param name="sizex">x-coord size</param>
        /// <param name="sizey">y-coord size</param>
        /// <param name="sizez">z-coord size</param>
        public Voxel(float sizex, float sizey, float sizez)
        {
            if (sizex <= 0) throw new TPCException("Tried to create Voxel with x-dimension that is not above zero:" + sizex);
            if (sizey <= 0) throw new TPCException("Tried to create Voxel with y-dimension that is not above zero:" + sizey);
            if (sizez <= 0) throw new TPCException("Tried to create Voxel with z-dimension that is not above zero:" + sizez);
            size = new Point(sizex, sizey, sizez);
        }
        /// <summary>
        /// Constructs voxel with desired size
        /// </summary>
        /// <param name="sizex">x-coord size</param>
        /// <param name="sizey">y-coord size</param>
        /// <param name="sizez">z-coord size</param>
        public Voxel(double sizex, double sizey, double sizez)
            : this((float)sizex, (float)sizey, (float)sizez)
        {
        }
        /// <summary>
        /// Constructs cubic voxel of size 1.0 x 1.0 x 1.0 units
        /// </summary>
        public Voxel()
        {
            size = new Point(1.0, 1.0, 1.0);           
        }
        /// <summary>
        /// Compares voxel equality
        /// </summary>
        /// <param name="v1">1st voxel</param>
        /// <param name="v2">2nd voxel</param>
        /// <returns>true if voxel dimensions are equal</returns>
		public static bool operator ==(Voxel v1, Voxel v2)
		{
			return v1.Equals(v2);
		}

        /// <summary>
        /// Compares voxels inequality
        /// </summary>
        /// <param name="v1">1st tested voxel</param>
        /// <param name="v2">2nd tested voxel</param>
        /// <returns>true if voxel sizes are ineqault at any axis</returns>
        public static bool operator !=(Voxel v1, Voxel v2)
        {
			return !(v1 == v2);
		}
        /// <summary>
        /// Multiplication operator
        /// </summary>
        /// <param name="p1">1st voxel size that is multiplied</param>
        /// <param name="d">multiplicator</param>
        public static Voxel operator *(Voxel p1, double d)
        {
            Voxel r = new Voxel();
            r.SizeX = (float)(p1.SizeX * d);
            r.SizeY = (float)(p1.SizeY * d);
            r.SizeZ = (float)(p1.SizeZ * d);
            return r;
        }
        /// <summary>
        /// Multiplication operator
        /// </summary>
        /// <param name="p1">1st voxel size that is multiplied</param>
        /// <param name="d">divisor</param>
        public static Voxel operator /(Voxel p1, double d)
        {
            if (d == 0) throw new TPCInvalidArgumentsException("Cannot divide voxel sizes by zero");
            Voxel r = new Voxel();
            r.SizeX = (float)(p1.SizeX / d);
            r.SizeY = (float)(p1.SizeY / d);
            r.SizeZ = (float)(p1.SizeZ / d);
            return r;
        }
        /// <summary>
        /// Compares two objects
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>true if object is voxel having equal dimensions</returns>
        public override bool Equals(object obj)
        {
			if (obj is Voxel) return this.Equals(obj as Voxel);
			else return false;
        }
        /// <summary>
        /// Returns a String that represents the current Object.
        /// </summary>
        /// <returns>A String that represents the current Object.</returns>
        public override string ToString()
        {
            return "Voxel[" + SizeX.ToString("G").Replace(',', '.') + " x " +
                              SizeY.ToString("G").Replace(',', '.') + " x " +
                              SizeZ.ToString("G").Replace(',', '.') + "]";
        }
        /// <summary>
        /// Operators != and == require implementation. Just calls 
        /// parent's method.
        /// </summary>
        /// <returns>hashcode for this object</returns>
        public override int GetHashCode()
        {
			return size.GetHashCode();
        }
        /// <summary>
        /// Gets array representation of voxel dimensions.
        /// </summary>
        /// <returns>array holding voxel size in x,y,z order</returns>
        public double[] ToArray() {
            return new double[] { size.X, size.Y, size.Z };
        }
        /// <summary>
        /// Implicit casting to 3D point object
        /// </summary>
        /// <param name="v">input voxel</param>
        /// <returns>point holding voxel size as its coordinates</returns>
        public static implicit operator Point(Voxel v) {
            return new Point(v.SizeX, v.SizeY, v.SizeZ);
        }

		public bool Equals(Voxel other)
		{
			return
				this.SizeX.Equals(other.SizeX) &&
				this.SizeY.Equals(other.SizeY) &&
				this.SizeZ.Equals(other.SizeZ);
		}
	}
}
