/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib
{
    /// <summary>
    /// Abstract class for any console-driven program. All programs 
    /// run from command prompt should inherit this class. This class 
    /// can perform unified help information printing etc.
    /// </summary>
    public abstract class ConsoleProgram
    {
        /// <summary>
        /// All valid parameter categories 
        /// </summary>
        public enum CategoryType{
            /// <summary>
            /// String without preceding tag e. g. -x
            /// </summary>
            STRING,
            /// <summary>
            /// Number without preceding tag e. g. -x
            /// </summary>
            NUMBER,
            /// <summary>
            /// Boolean value defined with tag e. g. -x
            /// </summary>
            PARAM_BOOL,
            /// <summary>
            /// Number with preceding tag e. g. -x
            /// </summary>
            PARAM_NUMBER,
            /// <summary>
            /// Multiple numbers (one or more) with preceding tag e. g. -x
            /// </summary>
            PARAM_NUMBERS,
            /// <summary>
            /// Character with preceding tag e. g. -x
            /// </summary>
            PARAM_CHAR,
            /// <summary>
            /// String with preceding tag e. g. -x
            /// </summary>
            PARAM_STRING
        }      
        
        /// <summary>
        /// Class for program parameter
        /// </summary>
        public class Parameter {
            /// <summary>
            /// Parameter name
            /// </summary>
            public string name;

            /// <summary>
            /// Second name of parameter
            /// </summary>
            public string second_name = "not set";

            /// <summary>
            /// Parameter description
            /// </summary>
            public string[] description;
            
            /// <summary>
            /// True for optional parameter
            /// </summary>
            public bool optional;

            /// <summary>
            /// Parameter type
            /// </summary>
            TypeCode type;

            /// <summary>
            /// Parameter category
            /// </summary>
            public CategoryType category_type;

            /// <summary>
            /// Parameter's value
            /// </summary>
            private object value;

            /// <summary>
            /// This is short description of the parameter, f.ex. -p=description
            /// </summary>
            public string parameter_short_description = "value";

            /// <summary>
            /// Set parameter's value
            /// </summary>
            /// <param name="o">new value</param>
            public void SetValue(object o) { value = o; }

            /// <summary>
            /// Checks, if parameter's value has been set.
            /// </summary>
            public bool HasValue {
                get
                {
                    if (value == null) return false;
                    else return true;
                }
            }

            /// <summary>
            /// Cast parameter's value to string and return it
            /// </summary>
            /// <exception cref="TPCException">if parameter does not have data of this type</exception>
            public string StringValue {
                get {
                    if (value == null) throw new TPCException("Parameter's value has not been set.");
                    if (type != TypeCode.String) throw new TPCException("Parameter type is not string");
                    return (string)value; }
                }

            /// <summary>
            /// Cast parameter's value to double and return it
            /// </summary>
            /// <exception cref="TPCException">if parameter does not have data of this type</exception>
            public double DoubleValue {
                get {
                    if (value == null) throw new TPCException("Parameter's value has not been set.");
                    if (type != TypeCode.Double) throw new TPCException("Parameter type is not double");
                    return (double)value; }
            }

            /// <summary>
            /// Cast parameter's values to double array and return it
            /// </summary>
            /// <exception cref="TPCException">if parameter does not have data of this type</exception>
            public double[] DoubleValueArray
            {
                get
                {
                    if (value == null) throw new TPCException("Parameter's value has not been set.");
                    return (double[])value;
                }
            }

            /// <summary>
            /// Cast parameter's value to char and return it
            /// </summary>
            /// <exception cref="TPCException">if parameter does not have data of this type</exception>
            public char CharValue{
                get{
                    if (value == null) throw new TPCException("Parameter's value has not been set.");
                    if (type != TypeCode.Char) throw new TPCException("Parameter type is not char");
                    return (char)value;
                }
            }

            /// <summary>
            /// Cast parameter's value to boolean and return it
            /// </summary>
            /// <exception cref="TPCException">if parameter does not have data of this type</exception>
            public bool BooleanValue{
                get{
                    if (value == null) throw new TPCException("Parameter's value has not been set.");
                    if (type != TypeCode.Boolean) throw new TPCException("Parameter type is not boolean");
                    return (bool)value;
                }
            }

            /// <summary>
            /// Constructs parameter.
            /// </summary>
            /// <param name="name">name</param>
            /// <param name="description">parameter behaviour (max. 56 chars)</param>
            /// <param name="optional">true for optional parameters</param>
            /// <param name="c_type">parameter category</param>
            public Parameter(string name, string description, bool optional, CategoryType c_type) { 
                this.name = name;
                this.description = new string[]{description};
                this.optional = optional;
                this.category_type = c_type;
                if (c_type.Equals(CategoryType.NUMBER)) this.type = TypeCode.Double;
                if (c_type.Equals(CategoryType.PARAM_BOOL)) this.type = TypeCode.Boolean;
                if (c_type.Equals(CategoryType.PARAM_CHAR))
                {
                    this.type = TypeCode.Char;
                    this.parameter_short_description = "character";
                }
                if (c_type.Equals(CategoryType.PARAM_NUMBER))
                {
                    this.type = TypeCode.Double;
                    this.parameter_short_description = "number";
                }
                if (c_type.Equals(CategoryType.PARAM_STRING))
                {
                    this.type = TypeCode.String;
                    this.parameter_short_description = "value";
                }
                if (c_type.Equals(CategoryType.STRING)) this.type = TypeCode.String;
            }
            /// <summary>
            /// Constructs parameter.
            /// </summary>
            /// <param name="name">name</param>
            /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
            /// <param name="optional">true for optional parameters</param>
            /// <param name="c_type">parameter category</param>
            public Parameter(string name, string[] description, bool optional, CategoryType c_type)
            {
                this.name = name;
                this.description = description;
                this.optional = optional;
                this.category_type = c_type;
                if (c_type.Equals(CategoryType.NUMBER)) this.type = TypeCode.Double;
                if (c_type.Equals(CategoryType.PARAM_BOOL)) this.type = TypeCode.Boolean;
                if (c_type.Equals(CategoryType.PARAM_CHAR))
                {
                    this.type = TypeCode.Char;
                    this.parameter_short_description = "character";
                }
                if (c_type.Equals(CategoryType.PARAM_NUMBER))
                {
                    this.type = TypeCode.Double;
                    this.parameter_short_description = "number";
                }
                if (c_type.Equals(CategoryType.PARAM_STRING))
                {
                    this.type = TypeCode.String;
                    this.parameter_short_description = "value";
                }
                if (c_type.Equals(CategoryType.STRING)) this.type = TypeCode.String;
            }
            /// <summary>
            /// Constructs parameter.
            /// </summary>
            /// <param name="name">name</param>
            /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
            /// <param name="optional">true for optional parameters</param>
            /// <param name="c_type">parameter category</param>
            /// <param name="short_description">parameters short description text</param>
            public Parameter(string name, string[] description, bool optional, CategoryType c_type, string short_description)
            {
                this.name = name;
                this.description = description;
                this.optional = optional;
                this.category_type = c_type;
                this.parameter_short_description = short_description;
                if (c_type.Equals(CategoryType.NUMBER)) this.type = TypeCode.Double;
                if (c_type.Equals(CategoryType.PARAM_BOOL)) this.type = TypeCode.Boolean;
                if (c_type.Equals(CategoryType.PARAM_CHAR)) this.type = TypeCode.Char;
                if (c_type.Equals(CategoryType.PARAM_NUMBER)) this.type = TypeCode.Double;
                if (c_type.Equals(CategoryType.PARAM_STRING)) this.type = TypeCode.String;
                if (c_type.Equals(CategoryType.STRING)) this.type = TypeCode.String;
            }
            /// <summary>
            /// Constructs parameter using also a second name.
            /// </summary>
            /// <param name="name">name</param>
            /// <param name="s_name">secondary name</param>
            /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
            /// <param name="optional">true for optional parameters</param>
            /// <param name="c_type">parameter category</param>
            /// <param name="short_description">parameters short description text</param>
            public Parameter(string name, string s_name, string[] description, bool optional, CategoryType c_type, string short_description)
            {
                this.name = name;
                this.second_name = s_name;
                this.description = description;
                this.optional = optional;
                this.category_type = c_type;
                this.parameter_short_description = short_description;
                if (c_type.Equals(CategoryType.NUMBER)) this.type = TypeCode.Double;
                if (c_type.Equals(CategoryType.PARAM_BOOL)) this.type = TypeCode.Boolean;
                if (c_type.Equals(CategoryType.PARAM_CHAR)) this.type = TypeCode.Char;
                if (c_type.Equals(CategoryType.PARAM_NUMBER)) this.type = TypeCode.Double;
                if (c_type.Equals(CategoryType.PARAM_STRING)) this.type = TypeCode.String;
                if (c_type.Equals(CategoryType.STRING)) this.type = TypeCode.String;
            }
        }// class Parameter

        /// <summary>
        /// Program name. Should not contain spaces
        /// </summary>
        public static string programname = "consoleprogram";

        /// <summary>
        /// Program version number string in format x.x.x.x
        /// </summary>
        public static string Version
        {
            get {
                return System.Reflection.Assembly.GetEntryAssembly().GetName().Version.ToString();
            }
        }

        /// <summary>
        /// Copyright string, e.g. Copyright Turku PET Centre 2007 
        /// </summary>
        public static string copyright = "";

        /// <summary>
        /// description of program behaviour.
        /// </summary>
        public static string description = "no description";

        /// <summary>
        /// Usage information, e.g. 'program inpufile outputfile [-o option]'
        /// </summary>
        public static string usage = "no usage information";

		/// <summary>
        /// Program parameters.
        /// </summary>
        private static List<Parameter> parameters = new List<Parameter>();

		/// <summary>
        /// If true, program should print verbose information.
        /// </summary>
        public static bool verbose = false;

		/// <summary>
        /// Maximum number of parameters, that belong to NUMBER-category.
        /// </summary>
        private static int numerical_parameters = 0;

		/// <summary>
        /// Maximum number of parameters, that belong to STRING-category.
        /// </summary>
        private static int string_parameters = 0;

		/// <summary>
        /// Number of given parameters, that belong to NUMBER-category.
        /// </summary>
        public static int num_param_set = 0;

		/// <summary>
        /// Number of given parameters, that belong to STRING-category.
        /// </summary>
        public static int string_param_set = 0;

        /// <summary>
        /// Checks validity of adding new parameter to list of exiswting ones
        /// </summary>
        /// <param name="type">parameter's category type</param>
        /// <exception cref="TPCException">If parameter of selected type can be added to list of existing parameters</exception>
        protected static void EnsureParameterValidity(CategoryType type) {
            //parameter of multiple numbers cannot co-exist with parameter of single number value without preceding tag e. g. -x
            //because single numbers could not be separated from the list
            if (type == CategoryType.PARAM_NUMBERS) {
                foreach (Parameter p in parameters) {
                    if (p.category_type == CategoryType.NUMBER) 
                        throw new TPCException("Cannot add multiple number parameter to the list, because it already has single number without preceding tag");
                    if (p.category_type == CategoryType.PARAM_NUMBERS)
                        throw new TPCException("Cannot add multiple number parameter to the list, because it already has multiple number parameter");
                }
            } else if (type == CategoryType.NUMBER)
            {
                foreach (Parameter p in parameters)
                {
                    if (p.category_type == CategoryType.PARAM_NUMBERS)
                        throw new TPCException("Cannot add multiple number parameter to the list, because it already has multiple number parameter");
                }
            }
        }

        /// <summary>
        /// Add a new parameter to parameter list.
        /// </summary>
        /// <param name="name">name</param>
        /// <param name="description">parameter behaviour (max. 56 chars)</param>
        /// <param name="optional">true for optional parameters</param>
        /// <param name="c_type">parameter category</param>
        public static void AddParameter(string name, string description, bool optional, CategoryType c_type)
        {
            EnsureParameterValidity(c_type);
            parameters.Add(new Parameter(name, description, optional, c_type));
            if (c_type.Equals(CategoryType.NUMBER)) numerical_parameters++;
            if (c_type.Equals(CategoryType.STRING)) string_parameters++;
        }

		/// <summary>
        /// Add a new parameter to parameter list.
        /// </summary>
        /// <param name="name">name</param>
        /// <param name="description">parameter behaviour (max. 56 chars)</param>
        /// <param name="optional">true for optional parameters</param>
        /// <param name="c_type">parameter category</param>
        /// <param name="short_description">Short description of the parameter, f.ex. -p=description</param>
        public static void AddParameter(string name, string description, bool optional, CategoryType c_type, string short_description)
        {
            EnsureParameterValidity(c_type);
            parameters.Add(new Parameter(name, new string[] { description }, optional, c_type, short_description));
            if (c_type.Equals(CategoryType.NUMBER)) numerical_parameters++;
            if (c_type.Equals(CategoryType.STRING)) string_parameters++;

        }

		/// <summary>
        /// Add a new parameter to parameter list.
        /// </summary>
        /// <param name="name">name</param>
        /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
        /// <param name="optional">true for optional parameters</param>
        /// <param name="c_type">parameter category</param>
        public static void AddParameter(string name, string[] description, bool optional, CategoryType c_type)
        {
            EnsureParameterValidity(c_type);
            parameters.Add(new Parameter(name, description, optional, c_type));
            if (c_type.Equals(CategoryType.NUMBER)) numerical_parameters++;
            if (c_type.Equals(CategoryType.STRING)) string_parameters++;

        }

		/// <summary>
        /// Add a new parameter to parameter list.
        /// </summary>
        /// <param name="name">name</param>
        /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
        /// <param name="optional">true for optional parameters</param>
        /// <param name="c_type">parameter category</param>
        /// <param name="short_description">Short description of the parameter, f.ex. -p=description</param>
        public static void AddParameter(string name, string[] description, bool optional, CategoryType c_type, string short_description)
        {
            EnsureParameterValidity(c_type);
            parameters.Add(new Parameter(name, description, optional, c_type, short_description));
            if (c_type.Equals(CategoryType.NUMBER)) numerical_parameters++;
            if (c_type.Equals(CategoryType.STRING)) string_parameters++;
        }

		/// <summary>
        /// Add a new parameter to parameter list. The parameter has also the second, optional name.
        /// </summary>
        /// <param name="name">name</param>
        /// <param name="s_name">Paremeter's second name</param>
        /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
        /// <param name="optional">true for optional parameters</param>
        /// <param name="c_type">parameter category</param>
        /// <param name="short_description">Short description of the parameter, f.ex. -p=description</param>
        public static void AddParameter(string name, string s_name, string[] description, bool optional, CategoryType c_type, string short_description)
        {
            EnsureParameterValidity(c_type);
            parameters.Add(new Parameter(name, s_name, description, optional, c_type, short_description));
            if (c_type.Equals(CategoryType.NUMBER)) numerical_parameters++;
            if (c_type.Equals(CategoryType.STRING)) string_parameters++;
        }

		/// <summary>
        /// Add a new parameter to parameter list. The parameter has also the second, optional name.
        /// </summary>
        /// <param name="name">name</param>
        /// <param name="s_name">Paremeter's second name</param>
        /// <param name="description">parameter behaviour (max. 56 chars / line)</param>
        /// <param name="optional">true for optional parameters</param>
        /// <param name="c_type">parameter category</param>
        public static void AddParameter(string name, string s_name, string[] description, bool optional, CategoryType c_type)
        {
            EnsureParameterValidity(c_type);
            parameters.Add(new Parameter(name, s_name, description, optional, c_type, "value"));
            if (c_type.Equals(CategoryType.NUMBER)) numerical_parameters++;
            if (c_type.Equals(CategoryType.STRING)) string_parameters++;
        }

        /// <summary>
        /// Returns true if parameter was given from command prompt.
        /// </summary>
        /// <param name="name">parameter's name</param>
        /// <returns>true if parameter is given from command prompt</returns>
        public static bool HasParameter(string name)
        {
            foreach (Parameter p in parameters)
            {
                if (p.name.Equals(name))
                    return p.HasValue;
            }
            return false;
        }

		/// <summary>
        /// Returns a parameter, which name was given.
        /// </summary>
        /// <param name="name">parameter's name</param>
        /// <returns>desired parameter</returns>
        /// <exception cref="TPCException">if parameter is not found</exception>
        public static Parameter GetParameter(string name)
        {
            foreach (Parameter p in parameters) {
                if (p.name.Equals(name) || p.second_name.Equals(name)) return p;
            }
            throw new TPCException("No parameter set for given name.");
        }

		/// <summary>
        /// Returns all parameters.
        /// </summary>
        /// <returns>all parameters</returns>
        public static Parameter[] GetParameters()
        {
            return parameters.ToArray();
        }

		/// <summary>
        /// Prints program info into stdout
        /// </summary>
        public static void PrintInfo()
        {
            Console.WriteLine();
            Console.WriteLine(programname+" "+Version+ " "+copyright);
            Console.WriteLine();
            Console.Write(description);
            Console.WriteLine("\r\n");
            Console.Write(usage);
            Console.WriteLine("\r\n");
            //resolve maximum parameter name length
            int max_parameter_name_length = int.MinValue;
            for (int i = 0; i < parameters.Count; i++) {
                if (parameters[i].name.Length + parameters[i].parameter_short_description.Length > max_parameter_name_length)
                {
                    max_parameter_name_length = parameters[i].name.Length + parameters[i].parameter_short_description.Length;
                }
            }
            for (int i = 0; i < parameters.Count; i++)
            {
                string name = parameters[i].name;
                
                if (parameters[i].category_type.Equals(CategoryType.PARAM_BOOL))
                    name = "-" + name;
                if (parameters[i].category_type.Equals(CategoryType.PARAM_CHAR))
                    name = "-" + name + "=<" + parameters[i].parameter_short_description + ">";
                if (parameters[i].category_type.Equals(CategoryType.PARAM_NUMBER))
                    name = "-" + name + "=<" + parameters[i].parameter_short_description + ">";
                if (parameters[i].category_type.Equals(CategoryType.PARAM_STRING))
                    name = "-" + name + "=<" + parameters[i].parameter_short_description + ">";
                if (!(parameters[i].category_type.Equals(CategoryType.NUMBER) || parameters[i].category_type.Equals(CategoryType.STRING)))
                {
                    
                    name = name.PadRight(max_parameter_name_length + 4, ' ') + ":  ";
                    Console.Write(name);
                    Console.WriteLine(parameters[i].description[0]);
                    if (parameters[i].description.Length > 1)
                    {
                        for (int ii = 1; ii < parameters[i].description.Length; ii++)
                        {
                            string line = parameters[i].description[ii];
                            line = line.PadLeft(max_parameter_name_length + 9 + line.Length, ' ');
                            Console.WriteLine(line);
                        }
                    }
                }
            }
            string text = "-" + "verbose";
            text = text.PadRight(max_parameter_name_length + 4, ' ') + ":  ";
            Console.WriteLine(text + "The software prints a lot of information about what it");
            text = "is doing.";
            text = text.PadLeft(max_parameter_name_length + 9 + text.Length, ' ');
            Console.WriteLine(text);

            text = "-" + "build";
            text = text.PadRight(max_parameter_name_length + 4, ' ') + ":  ";
            Console.WriteLine(text + "The software prints a build information text");
            
            Console.WriteLine();
            Console.WriteLine("library version:"+TPClib.Version.GetVersionInformation());
        }// PrintInfo

		/// <summary>
        /// Resolves program parameterNames. Exist running program with exit code 1 in case of errors.
        /// </summary>
        /// <param name="args">program arguments from command prompt</param>
        public static void ResolveParameters(string[] args)
        {
            string[] splitted_s;
            string left;
            string right;
            char[] delimiterChars = { '=' };
            double val;
            bool param_set = false; // used in for-loop
            num_param_set = 0;
            string_param_set = 0;
            //culture used to remove all culture specific behavior in order to force the same input type in all cultures
            System.Globalization.CultureInfo culture = new System.Globalization.CultureInfo("fi-FI");

            for (int i = 0; i < args.Length; i++) {

                if (args[i].Equals("--help") || args[i].Equals("--h") || args[i].Equals("-help") || args[i].Equals("-h") || args[i].Equals("/help") || args[i].Equals("/h")) 
                {
                    PrintInfo();
                    Environment.Exit(0);
                }
                if (args[i].Equals("--build") || args[i].Equals("-build"))
                {
                    Console.WriteLine();
                    Console.WriteLine(programname + " " + Version);
                    Console.WriteLine();
                    Console.WriteLine(copyright);
                    Console.WriteLine();
                    Console.WriteLine("library version:" + TPClib.Version.GetVersionInformation());
                    Environment.Exit(0);
                }
                if (args[i].Equals("-v") || args[i].Equals("-verbose") || args[i].Equals("--verbose"))
                {
                    verbose = true;
                    param_set = true;
                }
                
                // if argument is a number...
                String a = args[i];
                if (a.Contains(".")) a = a.Replace('.', ',');
                if (Double.TryParse(a, out val))
                {
                    int num = 0;
                    while (num < parameters.Count && !param_set)
                    {
                        if (num_param_set >= numerical_parameters)
                        {
                            PrintInfo();
                            Console.WriteLine();
                            Console.WriteLine("Wrong type of number parameter " + args[i] + ". Program takes only " + numerical_parameters + " numerical parameter(s).");
                            Environment.Exit(1);
                        }
                        if (parameters[num].category_type.Equals(CategoryType.NUMBER) && !(parameters[num].HasValue))
                        {
                            
                            parameters[num].SetValue(val);
                            num_param_set++;
                            param_set = true;
                        }
                        num++;
                    }
                    
                }
                else {
                    // if argument is a parameter...
                    if ((args[i].StartsWith("-") || args[i].StartsWith("/")) && !param_set)
                    {
                        if (args[i].Contains("="))
                        {
                            splitted_s = args[i].Split(delimiterChars);
                            left = splitted_s[0].Remove(0, 1);
                            right = splitted_s[1];
                            if (right.Equals("")) right = "not set";
                        }
                        else
                        {
                            left = args[i].Remove(0, 1);
                            right = "not set";
                        }
                        foreach (Parameter p in parameters)
                        {
                            if ((p.name.Equals(left) || p.second_name.Equals(left)) && !param_set)
                            {
                                if (p.category_type.Equals(CategoryType.PARAM_BOOL))
                                {
                                    p.SetValue(true);
                                    param_set = true;
                                }
                                if (p.category_type.Equals(CategoryType.PARAM_CHAR)) {
                                    if (right.Equals("not set")) {
                                        PrintInfo();
                                        Console.WriteLine();
                                        Console.WriteLine("Parameter -" + left + " uses only character values");
                                        Environment.Exit(1);
                                    }
                                    p.SetValue((right.ToCharArray(0, 1))[0]);
                                    param_set = true;
                                    
                                }
                                if (p.category_type.Equals(CategoryType.PARAM_NUMBER))
                                {
                                    if (right.Contains(".")) right = right.Replace('.', ',');
                                    if (Double.TryParse(right, System.Globalization.NumberStyles.Float, culture, out val)) p.SetValue(val);
                                    else {
                                        PrintInfo();
                                        Console.WriteLine();
                                        Console.WriteLine("Parameter -" + left + " uses only numerical values");
                                        Environment.Exit(1);
                                    }
                                    param_set = true;
                                }
                                if (p.category_type.Equals(CategoryType.PARAM_NUMBERS))
                                {
                                    if (right.Contains(".")) right = right.Replace('.', ',');
                                    List<double> values = new List<double>();
                                    //add numbers until parsing is not success anymore
                                    if (Double.TryParse(right, System.Globalization.NumberStyles.Float, culture, out val)) values.Add(val);
                                    while (i < args.Length-1 && !args[i+1].StartsWith("-") && Double.TryParse(args[i+1], out val)) {
                                        values.Add(val);
                                        i++;
                                    }
                                    if (values.Count > 0)
                                        p.SetValue(values.ToArray());
                                    else
                                    {
                                        PrintInfo();
                                        Console.WriteLine();
                                        Console.WriteLine("Parameter -" + left + " uses only numerical values");
                                        Environment.Exit(1);
                                    }
                                    param_set = true;                                
                                }
                                if (p.category_type.Equals(CategoryType.PARAM_STRING))
                                {
                                    if (right.Equals("not set"))
                                    {
                                        PrintInfo();
                                        Console.WriteLine();
                                        Console.WriteLine("Parameter -" + left + " uses only string values");
                                        Environment.Exit(1);
                                    }
                                    p.SetValue(right);
                                    param_set = true;
                                }
                            }
                        }
                        if (!param_set)
                        {
                            PrintInfo();
                            Console.WriteLine();
                            Console.WriteLine("Parameter -" + left + " not recognised");
                            Environment.Exit(1);
                        }
                    }
                    else
                    {
                        // if argument is a file name or string-type...
                        
                        int num = 0;
                        while (num < parameters.Count && !param_set)
                        {
                            if (string_param_set >= string_parameters)
                            {
                                PrintInfo();
                                Console.WriteLine();
                                Console.WriteLine("Wrong type of parameter " + args[i] + ". Program takes only " + string_parameters + " string parameter(s).");
                                Environment.Exit(1);
                            }
                            if (parameters[num].category_type.Equals(CategoryType.STRING) && !(parameters[num].HasValue))
                            {
                                
                                parameters[num].SetValue(args[i]);
                                string_param_set++;
                                param_set = true;
                            }
                            num++;
                        }
                    }
                }
                
                param_set = false;
            
            }//for
            foreach (Parameter p in parameters) {
                if (!(p.optional) && !(p.HasValue))
                {
                    PrintInfo();
                    Console.WriteLine();
                    Console.WriteLine("Compulsory parameter "+p.name+" is missing.");
                    Environment.Exit(1);
                }
            }
        }//ResolveParameters
    }// class
} // namespace
