/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;

namespace TPClib.Curve
{
    /// <summary>
    /// File containing curve
    /// </summary>
    public abstract class CurveFile : TPCFile
    {
        /// <summary>
        /// Filetype of file.
        /// </summary>
        public CurveFileType FileType;

        /// <summary>
        /// Curves in file.
        /// </summary>
        protected TACTable curves;
        /// <summary>
        /// Gets or sets curve tacs reference_times
        /// </summary>
        public virtual TACTable Curves
        {
            get
            {
                return curves;
            }
            set
            {
                curves = value;
            }
        }

        /// <summary>
        /// All the comment lines at the end of the file
        /// </summary>
        public string[] Comments = new string[0];

        /// <summary>
        /// Reads tacs from a file. Abstract function that will be overwritten
        /// </summary>
        public override void ReadFile()
        {
            throw new System.NotImplementedException();
        }

        /// <summary>
        /// Writes tacs into file.
        /// </summary>
        /// <exception cref="TPCException">if tacs was not written properly</exception>
        public override void WriteFile()
        {
            throw new System.NotImplementedException();
        }

        /// <summary>
        /// Reads image file.
        /// </summary>
        /// <param name="filename">image filename</param>
        /// <returns>image file that has all tacs in image</returns>
        /// <exception cref="TPCException">if reading of file failed</exception>
        public static CurveFile ReadFile(string filename)
        {
            CurveFileType filetype;
            try
            {
                filetype = CurveFile.ResolveFileFormat(filename);
            }
            catch (Exception)
            {
                throw new TPCException("Failed to recognize file " + filename);
            }
            switch (filetype)
            {
                case CurveFileType.CPT:
                    CPTCurveFile file_CPT = new CPTCurveFile(filename);
                    file_CPT.ReadFile();
                    return file_CPT;
                case CurveFileType.IDWC:
                    IDWCCurveFile file_IDWC = new IDWCCurveFile(filename);
                    file_IDWC.ReadFile();
                    return file_IDWC;
                case CurveFileType.Plasma_DFT:
                    DFTPlasmaCurveFile file_DFT = new DFTPlasmaCurveFile(filename);
                    file_DFT.ReadFile();
                    return file_DFT;
                case CurveFileType.Regional_DFT:
                    DFTRegionalCurveFile file_Regional_DFT = new DFTRegionalCurveFile(filename);
                    file_Regional_DFT.ReadFile();
                    return file_Regional_DFT;
                case CurveFileType.Unknown_DFT:
                    DFTCurveFile file_Unknown_DFT = new DFTCurveFile(filename);
                    file_Unknown_DFT.ReadFile();
                    return file_Unknown_DFT;
                case CurveFileType.IF:
                case CurveFileType.XML:
                default:
                    throw new TPCException("Cannot read curve data with this method since type " + filetype + " is not currently supported.");
            }
        }

        /// <summary>
        /// Resolves file format from available file formats in library.
        /// </summary>
        /// <param name="filename">full filename</param>
        /// <returns>curve filetype as enumerator</returns>
        public static CurveFileType ResolveFileFormat(string filename)
        {
            if (DFTPlasmaCurveFile.CheckFormat(filename)) return CurveFileType.Plasma_DFT;
            if (DFTRegionalCurveFile.CheckFormat(filename)) return CurveFileType.Regional_DFT;
            if (IFCurveFile.CheckFormat(filename)) return CurveFileType.IF;
            if (IDWCCurveFile.CheckFormat(filename)) return CurveFileType.IDWC;
            if (CPTCurveFile.CheckFormat(filename)) return CurveFileType.CPT;
            if (DFTCurveFile.CheckFormat(filename)) return CurveFileType.Unknown_DFT;
            throw new TPCException("Failed to recognize file format.");
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] ReadWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };
            List<string> comment_lines = new List<string>();

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else if (line[0] == '#')
                {
                    comment_lines.Add(line);
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            int oldLength = Comments.Length;
            Array.Resize<string>(ref Comments, oldLength + comment_lines.Count);
            Array.Copy(comment_lines.ToArray(), 0, Comments, oldLength, comment_lines.Count);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

    }
}
