/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using TPClib.Common;

namespace TPClib.Curve
{
    /// <summary>
    /// DFT file that contains Region Of Interest (ROI) tacs
    /// </summary>
    public class DFTRegionalCurveFile : DFTCurveFile
    {
        /// <summary>
        /// DFT file header
        /// </summary>
        public struct DFTRegionalFileHeader
        {
            /// <summary>
            /// StudyId. For example: exam0001
            /// </summary>
            public String StudyID;
            /// <summary>
            /// the first string in the DFT file, normally DFT1
            /// </summary>
            public String DFTHeading;
            /// <summary>
            /// Unit of time. Can be (min) or (sec) or (s).
            /// </summary>
            public TimeUnit timeunit;
        };
        /// <summary>
        /// This fileheader contains all DFT file header information like DFT1 exam01...
        /// </summary>
        public DFTRegionalFileHeader dftFileheader;

		/// <summary>
        /// Gets and sets the TACTable value
        /// </summary>
        new public RegionalTACTable Curves
        {
            get
            {
                return (RegionalTACTable)curves;
            }
            set
            {
                curves = value;
            }
        }

		/// <summary>
        /// Initializes DFTCurvefile object with filename. After this you can use ReadFile() and WriteFile() 
        /// method straight without filename parameter.
        /// </summary>
        /// <param name="filename">Name of the file</param>
        public DFTRegionalCurveFile(String filename) : base(filename)
        {
            // lets initialize all the controls first
            init();
        }

        /// <summary>
        /// Default constructor
        /// </summary>
        public DFTRegionalCurveFile() : base()
        {
            // lets initialize all the controls first
            init();
        }

        /// <summary>
        /// Generates a DFTRegionalfile with same header information as filein. Curve tacs is not included
        /// </summary>
        /// <param name="filein">DFTRegionalCurveFile object containing the initial information</param>
        public DFTRegionalCurveFile(DFTRegionalCurveFile filein) : base()
        {
            // lets initialize all the controls first
            init();

            this.dftFileheader.DFTHeading = filein.dftFileheader.DFTHeading;
            this.dftFileheader.StudyID = filein.dftFileheader.StudyID;
            this.dftFileheader.timeunit = filein.dftFileheader.timeunit;
            this.FileType = filein.FileType;

            if (filein.Comments != null)
            {
                int oldLength = Comments.Length;
                Array.Resize<string>(ref Comments, oldLength + filein.Comments.Length);
                Array.Copy(filein.Comments, 0, Comments, oldLength, filein.Comments.Length);
            }

            // TACTable information no curve tacs added
            Curves = new RegionalTACTable(filein.Curves.GetTimeCells());
            Curves.Timetype = filein.Curves.Timetype;
            Curves.Unit = filein.Curves.Unit;
        }

        /// <summary>
        /// Checks file format. Only DFT regional file format is recognized.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DFT plagional file</returns>
        public new static bool CheckFormat(string filename)
        {
            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
			string text;
			using (StreamReader reader = new StreamReader(new FileStream(filename, FileMode.Open, FileAccess.Read), new ASCIIEncoding()))
			{
				text = reader.ReadLine();
				StringReader strReader = new StringReader(text);
			}
            // if there is no header in the file we generate the columns manually
            return (text.Trim().Substring(0, 3).Equals("DFT"));
        }

        /// <summary>
        /// Reads the file. Filename must be given.
        /// </summary>
		public override void ReadFile()
		{
			if (filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");

			// we initialize all the objects first
			init();

			// we open the file and put its contents to String which is
			// read with StringReader. We can close the file after String is in memory
			string text;
			using (StreamReader reader = new StreamReader(new FileStream(filename, FileMode.Open, FileAccess.Read), new ASCIIEncoding()))
			{
				text = reader.ReadToEnd();
			}
			StringReader strReader = new StringReader(text);
			try
			{
				// if there is no header in the file we generate the columns manually
				if (!(text.Trim().Substring(0, 3).Equals("DFT")))
				{
					// this class does not read files without headers
					throw new TPCDFTCurveFileException("No correct 'DFT' header tag found.");
				}
				else
				{
					this.readHeader(ref strReader);
				}

				// readcurves method implemented in DFTCurvefile base class
				base.readCurves(ref strReader, this.dftFileheader.timeunit);
			}
			finally
			{
				strReader.Close();
			}
		}

        /// <summary>
        /// Creates a DFT file from contents of DFTRegionalcurvefile. Filename must be specified before
        /// calling this method
        /// </summary>
        public override void WriteFile()
        {
            if (this.filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            else WriteFile(this.filename);
        }

        /// <summary>
        /// Makes a DFT file in normal dft format (contains DFT1 heading and
        /// all region names)
        /// </summary>
        /// <param name="filename">Name of file</param>
        public override void WriteFile(String filename)
        {
            // Writing the text to file
			using (StreamWriter fileWriter = new StreamWriter(new FileStream(filename, FileMode.Create, FileAccess.Write), new ASCIIEncoding()))
			{
				fileWriter.AutoFlush = true;

				fileWriter.Write(this.GetHeaderAsString() + base.GetDataAsString(this.dftFileheader.timeunit));
			}
		}

		/// <summary>
        /// Returns this object as a string.
        /// </summary>
        /// <returns>string representation of this object</returns>
        public override string ToString()
        {
            return "DFTRegionalCurveFile[filename="+this.filename+" filetype="+this.FileType+" comments="+this.Comments+"]";
        }

		/// <summary>
        /// Returns header information as string.
        /// </summary>
        /// <returns>string representation of header information</returns>
        public string GetHeaderAsString()
        {
            StringBuilder sb = new StringBuilder();
            StringWriter writer = new StringWriter(sb);

			try
			{
				// if there is header tacs present we write it to file
				if (((RegionalTACTable)this.Curves).curveheader != null) writeHeader(ref writer);
			}
			catch (Exception e)
			{
				throw new TPCDFTCurveFileException("DFT File header writing failed: " + e);
			}
			finally
			{
				writer.Flush();
				writer.Close();
			}
            return sb.ToString();
        }

        /// <summary>
        /// Writes the curve tacs to stdout
        /// </summary>
        public void writeToStdOut()
        {
            Console.WriteLine(this.GetHeaderAsString() + base.GetDataAsString(this.dftFileheader.timeunit));
        }

        /// <summary>
        /// Converts timeframe tacs to min/sec. Only the unit in header is converted.
        /// </summary>
        /// <param name="time">The time unit</param>
        public void TimeConversion(TimeUnit time)
        {
            this.dftFileheader.timeunit = time;
        }

		/// <summary>
		/// Explicit cast to plasma curve file
		/// </summary>
		/// <param name="d">Regional curve file to cast</param>
		/// <returns>curve file as DFTPlasmaCurveFile</returns>
		public static explicit operator DFTPlasmaCurveFile(DFTRegionalCurveFile d)
        {
            DFTPlasmaCurveFile p = new DFTPlasmaCurveFile();
            p.Curves = d.Curves;
            p.filename = d.filename;
            p.FileType = d.FileType;

            DFTPlasmaCurveFile.DFTPlasmaFileHeader new_header = new DFTPlasmaCurveFile.DFTPlasmaFileHeader();
            DFTRegionalCurveFile.DFTRegionalFileHeader old_header = d.dftFileheader;
            new_header.timeunit = old_header.timeunit;
            

            p.dftFileheader = new_header;
            p.Comments = d.Comments;
            return p;
        }
        
        #region private members

        /// <summary>
        /// Initializes all the objects used in DFTRegionalCurveFile class
        /// </summary>
        private void init()
        {
            // Dft header tacs is assumed to be in default form if there is nothing yet
            this.dftFileheader.DFTHeading = "DFT";
            this.dftFileheader.StudyID = ".";
            this.dftFileheader.timeunit = TimeUnit.Unknown;

            this.Curves = new RegionalTACTable();
            curves = ((RegionalTACTable)Curves);
        }

        /// <summary>
        /// Reads all the header lines from the DFT file
        /// </summary>
        /// <param name="reader">gets a reference_times of streamreader</param>
        /// <returns>returns true if all success. Returns false if there is no header</returns>
        private void readHeader(ref StringReader reader)
        {
            RegionalTACTable curves = ((RegionalTACTable)this.Curves);

            //bool timeConverting = false; // if true, all timedata must be converted to seconds
            FrameTimeCell.FrameTimetype frametimetype = FrameTimeCell.FrameTimetype.START;
            string[] words;
            int NumRegions = 0;

            // Temporar header tacs storages.
            String[] RegionNames = null;
            String[] SecondaryRegionNames = null;
            String[] Planes = null;
            Double[] SizesOfRegions = null;

            try
            {
                // Line1: words should be in following order: DFT, regname1,regname2,regname3....
                words = ReadWordsFromLine(ref reader);

                // we store the caption (DFT1 or something else)
                dftFileheader.DFTHeading = words[0];

                // now we know the amount of region columns so we can allocate memory
                NumRegions = words.Length - 1; // there is one word too much (DFT)

                RegionNames = new String[NumRegions];
                SecondaryRegionNames = new String[NumRegions];
                Planes = new String[NumRegions];
                SizesOfRegions = new Double[NumRegions];

                // lets save the region names to RegionNames array
                for (int i = 1; i < words.Length; i++)
                {
                    RegionNames[i - 1] = words[i];
                    // all sizes of regions are first NaN bacause in some files there can be blanc spaces.
                    // and the blanc regions may remain zero if it is not marked as "." This prevents some rare
                    // exceptional cases
                    SizesOfRegions[i - 1] = Double.NaN;
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("Failed to read DFT file header (line1): " + e);
            }

            // Line2: the words should be in order: StudyID, secname1,secname2,secname3...
            try
            {
                words = ReadWordsFromLine(ref reader);
                this.dftFileheader.StudyID = words[0];
                for (int i = 1; i < words.Length; i++)
                {
                    SecondaryRegionNames[i - 1] = words[i];
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("Failed to read DFT file header (line2): " + e);
            }

            // Line3: the words should be in order: timeunit, plane1,plane2,plane3...
            try
            {
                words = ReadWordsFromLine(ref reader);

                // The Curves object is still null end we cant make it yet bacause we dont have
                // enough information. So we put unit information to temporar place "unitstring"
                String unitstring = words[0];
                this.Curves.Unit = DataUnit.Parse(unitstring);

                for (int i = 1; i < words.Length; i++)
                {
                    Planes[i - 1] = words[i];
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("Failed to read DFT file header (line3): " + e);
            }

            // Line4: the words should be in order: time,timeunit(min),regionsize1,regionsize2,regionsize3...
            try
            {
                words = ReadWordsFromLine(ref reader);

                // "Times" means that there is two time columns in the file
                if (words[0].Equals("Times")) frametimetype = FrameTimeCell.FrameTimetype.START_END;
                else frametimetype = FrameTimeCell.FrameTimetype.MID;

                // Time unit
                if (words[1].Equals("(min)"))
                {
                    this.dftFileheader.timeunit = TimeUnit.Minute;
                    //timeConverting = true; // because TACtable uses only seconds, we must convert all the timevalues
                }
                else // if there is something else than (min) we expect that all values are given in seconds
                {
                    this.dftFileheader.timeunit = TimeUnit.Second;
                }

                // region sizes
                for (int i = 2; i < words.Length; i++)
                {
                    if (words[i].Equals(".")) SizesOfRegions[i - 2] = Double.NaN;
                    else SizesOfRegions[i - 2] = System.Convert.ToDouble(words[i], new System.Globalization.CultureInfo("en-GB"));
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("Failed to read DFT file header (line4): " + e);
            }
            
            // headercell list must be initialized
            //curveheader = new List<CurveHeader>();

            // Now we must fill the tacs in CurveHeader list
            for (int i = 0; i < NumRegions; i++)
            {
                curves.curveheader.Add(new CurveHeader(
                    RegionNames[i],
                    SecondaryRegionNames[i],
                    Planes[i],
                    SizesOfRegions[i]));
            }

            // Now we know all information needed for new TACTable (numb of regions)
            for (int i = 0; i < NumRegions; i++) this.Curves.AddColumn();

            this.Curves.Timetype = frametimetype;
        }
        /// <summary>
        /// Writes all DFT header tacs.
        /// </summary>
        /// <param name="writer">writer used for writing</param>
        private void writeHeader(ref StringWriter writer)
        {
            // we generate curveheaders, if there is none
            if (Curves.Columns > 0 && ((RegionalTACTable)Curves).curveheader.Count == 0)
            {
                for (int i = 0; i < Curves.Columns; i++) ((RegionalTACTable)Curves).curveheader.Add(new CurveHeader());
            }

            // If there is normal curveheaders among header list, them must be converted to empty DFTCurveheaders            
            List<CurveHeader> headlist = new List<CurveHeader>();
            for (int i=0; i<Curves.curveheader.Count; i++ )
            {
                if (Curves.curveheader[i] is CurveHeader)
                    { headlist.Add(new CurveHeader(Curves.curveheader[i] as CurveHeader)); }
                else 
                    { headlist.Add(new CurveHeader()); }
            }

            // dftHeading: DFT1 for example
            writer.Write(FixStr(dftFileheader.DFTHeading, 20));

            // names of regions: putam,putam,cereb....
            foreach (CurveHeader ch in headlist ) { writer.Write(FixStr(ch.curveName, 20)); }
            writer.Write("\r\n");

            // Line2: StudyID, secondary names (dx,sin...)
            writer.Write(FixStr(dftFileheader.StudyID, 20));
            foreach (CurveHeader ch in headlist) { writer.Write(FixStr(ch.secondaryName, 20)); }
            writer.Write("\r\n");

            // Line3: curveUnit, plane1, palane2, plane3...
			if (Curves.Unit.Equals(DataUnit.Unknown)) writer.Write(FixStr(".", 20));
            else writer.Write(FixStr(Curves.Unit.ToString(), 20));

            foreach (CurveHeader ch in headlist) { writer.Write(FixStr(ch.plane, 20)); }
            writer.Write("\r\n");

            // Line4: tacs (min), sizes of regions....
            if (Curves.Timetype == FrameTimeCell.FrameTimetype.START_END)
            {
                // if there is both start and end tacs, we must write "Times (..unit..)"
                string ustring = dftFileheader.timeunit.ToString();
                if (dftFileheader.timeunit == TimeUnit.Minute) ustring = "(min)";
                else if (dftFileheader.timeunit == TimeUnit.Second) ustring = "(sec)";
                writer.Write(FixStr("Times " + ustring, 22));
            }
            else
            {
                // if there is only one time column, we write "Time (..unit..)"
                string ustring = dftFileheader.timeunit.ToString();
                if (dftFileheader.timeunit == TimeUnit.Minute) ustring = "(min)";
                else if (dftFileheader.timeunit == TimeUnit.Second) ustring = "(sec)";
                writer.Write(FixStr("Time " + ustring, 22));
            }

            // next we write planes
            foreach (CurveHeader ch in headlist)
            {
                if (Double.IsNaN(ch.ROIsize))
                {
                    // weight column can be "."
                    if (ch.curveName.ToLowerInvariant().Equals("weight")) writer.Write(FixStr(".", 20));
                    else writer.Write(FixStr(".",20));
                }
                else writer.Write(FixStr(ch.ROIsize.ToString("e11", new System.Globalization.CultureInfo("en-GB")), 20));
            }
            writer.Write("\r\n");
        }

        #endregion
    }
}
