/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Text;
using System.Data;
using System.IO;
using System.Globalization;

namespace TPClib.Curve
{
    /// <summary>
    /// The fit file format contains the parameterNames of
    /// different dataFunctions fitted to one or more tacs. 
    /// The file format is descripted in here:
    /// TPClib (.NET) > format_fit.doc
    /// </summary>
    public class FITFile : TPCFile
    {
        /// <summary>
        /// Contains the file header.
        /// </summary>
        private FITHeader h;

        /// <summary>
        /// Contains the parameter tacs.
        /// </summary>
        private ParameterTable t;

        /// <summary>
        /// Extension used for backup files
        /// </summary>
        public const string BackupExtension = ".bak";

        /// <summary>
        /// Contains the file header section.
        /// </summary>
        public FITHeader Header
        { 
            get { return h; }
            set { h = value; }
        }

        /// <summary>
        /// Holds the parameter tacs.
        /// </summary>
        public ParameterTable DataTable
        {
            get { return t; }
            set { t = value; }
        }

        /// <summary>
        /// Constrains the number of parameterNames per a line.
        /// </summary>
        public const int MAX_FITPARAMS = 6;

        /// <summary>
        /// Initializes a new instance of the FITfile class.
        /// </summary>
        public FITFile()
        {
            this.Header = new FITHeader();
            this.DataTable = CreateFITDataTable();  
        }

        /// <summary>
        /// Initializes a new instance of the FITfile class.
        /// </summary>
        /// <param name="filename">full filename</param>
        public FITFile(string filename)
        {
            base.filename = filename;
            this.Header = new FITHeader();
            this.DataTable = CreateFITDataTable();  
        }

        /// <summary>
        /// Encapsulates datatable creation. 
        /// </summary>
        /// <returns>A parameter table for FITfile format.</returns>
        private ParameterTable CreateFITDataTable() {

            // column headers
            string[] colName = new string[] { 
            "Start", "End",
            "dataNr", "WSS", "parNr", "Type", "parameterNames" };

            // init tacs table for fit
            ParameterTable pt = new ParameterTable(colName);

            if (pt.Parameters.Count != 7)
                throw new TPCFITfileException("Invariant broken");

            if (pt.Regions.Count != 0)
                throw new TPCFITfileException("Invariant broken");

            return pt;
        }

        /// <summary>
        /// Reads the FIT file. 
        /// </summary>
        /// <remarks>
        /// PRE:  true
        /// POST: The object contains header and parameter tacs
        /// </remarks>
        public override void ReadFile()
        {
            CheckFile(filename);

            // read the file to memory
            string text = File.ReadAllText(filename);

            // split the file to a header and tacs part
            int i = text.IndexOf("Region");
            if (i < 0)
                throw new TPCFITfileException("Wrong File Format");
            string header = text.Substring(0, i);
            string data = text.Substring(text.IndexOf("Region"));
            
            // parse Header
            Header.Parse(header);

            // parse Data
            using (StringReader sr = new StringReader(data))
            {
                // the 7th line: checks that label lines are correct
                String[] ls = sr.ReadLine().Split(new char[] { ' ' },
                    StringSplitOptions.RemoveEmptyEntries);

                if (ls.Length != 9)
                    throw new TPCFITfileException(
                        "file format error");
                
                String line;
                while ((line = sr.ReadLine()) != null)
                {
                    TableCell[] col = parseDataLine(line);
                    
                    if(8 != col.Length)
                        throw new TPCFITfileException("");

                    // cells
                    DataTable.Regions.Add((col[0] as RegionCell), new TableCell[] { col[1], col[2], col[3], col[4], col[5], col[6], col[7] });
                }
                
                if(DataTable.Regions.Count != Header.NumberOfVOIs)
                    throw new TPCFITfileException(
                    "TEST1 In header there is " + Header.NumberOfVOIs +
                    " rows " + "but there is " + DataTable.Regions.Count);
            }
        }

        /// <summary>
        /// Writes the FIT file to disk.
        /// </summary>
        /// <remarks>
        /// PRE: true
        /// POST: 
        /// </remarks>
        public override void WriteFile()
        {
            // Backup old file
            MakeBackup(filename);

            string header = Header.HeaderToString();
            string data = DataToString(DataTable);
            File.WriteAllText(filename, header+data, Encoding.ASCII);
        }

        /// <summary>
        /// Converts the file content to a string
        /// </summary>
        /// <remarks>
        /// PRE:  h and dt are not null
        /// POST: RESULT == (string representation)
        /// </remarks>
        private string DataToString(ParameterTable pt)
        {
            IFormatProvider NUMBER_FORMAT =
                System.Globalization.CultureInfo.CreateSpecificCulture("en-us");

            string s = "";
            s += "Region        Plane  Start   End     ";
            s += "dataNr WSS       parNr Type Parameters\r\n";

            for (int row = 0; row < Header.NumberOfVOIs; row++)
            {
                s += DataTable.Regions.GetHeader(row).Region.Name.PadRight(7);
                s += DataTable.Regions.GetHeader(row).Region.SecondaryName.PadRight(7);
                s += DataTable.Regions.GetHeader(row).Region.Plane.PadRight(9);
                                
                s += ((double)DataTable.Parameters.Get( "start", row )).ToString("0.000", NUMBER_FORMAT).PadRight(7);
                s += ((double)DataTable.Parameters.Get( "end", row )).ToString("0.000", NUMBER_FORMAT).PadRight(7);
                s += ((Int16)DataTable.Parameters.Get( "datanr", row)).ToString().PadLeft(6).PadRight(7);
                s += ((double)DataTable.Parameters.Get( "wss",row )).ToString("0.00E+000", NUMBER_FORMAT).PadRight(7);
                s += ((Int16)DataTable.Parameters.Get( "parnr",row )).ToString().PadLeft(6) + " ";
                s += ((Int16)DataTable.Parameters.Get( "type", row )).ToString("0000").PadRight(5);

                string pars = "";
                double[] table = (DataTable.Parameters.Get( "parameterNames", row ) as ParamTableCell).Parameters;
                if (table.Length > MAX_FITPARAMS)
                    throw new TPCFITfileException("Too many parameterNames");

                for (int i = 0; i < table.Length; i++)
                {
                    pars += table[i].ToString(
                        "0.000000E+000",NUMBER_FORMAT) + " ";
                }
                s += pars.Trim(); // trims away the last " "
                s += "\r\n";
            }

            return s;
        }

        /// <summary>
        /// Checks file format based on filename extension.
        /// </summary>
        /// <remarks>
        /// PRE: true
        /// POST: no expection is thrown if the filename
        /// isn't null, it exist, and has right extension.
        /// </remarks>
        /// <param name="filename">full path to a file</param>
        private static void CheckFile(string filename)
        {
            if (filename == null)
                throw new TPCFITfileException("Null pointer");

            FileInfo file = new FileInfo(filename);

            if (!file.Exists)
                throw new TPCFITfileException("File " + filename + " doesn't exist");

            // makes extension not case-sensitive.
            string ext = file.Extension;
            ext = ext.ToLower();
            if (ext.CompareTo(".fit") != 0)
                throw new TPCFITfileException("Wrong File Extension");
        }

        /// <summary>
        /// Parses file 
        /// </summary>
        /// <remarks>
        /// PRE:   line.Split(' ').Lenght == (from 9 to 15)
        /// POST:  RESULT.Length == 9 + RESULT[8].Length
        ///  RESULT[index] types: 0,1,2 String 
        ///  3,4,6 are Double, 5,7 integer and
        ///  8 is double table.
        /// </remarks>
        private TableCell[] parseDataLine(string line)
        {
            NumberFormatInfo nfi = new NumberFormatInfo();
            nfi.CurrencyDecimalSeparator = ".";

            char[] charSeparators = new char[] { ' ' };
            
            String[] cols = line.Split(charSeparators,
                StringSplitOptions.RemoveEmptyEntries);

            if( cols.Length < 9 )
                throw new TPCFITfileException(
                    "Too few Columns in a row");

            string region = cols[0].ToString();
            string name = cols[1].ToString();
            string plane = cols[2].ToString();

            // RegionHeader information
            RegionCell region_cell = new RegionCell(new RegionCell.RegionStruct(
                region, // region name
                name,   // secondary region name
                plane));// plane

            TableCell[] cells = new TableCell[8];

            cells[0] = (TableCell)region_cell;

            double start = Double.Parse(cols[3], nfi);
            cells[1] = new TableCell(start);

            double end = Double.Parse(cols[4], nfi);
            cells[2] = new TableCell(end);

            int dataNr = Int16.Parse(cols[5]);
            cells[3] = new TableCell(dataNr);

            double WSS = Double.Parse(cols[6], nfi);
            cells[4] = new TableCell(WSS);

            int ParNr = Int16.Parse(cols[7]);
            cells[5] = new TableCell(ParNr);
            
            if( ParNr < 1)
                throw new TPCFITfileException(
                "Error: " + ParNr + ", too few parameterNames");
            if( ParNr > MAX_FITPARAMS)
                throw new TPCFITfileException(
                "Error: " + ParNr + ", too many parameterNames");

            int type = Int16.Parse(cols[8]);
            cells[6] = new TableCell(type);

            double[] param = new double[ParNr];
            try
            {   
                for (int j = 0; j < ParNr; j++)
                {
                    param[j] = Double.Parse(cols[9 + j], nfi);
                }
            }
            catch (IndexOutOfRangeException)
            {
                throw new TPCFITfileException(
                "parNr doesn't match parameterNames table length");
            }

            cells[7] = new ParamTableCell(param);

            if(cols.Length != 9 + param.Length)
                throw new TPCFITfileException(
                    "parNr doesn't match");

            return cells;
        }

        /// <summary>
        /// Check for existing file and make backup, overwriting old backup file
        /// </summary>
        /// <exception cref="TPCFITfileException">
        /// Exception thrown, if file can't be opened or backup can't be made
        /// </exception>
        private void MakeBackup(string fname)
        {
            try
            {
                if (File.Exists(fname))
                {
                    string backup = fname + BackupExtension;
                    File.Copy(fname, backup, true);
                }
            }
            catch (Exception exc)
            {
                throw new TPCFITfileException(
                    "Read-only: \n" + exc.Message);
            }
        }
    }

    /// <summary>
    /// Exception class for exceptions occurring in class FITfile.
    /// </summary>
    public class TPCFITfileException : TPCException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCFITfileException()
            : base("No message")
        {
        }

        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCFITfileException(string message)
            : base(message)
        {
        }
    }
}
