/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace TPClib.Curve
{
    /// <summary>
    /// IDWC curve file
    /// </summary>
    public class IDWCCurveFile : CurveFile
    {
        // Public members
        # region public
        /// <summary>
        /// Default constructor
        /// </summary>
        public IDWCCurveFile()
        {
            this.Init();
        }

        /// <summary>
        /// Constructor with filename
        /// </summary>
        public IDWCCurveFile(string fname)
        {
            this.Init();
            this.filename = fname;
        }

        /// <summary>
        /// Write object tacs to IDWC file
        /// </summary>
        public override void WriteFile()
        {
            this.WriteFile(this.filename);
        }


        /// <summary>
        /// Write object tacs to IDWC file
        /// </summary>
        /// <param name="fname">Output file name</param>
        /// <exception cref="TPCIDWCFileException">
        /// Exception thrown, if file can't be opened or writing fails
        /// </exception>
        public void WriteFile(string fname)
        {
            try
            {
                if (File.Exists(fname) && IsReadOnly(fname)) throw new TPCIDWCFileException("File is read-only");
                // Backup old file
                MakeBackup();

                // Write IDWC file
                File.WriteAllText(fname, FileToString(), Encoding.ASCII);
            }

            // Writing failed
            catch (Exception exc)
            {
                throw new TPCIDWCFileException("Writing to file failed: " + exc.Message);
            }
        }

        /// <summary>
        /// Read IDWC file contents and create a Multicurve object, emptying its old contents
        /// </summary>
        public override void ReadFile()
        {
            this.ReadFile(this.filename);
        }

        /// <summary>
        /// Read IDWC file contents and create a Multicurve object, emptying its old contents
        /// </summary>
        /// <param name="fname">Name of the file to be read</param>
        public new void ReadFile(string fname) 
        {
            try
            {
                // Read file contents to a string
                string text = File.ReadAllText(fname, Encoding.ASCII);

                // Time frames list,
                // tacs list,
                // number of samples per region
                List<FrameTimeCell> frametimes = new List<FrameTimeCell>();
                List<WeightedTableCell> data = new List<WeightedTableCell>();
                int samples;

                // Read tacs to lists
                ReadData(text, out frametimes, out data, out samples);
                
                // Insert tacs into TACTable
                this.Curves = new TACTable(frametimes.GetRange(0, samples).ToArray());

                // Loop: read 'sample count' lines, make a new column to table
                // Repeat until no more lines
                while (data.Count > 0)
                {
                    // Take first region and add it to table
                    Curves.AddColumn(data.GetRange(0, samples).ToArray());
                    // Remove region from list
                    frametimes.RemoveRange(0, samples);
                    data.RemoveRange(0, samples);
                }
            }

            catch (Exception exc)
            {
                throw new TPCIDWCFileException("Reading file failed: " + exc.Message);
            }

        }

        /// <summary>
        /// Checks file format.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>
        /// True if file exists and is a valid, readable IDWC file.
        /// False in all other cases.
        /// </returns>
        public static bool CheckFormat(string filename)
        {
            bool return_value = false;

            // If file exists, try to read file with readFile()
            try
            {
                FileInfo file = new FileInfo(filename);
                if (file.Exists)
                {
                    IDWCCurveFile idwcfile = new IDWCCurveFile();
                    idwcfile.filename = filename;
                    idwcfile.ReadFile();
                    // If readfile throws an exception, return false
                    // File read correctly
                    return_value = true;
                }
            }
            catch
            {
                return_value = false;
            }
                return return_value;
        }

        /// <summary>
        /// Gives the string representation of this IDWC file
        /// </summary>
        /// <returns>String representation of this IDWC file</returns>
        public string FileToString()
        {
            StringBuilder sbuild = new StringBuilder();

            // Add comment lines to the beginning of string
            foreach (string comment in Comments)
            {
                sbuild.AppendLine(comment);
            }

            // Get the size of regions (== number of rows)
            sbuild.AppendLine(Curves.Rows.ToString("D", NUMBER_FORMAT));

            // NOTE: Region numbering starts from 1, but column numbering from 0
            for (int region = 0; region < Curves.Columns; region++)
            {
                sbuild.Append(RegionString(region));
            }

            return sbuild.ToString().Trim();
        }        

        # endregion

        // Private members
        #region private
        #region constants
        /// <summary>
        /// Delimiter characters between fields in a IDWC file
        /// </summary>
        private char[] DELIMS = { ';', ' ', '\t', '\n', '\r' };

        /// <summary>
        /// IDWC file comment lines
        /// </summary>
        private string[] COMMENTS = { "#", ";" };

        /// <summary>
        /// Extension used for backup files
        /// </summary>
        private string BACKUP_EXTENSION = ".bak";

        /// <summary>
        /// US English notation for numbers assumed
        /// </summary>
        private IFormatProvider NUMBER_FORMAT = System.Globalization.CultureInfo.CreateSpecificCulture("en-us");

        #endregion

        /// <summary>
        /// Standard class instance initialization
        /// Called in all constructors
        /// </summary>
        private void Init()
        {
            // New datatable: timetype is time frame mid time
            FileType = CurveFileType.IDWC;
            Comments = new String[0];
            this.Curves = new TACTable();
            this.filename = null;
        }

        /// <summary>
        /// Check for existing file and make backup, overwriting old backup file
        /// </summary>
        /// <exception cref="TPCIDWCFileException">
        /// Exception thrown, if file can't be opened or backup can't be made
        /// </exception>
        private void MakeBackup()
        {
            try
            {
                if (File.Exists(this.filename))
                {
                    string backup = this.filename + BACKUP_EXTENSION;
                    File.Copy(this.filename, backup, true);
                }
            }
            catch (Exception exc)
            {
                throw new TPCIDWCFileException("Can't open file for writing: " + exc.Message );
            }
        }

        /// <summary>
        /// Check if file is read-only
        /// </summary>
        /// <param name="filename">File to check</param>
        /// <returns>True, if file is read-only. False otherwise.</returns>
        private bool IsReadOnly(string filename)
        {
            return ((File.GetAttributes(this.filename) & FileAttributes.ReadOnly) == FileAttributes.ReadOnly);
        }

        /// <summary>
        /// Filters comments and empty lines
        /// </summary>
        /// <param name="sreader">StringReader holding the file to be filtered</param>
        private List<string> Filter(ref StringReader sreader)
        {
            List<string> filtered = new List<string>();
            string line;
            List<string> comment_lines = new List<string>();
            while ((line = sreader.ReadLine()) != null)
            {
                line.Trim();
                // Check for comments
                if (IsComment(line))
                {
                    comment_lines.Add(line);
                }
                // Non-comments and non-empty lines are returned
                else if (line.Length > 0)
                {
                    filtered.Add(line);
                }
            }

            int oldLength = Comments.Length;
            Array.Resize<string>(ref Comments, oldLength + comment_lines.Count);
            Array.Copy(comment_lines.ToArray(), 0, Comments, oldLength, comment_lines.Count);

            return filtered;
        }

        /// <summary>
        /// Checks if line is a comment line
        /// </summary>
        /// <param name="line">string to check</param>
        /// <returns>True, if line starts with comment string. False otherwise.</returns>
        private bool IsComment(string line)
        {
            foreach (string c in COMMENTS)
            {
                if (line.StartsWith(c))
                    return true;
            }
            return false;
        }

        /// <summary>
        /// Splits a string to fields and converts resulting strings to double
        /// </summary>
        /// <param name="line">String to split</param>
        /// <returns>Array holding values read from the argument string</returns>
        private double[] SplitLine(string line)
        {
            string[] splitted = line.Split(this.DELIMS,StringSplitOptions.RemoveEmptyEntries);
            double[] values = Array.ConvertAll<string, double>(splitted, new Converter<string, double>(stringToDouble));
            return values;
        }

        /// <summary>
        /// Converts a string to double precision floating point number
        /// </summary>
        /// <param name="str">String to parse</param>
        /// <returns>Numeric value represented by str</returns>
        private double stringToDouble(string str)
        {
            return Convert.ToDouble(str, NUMBER_FORMAT);
        }

        /// <summary>
        /// Reads time and sample values from string and inserts them into lists.
        /// </summary>
        /// <param name="text">String holding the contents of an IDWC file</param>
        /// <param name="times">List to which time frames are inserted</param>
        /// <param name="data">List to which weighted sample tacs is inserted</param>
        /// <param name="samples">Number of samples in each region</param>
        /// <returns>Number lines read</returns>
        /// <exception cref="TPCIDWCFileException">Exception thrown if file is not a valid IDWC file</exception>
        private int ReadData( string text,
                              out List<FrameTimeCell> times,
                              out List<WeightedTableCell> data,
                              out int samples)
        {
            StringReader sreader = new StringReader(text);
            times = new List<FrameTimeCell>();
            data = new List<WeightedTableCell>();

            // Filter comments
            List<string> filtered = Filter(ref sreader);

            // Read sample count from the first line and discard the line
            samples = Convert.ToInt16(filtered[0].ToString(NUMBER_FORMAT));
            filtered.RemoveAt(0);

            // Check that number of rows match the number of regions
            // and that number of samples is not negative
            if (samples > 0 && (filtered.Count % samples == 0))
            {

                // Read tacs into frame time and sample tacs lists
                foreach (string line in filtered)
                {
                    // Split text lines into tacs fields; each line should have four fields
                    double[] cells = SplitLine(line);
                    if (cells.Length != 4 )
                    {
                        throw new TPCIDWCFileException("Wrong number of fields");
                    }

                    // Get time frame mid time
                    FrameTimeCell ftc = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                    ftc.Value = cells[0];
                    times.Add(ftc);
                    // Get sample value and weight
                    WeightedTableCell wtc = new WeightedTableCell(cells[1], cells[2]);
                    data.Add(wtc);
                }
            }
            else if (samples < 0)
            {
                throw new TPCIDWCFileException("Negative number of samples not accepted");
            }
            return filtered.Count;
        }
        
        /// <summary>
        /// Gets tacs for one region from TACTable and makes a string in IDWC file format
        /// </summary>
        /// <param name="region">Region number</param>
        /// <returns>Region tacs in string</returns>
        private string RegionString(int region)
        {
            StringBuilder sbuild = new StringBuilder();

            // Get tacs from DataTable
            FrameTimeCell[] times = Curves.GetTimeCells();
            TableCell[] column = Curves.GetColumn(region);

            for (int row = 0; row < times.Length; row++)
            {
                // Add tacs to string
                sbuild.Append(times[row].Value.ToString("F1",NUMBER_FORMAT).PadLeft(6));
                sbuild.Append(((WeightedTableCell)column[row]).Value.ToString("F14",NUMBER_FORMAT).PadLeft(18));
                sbuild.Append(((WeightedTableCell)column[row]).Weight.ToString("F14",NUMBER_FORMAT).PadLeft(18));
                // NOTE: Region numbering starts from 1, but column numbering from 0
                sbuild.AppendLine((region+1).ToString("D",NUMBER_FORMAT).PadLeft(3));
            }
            return sbuild.ToString();
        }

        #endregion
    }
}
