/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.Curve
{
    /// <summary>
    /// CSV file that has Inveon specific column meanings
    /// </summary>
	public class InveonCsvFile : TPCFile
	{
        /// <summary>
        /// list os region tacs
        /// </summary>
		private IList<RegionTac> regions = new List<RegionTac>();

        /// <summary>
        /// Start of comment string
        /// </summary>
		private const string Comment = @"#";

        /// <summary>
        /// Total number of TACTables (region groups are in this file)
        /// </summary>
        public int Regions
        {
            get { return regions.Count; }
        }

        /// <summary>
        /// Constructor with parameter
        /// </summary>
        /// <param name="name">filename</param>
		public InveonCsvFile(string name)
		{
			this.filename = name;
		}

        /// <summary>
        /// Gets table for one region group
        /// </summary>
        /// <param name="i">region group index</param>
        /// <returns>table containing tacs for region group</returns>
		public TACTable this[int i]
		{
			get
			{
				return regions[i];
			}
		}

        /// <summary>
        /// Gets table for one region group
        /// </summary>
        /// <param name="name">name of the region</param>
        /// <returns>table containing tacs for region group</returns>
		public TACTable this[string name]
		{
			get
			{
				for (int i = 0; i < regions.Count; i++)
				{
					if (regions[i].Name.Equals(name)) return this[i];
				}
				throw new TPCException("No TAC found for region " + name);
		}
		}

        /// <summary>
        /// Reads data from file
        /// </summary>
		public override void ReadFile()
		{
            using (StreamReader sr = new StreamReader(new FileStream(filename, FileMode.Open, FileAccess.Read)))
			{
				string nextLine = Comment;
				while (!sr.EndOfStream && IsComment(nextLine))
				{
					nextLine = sr.ReadLine();
				}
				regions = RegionTac.ReadRegions(nextLine);

				while (!sr.EndOfStream)
				{
					do
					{
						nextLine = sr.ReadLine();
					} while (IsComment(nextLine));
					foreach (RegionTac rt in regions)
					{
						rt.Read(ref nextLine);
					}
				}
			}
		}

        /// <summary>
        /// Writes data into file
        /// </summary>
		public override void WriteFile()
		{
			List<string[]> allLines = new List<string[]>();
			int maxLength = 0;
			foreach (RegionTac rt in regions)
			{
				string[] lines = rt.GetAllLines();
				allLines.Add(lines);
				if (lines.Length > maxLength) maxLength = lines.Length;
			}

			using (StreamWriter sw = new StreamWriter(new FileStream(filename, FileMode.Create)))
			{
				for (int i = 0; i < maxLength; i++)
				{
					for(int j = 0; j < allLines.Count; j++)
					{
						if (allLines[j].Length > i) sw.Write(allLines[j][i]);
						else sw.Write(RegionTac.Row.EmptyRow);
						if (j < allLines.Count - 1) sw.Write(',');
						else sw.WriteLine();
					}
				}
			}
		}

        /// <summary>
        /// Returns true for comment line strings
        /// </summary>
        /// <param name="line">tested line</param>
        /// <returns>true of line is a comment line</returns>
		private static bool IsComment(string line)
		{
			return line.StartsWith(Comment);
		}

        /// <summary>
        /// TAC group class
        /// </summary>
		private class RegionTac
		{
			private const char Separator = '-';

			private static IFormatProvider nFormat = System.Globalization.NumberFormatInfo.InvariantInfo;

			public struct Row
			{
				public const string EmptyRow = @",,,,";

				public float midTime;
				public float value;
				public float upperBound;
				public float lowerBound;
				public float stdDev;

				public Row(float mT, float v, float uB, float lB, float sD)
				{
					midTime = mT;
					value = v;
					upperBound = uB;
					lowerBound = lB;
					stdDev = sD;
				}

				public override string ToString()
				{
					float tC = RegionTac.GetTimeConversion(TimeUnit.Minute);
					float dC = RegionTac.GetDataConversion(DataUnit.BecquerelsPerMillilitre);
					return
						(midTime / tC).ToString("0.000000", nFormat) + ',' +
						(value / dC).ToString("0.000000", nFormat) + ',' +
						(upperBound / dC).ToString("0.000000", nFormat) + ',' +
						(lowerBound / dC).ToString("0.000000", nFormat) + ',' +
						(stdDev / dC).ToString("0.000000", nFormat);
				}
			}

			private string regionName;

			private List<Row> rows = new List<Row>();

			private float timeConversion;

			private float dataConversion;

			public RegionTac(string region, TimeUnit tu, DataUnit du)
			{
				regionName = region;
				timeConversion = GetTimeConversion(tu);
				dataConversion = GetDataConversion(du);
			}

			public string Name { get { return this.regionName; } }

			public Row this[int i] { get { return rows[i]; } }

			public static RegionTac CreateRegion(ref string s)
			{
				string[] split = s.Split(new char[] { ',' }, 6, StringSplitOptions.RemoveEmptyEntries);

				RegionTac rt = null;

				if (split.Length > 4)
				{
					// Parse timeunit; default to minutes
					string[] timeHeader = split[0].Split(new char[] { Separator }, StringSplitOptions.RemoveEmptyEntries);
					string name = timeHeader[0].Trim();
					TimeUnit tu = TimeUnit.Unknown;
					if (timeHeader.Length > 1) tu = ParseTimeUnit(GetUnitString(timeHeader[1]));
					if (tu == TimeUnit.Unknown) tu = TimeUnit.Minute;

					// Parse dataunit; default to Bq/ml
					string[] dataHeader = split[1].Split(new char[] { Separator }, StringSplitOptions.RemoveEmptyEntries);
					DataUnit du = DataUnit.Unknown;
					if (dataHeader.Length > 1) du = DataUnit.Parse(GetUnitString(dataHeader[1]));
					if (du == DataUnit.Unknown) du = DataUnit.BecquerelsPerMillilitre;

					rt = new RegionTac(name, tu, du);
				}
				if (split.Length > 5)
				{
					s = split[5];
				}
				else s = String.Empty;

				return rt;
			}

			public void Add(float time, float val, float up, float low, float stdDev)
			{
				rows.Add(new Row(time * timeConversion, val * dataConversion, up * dataConversion, low * dataConversion, stdDev * dataConversion));
			}

			public bool Read(ref string s)
			{
				float mid, val, up, low, dev;
				if (Single.TryParse(ReadNext(ref s), System.Globalization.NumberStyles.Float, nFormat, out mid) &&
					Single.TryParse(ReadNext(ref s), System.Globalization.NumberStyles.Float, nFormat, out val) &&
					Single.TryParse(ReadNext(ref s), System.Globalization.NumberStyles.Float, nFormat, out up) &&
					Single.TryParse(ReadNext(ref s), System.Globalization.NumberStyles.Float, nFormat, out low) &&
					Single.TryParse(ReadNext(ref s), System.Globalization.NumberStyles.Float, nFormat, out dev))
				{
					Add(mid, val, up, low, dev);
					return true;
				}
				else return false;
			}

			public string[] GetAllLines()
			{
				string[] lines = new string[rows.Count + 1];
				lines[0] = HeaderToString();
				for (int i = 0; i < rows.Count; i++)
				{
					lines[i + 1] = rows[i].ToString();
				}
				return lines;
			}

            /// <summary>
            /// Implicit casting to TACTable
            /// </summary>
            /// <param name="rt">group of TACs</param>
            /// <returns>casted object</returns>
			public static implicit operator TACTable(RegionTac rt)
			{
				TACTable tt = new TACTable();
                tt.AddColumn();
                tt.AddColumn();
                tt.AddColumn();
                tt.AddColumn();
                foreach (Row r in rt.rows)
				{
					tt.AddRow(new FrameTimeCell(FrameTimeCell.FrameTimetype.MID, r.midTime, r.midTime),
                        new TableCell[] { new TableCell(r.value), new TableCell(r.upperBound), new TableCell(r.lowerBound), new TableCell(r.stdDev) });
				}
				return tt;
			}

			public static IList<RegionTac> ReadRegions(string s)
			{
				List<RegionTac> regionList = new List<RegionTac>();
				while (s.Length > 0)
				{
					RegionTac rt = CreateRegion(ref s);
					if (rt is RegionTac) regionList.Add(rt);
				}
				return regionList;
			}

			private static string GetUnitString(string s)
			{
				string sub = string.Empty;
				int start = s.IndexOf('(');
				int end = s.LastIndexOf(')');
				if(start >= 0 && end > 0) sub = s.Substring(start + 1, end - start - 1);
				return sub;
			}

			private string HeaderToString()
			{
				return
					regionName + " " + Separator + @" Time " + @"(min)" + ',' +
					regionName + " " + Separator + @" VI " + @"(Bq/ml)" + ',' +
					regionName + " " + Separator + @" VI " + @"(Bq/ml) (upped bound)" + ',' +
					regionName + " " + Separator + @" VI " + @"(Bq/ml) (lower bound)" + ',' +
					regionName + " " + Separator + @" VI " + @"(Bq/ml) (standard deviation)";
			}

			private static string ReadNext(ref string s)
			{
				string[] split = s.Split(new char[] { ',' }, 2, StringSplitOptions.RemoveEmptyEntries);
				if (split.Length > 1) s = split[1];
				else s = String.Empty;
				return split[0];
			}

			private static float GetDataConversion(DataUnit dUnit)
			{
				try
				{
					return (float)DataUnit.ConvertValue(dUnit, DataUnit.BecquerelsPerMillilitre, 1.0);
				}
				catch { return 1.0f; }
			}

			private static float GetTimeConversion(TimeUnit t)
			{
				try
				{
					return (float)TimeUnit.ConvertValue(t, TimeUnit.Millisecond, 1.0);
				}
				catch { return 1.0f; }
			}

			private static TimeUnit ParseTimeUnit(string s)
			{
				return TimeUnit.Parse(s);
			}

			private static DataUnit ParseDataUnit(string s)
			{
				return DataUnit.Parse(s);
			}
		}
	}
}
