/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Curve
{
    /// <summary>
    /// Table cell which can contain varying amount of Double parameterNames
    /// </summary>
    public class ParamTableCell : TableCell
    {
        /// <summary>
        /// Array of parameterNames
        /// </summary>
        public Double[] Parameters;

        /// <summary>
        /// Creates a ParamTableCell with given parameterNames
        /// </summary>
		/// <param name="parameters">An array of doubles</param>
        public ParamTableCell(Double[] parameters)
        {
            Value = double.NaN;

            if (parameters.Length > 0)
            {
                Parameters = new Double[parameters.Length];
                for (int i = 0; i < parameters.Length; i++) Parameters[i] = parameters[i];
            }
        }

        /// <summary>
        /// Writes the first parameter to string
        /// </summary>
        /// <returns>first parameter as String</returns>
        public override String ToString()
        {
            if (Parameters.Length > 0) return Parameters[0].ToString();
            else return Double.NaN.ToString();
        }

        /// <summary>
        /// Returns copy of the object
        /// </summary>
        /// <returns>Copy of the object.</returns>
        public override TableCell Clone()
        {
            ParamTableCell cell = new ParamTableCell(Parameters);

            cell.Value = this.Value;
            return cell;
        }

        /// <summary>
        /// Checks if two REScells are identical by contents
        /// </summary>
        /// <param name="obj">Another rescell</param>
        /// <returns>True if the cells are identical.</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is ParamTableCell)) return false;

            ParamTableCell rc = (obj as ParamTableCell);

            if (Parameters == null && rc.Parameters == null) return true;
            else if (Parameters == null || rc.Parameters == null) return false;

            if (Parameters.Length != rc.Parameters.Length) return false;

            for (int i = 0; i < Parameters.Length; i++)
            {
                if (!DoubleCheck(Parameters[i], rc.Parameters[i])) return false;
            }

            return true;
        }


        /// <summary>
        /// Compares this region_cell to another
        /// </summary>
        /// <param name="obj">other RESCell object</param>
        /// <returns>0: this equals obj, 1: this smaller than obj, -1: this greater than obj</returns>
        public override int CompareTo(object obj)
        {
            if (!(obj is ParamTableCell)) throw new TPCTableCellException("Cannot compare ParamTableCell to other than ParamTableCell.");

            if ((obj as ParamTableCell).Parameters.Length == 0) return -1;

            if (Parameters.Length > 0) return Parameters[0].CompareTo((obj as ParamTableCell).Parameters[0]);
            else return 1;
        }
        /// <summary>
        /// Gets hash code of this object. Default bahaviour.
        /// </summary>
        /// <returns>returns hash code with default implementation</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}