/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// 
	/// </summary>
	public enum Endianness : byte
	{
		/// <summary>
		/// 
		/// </summary>
		BigEndian,

		/// <summary>
		/// 
		/// </summary>
		LittleEndian
	}

	/// <summary>
	/// 
	/// </summary>
	public class EndianWriter : BinaryWriter
	{
		private EndianWriter(Stream s) : base(s) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="s"></param>
		/// <param name="e"></param>
		/// <returns></returns>
		public static BinaryWriter CreateWriter(Stream s, Endianness e = Endianness.LittleEndian)
		{
			if (e == Endianness.LittleEndian) return new BinaryWriter(s);
			else return new EndianWriter(s);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="ch"></param>
		public override void Write(char ch)
		{
			ReverseWrite(BitConverter.GetBytes(ch));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(double value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(float value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(int value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(short value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(long value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(uint value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(ulong value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void Write(ushort value)
		{
			ReverseWrite(BitConverter.GetBytes(value));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="chars"></param>
		public override void Write(char[] chars)
		{
			foreach (char c in chars)
			{
				this.Write(c);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="chars"></param>
		/// <param name="index"></param>
		/// <param name="count"></param>
		public override void Write(char[] chars, int index, int count)
		{
			if (index + count > chars.Length) throw new ArgumentException();

			for (int i = index; i < index + count; i++)
			{
				this.Write(chars[i]);
			}
		}

		private void ReverseWrite(byte[] buf)
		{
			Array.Reverse(buf);
			this.Write(buf);
		}
	}

	/// <summary>
	/// 
	/// </summary>
	public class EndianReader : BinaryReader
	{
		private EndianReader(Stream s) : base(s) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="s"></param>
		/// <param name="e"></param>
		/// <returns></returns>
		public static BinaryReader CreateReader(Stream s, Endianness e = Endianness.LittleEndian)
		{
			if (e == Endianness.LittleEndian) return new BinaryReader(s);
			else return new EndianReader(s);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override char ReadChar()
		{
			return BitConverter.ToChar(ReverseRead(2), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override float ReadSingle()
		{
			return BitConverter.ToSingle(ReverseRead(4), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override double ReadDouble()
		{
			return BitConverter.ToDouble(ReverseRead(8), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override short ReadInt16()
		{
			return BitConverter.ToInt16(ReverseRead(2), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override int ReadInt32()
		{
			return BitConverter.ToInt32(ReverseRead(4), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override long ReadInt64()
		{
			return BitConverter.ToInt64(ReverseRead(8), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override ushort ReadUInt16()
		{
			return BitConverter.ToUInt16(ReverseRead(2), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override uint ReadUInt32()
		{
			return BitConverter.ToUInt32(ReverseRead(4), 0);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override ulong ReadUInt64()
		{
			return BitConverter.ToUInt64(ReverseRead(8), 0);
		}

		private byte[] ReverseRead(int bytes)
		{
			byte[] buffer = this.ReadBytes(bytes);
			Array.Reverse(buffer);
			return buffer;
		}
	}
}
