/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Analyze 7.5 File class
	/// </summary>
	public class Analyze75File : AnalyzeFile
	{
		// Delegates for IO operations
		private delegate float GetValue();
		private delegate void WriteValue(float f);

		/// <summary>
		/// Image data scaling, slope coefficient
		/// </summary>
		private float Slope
		{
			get
			{
				return (float)(this.imgHeader[@"scl_slope"]);
			}
			set
			{
				this.imgHeader[@"scl_slope"] = (IConvertible)value;
			}
		}

		/// <summary>
		/// Image data scaling, intercept
		/// </summary>
		private float Intercept
		{
			get
			{
				return (float)(this.imgHeader[@"scl_inter"]);
			}
			set
			{
				this.imgHeader[@"scl_inter"] = (IConvertible)value;
			}
		}

		/// <summary>
		/// Default constructor
		/// </summary>
		public Analyze75File()
		{
			this.imgHeader = new Analyze75Header();
			this.filetype = FileType.Analyze75;
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fn">File name</param>
		public Analyze75File(string fn)
			: this()
		{
			this.filename = fn;
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fn">File name</param>
		/// <param name="img">Image</param>
		/// <param name="hdr">Image header</param>
		public Analyze75File(string fn, Image img, ImageHeader hdr)
		{
			this.filename = fn;
			this.imgHeader = new Analyze75Header(hdr);
			this.image = img;
			this.imgHeader.Dim = img.Dim;
			this.filetype = FileType.Analyze75;
		}

        /// <summary>
        /// Constructs an empty (all-zero) Analyze75 file with filename and dimensions.
        /// </summary>
        /// <param name="filename">File name</param>
        /// <param name="dimx">x-dimension</param>
        /// <param name="dimy">y-dimension</param>
        /// <param name="dimz">z-dimension</param>
        public Analyze75File(string filename, int dimx, int dimy, int dimz)
        {
            if (filename.EndsWith(".img") || filename.EndsWith(".hdr"))
                this.filename = filename.Remove(filename.Length - 4);
            else
                this.filename = filename;
            this.imgHeader = new Analyze75Header();
            this.image = new Image(new int[] { dimx, dimy, dimz });
            this.imgHeader.Dim = image.Dim;
            this.filetype = FileType.Analyze75;
        }

        /// <summary>
        /// Constructs an empty (all-zero) Analyze75 file with filename and dimensions.
        /// </summary>
        /// <param name="filename">File name</param>
        /// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
        public Analyze75File(string filename, int[] dim)
        {
            if (filename.EndsWith(".img") || filename.EndsWith(".hdr"))
                this.filename = filename.Remove(filename.Length - 4);
            else
                this.filename = filename;
            this.imgHeader = new Analyze75Header();
            this.image = new Image(dim);
            this.imgHeader.Dim = this.image.Dim;
            this.filetype = FileType.Analyze75;
        }

		/// <summary>
		/// Write image to file
		/// </summary>
		/// <param name="img">Image</param>
		public override void WriteFile(ref Image img)
		{
			// Always write the image in the default Analyze 7.5 orientation
			Image.Flip(Header.Orientation, Analyze75Header.DefaultOrientation, img);
			base.WriteFile(ref img);
			// Flip the image back
			Image.Flip(Analyze75Header.DefaultOrientation, Header.Orientation, img);
		}

		/// <summary>
		/// Header as Analyze 7.5 header
		/// </summary>
		public Analyze75Header AnalyzeHeader { get { return Header as Analyze75Header; } }

		/// <summary>
		/// Image header
		/// </summary>
		public override ImageHeader Header
		{
			get
			{
				return imgHeader;
			}
			set
			{
				imgHeader = new Analyze75Header(value);
			}
		}

		/// <summary>
		/// Read header from this file
		/// </summary>
		/// <returns>Header of this file</returns>
		protected override AnalyzeHeader ReadAnalyzeHeader()
		{
			return new Analyze75Header(ResolveHeaderFilename(this.filename));
		}

		/// <summary>
		/// Check Analyze75 format validness of a file
		/// </summary>
		/// <param name="fn">File to check</param>
		/// <returns>True, if file is in Nifti format</returns>
		public static bool CheckFormat(string fn)
		{
			string filename = ResolveHeaderFilename(fn);

			if (File.Exists(filename))
			{
				using (FileStream fs = new FileStream(filename, FileMode.Open))
				{
					return CheckFormat(fs);
				}
			}
			else return false;
		}

		/// <summary>
		/// Check Nifti format validness of a stream
		/// </summary>
		/// <param name="s">Stream to check</param>
		/// <returns>True, if stream data is in Nifti format</returns>
		public static bool CheckFormat(Stream s)
		{
			Analyze75Header head = new Analyze75Header();
			return head.Read(s);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="slope"></param>
		/// <param name="intercept"></param>
		/// <param name="dt"></param>
		/// <param name="min"></param>
		/// <param name="max"></param>
		protected override void GetScalingFactors(out float slope, out float intercept, DataType dt, float min, float max)
		{
			slope = 1.0f;
			intercept = 0.0f;

			// Use only slope for Analyze 7.5 for compatibility reasons
			if (dt != DataType.FLT32 && dt != DataType.FLT64)
			{
				// Calculate slope
				double absMax = Math.Max(Math.Abs(min), Math.Abs(max));
				double typeMax = Math.Max(MinValue(dt), MaxValue(dt));

				slope = (float)(absMax / typeMax);
			}
		}

		/// <summary>
		/// Resolve the header file name
		/// </summary>
		/// <returns>Full name of the header file</returns>
		private static string ResolveHeaderFilename(string fn)
		{
			string dir = Path.GetDirectoryName(fn);
			string file = Path.GetFileNameWithoutExtension(fn);
			return (dir + Path.DirectorySeparatorChar + file + HEADEREXTENSION);
		}

		/// <summary>
		/// Resolve the data file name
		/// </summary>
		/// <returns>Full name of the data file</returns>
		private static string ResolveDataFilename(string fn)
		{
			string dir = Path.GetDirectoryName(fn);
			string file = Path.GetFileNameWithoutExtension(fn);
			return (dir + Path.DirectorySeparatorChar + file + DATAEXTENSION);
		}
	}

	#region Old Analyze 7.5
	/*    /// <summary>
    /// Mayo Clinic Analyze 7.5 format is a file format for
    /// storing MRI tacs.
    /// An Analyze 7.5 tacs set consists of two files:
    /// Header file (something.hdr): Provides dimensional,
    /// identifying and some processing history information
    /// Image file (something.img): Stream of voxels, whose
    /// datatype and ordering are described by the header file
    /// </summary>
    public class Analyze75File : ImageFile
    {
		public const string AnalyzeHeaderExt = ".hdr";
		public const string AnalyzeImageExt = ".img";

        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public static event IOProcessEventHandler IOProgress;

		/// <summary>
        /// Analyze Header
        /// </summary>
        public Analyze75Header analyzeHeader;

		/// <summary>
        /// Resolves Analyze75 datatype from ImageFile dataype
        /// </summary>
        /// <param name="type">imagefile datatype</param>
        /// <returns>datatype in Analyze75 enumeration</returns>
        public Analyze75Header.Datatype ResolveAnalyze75Datatype(ImageFile.DataType type) {
            switch(type) {
                case DataType.BIT8_U:
                    return Analyze75Header.Datatype.UNSIGNED_CHAR;
                case DataType.BIT16_S:
                    return Analyze75Header.Datatype.SIGNED_SHORT;
                case DataType.BIT32_S:
                    return Analyze75Header.Datatype.SIGNED_INT;
                case DataType.COLRGB:
                    return Analyze75Header.Datatype.RGB;
                case DataType.FLT32:
                    return Analyze75Header.Datatype.FLOAT;
                case DataType.FLT64:
                    return Analyze75Header.Datatype.DOUBLE;
                case DataType.BIT1:
                    return Analyze75Header.Datatype.BINARY;
                default:
                    return Analyze75Header.Datatype.ALL;
            }
        }

		/// <summary>
        /// Returns true if image is dynamic,
        /// otherwise false for static image.
        /// </summary>
        public bool IsDynamic
        {
            get { return analyzeHeader.IsDynamic; }
        }

        /// <summary>
        /// Creates and assings the basename to this class.
        /// </summary>
        /// <param name="filename">full path to filename</param>
        public Analyze75File(string filename)
        {
            if (filename.EndsWith(".img") || filename.EndsWith(".hdr"))
                this.filename = filename.Remove(filename.Length-4);
            else
                this.filename = filename;
            filetype = FileType.Analyze75;

            header = new ImageHeader();
            analyzeHeader = new  Analyze75Header();
            image = null;
        }

		/// <summary>
        /// Constructs Analyze75 file.
        /// </summary>
        /// <param name="filename">basename of Analyze file</param>
        /// <param name="image">image data</param>
        /// <param name="header">image header</param>
        public Analyze75File(string filename, Image image, ImageHeader header)
            : this(filename)
        {
            filetype = FileType.Analyze75;
            this.image = image;
            if (image is DynamicImage)
            {
                analyzeHeader.dime.dim[0] = 4;
                analyzeHeader.dime.dim[4] = (short)(image as DynamicImage).frames;
                if (image.dim.Length > IntLimits.GATES)
                {
                    analyzeHeader.dime.dim[0] = 5;
                    analyzeHeader.dime.dim[5] = (short)(image as DynamicImage).gates;
                }
                if (image.dim.Length > IntLimits.BEDS)
                {
                    analyzeHeader.dime.dim[0] = 6;
                    analyzeHeader.dime.dim[6] = (short)(image as DynamicImage).beds;
                }
            }
            else {
                analyzeHeader.dime.dim[0] = 3;
            }
            analyzeHeader.dime.dim[3] = (short)image.dimz;
            analyzeHeader.dime.datatype = (short)this.ResolveAnalyze75Datatype(header.Datatype);
            this.header = new DynamicImageHeader(header);
            this.header.Datatype = DataType.BIT16_S;
            this.header.Dim = image.dim;
            this.header.Dataunit = header.Dataunit;
        }

		/// <summary>
        /// Constructs an empty (all-zero) Analyze75 file with filename and dimensions.
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="dimx">x-dimension</param>
        /// <param name="dimy">y-dimension</param>
        /// <param name="dimz">z-dimension</param>
        public Analyze75File(string filename, int dimx, int dimy, int dimz)
        {
            if (filename.EndsWith(".img") || filename.EndsWith(".hdr"))
                base.filename = filename.Remove(filename.Length - 4);
            else
                base.filename = filename;
            this.filetype = FileType.Analyze75;
            header = new ImageHeader();
            analyzeHeader = new Analyze75Header();
            image = new Image(new int[] {dimx, dimy, dimz});
            header.Dim = image.dim;
            header.Datatype = TPClib.Image.ImageFile.DataType.BIT16_S;
        }

		/// <summary>
        /// Constructs an empty (all-zero) Analyze75 file with filename and dimensions.
        /// </summary>
        /// <param name="filename"></param>
        /// <param name="dim">dimensions {x,y,z,t} set to 1 if omitted</param>
        public Analyze75File(string filename, int[] dim)
        {
            if (filename.EndsWith(".img") || filename.EndsWith(".hdr"))
                base.filename = filename.Remove(filename.Length - 4);
            else
                base.filename = filename;
            this.filetype = FileType.Analyze75;
            header = new ImageHeader();
            analyzeHeader = new  Analyze75Header();
            image = new Image(dim);
            header.Dim = image.dim;
            header.Datatype = TPClib.Image.ImageFile.DataType.BIT16_S;
        }

		/// <summary>
        /// Help routine for ReadFile
        /// </summary>
        /// <returns></returns>
        private delegate float GetValue();

        /// <summary>
        /// Reads the .img and .hdr from 
        /// the file to the tacs structures.
        /// </summary>
        public override void ReadFile()
        {
            //read the header tacs
            ReadHeader();

            //create image
            if (header is DynamicImageHeader)
            {
                image = new DynamicImage(header.Dim);
            }
            //read voxel tacs
            GetPixelData(ref image, 0, filename + ".img", header.Dim, header.Datatype, 0, analyzeHeader.ScaleFactor);
        }

        /// <summary>
        /// Writes both the header and image tacs to the disk.
        /// </summary>
        /// <param name="image">image tacs that is written</param>
        public override void WriteFile(ref Image image)
        {
            //update bits per pixel
            switch(header.Datatype) {
                case DataType.BIT1:
                    analyzeHeader.dime.datatype = 1;
                    analyzeHeader.dime.bitpix = 1;
                    break;
                case DataType.BIT8_U:
                    analyzeHeader.dime.datatype = 2;
                    analyzeHeader.dime.bitpix = 8;
                    break;
                case DataType.BIT16_S:
                    analyzeHeader.dime.datatype = 4;
                    analyzeHeader.dime.bitpix = 16;
                    break;
                case DataType.BIT16_U:
                case DataType.SUNI2:
                case DataType.VAXI16:
                    analyzeHeader.dime.bitpix = 16;
                    break;
                case DataType.BIT32_S:
                    analyzeHeader.dime.datatype = 8;
                    analyzeHeader.dime.bitpix = 32;
                    break;
                case DataType.BIT32_U:
                case DataType.FLT32:
                case DataType.SUNI4:
                case DataType.VAXFL32:
                case DataType.VAXI32:
                    analyzeHeader.dime.bitpix = 32;
                    break;
                case DataType.ASCII:
                case DataType.BIT64_S:
                case DataType.BIT64_U:
                case DataType.FLT64:
                    analyzeHeader.dime.bitpix = 64;
                    break;
                case DataType.COLRGB:
                    analyzeHeader.dime.bitpix = 24;
                    break;
                case DataType.BIT8_S:
                default:
                    analyzeHeader.dime.bitpix = 8;
                    break;
            }
            
            // update dimension information
            // from the general header structure
            int n = image.dim.Length;
            //overwrite dimension number if it was set larger by analyzeheader
            if (analyzeHeader.dime.dim[4] > 0) n = 4;
            analyzeHeader.dime.dim[0] = (short)(n);
            analyzeHeader.dime.dim[1] = (short)image.dim[0];
            analyzeHeader.dime.dim[2] = (short)image.dim[1];
            analyzeHeader.dime.dim[3] = (short)image.dim[2];
            //use image dimension if it is available, else do no modifications
            if (image.dim.Length > 3) analyzeHeader.dime.dim[4] = (short)(image.dim[3]);
            //higher dimension definitions are not supported
            analyzeHeader.dime.dim[5] = 0;
            analyzeHeader.dime.dim[6] = 0;
            analyzeHeader.dime.dim[7] = 0;

            //update voxel size
            analyzeHeader.dime.pixdim[0] = 3.0f;
            analyzeHeader.dime.pixdim[1] = header.SizeX;
            analyzeHeader.dime.pixdim[2] = header.SizeY;
            analyzeHeader.dime.pixdim[3] = header.SizeZ;

            //update patient ID field
            analyzeHeader.hist.patient_id = Array.ConvertAll<char, byte>(header.PatientID.PadRight(10, ' ').Substring(0, 10).ToCharArray(), new Converter<char, byte>(delegate(char c) { return (byte)c; }));
            analyzeHeader.key.db_name = Array.ConvertAll<char, byte>(filename.PadRight(18, ' ').Substring(0, 18).ToCharArray(), new Converter<char, byte>(delegate(char c) { return (byte)c; }));
            
            // writes image tacs
            string fname = filename + ".img";
			using (FileStream stream = File.Open(fname, FileMode.CreateNew))
			{
				float min = image.GetMin();
				float max = image.GetMax();
				switch (analyzeHeader.DataType)
				{
					case (short)Analyze75Header.Datatype.UNSIGNED_CHAR:
						// calculates the scale factor.
						if (Math.Abs(min) <= Math.Abs(max))
							analyzeHeader.ScaleFactor = max / (byte.MaxValue);
						else
							analyzeHeader.ScaleFactor = min / (byte.MinValue);
						//set scale factor to 1 if tacs is unified (evade div-by-zero)
						if (min == max)
							analyzeHeader.ScaleFactor = 1.0f;
						WritePixelData(ref image, 0, stream, image.dim, DataType.BIT8_U, 0, analyzeHeader.ScaleFactor);
						break;
					case (short)Analyze75Header.Datatype.SIGNED_SHORT:
						// calculates the scale factor.
						if (Math.Abs(min) <= Math.Abs(max))
							analyzeHeader.ScaleFactor = max / (Int16.MaxValue);
						else
							analyzeHeader.ScaleFactor = min / (Int16.MinValue);
						//set scale factor to 1 if tacs is unified (evade div-by-zero)
						if (min == max)
							analyzeHeader.ScaleFactor = 1.0f;
						WritePixelData(ref image, 0, stream, image.dim, DataType.BIT16_S, 0, analyzeHeader.ScaleFactor);
						break;
					case (short)Analyze75Header.Datatype.SIGNED_INT:
						// calculates the scale factor.
						if (Math.Abs(min) <= Math.Abs(max))
							analyzeHeader.ScaleFactor = max / (Int32.MaxValue);
						else
							analyzeHeader.ScaleFactor = min / (Int32.MinValue);
						//set scale factor to 1 if tacs is unified (evade div-by-zero)
						if (min == max)
							analyzeHeader.ScaleFactor = 1.0f;
						WritePixelData(ref image, 0, stream, image.dim, DataType.BIT32_S, 0, analyzeHeader.ScaleFactor);
						break;
					case (short)Analyze75Header.Datatype.FLOAT:
					case (short)Analyze75Header.Datatype.COMPLEX:
						// float and double scale factors have to be 1.0f
						if (1.0f != analyzeHeader.ScaleFactor)
							throw new TPCAnalyzeFileException("Wrong scale factor");
						WritePixelData(ref image, 0, stream, image.dim, header.Datatype, 0, analyzeHeader.ScaleFactor);
						break;
					case (short)Analyze75Header.Datatype.BINARY:
						WritePixelData(ref image, 0, stream, image.dim, DataType.BIT1, 0, 1.0f);
						break;
					default:
						string msg = "Unrecognized tacs type in Analyze75: datatype=" + analyzeHeader.DataType + " bits per pixel: " + analyzeHeader.BitsPerPixel;
						throw new TPCAnalyzeFileException(msg);
				}
			}
            // writes header tacs
            analyzeHeader.Write(filename + ".hdr", header);
        }

		/// <summary>
		/// Get a stream to this image's data
		/// </summary>
		/// <param name="limits">Stream window limits</param>
		/// <returns>An image data stream</returns>
		public ImageStream GetStream(IntLimits limits)
		{
			return new ImageStream(this.filename + AnalyzeImageExt, this.header.Dim, limits, FileMode.OpenOrCreate, 0, this.header.Datatype);
		}

		/// <summary>
		/// Get a stream to this image's data
		/// </summary>
		/// <returns>An image data stream</returns>
		public ImageStream GetStream()
		{
			return GetStream(this.header.Dim);
		}

		/// <summary>
        /// Checks file format. Only SPM format is recognized.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a Analyze7.5 file (SPM format)</returns>
        /// <exception cref="TPCAnalyzeFileException">if there was a read problem</exception>
        public static bool CheckFormat(string filename)
        {
            String ext = Path.GetExtension(filename);
            String basename = Path.GetFileNameWithoutExtension(filename);
            String path = Path.GetDirectoryName(filename);
            //if extension exists it must be valid extension
            if (ext.Length > 0)
            {
                if (!ext.EndsWith(".hdr") && !ext.EndsWith(".img")) return false;
            }
            //header file <basename>.hdr must exist
            if (!File.Exists(path + Path.DirectorySeparatorChar + basename + ".hdr") || 
                !File.Exists(path + Path.DirectorySeparatorChar + basename + ".img"))
            {
                return false;
            }
            //try to read Analyze75 header tacs
            try
            {
                Analyze75Header h = new Analyze75Header();
                ImageHeader hdr = new ImageHeader();
                h.Read(path + Path.DirectorySeparatorChar + basename + ".hdr", ref hdr);
            }
            catch(FileNotFoundException) {
                throw new TPCAnalyzeFileException("Header file was not found");
            }
            catch (IOException e)
            {
                throw new TPCAnalyzeFileException("Read error while determining Analyze75 file format:" + e);
            }
            //file format is not recognized if TPCAnalyzeFileException is thrown
            catch (TPCAnalyzeFileException)
            {
                return false;
            }
            catch (Exception)
            {
                throw new TPCAnalyzeFileException("Unknown error while determining Analyze75 file format.");
            }
            return true;
        }

		/// <summary>
        /// Writes single frame into file. If frame number is larger than 
        /// current number of frames in file, then file size is grown.
        /// </summary>
        /// <remarks>The header information including scaling factor is not updated by this method.</remarks>
        /// <param name="image">image tacs that is written</param>
        /// <param name="frame_No">frame number that is written [1..no frames+1]</param>
        public override void WriteFrame(ref Image image, int frame_No)
        {
            if (frame_No < 1 || frame_No > analyzeHeader.Frames + 1)
                throw new TPCInvalidArgumentsException("Frame number "+frame_No+" out of bounds [1..no frames +1]");
            
            string fname = filename + ".img";
			using (FileStream stream = File.Open(fname, FileMode.OpenOrCreate))
			{
				using (
				BinaryWriter writer = new BinaryWriter(stream))
				{

					// This image can be dynamic or non-dynamic.
					Image img = image;

					//seek to right position in file before writing
					int voxels_in_frame = image.dimx * image.dimy * image.dimz;
					switch (analyzeHeader.DataType)
					{
						case (short)Analyze75Header.Datatype.SIGNED_SHORT:
							writer.BaseStream.Seek(2 * voxels_in_frame * (frame_No - 1), SeekOrigin.Begin);
							break;
						case (short)Analyze75Header.Datatype.FLOAT:
						case (short)Analyze75Header.Datatype.COMPLEX:
							writer.BaseStream.Seek(4 * voxels_in_frame * (frame_No - 1), SeekOrigin.Begin);
							break;
						default:
							throw new TPCAnalyzeFileException("Datatype is unsupported, type: " + analyzeHeader.DataType + " bpp: " + analyzeHeader.BitsPerPixel);
					}

					if (analyzeHeader.DataType == (short)Analyze75Header.Datatype.SIGNED_SHORT)
					{
						for (int i = 0; i < img.DataLength; i++)
						{
							if (analyzeHeader.IsLittleEndian)
							{
								Decimal d = (decimal)(img.GetArray()[i] / analyzeHeader.ScaleFactor);
								Int16 integer = (Int16)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
								writer.Write(integer);
							}
							else
							{
								Decimal d = (decimal)(img.GetArray()[i] / analyzeHeader.ScaleFactor);
								Int16 integer = (Int16)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
								byte[] b = BitConverter.GetBytes(integer);
								writer.Write(b[1]);
								writer.Write(b[0]);
							}
						}
					}
					else if (analyzeHeader.DataType == (short)Analyze75Header.Datatype.FLOAT ||
						analyzeHeader.DataType == (short)Analyze75Header.Datatype.COMPLEX)
					{
						// float and double scale factors have to be 1.0f
						if (1.0f != analyzeHeader.ScaleFactor)
							throw new TPCAnalyzeFileException("Wrong scale factor");

						for (int i = 0; i < img.DataLength; i++)
						{
							writer.Write(img.GetValue(i));
						}
					}
					else
					{
						string msg = "Unrecognized tacs type in Analyze75: " + analyzeHeader.DataType + " bits per pixel: " + analyzeHeader.BitsPerPixel;
						throw new TPCAnalyzeFileException(msg);
					}
				}
			}
        }

		/// <summary>
        /// Reads only header tacs from file. 
        /// </summary>
        /// <returns>Header information in file</returns>
        public override ImageHeader ReadHeader()
        {
            // reads the header tacs
            analyzeHeader = new Analyze75Header();
            analyzeHeader.Read(filename + ".hdr", ref header);
            return header;
        }

        /// <summary>
        /// Writes pixel tacs into image file. Note that endianness in analyseHeader variable 
        /// affects writing.
        /// </summary>
        /// <param name="data">source array</param>
        /// <param name="offset">offset where reading begins</param>
        /// <param name="stream">output stream that is expected to be open</param>
        /// <param name="dim">output tacs dimensions</param>
        /// <param name="datatype">tacs type to be written</param>
        /// <param name="position">file position where writing is started</param>
        /// <param name="scale">scale factor</param>
        public void WritePixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
        {
            stream.Seek(position, SeekOrigin.Begin);
			using (BinaryWriter writer = new BinaryWriter(stream))
			{
				int dataLength = dim.GetProduct();
				switch (datatype)
				{
					case ImageFile.DataType.BIT1:
						byte current_byte = (byte)0;
						int bit_position = 0;
						for (int i = 0; i < dataLength; i++)
						{
							if (data[offset + i] != 0)
							{
								current_byte = (byte)(current_byte | (1 << bit_position++));
							}
							else
							{
								current_byte = (byte)(current_byte & ~(1 << bit_position++));
							}
							if (bit_position % 8 == 0)
							{
								writer.Write(current_byte);
								bit_position = 0;
							}
						}
						if (bit_position % 8 != 0)
						{
							while (bit_position < 8)
								current_byte = (byte)(current_byte & ~(1 << bit_position++));
							writer.Write(current_byte);
						}
						break;
					case ImageFile.DataType.BIT8_U:
						for (int i = 0; i < dataLength; i++)
						{
							Decimal d = (decimal)(data[offset + i] / scale);
							byte integer = (byte)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
							writer.Write(integer);
						}
						break;
					case ImageFile.DataType.BIT16_S:
						for (int i = 0; i < dataLength; i++)
						{
							if (analyzeHeader.IsLittleEndian)
							{
								Decimal d = (decimal)(data[offset + i] / scale);
								Int16 integer = (Int16)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
								writer.Write(integer);
							}
							else
							{
								Decimal d = (decimal)(data[offset + i] / scale);
								Int16 integer = (Int16)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
								byte[] b = BitConverter.GetBytes(integer);
								writer.Write(b[1]);
								writer.Write(b[0]);
							}
						}
						break;
					case ImageFile.DataType.BIT32_S:
						for (int i = 0; i < dataLength; i++)
						{
							if (analyzeHeader.IsLittleEndian)
							{
								Decimal d = (decimal)(data[offset + i] / scale);
								Int32 integer = (Int32)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
								writer.Write(integer);
							}
							else
							{
								Decimal d = (decimal)(data[offset + i] / scale);
								Int32 integer = (Int32)(Decimal.Round(d, 0, MidpointRounding.AwayFromZero));
								byte[] b = BitConverter.GetBytes(integer);
								writer.Write(b[3]);
								writer.Write(b[2]);
								writer.Write(b[1]);
								writer.Write(b[0]);
							}
						}
						break;
					case ImageFile.DataType.FLT32:
						{
							// float and double scale factors have to be 1.0f
							if (1.0f != scale)
								throw new TPCAnalyzeFileException("Wrong scale factor");
							for (int i = 0; i < dataLength; i++)
							{
								writer.Write(data[offset + i]);
							}
							break;
						}
					default:
						throw new TPCAnalyzeFileException("Dataype " + datatype + " is not supported by Analyze7.5 format.");
				}
			}
        }

		/// <summary>
        /// Gets pixel tacs from image. It is strongly suggested that the file header is read 
        /// first in order to fill proper header fields that might be needed for reading.
        /// </summary>
        /// <param name="data">target array</param>
        /// <param name="offset">offset where copying begins to write tacs</param>
        /// <param name="stream">input stream that is expected to be open and stays open</param>
        /// <param name="dim">input tacs dimensions</param>
        /// <param name="datatype">tacs type to be read</param>
        /// <param name="position">file position where reading is started</param>
        /// <param name="scale">scale factor</param>
        public override void GetPixelData(ref Image data, int offset, Stream stream, IntLimits dim, DataType datatype, long position, float scale)
        {
            //initialization for event sending
            IOProcessEventHandler pevent = null;
            int voxels_in_plane = 0;
            // opens binary reader
			using (BinaryReader br = new BinaryReader(stream))
			{

				// collect information from header.
				int bpp = analyzeHeader.BitsPerPixel;
				bool isDynamic = analyzeHeader.IsDynamic;
				bool isLittleEndian = analyzeHeader.IsLittleEndian;
				//bit position is used only for binary data reading
				byte bit_position = 0;
				//current byte is used only for binary data reading
				byte current_byte = 0;

				GetValue getValue;

				switch (datatype)
				{
					case DataType.BIT1:
						getValue = delegate()
						{
							//read next byte and reset counter
							if (bit_position % 8 == 0)
							{
								current_byte = br.ReadByte();
								bit_position = 0;
							}
							return ((current_byte & (1 << bit_position++)) != 0 ? 1 : 0);
						};
						break;
					case DataType.BIT8_U:
						getValue = delegate()
						{
							byte ubute = br.ReadByte();
							return ubute;
						};
						break;
					case DataType.BIT16_S:
						if (isLittleEndian)
							getValue = delegate()
							{
								Int16 integer = br.ReadInt16();
								return integer * scale;
							};
						else
							getValue = delegate()
							{
								byte a = br.ReadByte();
								byte b = br.ReadByte();
								byte[] table = new byte[2] { b, a };
								Int16 integer = BitConverter.ToInt16(table, 0);
								return integer * scale;
							};
						break;
					case DataType.BIT16_U:

						if (isLittleEndian)
							getValue = delegate()
							{
								UInt16 integer = br.ReadUInt16();
								return integer * scale;
							};
						else
							getValue = delegate()
							{
								byte a = br.ReadByte();
								byte b = br.ReadByte();
								byte[] table = new byte[2] { b, a };
								UInt16 integer = BitConverter.ToUInt16(table, 0);
								return integer * scale;
							};
						break;
					case DataType.BIT32_S:
						if (isLittleEndian)
							getValue = delegate()
							{
								Int32 integer = br.ReadInt32();
								return integer * scale;
							};
						else
							getValue = delegate()
							{
								byte[] a = br.ReadBytes(4);
								byte[] table = new byte[4] { a[3], a[2], a[1], a[0] };
								Int32 integer = BitConverter.ToInt32(table, 0);
								return integer * scale;
							};
						break;
					case DataType.FLT32:
						if (isLittleEndian)
							getValue = delegate()
							{
								return br.ReadSingle();
							};
						else
							getValue = delegate()
							{
								byte[] bytes = br.ReadBytes(4);
								this.swapBytes(bytes, DataType.FLT32);
								return BitConverter.ToSingle(bytes, 0);
							};
						break;
					default:
						throw new TPCAnalyzeFileException("Data is unsupported, type: " + datatype + " bpp: " + bpp);
				}

				// start reading tacs.
				this.image = isDynamic ? new DynamicImage(header.Dim) : new Image(header.Dim);
				voxels_in_plane = header.DimX * header.DimY;
				for (int i = 0; i < this.image.DataLength; i++)
				{
					data[i] = getValue();
					//send event after each plane
					if (i > 0 && i % voxels_in_plane == 0)
					{
						pevent = IOProgress;
						if (pevent != null)
							pevent.Invoke(this, new IOProgressEventArgs(100.0f * (float)(i / (float)image.DataLength), System.Reflection.MethodBase.GetCurrentMethod()));
					}
				}

			}
			pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod()));
        }

        /// <summary>
        /// Reads subregion from file.
        /// </summary>
        /// <param name="region">subregion that is read from file</param>
        public override void ReadSubImage(IntLimits region)
        {
            ReadFile();
            header.Dim = region;
            image = image.GetSubImage(region);
        }
    }*/
	#endregion
}
