/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Base class for all headers derived from Analyze file format
	/// </summary>
	public abstract class AnalyzeHeader : ImageHeader
	{
		#region Header construction

		/// <summary>
		/// Header field types
		/// </summary>
		protected enum HeaderFieldType : byte
		{
			/// <summary>
			/// Single byte
			/// </summary>
			BYTE,
			/// <summary>
			/// Single character (unsigned byte)
			/// </summary>
			CHAR,
			/// <summary>
			/// 16 bit integer
			/// </summary>
			SHORT,
			/// <summary>
			/// 32 bit integer
			/// </summary>
			INT,
			/// <summary>
			/// 64 bit integer
			/// </summary>
			LONG,
			/// <summary>
			/// Single precision floating point
			/// </summary>
			FLOAT,
			/// <summary>
			/// Double precision floating point
			/// </summary>
			DOUBLE,
			/// <summary>
			/// Unicode string
			/// </summary>
			STRING
		}

		/// <summary>
		/// Header field description
		/// </summary>
		protected class HeaderFieldDescription
		{
			/// <summary>
			/// Name of the field
			/// </summary>
			public string name;

			/// <summary>
			/// Field type
			/// </summary>
			public HeaderFieldType type;

			/// <summary>
			/// Field lenght in elements
			/// </summary>
			public int length;

			/// <summary>
			/// Constructor
			/// </summary>
			/// <param name="n">Name</param>
			/// <param name="t">Type</param>
			/// <param name="l">Length</param>
			public HeaderFieldDescription(string n, HeaderFieldType t, int l)
			{
				name = n;
				type = t;
				length = l;
			}
		}

		/// <summary>
		/// Header description (list of header field descriptions)
		/// </summary>
		protected class HeaderDescription : System.Collections.Generic.List<HeaderFieldDescription>
		{
			/// <summary>
			/// Create new header description
			/// </summary>
			/// <param name="names">Header field names</param>
			/// <param name="types">Header field types</param>
			/// <param name="lengths">Header field lengths</param>
			public HeaderDescription(string[] names, HeaderFieldType[] types, int[] lengths)
			{
				if (names.Length != lengths.Length || names.Length != types.Length) throw new TPCException("Description array lengths don't match.");
				for (int i = 0; i < names.Length; i++)
				{
					Add(new HeaderFieldDescription(names[i], types[i], lengths[i]));
				}
			}
		}

		/// <summary>
		/// Create, read and write header byte data
		/// </summary>
		protected static class HeaderConstructor
		{
			/// <summary>
			/// Create an empty header
			/// </summary>
			/// <param name="hd">Header description</param>
			/// <returns>Header with all fields initialized with default values</returns>
			public static ImageHeader MakeHeader(HeaderDescription hd)
			{
				ImageHeader header = new ImageHeader();
				foreach (HeaderFieldDescription hfd in hd)
				{
					IConvertible[] items = MakeHeaderItem(hfd);
					if (items.Length == 1) header[hfd.name] = items[0];
					else if (hfd.type == HeaderFieldType.STRING)
					{
						header[hfd.name] = String.Empty;
					}
					else header[hfd.name] = items;
				}
				return header;
			}

			private static IConvertible[] MakeHeaderItem(HeaderFieldDescription hfd)
			{
				IConvertible[] item = new IConvertible[hfd.length];
				for (int i = 0; i < hfd.length; i++)
				{
					item[i] = MakeValue(hfd.type);
				}
				return item;
			}

			private static IConvertible MakeValue(HeaderFieldType type)
			{
				IConvertible val;
				switch (type)
				{
					case HeaderFieldType.STRING:
					case HeaderFieldType.BYTE:
						val = new Byte();
						break;
					case HeaderFieldType.CHAR:
						val = new Char();
						break;
					case HeaderFieldType.SHORT:
						val = new Int16();
						break;
					case HeaderFieldType.INT:
						val = new Int32();
						break;
					case HeaderFieldType.LONG:
						val = new Int64();
						break;
					case HeaderFieldType.FLOAT:
						val = new Single();
						break;
					case HeaderFieldType.DOUBLE:
						val = new Double();
						break;
					default:
						val = null;
						break;
				}
				return val;
			}

			private static string MakeString(byte[] bytes)
			{
				string s = "";
				int i = 0;
				while (i < bytes.Length && bytes[i] != 0)
				{
					s += (char)bytes[i];
					i++;
				}
				return s;
			}

			/// <summary>
			/// Write header bytes
			/// </summary>
			/// <param name="hd">Header description</param>
			/// <param name="bw">Output writer</param>
			/// <param name="header">Header to write</param>
			/// <returns>Number of bytes written</returns>
			public static long WriteHeader(HeaderDescription hd, BinaryWriter bw, ImageHeader header)
			{
				long bytesWritten = -bw.BaseStream.Position;
				IConvertible[] values;

				foreach (HeaderFieldDescription hfd in hd)
				{
					values = new IConvertible[hfd.length];

					if (values.Length == 1)
					{
						values[0] = (IConvertible)(header[hfd.name]);
					}

					// Convert strings to byte arrays
					else if (hfd.type == HeaderFieldType.STRING)
					{
						String s = ((IConvertible)(header[hfd.name])).ToString();
						char[] chars = s.ToCharArray();
						for (int i = 0; i < values.Length; i++)
						{
							if (i < chars.Length) values[i] = chars[i];
							else values[i] = 0;
						}
					}

					else
					{
						IConvertible[] headerValues = (IConvertible[])header[hfd.name];
						for (int i = 0; i < values.Length; i++)
						{
							if (i < headerValues.Length) values[i] = headerValues[i];
							else values[i] = 0;
						}
					}

					WriteHeaderItem(bw, hfd, values);
				}

				return (bytesWritten += bw.BaseStream.Position);
			}

			private static void WriteHeaderItem(BinaryWriter bw, HeaderFieldDescription hfd, IConvertible[] values)
			{
				foreach (IConvertible val in values)
				{
					WriteValue(bw, hfd.type, val);
				}
			}

			private static void WriteValue(BinaryWriter bw, HeaderFieldType type, IConvertible val)
			{
				IFormatProvider format = System.Globalization.NumberFormatInfo.InvariantInfo;

				switch (type)
				{
					case HeaderFieldType.STRING:
					case HeaderFieldType.BYTE:
						bw.Write(val.ToByte(format));
						break;
					case HeaderFieldType.CHAR:
						bw.Write(val.ToByte(format));
						break;
					case HeaderFieldType.SHORT:
						bw.Write(val.ToInt16(format));
						break;
					case HeaderFieldType.INT:
						bw.Write(val.ToInt32(format));
						break;
					case HeaderFieldType.LONG:
						bw.Write(val.ToInt64(format));
						break;
					case HeaderFieldType.FLOAT:
						bw.Write(val.ToSingle(format));
						break;
					case HeaderFieldType.DOUBLE:
						bw.Write(val.ToDouble(format));
						break;
					default:
						throw new TPCException("Undefined field type");
				}
			}

			/// <summary>
			/// Read header
			/// </summary>
			/// <param name="hd">Header description</param>
			/// <param name="br">Input reader</param>
			/// <param name="header">Header to fill with the data read</param>
			/// <returns>Number of bytes read</returns>
			public static long ReadHeader(HeaderDescription hd, BinaryReader br, ImageHeader header)
			{
				long bytesRead = -br.BaseStream.Position;
				foreach (HeaderFieldDescription hfd in hd)
				{
					IConvertible[] items = ReadHeaderItem(br, hfd);
					if (items.Length == 1) header[hfd.name] = items[0];
					else if (hfd.type == HeaderFieldType.STRING)
					{
						byte[] str = Array.ConvertAll<IConvertible, byte>(items, Convert.ToByte);
						header[hfd.name] = MakeString(str);
					}
					else header[hfd.name] = items;
				}
				return (bytesRead += br.BaseStream.Position);
			}

			private static IConvertible[] ReadHeaderItem(BinaryReader br, HeaderFieldDescription hfd)
			{
				IConvertible[] item = new IConvertible[hfd.length];
				for (int i = 0; i < hfd.length; i++)
				{
					item[i] = ReadValue(br, hfd.type);
				}
				return item;
			}

			private static IConvertible ReadValue(BinaryReader br, HeaderFieldType type)
			{
				IConvertible val;
				switch (type)
				{
					case HeaderFieldType.STRING:
					case HeaderFieldType.BYTE:
					case HeaderFieldType.CHAR:
						val = br.ReadByte();
						break;
					case HeaderFieldType.SHORT:
						val = br.ReadInt16();
						break;
					case HeaderFieldType.INT:
						val = br.ReadInt32();
						break;
					case HeaderFieldType.LONG:
						val = br.ReadUInt64();
						break;
					case HeaderFieldType.FLOAT:
						val = br.ReadSingle();
						break;
					case HeaderFieldType.DOUBLE:
						val = br.ReadDouble();
						break;
					default:
						val = null;
						break;
				}
				return val;
			}
		}

		#endregion

		/// <summary>
		/// Main header length in bytes
		/// </summary>
		protected const int HEADERSIZE = 348;

		/// <summary>
		/// Description of the byte representation of this header
		/// </summary>
		protected abstract HeaderDescription ByteDescription { get; }

		/// <summary>
		/// Size of this header in bytes
		/// </summary>
		public virtual int HeaderSize
		{
			get
			{
				return HEADERSIZE;
			}
		}

		/// <summary>
		/// BigEndian data flag
		/// </summary>
		public bool bigEndian = false;

		/// <summary>
		/// Image data datatype
		/// </summary>
		public override ImageFile.DataType Datatype
		{
			get
			{
				return AnalyzeFile.ResolveDataType((AnalyzeFile.AnalyzeDataType)this[@"datatype"]);
			}
			set
			{
				// Set both datatype and bits per pixel fields
				this.SetValue(@"datatype", AnalyzeFile.ResolveAnalyzeDataType(value));
				this.SetValue(@"bitpix", AnalyzeFile.BitsPerPixel(value));
			}
		}

		/// <summary>
		/// Image description
		/// </summary>
		public override string Description
		{
			get
			{
				return (String)(this[@"descrip"]);
			}
			set
			{
				this[@"descrip"] = value;
			}
		}

		/// <summary>
		/// Image dimensions
		/// </summary>
		public override IntLimits Dim
		{
			get
			{
				IConvertible[] dims = (IConvertible[])this[@"dim"];
				short[] dimField = Array.ConvertAll<IConvertible, short>(dims, Convert.ToInt16);
				int[] dimensions = new int[dimField[0]];
				Array.Copy(dimField, 1, dimensions, 0, dimensions.Length);
				return new IntLimits(dimensions);
			}
			set
			{
				IConvertible[] dimField = new IConvertible[8];
				dimField[0] = (short)(value.Length);
				for (int i = 1; i < 8; i++)
				{
					dimField[i] = (short)(value.GetDimension(i - 1));
				}
				this[@"dim"] = dimField;
			}
		}

		/// <summary>
		/// Voxel dimensions
		/// </summary>
		public override Voxel Siz
		{
			get
			{
				IConvertible[] pdims = (IConvertible[])this[@"pixdim"];
				float[] pixdims = Array.ConvertAll<IConvertible, float>(pdims, Convert.ToSingle);
				return new Voxel(pixdims[1], pixdims[2], pixdims[3]);
			}
			set
			{
				// For now, support pixdim values for 3 first dimensions
				IConvertible[] pixdims = new IConvertible[8];

				pixdims[0] = 1;
				pixdims[1] = value.SizeX;
				pixdims[2] = value.SizeY;
				pixdims[3] = value.SizeZ;

				for (int i = 4; i < pixdims.Length; i++)
				{
					pixdims[i] = 1.0f;
				}

				this[@"pixdim"] = pixdims;
			}
		}

		/// <summary>
		/// Default constructor. Creates a static image.
		/// </summary>
		protected AnalyzeHeader() : base() { Init(); }

		protected AnalyzeHeader(ImageHeader hdr)
			: this()
		{
			this.PatientName = hdr.PatientName;
			this.PatientID = hdr.PatientID;
			this.Dim = new IntLimits(hdr.Dim);
			this.Siz = new Voxel(hdr.Siz);
			this.Description = hdr.Description;
			this.Datatype = hdr.Datatype;
			this.Dataunit = hdr.Dataunit;
			this.Modality = hdr.Modality;
			this.DoseStartTime = hdr.DoseStartTime;
			this.InjectedDose = hdr.InjectedDose;
			this.Isotope = hdr.Isotope;
			this.Radiopharma = hdr.Radiopharma;
			this.IsDynamic = hdr.IsDynamic;
			this.FrameDurations = hdr.FrameDurations;
			this.FrameStartTimes = hdr.FrameStartTimes;

			if (hdr is AnalyzeHeader)
			{
				this.Slope = (hdr as AnalyzeHeader).Slope;
				this.Intercept = (hdr as AnalyzeHeader).Intercept;
			}
		}

		private void Init()
		{
			this.AddRange(HeaderConstructor.MakeHeader(ByteDescription));

			// 16-b integer data as default
			this.Datatype = ImageFile.DataType.BIT16_S;

			// Fill defaults; everything not listed is set to 0
			// Must be 348
			this[@"sizeof_hdr"] = HEADERSIZE;
			// These are fields in the Analyze header_key substruct, unused in Nifti.
			// Some software expect these values.
			this[@"extents"] = 16384;
			this[@"regular"] = 'r';
		}

		/// <summary>
		/// Read header from a stream
		/// </summary>
		/// <param name="s">Input stream</param>
		/// <returns>True, if header was succesfully read</returns>
		public virtual bool Read(Stream s)
		{
			// Clear old contents
			this.Clear();

			bool success = false;

			Endianness endian = NiftiFile.CheckEndian(s);
			this.bigEndian = (endian == Endianness.BigEndian);

			long bytesRead = 0;
			using (BinaryReader br = EndianReader.CreateReader(s, endian))
			{
				bytesRead = HeaderConstructor.ReadHeader(ByteDescription, br, this);
			}

			// Check header validity
			if (bytesRead == HEADERSIZE && (int)this[@"sizeof_hdr"] == HEADERSIZE) success = true;

			return success;
		}

		/// <summary>
		/// Write header to a stream
		/// </summary>
		/// <param name="s">Output stream</param>
		/// <returns>True, if header was succesfully written</returns>
		public virtual bool Write(Stream s)
		{
			bool success = false;

			using (BinaryWriter bw = new BinaryWriter(s))
			{
				// Write the main header
				long bytesWritten = HeaderConstructor.WriteHeader(ByteDescription, bw, this);

				if (bytesWritten == HEADERSIZE) success = true;
			}
			return success;
		}

		/// <summary>
		/// Image data scaling, slope coefficient
		/// </summary>
		public virtual float Slope
		{
			get
			{
				return (float)(this[@"scl_slope"]);
			}
			set
			{
				this[@"scl_slope"] = (IConvertible)value;
			}
		}

		/// <summary>
		/// Image data scaling, intercept
		/// </summary>
		public virtual float Intercept
		{
			get
			{
				return (float)(this[@"scl_inter"]);
			}
			set
			{
				this[@"scl_inter"] = (IConvertible)value;
			}
		}
	}
}
